﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_STREAM_SOUND_FILE_READER_H_
#define NW_SND_STREAM_SOUND_FILE_READER_H_

#include <nw/snd/snd_StreamSoundFile.h>
#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_Config.h>

namespace nw {
namespace snd {
namespace internal {

class StreamSoundFileReader
{
public:
    struct TrackInfo
    {
        u8 volume;
        u8 pan;
        u8 span;
        u8 flags;   // enum StreamSoundFile::SurroundMode が入る
        u8 channelCount;
        u8 globalChannelIndex[ WAVE_CHANNEL_MAX ];

        TrackInfo() : volume(0), pan(0), channelCount(0)
        {
            std::memset(globalChannelIndex, 0, sizeof(u8)*WAVE_CHANNEL_MAX);
        }
    };

    StreamSoundFileReader();
    void Initialize( const void* streamSoundFile );
    void Finalize();
    bool IsAvailable() const { return m_pHeader != NULL; }
    bool IsTrackInfoAvailable() const;
    bool IsOriginalLoopAvailable() const;

    bool IsValidFileHeader( const void* streamSoundFile ) const;

    bool ReadStreamSoundInfo( StreamSoundFile::StreamSoundInfo* strmInfo ) const;
    bool ReadStreamTrackInfo( TrackInfo* trackInfo, int trackIndex ) const;
    bool ReadDspAdpcmChannelInfo(
            DspAdpcmParam* param,
            DspAdpcmLoopParam* loopParam,
            int channelIndex ) const;

    NW_INLINE u32 GetChannelCount() const
    {
        NW_NULL_ASSERT( m_pInfoBlockBody );
        return m_pInfoBlockBody->GetChannelInfoTable()->GetChannelCount();
    }

    NW_INLINE u32 GetTrackCount() const
    {
        NW_NULL_ASSERT( m_pInfoBlockBody );
        return m_pInfoBlockBody->GetTrackInfoTable()->GetTrackCount();
    }

    NW_INLINE u32 GetSeekBlockOffset() const
    {
        if ( IsAvailable() && m_pHeader->HasSeekBlock() )
        {
            return m_pHeader->GetSeekBlockOffset();
        }
        return 0;
    }

    NW_INLINE u32 GetSampleDataOffset() const
    {
        if ( IsAvailable() )
        {
            u32 result =  m_pHeader->GetDataBlockOffset() +
                sizeof( ut::BinaryBlockHeader ) +
                m_pInfoBlockBody->GetStreamSoundInfo()->sampleDataOffset.offset;
            return result;
        }
        return 0;
    }

    // ファイル先頭から REGN ブロック内のデータの開始位置までのオフセットを返す
    // ([ブロックヘッダー] および [ブロックヘッダーのうしろからデータ開始位置] も考慮)
    NW_INLINE u32 GetRegionDataOffset() const
    {
        NW_NULL_ASSERT( m_pInfoBlockBody );
        if ( IsAvailable() && m_pHeader->HasRegionBlock() )
        {
            u32 result = m_pHeader->GetRegionBlockOffset() +
                sizeof( ut::BinaryBlockHeader ) +
                m_pInfoBlockBody->GetStreamSoundInfo()->regionDataOffset.offset;
            return result;
        }
        return 0;
    }

    // リージョン情報 1 つあたりのバイト数を返す
    NW_INLINE u32 GetRegionInfoBytes() const
    {
        NW_NULL_ASSERT( m_pInfoBlockBody );
        u16 result = m_pInfoBlockBody->GetStreamSoundInfo()->regionInfoBytes;
        return result;
    }
private:
    const StreamSoundFile::FileHeader*      m_pHeader;
    const StreamSoundFile::InfoBlockBody*   m_pInfoBlockBody;
};

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_STREAM_SOUND_FILE_READER_H_ */

