﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_STREAM_SOUND_H_
#define NW_SND_STREAM_SOUND_H_

#include <nw/snd/snd_Config.h>
#include <nw/snd/snd_SoundInstanceManager.h>
#include <nw/snd/snd_BasicSound.h>
#include <nw/snd/snd_StreamSoundPlayer.h>
#include <nw/snd/snd_StreamBufferPool.h>
#include <nw/snd/snd_Debug.h>

namespace nw {
namespace snd {

class StreamSoundHandle;

namespace internal {

class StreamSound;
typedef SoundInstanceManager<StreamSound> StreamSoundInstanceManager;

class StreamSound : public BasicSound
{
    friend class nw::snd::StreamSoundHandle;

public:
    NW_UT_RUNTIME_TYPEINFO(BasicSound)

    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */
public:
    explicit StreamSound( StreamSoundInstanceManager& manager );

    void SetCacheBuffer(void* cache, size_t cacheSize)
    {
        m_pCacheBuffer = cache;
        m_CacheSize = cacheSize;
    }
    bool IsCacheAvailable() const
    {
        if (m_pCacheBuffer != NULL && m_CacheSize != 0)
        {
            return true;
        }
        return false;
    }
    void* GetCacheBuffer() { return m_pCacheBuffer; }
    size_t GetCacheSize() { return m_CacheSize; }

    void Setup(const driver::StreamSoundPlayer::SetupArg& arg);
    void Prepare(const driver::StreamSoundPlayer::PrepareBaseArg& arg);
    void PreparePrefetch(const void* strmPrefetchFile, const driver::StreamSoundPlayer::PrepareBaseArg& arg);

    virtual void Initialize();
    virtual void Finalize();
    virtual bool IsPrepared() const;
    bool IsSuspendByLoadingDelay() const;

    // トラックパラメータ
    void SetTrackVolume( u32 trackBitFlag, f32 volume, int frames = 0 );
    void SetTrackInitialVolume( u32 trackBitFlag, u32 volume );
    void SetTrackOutputLine( u32 trackBitFlag, u32 lineFlag );
    void ResetTrackOutputLine( u32 trackBitFlag );

    void SetTrackChannelMixParameter( u32 trackBitFlag, u32 srcChNo, const MixParameter& mixParam );
    void SetTrackMainOutVolume( u32 trackBitFlag, f32 volume );
    void SetTrackPan( u32 trackBitFlag, f32 pan );
    void SetTrackSurroundPan( u32 trackBitFlag, f32 span );
    void SetTrackMainSend( u32 trackBitFlag, f32 send );
    void SetTrackFxSend( u32 trackBitFlag, AuxBus bus, f32 send );

    void SetTrackDrcChannelMixParameter( u32 drcIndex, u32 trackBitFlag, u32 srcChNo, const MixParameter& mixParam );
    void SetTrackDrcOutVolume( u32 drcIndex, u32 trackBitFlag, f32 volume );
    void SetTrackDrcPan( u32 drcIndex, u32 trackBitFlag, f32 pan );
    void SetTrackDrcSurroundPan( u32 drcIndex, u32 trackBitFlag, f32 span );
    void SetTrackDrcMainSend( u32 drcIndex, u32 trackBitFlag, f32 send );
    void SetTrackDrcFxSend( u32 drcIndex, u32 trackBitFlag, AuxBus bus, f32 send );

    // 情報取得
    bool ReadStreamDataInfo( StreamDataInfo* info ) const;
    long GetPlayLoopCount() const;
    long GetPlaySamplePosition(bool isOriginalSamplePosition) const;
    NW_INLINE u32 GetAvailableTrackBitFlag( u32 srcChNo ) const
    {
        NW_ASSERT_MAXLT( srcChNo, WAVE_CHANNEL_MAX );
        return m_AvailableTrackBitFlag[ srcChNo ];
    }
    f32 GetFilledBufferPercentage() const;

    // デバッグ関数
    DebugSoundType GetSoundType() const { return DEBUG_SOUND_TYPE_STRMSOUND; }

    void SetLoaderManager(driver::StreamSoundLoaderManager& manager)
    {
        m_PlayerInstance.SetLoaderManager(&manager);
    }

protected:
    /* override */ bool IsAttachedTempSpecialHandle();
    /* override */ void DetachTempSpecialHandle();

    /* override */ void UpdateMoveValue();

    /* override */ driver::BasicSoundPlayer* GetBasicSoundPlayerHandle()
    {
        return &m_PlayerInstance;
    }

    virtual void OnUpdatePlayerPriority();

private:
    /* override */ void OnUpdateParam();
    StreamSoundHandle* m_pTempSpecialHandle;
    StreamSoundInstanceManager& m_Manager;

    MoveValue<f32, int> m_TrackVolume[ STRM_TRACK_NUM ];

    u16 m_AllocTrackFlag;
    bool m_InitializeFlag;
    u8 padding;

    u32 m_AvailableTrackBitFlag[ WAVE_CHANNEL_MAX ];

    void* m_pCacheBuffer;
    size_t m_CacheSize;
    driver::StreamSoundPlayer m_PlayerInstance;
};

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_STREAM_SOUND_H_ */

