﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_SoundSystem.h
 *
 * @file snd_SoundSystem.h
 */

#ifndef NW_SND_SOUND_SYSTEM_H_
#define NW_SND_SOUND_SYSTEM_H_

#include <nw/config.h>
#include <nw/snd/snd_Config.h>
#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_HardwareManager.h>
#include <nw/snd/snd_MultiVoiceManager.h>
#include <nw/snd/snd_SoundThread.h>
#include <nw/snd/snd_MmlParser.h>
#include <nw/snd/snd_TaskThread.h>
#include <nw/snd/snd_ThreadStack.h>
#include <nw/snd/snd_RemoteSpeaker.h>
#include <nw/snd/snd_RemoteSpeakerManager.h>
#include <nw/snd/snd_Util.h>

namespace nw {
namespace snd {

namespace internal {
    class NwVoiceRenderer;
}
//---------------------------------------------------------------------------
//! @brief  サウンドシステム全体に関わる設定を行います。
//!
//! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
class SoundSystem
{
public:

    //---------------------------------------------------------------------------
    //! @brief  サウンドシステムの初期化に使用する構造体です。
    //!
    //! @see SoundSystem::Initialize
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    struct SoundSystemParam
    {
        //---------------------------------------------------------------------------
        //! @brief    サウンドスレッドのプライオリティです。初期値は 4 です。
        //---------------------------------------------------------------------------
        s32 soundThreadPriority;

        //---------------------------------------------------------------------------
        //! @brief    サウンドスレッドのスタックサイズです。8 の倍数で指定してください。
        //!           初期値は 16KB です。
        //---------------------------------------------------------------------------
        size_t soundThreadStackSize;

        //---------------------------------------------------------------------------
        //! @briefprivate
        //!
        //! @brief    サウンドスレッドのコマンドバッファサイズです。4 の倍数で指定してください。
        //!           初期値は 128KB です。
        //---------------------------------------------------------------------------
        size_t soundThreadCommandBufferSize;

        //! @briefprivate
        size_t voiceCommandBufferSize;

        //---------------------------------------------------------------------------
        //! @brief    サウンドデータロードスレッドのプライオリティです。初期値は 3 です。
        //---------------------------------------------------------------------------
        s32 taskThreadPriority;

        //---------------------------------------------------------------------------
        //! @brief    サウンドデータロードスレッドのスタックサイズです。
        //!           8 の倍数で指定してください。初期値は 16KB です。
        //---------------------------------------------------------------------------
        size_t taskThreadStackSize;

        //---------------------------------------------------------------------------
        //! @briefprivate
        //!
        //! @brief    サウンドデータロードスレッドのコマンドバッファサイズです。
        //!           4 の倍数で指定してください。初期値は 8KB です。
        //---------------------------------------------------------------------------
        size_t taskThreadCommandBufferSize;

        //---------------------------------------------------------------------------
        //! @briefprivate
        //! @brief    サウンドスレッド処理負荷計測するかどうかを示すフラグです。
        //!           初期値は true です。
        //!           false の場合は、@ref GetSoundThreadTickCount は正常に動作しません。
        //---------------------------------------------------------------------------
        bool enableGetSoundThreadTick;

        //---------------------------------------------------------------------------
        //! @briefprivate
        //!
        //! @brief  NW レンダラを利用するかどうかを示すフラグです。
        //!         初期値は false です。
        //!         true にすると、次の制約がありますが、
        //!         48KHz でボイスのレンダリングすることが可能です。
        //!
        //!         - @ref GetRequiredMemSize のサイズが増えます
        //!         - サウンドスレッドの処理負荷が増えます
        //!         - エフェクト・フィルターは効きません
        //---------------------------------------------------------------------------
        bool enableNwRenderer;

        //---------------------------------------------------------------------------
        //! @briefprivate
        //!
        //! @brief    NW レンダラ用シンセサイズバッファの数です。
        //---------------------------------------------------------------------------
        u32 nwVoiceSynthesizeBufferCount;

        //! サウンドスレッドプライオリティのデフォルト値です。
        static const s32 DEFAULT_SOUND_THREAD_PRIORITY;

        //! サウンドデータロードスレッドプライオリティのデフォルト値です。
        static const s32 DEFAULT_TASK_THREAD_PRIORITY;

        //! @brief サウンドスレッドスタックサイズのデフォルト値です。
        //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
        static const u32 DEFAULT_SOUND_THREAD_STACK_SIZE = 16 * 1024;

        //! @brief サウンドデータロードスレッドスタックサイズのデフォルト値です。
        //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
        static const u32 DEFAULT_TASK_THREAD_STACK_SIZE = 16 * 1024;

        //---------------------------------------------------------------------------
        //! @briefprivate
        //!
        //! @brief    サウンドスレッドのコマンドバッファサイズのデフォルト値です。
        //---------------------------------------------------------------------------
        static const u32 DEFAULT_SOUND_THREAD_COMMAND_BUFFER_SIZE = 128 * 1024;

        //---------------------------------------------------------------------------
        //! @briefprivate
        //!
        //! @brief    サウンドデータロードスレッドのコマンドバッファサイズのデフォルト値です。
        //---------------------------------------------------------------------------
        static const u32 DEFAULT_TASK_THREAD_COMMAND_BUFFER_SIZE = 8 * 1024;

        //---------------------------------------------------------------------------
        //! @brief    コンストラクタです。
        //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        SoundSystemParam();
    };


    //---------------------------------------------------------------------------
    //! @brief      サウンドライブラリが扱える最大ボイス数です。
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static const u32 MAX_VOICE_COUNT = internal::driver::HardwareManager::MAX_VOICE_COUNT;

    //----------------------------------------
    //! @name システム
    //@{
    //---------------------------------------------------------------------------
    //! @brief  サウンドライブラリの初期化を行います。
    //!
    //!         サウンドシステムが使用するワークメモリをユーザが渡します。
    //!         @ref SoundSystemParam 構造体に、初期化に必要なパラメータを設定し、
    //!         @ref nw::snd::SoundSystem::GetRequiredMemSize で、
    //!         必要なメモリサイズを取得してください。
    //!
    //!         @ref SoundSystemParam 構造体で指定した優先度・スタックサイズで、
    //!         サウンド処理を行うスレッドと、
    //!         ストレージからサウンドデータをロードするためのスレッドの初期化、実行します。
    //!
    //!         Cafe SDK の AX の初期化と、サウンドシステムの初期化は同一コアで実行してください。
    //!         別々のコアで実行した場合は正しく動作しません。
    //!
    //! @param[in]    param         初期化パラメータです。
    //! @param[in]    workMem       サウンドシステムのワークメモリへのポインタです。
    //!                             4 バイト境界に配置されている必要があります。
    //! @param[in]    workMemSize   ワークメモリのサイズです。
    //!
    //! @see Finalize
    //! @see GetRequiredMemSize
    //! @see SoundSystemParam
    //! @see IsInitialized
    //!
    //! @date 2014/06/10 コアに関する制限についての記述を追加
    //! @date 2011/09/09 workMem のアラインメントの記述を修正 (128バイト→4バイト)
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void Initialize(
        const SoundSystemParam& param,
        uptr workMem,
        size_t workMemSize
    );

    //---------------------------------------------------------------------------
    //! @brief        サウンドライブラリの初期化に必要なメモリサイズを取得します。
    //!
    //!               param に設定されている初期化パラメータに基づいて、
    //!               サウンドシステムのセットアップに必要なメモリサイズを計算します。
    //!
    //!               この関数で取得したサイズのメモリ領域を
    //!               @ref nw::snd::SoundSystem::Initialize に渡してください。
    //!
    //! @param[in]    param   初期化パラメータです。
    //!
    //! @return       サウンドシステムの初期化に必要なメモリサイズを返します。
    //!
    //! @see  Initialize
    //! @see  SoundSystemParam
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static size_t GetRequiredMemSize( const SoundSystemParam& param );


    //! @briefprivate
    //! @param param :private
    //! @return :private
    static size_t detail_GetRequiredDriverCommandManagerMemSize( const SoundSystemParam& param );

    //! @briefprivate
    //! @param param :private
    //! @param workMem :private
    //! @param workMemSize :private
    static void detail_InitializeDriverCommandManager(
        const SoundSystemParam& param,
        uptr workMem,
        size_t workMemSize );

    //---------------------------------------------------------------------------
    //! @brief        サウンドシステムの終了処理を行います。
    //!
    //!               @ref SoundSystem::Initialize で作成された、
    //!               サウンド処理を行うスレッドと
    //!               ストレージからサウンドデータをロードするためのスレッドを終了させます。
    //!
    //! @see Initialize
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void Finalize();

    //---------------------------------------------------------------------------
    //! @brief        サウンドフレーム処理から呼び出されるコールバック関数を設定します。
    //!
    //! @param[in] callback   サウンドフレーム処理から呼び出されるコールバック関数です。
    //! @param[in] arg        コールバック関数に渡されるユーザー引数です。
    //!
    //! @see Initialize
    //! @see SoundFrameUserCallback
    //! @see ClearSoundFrameUserCallback
    //!
    //! @date 2011/07/21 初版
    //---------------------------------------------------------------------------
    static void SetSoundFrameUserCallback( SoundFrameUserCallback callback, uptr arg )
    {
        internal::driver::SoundThread::GetInstance().RegisterSoundFrameUserCallback(
                callback, arg );
    }

    //---------------------------------------------------------------------------
    //! @brief        サウンドフレーム処理から呼び出されるコールバック関数を削除します。
    //!
    //! @see SetSoundFrameUserCallback
    //!
    //! @date 2011/07/21 初版
    //---------------------------------------------------------------------------
    static void ClearSoundFrameUserCallback()
    {
        internal::driver::SoundThread::GetInstance().ClearSoundFrameUserCallback();
    }

    //---------------------------------------------------------------------------
    //! @brief        サウンドシステムが初期化されているかどうかを取得します。
    //!
    //!               @ref Initialize が呼び出され、
    //!               サウンドシステムの初期化が完了しているかどうかを調べます。
    //!
    //! @return       サウンドシステムの初期化が完了していれば true を、
    //!               そうでなければ false を返します。
    //!
    //! @see  Initialize
    //! @see  Finalize
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static bool IsInitialized();


    //---------------------------------------------------------------------------
    //! @brief      サウンドスレッドをロックします。
    //!
    //!             長時間ロックし続けると、 nw::snd のサウンド処理が滞り、
    //!             音が途切れるなどの悪影響があります。
    //!
    //! @see  UnlockSoundThread
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void LockSoundThread()
    {
        internal::driver::SoundThread::GetInstance().Lock();
    }

    //---------------------------------------------------------------------------
    //! @brief      サウンドスレッドをアンロックします。
    //!
    //!             @ref LockSoundThread 関数の説明もあわせてご覧ください。
    //!
    //! @see  LockSoundThread
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void UnlockSoundThread()
    {
        internal::driver::SoundThread::GetInstance().Unlock();
    }

    //---------------------------------------------------------------------------
    //! @brief      オブジェクトの生成時からオブジェクトの存在するスコープを抜けるまでの間、
    //!             サウンドスレッドをロックします。
    //!
    //!             使用に関する注意点は、@ref LockSoundThread 関数と同じです。
    //!
    //! @see LockSoundThread
    //! @see UnlockSoundThread
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    class SoundThreadScopedLock
    {
    public:
        //---------------------------------------------------------------------------
        //! @brief  コンストラクタです。
        //!
        //!         内部で @ref LockSoundThread 関数を呼び出します。
        //!
        //! @see  LockSoundThread
        //!
        //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        SoundThreadScopedLock()
        {
            LockSoundThread();
        }
        //---------------------------------------------------------------------------
        //! @brief  デストラクタです。
        //!
        //!         内部で @ref UnlockSoundThread 関数を呼び出します。
        //!
        //! @see  UnlockSoundThread
        //!
        //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        ~SoundThreadScopedLock()
        {
            UnlockSoundThread();
        }
    private:
        NW_DISALLOW_COPY_AND_ASSIGN( SoundThreadScopedLock );
    };



    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief  サウンドライブラリのスリープ準備処理を実行します。
    //!
    //!         スリープに入る前に、本関数を必ず呼び出してください。
    //!
    //!         スリープの準備では、
    //!         サウンドデータロードスレッドをロックします。
    //!
    //!         サウンドデータロードスレッドでロード中のデータがある場合は、
    //!         ロードが終わるまで本関数内でブロックされます。
    //!
    //!         スリープを解除する場合は、@ref LeaveSleep 関数を呼び出してください。
    //!
    //! @see  LeaveSleep
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void EnterSleep()
    {
        if ( s_IsEnterSleep )
        {
            return;
        }

        s_IsEnterSleep = true;
        s_IsStreamLoadWait = true;
        internal::TaskThread::GetInstance().Lock();

        // サウンドスレッドは SDK 側で止められる
    }

    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief  サウンドライブラリのスリープ終了処理を実行します。
    //!
    //!         スリープを解除した後に、必ず呼び出してください。
    //!
    //! @see  EnterSleep
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void LeaveSleep()
    {
        if ( ! s_IsEnterSleep )
        {
            return;
        }

        // サウンドスレッドは SDK 側で復帰する
        internal::TaskThread::GetInstance().Unlock();
        s_IsStreamLoadWait = false;
        s_IsEnterSleep = false;
    }

    //! @briefprivate
    //! @return :private
    static bool detail_IsStreamLoadWait()
    {
        return s_IsStreamLoadWait;
    }

    //! @briefprivate
    //! @return :private
    static bool detail_IsInitializedNwRenderer()
    {
        return s_IsInitializedNwRenderer;
    }

    //! @briefprivate
    //! @param isEnabled :private
    static void detail_SetStreamOpenFailureHaltEnabled(bool isEnabled)
    {
        s_IsStreamOpenFailureHalt = isEnabled;
    }

    //! @briefprivate
    //! @return :private
    static bool detail_IsStreamOpenFailureHaltEnabled()
    {
        return s_IsStreamOpenFailureHalt;
    }
    //@}

    //----------------------------------------
    //! @name Cafe リモコン
    //@{
    //---------------------------------------------------------------------------
    //! @brief  @ref RemoteSpeaker クラスのインスタンスを取得します。
    //!
    //!         channelIndex で指定した番号の @ref RemoteSpeaker
    //!         クラスのインスタンスを取得します。
    //!
    //!         有効なチャンネル番号は 0～3 です。
    //!         範囲外の値を渡すと、Debug / Development 版ではアサートになります。
    //!         Release 版での動作は不定です。
    //!
    //! @param[in] channelIndex リモコンのチャンネル番号です。0～3 を入れる必要があります。
    //!
    //! @return @ref RemoteSpeaker クラスのインスタンスを返します。
    //!
    //! @see  RemoteSpeaker
    //!
    //! @date 2011/08/10 初版
    //---------------------------------------------------------------------------
    static RemoteSpeaker& GetRemoteSpeaker( int channelIndex )
    {
        return internal::RemoteSpeakerManager::GetInstance().GetRemoteSpeaker( channelIndex );
    }
    //@}

    //----------------------------------------
    //! @name 出力モード
    //@{
    //---------------------------------------------------------------------------
    //! @brief  サウンドの出力モードを設定します。
    //!
    //!         @ref Initialize を呼び出したときに、
    //!         本体メニューの設定と同じモードに初期化されます。
    //!         具体的には下記のとおりです。
    //!
    //!         - 本体メニューの設定が「モノラル」のとき、OUTPUT_MODE_MONO に設定されます
    //!         - 本体メニューの設定が「ステレオ」のとき、OUTPUT_MODE_STEREO に設定されます
    //!         - 本体メニューの設定が「サラウンド」のとき、OUTPUT_MODE_SURROUND に設定されます
    //!
    //!         現在の出力モードを知りたい場合は、@ref GetOutputMode を使用してください。
    //!
    //!         ■ ミックスパラメータの計算方法
    //!
    //!         本関数で設定する出力モードは、ミックスパラメータ
    //!         (フロントの L/R、リアの L/R の音量。
    //!         以下、それぞれ fL, fR, rL, rR と表現します)
    //!         の計算方法に影響を与えます。
    //!
    //!         たとえば OUTPUT_MODE_MONO を設定すると、SoundHandle などで設定する
    //!         パン・サラウンドパンの値によらず、ミックスパラメータのフロント L/R
    //!         が同じ値になり、リアの L/R にはゼロが設定されます。
    //!
    //!         - fL = fR = M (ボリュームに応じた値)
    //!         - rL = rR = 0.0f
    //!
    //!         OUTPUT_MODE_STEREO を設定すると、パンの値を反映して、
    //!         フロントの L/R が別の値 (パンを中央に振っている場合は同じ値)
    //!         になりますが、サラウンドパンは無視され、リアの L/R はゼロが設定されます。
    //!
    //!         - fL = L (ボリュームとパンに応じた値)
    //!         - fR = R (ボリュームとパンに応じた値)
    //!         - rL = rR = 0.0f
    //!
    //!         OUTPUT_MODE_SURROUND を設定すると、パンとサラウンドパンの値が反映され、
    //!         フロントの L/R、リアの L/R がそれぞれ別の値が設定されます
    //!         (それぞれパンを中央に振っている場合は同じ値になります)。
    //!
    //!         - fL = surFL (ボリュームとパン・サラウンドパンに応じた値)
    //!         - fR = surFR (ボリュームとパン・サラウンドパンに応じた値)
    //!         - rL = surRL (ボリュームとパン・サラウンドパンに応じた値)
    //!         - rR = surRR (ボリュームとパン・サラウンドパンに応じた値)
    //!
    //!
    //!         ■ ミックスパラメータのダウンミックス
    //!
    //!         上記のように計算されたフロントの L/R およびリアの L/R は、
    //!         さらに、本体メニューの設定に応じてダウンミックスされます。
    //!
    //!         本体メニューがモノラルの場合、ここで設定するモードが
    //!         OUTPUT_MODE_STEREO および OUTPUT_MODE_SURROUND
    //!         のときに、次のようにダウンミックスされます。
    //!
    //!         (OUTPUT_MODE_STEREO)
    //!         - fL = fR = L+R/2
    //!
    //!         (OUTPUT_MODE_SURROUND)
    //!         - fL = fR = ( (surFL + surRL) + (surFR + surRR) ) / 2
    //!
    //!         本体メニューがステレオの場合、ここで設定するモードが、
    //!         OUTPUT_MODE_SURROUND のときに次のようにダウンミックスされます。
    //!
    //!         - fL = surFL + sufRL
    //!         - rL = surFR + sufRR
    //!
    //!         そのほかのケースではダウンミックスは行われません。
    //!
    //!
    //!         ■ ダウンミックスによる音量感の変化
    //!
    //!         パンやサラウンドパンを振ると、
    //!         ダウンミックス後の音量感が変わる可能性があります。
    //!
    //!         たとえば、SetPan(1.0f) および SetSurroundPan(2.0f) とすると、
    //!         OUTPUT_MODE_MONO では、
    //!
    //!         - M = 0.71f
    //!
    //!         OUTPUT_MODE_STEREO では、
    //!
    //!         - L = 0.00f
    //!         - R = 1.00f
    //!
    //!         OUTPUT_MODE_SURROUND では、
    //!
    //!         - surFL = 0.00f
    //!         - surFR = 0.00f
    //!         - surRL = 0.00f
    //!         - surRR = 0.80f
    //!
    //!         のような値が計算されます。本体メニューのモードがモノラルの場合、
    //!         上で示した計算式によってダウンミックスされるため、
    //!
    //!         - 本体メニューがモノラル : fL( 0.71) fR( 0.71) rL( 0.00) rR( 0.00)
    //!         - 本体メニューがステレオ : fL( 0.50) fR( 0.50) rL( 0.00) rR( 0.00)
    //!         - 本体メニューがサラウンド: fL( 0.40) fR( 0.40) rL( 0.00) rR( 0.00)
    //!
    //!         となり、かなり音量が変わってしまうことになります。
    //!
    //!         このような音量の変化を懸念される場合は、
    //!         本関数で出力モードを変化できるようにするシーンを絞ったりするなど、
    //!         ユーザー側で工夫する必要があります。
    //!
    //! @param[in]  mode    サウンドの出力モードです。
    //! @param[in]  device  出力先デバイスです。
    //!
    //! @see  OutputMode
    //! @see  OutputDevice
    //! @see  GetOutputMode
    //!
    //! @date 2012/08/02 計算方法の違いやダウンミックスについて追記
    //! @date 2012/04/24 device 引数を追加
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void SetOutputMode( OutputMode mode, OutputDevice device = OUTPUT_DEVICE_MAIN )
    {
        internal::driver::HardwareManager::GetInstance().SetOutputMode( mode, device );
    }

    //---------------------------------------------------------------------------
    //! @brief        現在のサウンド出力モードを取得します。
    //!
    //! @return       現在のサウンド出力モードを返します。
    //!
    //! @param[in]  device  出力先デバイスです。
    //!
    //! @see  OutputMode
    //! @see  OutputDevice
    //! @see  SetOutputMode
    //!
    //! @date 2012/04/24 device 引数を追加
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static OutputMode GetOutputMode( OutputDevice device = OUTPUT_DEVICE_MAIN )
    {
        return internal::driver::HardwareManager::GetInstance().GetOutputMode( device );
    }

    //! @briefprivate
    static void detail_UpdateEndUserOutputMode()
    {
        internal::driver::HardwareManager::GetInstance().UpdateEndUserOutputMode();
    }
    //@}

    //----------------------------------------
    //! @name 出力先
    //@{
    //---------------------------------------------------------------------------
    //! @brief        指定した出力ラインについて、出力先デバイスのビットフラグを設定します。
    //!               ただし OUTPUT_LINE_INDEX_RESERVED_MAX より前の出力ラインは
    //!               予約済みの出力ラインとなっておりまして、出力先デバイスの変更はできません。
    //!
    //! @param[in]  index             出力ラインのインデックスです。
    //! @param[in]  outputDeviceFlag  出力先デバイスのビットフラグです。
    //!
    //! @see  OutputDeviceIndex
    //! @see  GetOutputDeviceFlag
    //!
    //! @date 2013/05/20 初版
    //---------------------------------------------------------------------------
    static void SetOutputDeviceFlag( OutputLineIndex index, u8 outputDeviceFlag )
    {
        internal::driver::HardwareManager::GetInstance().SetOutputDeviceFlag( index, outputDeviceFlag );
    }

    //---------------------------------------------------------------------------
    //! @brief        指定した出力ラインについて、出力先デバイスのビットフラグを取得します。
    //!
    //! @return       指定した出力ラインについて出力先デバイスのビットフラグを返します。
    //!
    //! @param[in]  index  出力ラインのインデックスです。
    //!
    //! @see  OutputDeviceIndex
    //! @see  SetOutputDeviceFlag
    //!
    //! @date 2013/05/20 初版
    //---------------------------------------------------------------------------
    static u8 GetOutputDeviceFlag( OutputLineIndex index )
    {
        return internal::driver::HardwareManager::GetInstance().GetOutputDeviceFlag( index );
    }
    //@}


    //----------------------------------------
    //! @name マスターボリューム
    //@{
    //---------------------------------------------------------------------------
    //! @brief        マスターボリュームを設定します。
    //!
    //!               マスターボリュームは、
    //!               サウンドライブラリで発音される音全てに対して影響を与えます。
    //!               サウンドライブラリ以外で発音された音には影響を与えません。
    //!
    //!               この関数で指定する値は、他のどの音量パラメータとも独立して動作し、
    //!               それらは全て重ね合わされます。
    //!
    //!               音量 volume は、0.0 以上の倍率で指定します。
    //!               すなわち、1.0 を指定すると音量に影響を与えません。
    //!               0.0 を指定すると発音されなくなります。
    //!               デフォルト値は 1.0 です。
    //!
    //!               他の音量パラメータと重ね合わされたあと、最終的な音量は
    //!               0.0 ～ 2.0 の範囲でクランプされます。
    //!               この関数で 2.0 を設定したとしても、元サウンドの音量は 2
    //!               倍にならない可能性があることに注意してください。
    //!
    //!               音量の変化は fadeTimes で指定した時間をかけて行われます。
    //!               音量の変化途中にさらにこの関数を呼び出した場合は、
    //!               その時点での変化途中の音量値を基点として、
    //!               新しく指定した時間をかけて音量を変化させます。
    //!
    //! @param[in] volume     音量の倍率 ( 0.0 ～ ) です。
    //! @param[in] fadeTimes  音量変化にかける時間 (ミリ秒) です。
    //!
    //! @see  GetMasterVolume
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void SetMasterVolume( float volume, int fadeTimes )
    {
        internal::driver::HardwareManager::GetInstance().SetMasterVolume( volume, fadeTimes );
    }
    // TODO: ★ 「AUX に送られる前に反映されるので～」を書いていない

    //---------------------------------------------------------------------------
    //! @brief        現在のマスターボリュームを取得します。
    //!
    //!               現在設定されている @ref SetMasterVolume で設定された
    //!               マスターボリュームの値を取得します。
    //!
    //!               音量が時間変化している間に値を取得した場合は、
    //!               その時点での変化途中の音量の値を返します。
    //!
    //! @return       現在のマスターボリュームの値を返します。
    //!
    //! @see          SetMasterVolume
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static float GetMasterVolume()
    {
        return internal::driver::HardwareManager::GetInstance().GetMasterVolume();
    }
    //@}

    //----------------------------------------
    //! @name エフェクト
    //@{

    //---------------------------------------------------------------------------
    //! @brief      指定したエフェクトを AUX バスに追加します。
    //!
    //!             エフェクトは、指定したバスそれぞれで独立して処理されます。
    //!             エフェクト (@ref nw::snd::FxBase を継承したエフェクト) は、
    //!             1 つのバスに複数のエフェクトを登録することができます。
    //!             エフェクトは、追加した順番で処理されます。
    //!
    //!             途中でエフェクトの接続順を変更したい場合は、
    //!             @ref nw::snd::SoundSystem::ClearEffect
    //!             を呼び出して一度すべてのエフェクトの登録を削除した後、
    //!             改めて追加しなおしてください。
    //!
    //!             エフェクトを追加する際、内部で @ref nw::snd::FxBase::Initialize
    //!             を呼び出しています。
    //!
    //!             DRC にチャンネル数指定可能なエフェクトを設定する場合、6ch のエフェクトを設定することはできません。
    //!             Release ビルドの場合はエフェクトが反映されず、
    //!             Debug / Develop ビルドの場合はアサートが発生するためご注意ください。
    //!
    //!             エフェクト処理の概要については @ref SoundSystem クラスの説明をご覧ください。
    //!
    //! @param[in]  bus         エフェクトを追加するバスです。
    //! @param[in]  effect      エフェクトです。
    //! @param[in]  device      エフェクトを設定するデバイスです。
    //!
    //! @return     エフェクトの開始に成功したら true を、失敗したら false を返します。
    //!
    //! @see AuxBus
    //! @see FxBase
    //! @see ClearEffect
    //! @see OutputDevice
    //!
    //! @date 2012/03/22 device 引数を追加
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static bool AppendEffect( AuxBus bus, FxBase* effect, OutputDevice device = OUTPUT_DEVICE_MAIN );

    //---------------------------------------------------------------------------
    //! @brief        指定した AUX バスに登録されているエフェクトを全てバスから削除します。
    //!
    //!               エフェクトを終了させる際、 fadeTimes でフェードアウトをかけることができます。
    //!               フェードアウトが完了するまでエフェクトは終了しません。
    //!               フェードアウトが完了したかどうかを知るには
    //!               @ref nw::snd::SoundSystem::IsFinishedClearEffect を呼び出してください。
    //!
    //!               エフェクトをバスから削除する際、内部で
    //!               @ref nw::snd::FxBase::Finalize を呼び出しています。
    //!
    //! @param[in]  bus         エフェクトバスです。
    //! @param[in]  fadeTimes   フェード時間 (ミリ秒) です。
    //! @param[in]  device      エフェクトを設定したデバイスです。
    //!
    //! @see IsFinishedClearEffect
    //! @see AuxBus
    //! @see FxBase
    //! @see OutputDevice
    //!
    //! @date 2012/03/22 device 引数を追加
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void ClearEffect( AuxBus bus, int fadeTimes = 0, OutputDevice device = OUTPUT_DEVICE_MAIN );

    //---------------------------------------------------------------------------
    //! @brief        ClearEffect が完了したかどうかを調べます。
    //!
    //!               @ref ClearEffect が呼ばれる前の動作は不定です。
    //!
    //! @param[in]  bus     エフェクトバスです。
    //! @param[in]  device  エフェクトを設定したデバイスです。
    //!
    //! @return       @ref ClearEffect の処理が完了していれば true を、
    //!               そうでなければ false を返します。
    //!
    //! @see ClearEffect
    //! @see AuxBus
    //!
    //! @date 2012/03/22 device 引数を追加
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static bool IsFinishedClearEffect( AuxBus bus, OutputDevice device = OUTPUT_DEVICE_MAIN )
    {
        return internal::driver::HardwareManager::GetInstance().IsFinishedClearEffect( bus, device );
    }
    //@}

    //----------------------------------------
    //! @name SRC タイプ
    //@{
    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief      SRC タイプ (サンプリングレートコンバータの補間方法) を設定します。
    //!
    //!             デフォルトでは SRC_TYPE_4TAP が設定されています。
    //!
    //!             ウェーブサウンドおよびストリームサウンドに影響がありますが、
    //!             シーケンスサウンドには影響がありません。
    //!             シーケンスサウンドは、SoundMaker のインストリストおよび
    //!             パーカッションリスト上で設定された「補間タイプ」が影響します。
    //!
    //! @param[in]    type    SRC タイプです。
    //!
    //! @see nw::snd::SrcType
    //! @see GetSrcType
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void SetSrcType( SrcType type )
    {
        internal::driver::HardwareManager::GetInstance().SetSrcType( type );
    }

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief        現在設定されている SRC タイプ
    //!               (サンプリングレートコンバータの補間方法) を取得します。
    //!
    //!               デフォルトでは SRC_TYPE_4TAP が設定されています。
    //!
    //! @return       現在設定されている SRC タイプの値を返します。
    //!
    //! @see nw::snd::SrcType
    //! @see SetSrcType
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static SrcType GetSrcType()
    {
        return internal::driver::HardwareManager::GetInstance().GetSrcType();
    }
    //@}

    //----------------------------------------
    //! name リセット準備
    //{

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief        リセット前に必要なサウンドの処理を行います。
    //!
    //!               この関数は、リセット時にノイズが発生しないように再生中の音を緩やかに消します。
    //!               一度この関数が呼ばれた後は、再び発音することができません。
    //!
    //!               ノイズが発生しないようにリセットを行うためには、
    //!               @ref WaitForResetReady を呼び出して、
    //!               この関数の処理が完了するのを待つ必要があります。
    //!
    //! @see  WaitForResetReady
    //!
    //! @date 2010/01/15 初版
    //---------------------------------------------------------------------------
    static void PrepareReset()
    {
        internal::driver::HardwareManager::GetInstance().PrepareReset();
    }

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief        サウンドのリセット準備処理が完了するのを待ちます。
    //!
    //!               この関数は、@ref PrepareReset が呼び出された後に、
    //!               その処理が完了するのを待ちます。
    //!
    //! @see  PrepareReset
    //!
    //! @date 2010/01/15 初版
    //---------------------------------------------------------------------------
    static void WaitForResetReady();
    //@}

    //----------------------------------------
    //! @name パフォーマンス計測
    //@{
    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief        サウンドスレッド処理にかかった累積時間および処理回数を取得します。
    //!
    //!               サウンドスレッドの処理は、定期的 (約 5ms ごと。正確には、
    //!               @ref SOUND_THREAD_INTERVAL_USEC マイクロ秒ごと) に実行されるため、
    //!               本関数を 1/60 秒 (約 16.67ms) ごとに呼び出した場合、
    //!               処理回数は約 3～4 回になります。
    //!               累積処理時間は、この約 3～4 回分のサウンドスレッド処理にかかった時間になります。
    //!
    //!               処理に要した時間は、
    //!               サウンドスレッドを開始した時間と終了した時間の差分で計算します。
    //!               したがって、サウンドスレッドの実行中に他のスレッドに切り替わった場合、
    //!               それらの実行時間を含んだ値を返すことに注意してください。
    //!
    //!               処理時間には、NW4C サウンドライブラリ層の処理にかかった時間に加え、
    //!               nn::snd::WaitForDspSync および nn::snd::SendParameterToDsp
    //!               にかかった時間も含まれます。
    //!
    //!               累積時間及び処理回数は @ref ClearSoundThreadTickCount 関数によって、
    //!               ゼロにクリアすることができます。
    //!
    //! @param[out]   tick    サウンドスレッド内部の累積処理時間です。
    //! @param[out]   count   サウンドスレッド処理回数です。
    //!
    //! @see ClearSoundThreadTickCount
    //! @see SOUND_THREAD_INTERVAL_USEC
    //!
    //! @date 2010/12/10 PC 版に関する記述を削除
    //! @date 2010/05/08 GetSoundThreadProcessTick 関数を廃止し、本関数を追加
    //---------------------------------------------------------------------------
    static void GetSoundThreadTickCount( nw::ut::Tick& tick, int& count )
    {
      #ifdef NW_PLATFORM_CTR
        internal::driver::SoundThread::GetInstance().GetTickCount( tick, count );
      #else
        NW_UNUSED_VARIABLE( tick );
        NW_UNUSED_VARIABLE( count );
      #endif
    }

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief      サウンドスレッド処理にかかった累積時間および処理回数を
    //!             ゼロにリセットします。
    //!
    //! @see  GetSoundThreadTickCount
    //!
    //! @date 2010/05/08 初版
    //---------------------------------------------------------------------------
    static void ClearSoundThreadTickCount()
    {
      #ifdef NW_PLATFORM_CTR
        internal::driver::SoundThread::GetInstance().ClearTickCount();
      #endif
    }

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @param sampleRate :private
    //---------------------------------------------------------------------------
    static void SetNwVoiceSampleRate( u32 sampleRate );

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief        サウンドスレッド処理間隔をマイクロ秒換算した値です。
    //!
    //!               @ref GetSoundThreadTickCount で得られた count と tick に対し、
    //!               ( tick.ToTimeSpan().GetMicroSeconds() * 100 ) /
    //!               ( count * SOUND_THREAD_INTERVAL_USEC ) とすると、
    //!               count 回のサウンドスレッド処理で、
    //!               CPU を何 % 占有しているかが算出できます。
    //!
    //! @see GetSoundThreadTickCount
    //!
    //! @date 2010/05/08 初版
    //---------------------------------------------------------------------------
    static const int SOUND_THREAD_INTERVAL_USEC = nw::snd::internal::driver::HardwareManager::SOUND_FRAME_INTERVAL_USEC;
    //@}

    //----------------------------------------
    //! @name シーケンス変数出力設定
    //@{
    // シーケンス変数のプリント ON/OFF を設定する
    //---------------------------------------------------------------------------
    //! @brief        printvar シーケンスコマンドを有効にします。
    //!
    //!               デフォルトでは無効になっています。
    //!               詳しくは、シーケンスデータマニュアルをご覧ください。
    //!
    //!               printvar シーケンスコマンドを有効にすると、
    //!               無効のときと比べて printvar コマンド処理で余分に処理がかかるので、
    //!               シーケンス変数デバッグをしない場合は無効にしておくことを推奨します。
    //!
    //! @see  DisableSeqPrintVar
    //! @see  IsEnabledSeqPrintVar
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void EnableSeqPrintVar() { internal::driver::MmlParser::EnablePrintVar( true ); }

    //---------------------------------------------------------------------------
    //! @brief        printvar シーケンスコマンドを無効にします。
    //!
    //!               詳しくは、シーケンスデータマニュアルをご覧ください。
    //!
    //! @see  EnableSeqPrintVar
    //! @see  IsEnabledSeqPrintVar
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void DisableSeqPrintVar() { internal::driver::MmlParser::EnablePrintVar( false ); }

    //---------------------------------------------------------------------------
    //! @brief        printvar シーケンスコマンドが有効かどうかを取得します。
    //!
    //!               true のときは有効、false のときは無効です。
    //!
    //!               printvar シーケンスコマンドを有効にすると、
    //!               無効のときと比べて printvar コマンド処理で余分に処理がかかるので、
    //!               シーケンス変数デバッグをしない場合は無効にしておくことを推奨します。
    //!
    //! @return       現在、printvar シーケンスコマンドが有効か無効かを取得します。
    //!
    //! @see  EnableSeqPrintVar
    //! @see  DisableSeqPrintVar
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static bool IsEnabledSeqPrintVar()
    {
        return internal::driver::MmlParser::IsEnabledPrintVar();
    }
    //@}

    //----------------------------------------
    //! @name 使用ボイス数
    //@{
    //---------------------------------------------------------------------------
    //! @brief        サウンドライブラリで現在発音中のボイス数を取得します。
    //!
    //! @return       現在発音中のボイス数を返します。
    //!
    //! @see GetMaxVoiceCount
    //! @see SetMaxVoiceCount
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static int GetVoiceCount()
    {
        return internal::driver::MultiVoiceManager::GetInstance().GetVoiceCount();
    }

    //---------------------------------------------------------------------------
    //! @brief        サウンドライブラリで使用できる最大ボイス数を取得します。
    //!
    //!               デフォルトでは、@ref MAX_VOICE_COUNT を返します。
    //!
    //! @return       最大ボイス数を返します。
    //!
    //! @see GetVoiceCount
    //! @see SetMaxVoiceCount
    //! @see MAX_VOICE_COUNT
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static int GetMaxVoiceCount()
    {
        return s_MaxVoiceCount;
    }

    //---------------------------------------------------------------------------
    //! @brief        サウンドライブラリで使用する最大ボイス数を設定します。
    //!
    //!               @ref Initialize より先に呼ぶ必要があります。
    //!
    //!               設定できる最大値は @ref MAX_VOICE_COUNT です。
    //!               これは、@ref GetMaxVoiceCount で取得できるデフォルト値と同じです。
    //!
    //!               最大ボイス数を削減することで、サウンドの処理量を抑えることができます。
    //!
    //!               最大ボイス数として、1 ～ @ref MAX_VOICE_COUNT を指定する必要があります。
    //!               範囲外の値を指定した場合は、Debug 版、Development 版ではアサートで停止し、
    //!               Release 版では 1 ～ @ref MAX_VOICE_COUNT の範囲に丸められます。
    //!
    //! @param[in] count  最大ボイス数です。
    //!
    //! @see Initialize
    //! @see GetVoiceCount
    //! @see GetMaxVoiceCount
    //! @see MAX_VOICE_COUNT
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void SetMaxVoiceCount( int count )
    {
        NW_MINMAX_ASSERT( count, 1, nw::snd::internal::driver::HardwareManager::MAX_VOICE_COUNT );
        count = ut::Clamp<int>( count, 1, nw::snd::internal::driver::HardwareManager::MAX_VOICE_COUNT );
        s_MaxVoiceCount = count;
    }
    //@}

    //----------------------------------------
    //! @name   biquad フィルタ
    //@{
    //---------------------------------------------------------------------------
    //! @brief  ユーザー定義の biquad フィルタを登録します。
    //!
    //!         @ref BiquadFilterCallback クラスを継承してユーザーが実装した
    //!         biquad フィルタを登録します。
    //!
    //!         type には、@ref BiquadFilterType で定義されている、
    //!         BIQUAD_FILTER_TYPE_USER_MIN ～ BIQUAD_FILTER_TYPE_USER_MAX
    //!         の値を指定します。
    //!         範囲外の値を設定すると、Debug 版 / Development 版ではアサートで停止します。
    //!         Release 版では無視されます。
    //!
    //!         ここで設定したコールバック関数はサウンドスレッドから呼び出されます。
    //!         排他制御が必要な場合は、
    //!         @ref SoundThreadScopedLock クラスあるいは、
    //!         @ref LockSoundThread / @ref UnlockSoundThread 関数を利用する必要があります。
    //!
    //!         コールバック関数はすみやかに処理を終える必要があります。
    //!         処理が長引くと、ノイズが発生する可能性が高くなります。
    //!         たとえば、コールバック関数内でブロックする可能性のある API
    //!         (クリティカルセクションなど) を呼び出すと、
    //!         処理を終えるのに時間がかかる可能性があります。
    //!
    //! @param[in] type     フィルタの種類です
    //!                     (BIQUAD_FILTER_TYPE_USER_MIN ～BIQUAD_FILTER_TYPE_USER_MAX)。
    //! @param[in] biquad   ユーザーが実装した biquad フィルタクラスです。
    //!
    //! @see BiquadFilterCallback クラス
    //! @see BiquadFilterType
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void SetBiquadFilterCallback( int type, const BiquadFilterCallback* biquad )
    {
        NW_MINMAX_ASSERT( type, BIQUAD_FILTER_TYPE_USER_MIN, BIQUAD_FILTER_TYPE_USER_MAX );
        internal::driver::HardwareManager::GetInstance().SetBiquadFilterCallback( type, biquad );
    }
    //@}

#ifdef NW_PLATFORM_CAFE
    //----------------------------------------
    //! @name   パフォーマンス計測
    //@{
    //---------------------------------------------------------------------------
    //! @brief  プロファイルリーダーを登録します。
    //!
    //! @param[in] profileReader 登録するプロファイルリーダーです。
    //!
    //! @see ProfileReader クラス
    //!
    //! @date 2012/02/07 初版
    //---------------------------------------------------------------------------
    static void RegisterProfileReader( ProfileReader& profileReader )
    {
        internal::driver::SoundThread::GetInstance().RegisterProfileReader( profileReader );
    }

    //---------------------------------------------------------------------------
    //! @brief  プロファイルリーダーの登録を解除します。
    //!
    //! @param[in] profileReader 登録を解除するプロファイルリーダーです。
    //!
    //! @see ProfileReader クラス
    //!
    //! @date 2012/02/07 初版
    //---------------------------------------------------------------------------
    static void UnregisterProfileReader( ProfileReader& profileReader )
    {
        internal::driver::SoundThread::GetInstance().UnregisterProfileReader( profileReader );
    }
    //@}
#endif

    //---------------------------------------------------------------------------
    //! @brief @ref nw::snd::FinalMixCallback をリストの末尾に追加します。
    //!
    //!        追加した @ref nw::snd::FinalMixCallback は先頭から順に呼び出されます。
    //!
    //! @param[in] userCallback 追加する @ref nw::snd::FinalMixCallback です。
    //!
    //! @see FinalMixCallback クラス
    //!
    //! @date 2013/10/02 初版
    //---------------------------------------------------------------------------
    static void AppendFinalMixCallback( FinalMixCallback* userCallback )
    {
        internal::driver::HardwareManager::GetInstance().AppendFinalMixCallback( userCallback );
    }

    //---------------------------------------------------------------------------
    //! @brief @ref nw::snd::FinalMixCallback をリストの先頭に追加します。
    //!
    //!        追加した @ref nw::snd::FinalMixCallback は先頭から順に呼び出されます。
    //!
    //! @param[in] userCallback 追加する @ref nw::snd::FinalMixCallback です。
    //!
    //! @see FinalMixCallback クラス
    //!
    //! @date 2013/10/02 初版
    //---------------------------------------------------------------------------
    static void PrependFinalMixCallback( FinalMixCallback* userCallback )
    {
        internal::driver::HardwareManager::GetInstance().PrependFinalMixCallback( userCallback );
    }

    //---------------------------------------------------------------------------
    //! @brief @ref nw::snd::FinalMixCallback をリストから削除します。
    //!
    //! @param[in] userCallback 削除する @ref nw::snd::FinalMixCallback です。
    //!
    //! @see FinalMixCallback クラス
    //!
    //! @date 2013/10/02 初版
    //---------------------------------------------------------------------------
    static void EraseFinalMixCallback( FinalMixCallback* userCallback )
    {
        internal::driver::HardwareManager::GetInstance().EraseFinalMixCallback( userCallback );
    }

    //! @briefprivate
    //! @param audioFrameCount :private
    static void VoiceCommandProcess(u32 audioFrameCount);

#ifdef NW_PLATFORM_CAFE
    //----------------------------------------
    //! @name   メモリ管理
    //@{
    //---------------------------------------------------------------------------
    //! @brief     指定メモリ領域が DSP レンダリング時に有効かどうかを調べます。
    //!
    //!            ハードウェアの制約により、DSP レンダリングに利用する波形データ領域は
    //!            512 MB 境界をまたぐことができません。
    //!
    //!            上記制約に該当すると、この関数は false を返します。
    //!
    //!            詳しくは CafeSDK リファレンスマニュアルの AXCheckVoiceOffsets() を参照ください。
    //!
    //! @param[in] ptr  メモリ領域へのポインタです。
    //! @param[in] size メモリ領域のサイズです。
    //!
    //! @return    有効なメモリ領域の場合は true, そうでない場合は false を返します。
    //!
    //! @date 2012/07/13 初版
    //---------------------------------------------------------------------------
    static bool IsValidMemoryForDsp( const void* ptr, size_t size )
    {
        return internal::Util::IsValidMemoryForDsp( ptr, size );
    }
    //@}
#endif

private:
    static int s_MaxVoiceCount;

    static void* s_LoadThreadStackPtr;
    static u32 s_LoadThreadStackSize;
    static void* s_SoundThreadStackPtr;
    static u32 s_SoundThreadStackSize;

    static bool s_IsInitialized;
    static bool s_IsStreamLoadWait;
    static bool s_IsEnterSleep;
    static bool s_IsInitializedDriverCommandManager;
    static bool s_IsInitializedNwRenderer;
    static bool s_IsStreamOpenFailureHalt;

    static internal::NwVoiceRenderer* s_pNwVoiceRenderer;
};

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SOUND_SYSTEM_H_ */

