﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SOUND_INSTANCE_MANAGER_H_
#define NW_SND_SOUND_INSTANCE_MANAGER_H_

#include <nw/ut/ut_LinkList.h>
#include <nw/snd/snd_BasicSound.h>

namespace nw {
namespace snd {
namespace internal {

/* ========================================================================
        SoundInstanceManager class
   ======================================================================== */

template < typename Sound >
class SoundInstanceManager
{
    /* ------------------------------------------------------------------------
            typename definition
       ------------------------------------------------------------------------ */
public:
    typedef ut::LinkList< Sound, offsetof(Sound,m_PriorityLink)> PriorityList;
    typedef typename PriorityList::Iterator Iterator;

    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */
public:
    SoundInstanceManager() : m_pBuffer(NULL), m_BufferSize(0)
    {
    }

    ~SoundInstanceManager()
    {
    }

    size_t GetRequiredMemSize( int instanceCount ) const
    {
        return sizeof(Sound) * instanceCount;
    }

    /*---------------------------------------------------------------------------*
      Name:         Create

      Description:  指定したメモリ領域をインスタンスマネージャに割り当てます

      Arguments:    buffer - メモリ領域の開始アドレス
                    size   - メモリ領域のサイズ

      Returns:      無し
     *---------------------------------------------------------------------------*/
    unsigned long Create( void* buffer, unsigned long size )
    {
        NW_NULL_ASSERT( buffer );

        char* ptr = static_cast<char*>(buffer);
        const unsigned long numObjects = size / sizeof(Sound);
        for( unsigned long i=0 ;i<numObjects; i++ ){
            Sound* sound = new( ptr ) Sound( *this );
            m_FreeList.PushBack( sound );
            ptr += sizeof(Sound);
        }

        m_pBuffer = buffer;
        m_BufferSize = size;
        return numObjects;
    }

    /*---------------------------------------------------------------------------*
      Name:         Destroy

      Description:  指定したメモリ領域をインスタンスマネージャから解放します
                    解放するためにはメモリ領域が Free されている必要があります。

      Arguments:    buffer - メモリ領域の開始アドレス
                    size   - メモリ領域のサイズ

      Returns:      無し
     *---------------------------------------------------------------------------*/
    void Destroy()
    {
        NW_ASSERT( m_PriorityList.IsEmpty() );

        char* ptr = static_cast<char*>(m_pBuffer);
        const unsigned long numObjects = m_BufferSize / sizeof(Sound);
        for ( unsigned long i = 0; i < numObjects; i++ )
        {
            Sound* sound = reinterpret_cast<Sound*>(ptr);
            sound->~Sound();
            ptr += sizeof(Sound);
        }

        m_FreeList.Clear();
        m_PriorityList.Clear();
    }

    /*---------------------------------------------------------------------------*
      Name:         Alloc

      Description:  インスタンスを確保する
                    マネージャーからインスタンスを確保します。

      Arguments:    priority - プライオリティ
                    ambientPriority - アンビエントプライオリティ

      Returns:      確保したインスタンス、確保できなければNULL
     *---------------------------------------------------------------------------*/
    Sound* Alloc( int priority, int ambientPriority )
    {
        int allocPriority = priority + ambientPriority;
        allocPriority = ut::Clamp( allocPriority, BasicSound::PRIORITY_MIN, BasicSound::PRIORITY_MAX );

        Sound* sound = NULL;
        while( sound == NULL )
        {
            if ( ! m_FreeList.IsEmpty() )
            {
                sound = &m_FreeList.GetFront();
                m_FreeList.PopFront();
            }
            else
            {
                // get lowest priority sound
                Sound* lowPrioSound = GetLowestPrioritySound();
                if ( lowPrioSound == NULL ) return NULL;
                if ( allocPriority < lowPrioSound->CalcCurrentPlayerPriority() ) return NULL;

                NW_WARNING(
                    ! internal::Debug_GetWarningFlag(
                        internal::Debug_GetDebugWarningFlagFromSoundType(
                            lowPrioSound->GetSoundType() ) ),
                    "Sound (id:0x%08x) is stopped for not enough %s sound instance.",
                    lowPrioSound->GetId(),
                    internal::Debug_GetSoundTypeString( lowPrioSound->GetSoundType() )
                );
                lowPrioSound->Stop( 0 );
            }
        }

        sound->Initialize();
        sound->SetPriority( priority, ambientPriority );

        InsertPriorityList( sound, allocPriority );
        return sound;
    }

    /*---------------------------------------------------------------------------*
      Name:         Free

      Description:  インスタンスをマネージャーに解放する
                    指定したインスタンスをマネージャーに対して解放します。
                    解放するインスタンスは、あらかじめ停止されている必要があります。

      Arguments:    sound - インスタンス

      Returns:      無し
     *---------------------------------------------------------------------------*/
    void Free( Sound* sound )
    {
        NW_NULL_ASSERT( sound );

        RemovePriorityList( sound );

        sound->Finalize();
        m_FreeList.PushBack( sound );
    }

    /*---------------------------------------------------------------------------*
      Name:         UpdatePriority

      Description:  引数に指定したサウンドのプライオリティをプライオリティリストに
                    反映させる

      Arguments:    sound - インスタンス
                    priority - プライオリティ

      Returns:      無し
     *---------------------------------------------------------------------------*/
    void UpdatePriority( Sound* sound, int priority )
    {
        RemovePriorityList( sound );
        InsertPriorityList( sound, priority );
    }

    /*---------------------------------------------------------------------------*
      Name:         SortPriorityList

      Description:  プライオリティリストをソートする

      Arguments:    無し

      Returns:      無し
     *---------------------------------------------------------------------------*/
    void SortPriorityList()
    {
        if ( m_PriorityList.GetSize() < 2 ) return;

        static const int TMP_NUM =
            internal::BasicSound::PRIORITY_MAX - internal::BasicSound::PRIORITY_MIN + 1;
        PriorityList tmplist[ TMP_NUM ]; // notice: large stack

        while ( !m_PriorityList.IsEmpty() )
        {
            Sound& front = m_PriorityList.GetFront();
            m_PriorityList.PopFront();
            tmplist[ front.CalcCurrentPlayerPriority() ].PushBack( &front );
        }
        for ( int i=0; i<TMP_NUM; i++ )
        {
            while ( !tmplist[i].IsEmpty() )
            {
                Sound& front = tmplist[i].GetFront();
                tmplist[i].PopFront();
                m_PriorityList.PushBack( &front );
            }
        }
    }

    /*---------------------------------------------------------------------------*
      Name:         GetLowestPrioritySound

      Description:  最もプライオリティの低いサウンドを取得する

      Arguments:    なし

      Returns:
     *---------------------------------------------------------------------------*/
    Sound* GetLowestPrioritySound()
    {
        if ( m_PriorityList.IsEmpty() ) return NULL;
        return &m_PriorityList.GetFront();
    }

    /*---------------------------------------------------------------------------*
      Name:         GetActiveCount

      Description:  現在アクティブなサウンド数を取得する

      Arguments:    なし

      Returns:      サウンド数
     *---------------------------------------------------------------------------*/
    unsigned long GetActiveCount() const
    {
        return m_PriorityList.GetSize();
    }

    /*---------------------------------------------------------------------------*
      Name:         GetFreeCount

      Description:  マネージャーからAlloc可能な残りインスタンス数を取得する

      Arguments:    無し

      Returns:      インスタンス数
     *---------------------------------------------------------------------------*/
    int GetFreeCount() const { return m_FreeList.GetSize(); }

    const PriorityList& GetSoundList() const { return m_PriorityList; }
    PriorityList& GetFreeList() { return m_FreeList; }

private:
    void InsertPriorityList( Sound* sound, int priority )
    {
        // プライオリティリストへ追加
        Iterator itr = m_PriorityList.GetBeginIter();
        while ( itr != m_PriorityList.GetEndIter() )
        {
            if ( priority < itr->CalcCurrentPlayerPriority() ) break;
            (void)++itr;
        }
        m_PriorityList.Insert( itr, sound );
    }
    void RemovePriorityList( Sound* sound ) { m_PriorityList.Erase( sound ); }

    void* m_pBuffer;
    u32 m_BufferSize;

    // NOTE: m_PriorityList と m_FreeList はどちらも、Sound::m_PriorityLink
    //       を管理するが、同時に 2 つのリストの管理下に置かれることは無いので、
    //       問題ない
    PriorityList m_PriorityList;
    PriorityList m_FreeList;
};

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SOUND_INSTANCE_MANAGER_H_ */

