﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_SoundArchivePlayer.h
 *
 * @file snd_SoundArchivePlayer.h
 */

#ifndef NW_SND_SOUND_ARCHIVE_PLAYER_H_
#define NW_SND_SOUND_ARCHIVE_PLAYER_H_

#include <nw/snd/snd_SoundStartable.h>
#include <nw/snd/snd_NoteOnCallback.h>
#include <nw/snd/snd_SoundArchive.h>
#include <nw/snd/snd_BasicSound.h>
#include <nw/snd/snd_SequenceSoundPlayer.h> // SequenceUserprocCallback
#include <nw/snd/snd_SequenceSound.h>       // SequenceSoundInstanceManager
#include <nw/snd/snd_WaveSound.h>           // WaveSoundInstanceManager
#include <nw/snd/snd_StreamSound.h>         // StreamSoundInstanceManager
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_SoundInstanceManager.h>
#include <nw/snd/snd_MmlSequenceTrackAllocator.h>
#include <nw/snd/snd_MmlParser.h>

namespace nw {
namespace snd {

class SoundHandle;
class SoundHeap;
class SoundActor;
class SoundPlayer;
class SoundDataManager;

namespace internal {

class ExternalSoundPlayer;
class Channel;
class StreamSound;
class WaveSound;
class PlayerHeap;
class SoundArchiveFilesHook;

}

//---------------------------------------------------------------------------
//! @brief  サウンドアーカイブ中のサウンドを再生するクラスです。
//!
//!         SoundArchivePlayer で再生したサウンドは、
//!         ハンドルクラスを用いて操作することができます。
//!
//!         ダウンロード販売・追加コンテンツ・パッチなどでサウンドアーカイブファイル
//!         (*.bfsar) やストリームサウンドファイル (*.bfstm) を SD カードに配置する場合は、
//!         次の点に注意してください。
//!
//!         SD カード上にあるストリームサウンドファイル (*.bfstm) を再生したり、
//!         SD カード上にあるサウンドアーカイブファイル (*.bfsar) の中のデータを
//!         プレイヤーヒープを使って頻繁に再生すると、
//!         場合によっては SD カードの「データ化け」を引き起こす可能性があります。
//!
//!         「データ化け」を抑制するには、
//!         「頻繁に再生する音をプレイヤーヒープを使って鳴らさない」
//!         「ストリームサウンド再生用にファイルリードキャッシュを利用するようにする」
//!         などの方法があります。
//!
//!         ストリームサウンド再生用のファイルリードキャッシュについては、
//!         @ref InitializeParam 構造体を渡す @ref Initialize 関数の説明をご確認ください。
//!
//! @see SoundHandle クラス
//! @see SoundStartable インターフェイス
//!
//! @date 2012/06/14 SD カード上のデータの「データ化け」について追記
//! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
class SoundArchivePlayer : public SoundStartable
{
private:
    static const int DEFAULT_STREAM_BLOCK_COUNT = 5;

    /* ------------------------------------------------------------------------
            SeqCallback class
       ------------------------------------------------------------------------ */
    class SequenceNoteOnCallback : public internal::driver::NoteOnCallback
    {
        NW_DISALLOW_COPY_AND_ASSIGN(SequenceNoteOnCallback);

    public:
        SequenceNoteOnCallback() : m_pSoundArchivePlayer( NULL ) { }
        void Initialize( const SoundArchivePlayer& player )
        {
            m_pSoundArchivePlayer = &player;
        }

        virtual internal::driver::Channel* NoteOn(
            internal::driver::SequenceSoundPlayer* seqPlayer,
            u8 bankIndex,
            const internal::driver::NoteOnInfo& noteOnInfo
        );

    private:
        const SoundArchivePlayer* m_pSoundArchivePlayer;
    };
    friend class SoundArchivePlayer::SequenceNoteOnCallback;

    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */
public:
    //----------------------------------------
    //! @name コンストラクタ/デストラクタ
    //@{
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    SoundArchivePlayer();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    virtual ~SoundArchivePlayer();
    //@}

    //----------------------------------------
    //! @name 初期化
    //@{
    //---------------------------------------------------------------------------
    //! @brief      初期化に必要なメモリのサイズを取得します。
    //!
    //!             userParamSizePerSound 引数については、
    //!             @ref Initialize の説明も参照してください。
    //!
    //! @param[in]    arc  プレイヤーで使用するサウンドアーカイブです。
    //! @param[in]    userParamSizePerSound 各サウンドインスタンスに持たせるユーザーパラメータのバッファサイズです。
    //!
    //! @return       初期化に必要なメモリサイズを返します。
    //!
    //! @see Initialize
    //!
    //! @date 2012/07/24 userParamSizePerSound を追加
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    size_t GetRequiredMemSize( const SoundArchive* arc, size_t userParamSizePerSound = 0 ) const;

    //---------------------------------------------------------------------------
    //! @brief        初期化に必要なストリームバッファのサイズを取得します。
    //!
    //!               @ref Initialize に渡すストリームバッファの最小サイズを取得します。
    //!               この関数で取得した値より大きいサイズ (4 倍まで可)
    //!               のバッファを渡す必要があります。
    //!
    //! @param[in]    arc             プレイヤーで使用するサウンドアーカイブです。
    //!
    //! @return       初期化に必要なストリームバッファサイズを返します。
    //!
    //! @see Initialize
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    size_t GetRequiredStreamBufferSize( const SoundArchive* arc ) const;

    //---------------------------------------------------------------------------
    //! @brief        初期化に必要なストリームバッファサイズの倍率を取得します。
    //!
    //!               @ref Initialize に渡すストリームバッファについて、
    //!               @ref GetRequiredStreamBufferSize で得られる最小サイズの何倍が必要かを取得します。
    //!               この関数で取得した倍率より大きいサイズのバッファを渡す必要があり、
    //!               不足している場合はプリフェッチデータを一部活用できない場合があります。
    //!
    //! @param[in]    arc             プレイヤーで使用するサウンドアーカイブです。
    //!
    //! @return       初期化に必要なストリームバッファサイズの倍率を返します。
    //!
    //! @see Initialize
    //! @see GetRequiredStreamBufferSize
    //!
    //! @date 2013/11/11 初版
    //---------------------------------------------------------------------------
    s32 GetRequiredStreamBufferTimes( const SoundArchive* arc ) const;

    //---------------------------------------------------------------------------
    //! @brief  サウンドアーカイブプレイヤーを初期化します。
    //!
    //!         サウンドアーカイブプレイヤーを使用する前に、
    //!         初期化を行う必要があります。
    //!         初期化を行うことにより、
    //!         サウンドアーカイブプレイヤーがサウンドアーカイブと
    //!         サウンドデータマネージャ関連付けられ、
    //!         そのサウンドアーカイブ中のデータを再生できるようになります。
    //!
    //!         サウンドアーカイブプレイヤーが必要とするメモリのサイズは
    //!         @ref GetRequiredMemSize 及び @ref GetRequiredStreamBufferSize
    //!         で取得することができます。
    //!
    //!         必要なストリームバッファサイズが 0 の場合、
    //!         ストリームバッファへのポインタ strmBuffer に NULL を渡すことができます。
    //!
    //!         ストリームバッファを @ref GetRequiredStreamBufferSize
    //!         で取得したサイズより大きく設定すると、
    //!         ストリームサウンドの音途切れが発生しにくくなりますが、
    //!         データをロードしてから実際に再生するまでの遅延時間が大きくなります。
    //!
    //!         ストリームバッファは、@ref GetRequiredStreamBufferSize で取得したサイズの
    //!         4 倍まで設定することができます。
    //!         4 倍より大きなサイズを設定すると、Debug 版 / Develop 版ではアサートで
    //!         停止します。Release 版での挙動は不定です。
    //!
    //!         userParamSizePerSound 引数を使うと、
    //!         各サウンドインスタンスにユーザーパラメータ (ユーザーが自由に読み書きできる領域)
    //!         を持たせることができます。
    //!
    //!         この「ユーザーパラメータ」は、SoundMaker で設定する「ユーザーパラメータ」とは異なります。
    //!         SoundMaker で設定する「ユーザーパラメータ」は各サウンド ID にひも付けられるパラメータで
    //!         @ref SoundArchive::ReadSoundUserParam で取得することができるリードオンリーのデータです。
    //!         一方、ここで扱う「ユーザーパラメータ」は、
    //!         各サウンドインスタンスにひも付けられるパラメータで、
    //!         プログラム内で読み書きできるという違いがあります。
    //!
    //!         また、SoundMaker のユーザーパラメータは u32 を 4 つまでしか保持できませんが、
    //!         こちらは、userParamSizePerSound 引数で渡したサイズだけ値を保持することが可能です。
    //!
    //!         userParamSizePerSound に 1 以上の値を渡して初期すると、サウンド再生時に
    //!         @ref SoundHandle::GetUserParam 関数を使うことで、
    //!         ユーザーパラメータを取得することができます。
    //!         userParamSizePerSound に 0 を渡すと、SoundHandle:*GetUserParam は NULL が返ります。
    //!         ユーザーパラメータは、サウンド再生開始時にゼロクリアされます。
    //!
    //!         各サウンドインスタンスが保持しているユーザーパラメータの実際のサイズは、
    //!         @ref GetSoundUserParamSize で取得することができます。
    //!
    //!         userParamSizePerSound には、@ref GetRequiredMemSize の userParamSizePerSound
    //!         引数と同じ値を渡す必要があります。
    //!
    //! @param[in]    arc             プレイヤーで使用するサウンドアーカイブです。
    //! @param[in]    manager         プレイヤーで使用するサウンドデータマネージャです。
    //! @param[in]    buffer          バッファへのポインタです。
    //!                               32 バイト境界に配置されている必要があります。
    //! @param[in]    size            バッファサイズです。
    //! @param[in]    strmBuffer      ストリームバッファへのポインタです。
    //!                               8 バイト境界に配置されている必要があります。
    //! @param[in]    strmBufferSize  ストリームバッファのサイズです。
    //! @param[in]    userParamSizePerSound  1サウンドあたりのユーザーパラメータのサイズです。
    //!
    //! @return       初期化に成功したら true を、失敗したら false を返します。
    //!
    //! @see GetRequiredMemSize
    //! @see GetRequiredStreamBufferSize
    //! @see Finalize
    //! @see IsAvailable
    //!
    //! @date 2012/07/24 userParamSizePerSound 引数を追加
    //! @date 2011/09/09 buffer および strmBuffer のアラインメントの記述を修正
    //!                  (buffer:128バイト→32バイト、strmBuffer:128バイト→8バイト)
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool Initialize(
        const SoundArchive* arc,
        const SoundDataManager* manager,
        void*   buffer,
        u32     size,
        void*   strmBuffer,
        u32     strmBufferSize,
        size_t  userParamSizePerSound = 0
    );

    //---------------------------------------------------------------------------
    //! @brief  サウンドアーカイブプレイヤーの初期化に使用する構造体です。
    //!
    //!         本構造体を引数に取る @ref Initialize 関数を利用する際に、ご使用ください。
    //!
    //!         本構造体を利用した @ref Initialize 関数を呼ぶことで、
    //!         ストリームサウンド再生時のファイルリードキャッシュが有効になり、
    //!         ファイルシステムへのアクセス頻度を制御することができます。
    //!
    //!         userParamSizePerSound については、
    //!         @ref Initialize の説明も参照してください。
    //!
    //! @see Initialize
    //! @see GetRequiredMemSize
    //! @see GetRequiredStreamBufferSize
    //! @see GetRequiredStreamCacheSize
    //!
    //! @date 2012/07/24 userParamSizePerSound を追加
    //! @date 2012/06/14 初版
    //---------------------------------------------------------------------------
    struct InitializeParam
    {
        //! サウンドアーカイブプレイヤーで使用するサウンドアーカイブです。
        const SoundArchive* soundArchive;

        //! サウンドアーカイブプレイヤーで使用するサウンドデータマネージャです。
        const SoundDataManager* soundDataManager;

        //! サウンドアーカイブプレイヤー内で生成するインスタンスやプレイヤーヒープに使われるバッファです。
        //! @ref GetRequiredMemSize で得られたサイズのメモリ領域を渡す必要があります。
        //! 32 バイト境界に配置されている必要があります。
        void* setupBuffer;

        //! setupBuffer のサイズです。
        //! @ref GetRequiredMemSize で得られたサイズを渡す必要があります。
        u32 setupBufferSize;

        //! ストリームサウンド再生に使われるバッファです。
        //! @ref GetRequiredStreamBufferSize で得られたサイズか、その 4 倍までのサイズのメモリ領域を渡す必要があります。
        //! GetRequiredStreamBufferSize で得られたサイズが 0 の場合は、NULL を指定します。
        //! 8 バイト境界に配置されている必要があります。
        void* streamBuffer;

        //! streamBuffer のサイズです。
        //! @ref GetRequiredStreamBufferSize で得られたサイズか、その 4 倍までのサイズを渡す必要があります。
        u32 streamBufferSize;

        //! ストリームサウンド再生時のファイルリードキャッシュ用のバッファです。
        //! @ref GetRequiredStreamCacheSize で得られたサイズを渡す必要があります。
        //! FsSoundArchive::BUFFER_ALIGN_SIZE バイト境界に配置されている必要があります。
        void* streamCacheBuffer;

        //! streamCacheBuffer のサイズです。
        //! @ref GetRequiredStreamCacheSize で得られたサイズを渡す必要があります。
        u32 streamCacheSize;

        //! 1サウンドあたりのユーザーパラメータのサイズです。
        size_t userParamSizePerSound;

        //! @brief コンストラクタです
        //! @date 2012/06/14 初版
        InitializeParam()
        : soundArchive(NULL)
        , soundDataManager(NULL)
        , setupBuffer(NULL)
        , setupBufferSize(0)
        , streamBuffer(NULL)
        , streamBufferSize(0)
        , streamCacheBuffer(NULL)
        , streamCacheSize(0)
        , userParamSizePerSound(0)
        {}
    };

    //---------------------------------------------------------------------------
    //! @brief      ストリームサウンド再生時のファイルリードキャッシュ用のバッファサイズを取得します。
    //!
    //!             サウンドアーカイブ (*.bfsar) で定義されているストリームサウンドの数
    //!             (SoundMaker の [プロジェクトの設定] - [サウンドアーカイブ] - [ストリームサウンドの数]
    //!             で指定します) に比例したサイズを返します。
    //!
    //!             cacheSizePerSound に渡すサイズの目安は下記のとおりです。
    //!
    //!             - モノラルのストリームサウンドしか無い場合は、8KB の倍数
    //!             - ステレオ～ (マルチトラックストリームを含む) がある場合は、16KB の倍数
    //!
    //!             キャッシュを利用することで、以下のメリット・デメリットがあります。
    //!
    //!             ■メリット @n
    //!             - ストリームサウンドファイルが SD カード上にある場合は、
    //!               SD カードへのアクセス回数が減るためデータ化けが起こる可能性を抑制できる
    //!             - ストリーム再生時の音途切れを緩和できる
    //!               (次のステージのデータを裏読みしたり、ムービーを同時に再生したり、
    //!                複数のストリームサウンドを同時に再生したりするケースで音途切れを緩和できます)
    //!
    //!             ■デメリット @n
    //!             - キャッシュはすべてのストリームサウンドインスタンスに対して設定されるため、
    //!               キャッシュを必要としないサウンドに対してもキャッシュが設定され、メモリを余計に消費する
    //!
    //! @param[in]  arc                 サウンドアーカイブプレイヤーで利用するサウンドアーカイブです。
    //! @param[in]  cacheSizePerSound   1 ストリームサウンドあたりのファイルリードキャッシュサイズです。
    //!
    //! @return     ストリームサウンド再生時のファイルリードキャッシュ用のバッファサイズを返します。
    //!
    //! @see InitializeParam 構造体
    //!
    //! @date 2012/06/14 初版
    //---------------------------------------------------------------------------
    size_t GetRequiredStreamCacheSize(const SoundArchive* arc, size_t cacheSizePerSound) const;

    //---------------------------------------------------------------------------
    //! @brief      サウンドアーカイブプレイヤーを初期化します。
    //!
    //!             サウンドアーカイブプレイヤーを使用する前に、初期化を行う必要があります。
    //!             初期化を行うことによりサウンドアーカイブプレイヤーが、
    //!             サウンドアーカイブとサウンドデータマネージャが関連付けられ、
    //!             サウンドアーカイブ中のデータを再生できるようになります。
    //!
    //!             サウンドアーカイブプレイヤーで必要とするメモリのサイズは、
    //!             下記の API で求めることができます。
    //!
    //!             - @ref GetRequiredMemSize
    //!             - @ref GetRequiredStreamBufferSize
    //!             - @ref GetRequiredStreamCacheSize
    //!
    //!             上記の API で取得したメモリサイズと、該当するサイズのメモリ領域を
    //!             @ref InitializeParam に格納して、本関数に渡します。
    //!
    //!             ストリームバッファを @ref GetRequiredStreamBufferSize
    //!             で取得したサイズより大きく設定すると、
    //!             ストリームサウンドの音途切れが発生しにくくなりますが、
    //!             データをロードしてから実際に再生するまでの遅延時間が大きくなります。
    //!
    //!             ストリームバッファは、@ref GetRequiredStreamBufferSize で取得したサイズの
    //!             4 倍まで設定することができます。
    //!             4 倍より大きなサイズを設定すると、Debug 版 / Development 版ではアサートで
    //!             停止します。Release 版での挙動は不定です。
    //!
    //!             ファイルリードキャッシュを適切に設定することで、ストリームサウンド再生時の
    //!             ファイルシステムへのアクセス頻度を減らすことができます。
    //!             キャッシュ用のバッファサイズについては @ref GetRequiredStreamCacheSize
    //!             の説明をご参照ください。
    //!
    //! @param[in] param    初期化パラメータです。
    //!
    //! @see InitializeParam 構造体
    //! @see GetRequiredMemSize
    //! @see GetRequiredStreamBufferSize
    //! @see GetRequiredStreamCacheSize
    //! @see Finalize
    //! @see IsAvailable
    //!
    //! @return     初期化に成功したら true を、失敗したら false を返します。
    //!
    //! @date 2012/06/14 初版
    //---------------------------------------------------------------------------
    bool Initialize(const InitializeParam& param);

    //---------------------------------------------------------------------------
    //! @brief        サウンドアーカイブプレイヤーを破棄します。
    //!
    //!               破棄は以下のとおり行われます。
    //!
    //!               - このサウンドアーカイブプレイヤーを使って再生中のサウンドを停止します。
    //!               - 初期化で割り当てたメモリ領域を開放します。
    //!               - 初期化されたサウンドアーカイブとの関連が外れます。
    //!
    //! @see Initialize
    //! @see IsAvailable
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief        利用可能な状態かどうかを調べます。
    //!
    //!               サウンドアーカイブプレイヤーを利用するためには
    //!               @ref Initialize を呼び出して初期化を完了させる必要があります。
    //!               また、@ref Finalize を呼び出すと、
    //!               サウンドアーカイブプレイヤーの終了処理が行われ、
    //!               利用できない状態になります。
    //!
    //! @return       サウンドアーカイブプレイヤーが利用可能な状態なら true を、
    //!               そうでなければ false を返します。
    //!
    //! @see Initialize
    //! @see Finalize
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool IsAvailable() const;
    //@}

    //----------------------------------------
    //! @name 更新
    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブプレイヤーの更新処理を行います。
    //!
    //!           ビデオフレームごとに呼び出す必要はありませんが、
    //!           1 ゲームフレームに 1 回など、定期的に呼び出してください。
    //!
    //!           この関数では、プレイヤーやサウンドに対するパラメータ設定や、
    //!           フェイドイン・フェイドアウトなどの連続的なパラメータ変更を
    //!           反映するための処理が行われるため、呼び出し頻度が変えると、
    //!           フェードインなどの時間が変わってしまいます。
    //!
    //! @see SoundPlayer::StopAllSound
    //! @see SoundPlayer::PauseAllSound
    //! @see SoundActor::StopAllSound
    //! @see SoundActor::PauseAllSound
    //! @see Sound3DActor::StopAllSound
    //! @see Sound3DActor::PauseAllSound
    //! @see SoundHandle::Stop
    //! @see SoundHandle::Pause
    //! @see SoundHandle::FadeIn
    //! @see SoundHandle::SetVolume
    //! @see SoundHandle::GetRemainingFadeFrames
    //! @see SoundHandle::GetRemainingPauseFadeFrames
    //! @see StreamSoundHandle::Stop
    //! @see StreamSoundHandle::Pause
    //! @see StreamSoundHandle::FadeIn
    //! @see StreamSoundHandle::SetVolume
    //! @see StreamSoundHandle::SetTrackVolume
    //! @see WaveSoundHandle::Stop
    //! @see WaveSoundHandle::Pause
    //! @see WaveSoundHandle::FadeIn
    //! @see WaveSoundHandle::SetVolume
    //! @see SequenceSoundHandle::Stop
    //! @see SequenceSoundHandle::Pause
    //! @see SequenceSoundHandle::FadeIn
    //! @see SequenceSoundHandle::SetVolume
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void Update();
    //@}

    //----------------------------------------
    //! @name インスタンス取得

    //---------------------------------------------------------------------------
    //! @brief    プレイヤーに関連付けられているサウンドアーカイブを取得します。
    //!
    //! @return   プレイヤーに関連付けられているサウンドアーカイブを返します。
    //!
    //! @see SoundArchive クラス
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    const SoundArchive& GetSoundArchive() const;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のサウンドプレイヤーを取得します。
    //!
    //! @param[in]    playerId    プレイヤーの ID です。
    //!
    //! @return   指定した ID のサウンドプレイヤーを返します。
    //!
    //! @see SoundPlayer クラス
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    SoundPlayer& GetSoundPlayer( SoundArchive::ItemId playerId );

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のサウンドプレイヤーを取得します。
    //!
    //! @param[in]    pPlayerName     プレイヤーのラベル文字列です。
    //!
    //! @return   指定した ID のサウンドプレイヤーを返します。
    //!
    //! @see SoundPlayer クラス
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    SoundPlayer& GetSoundPlayer( const char* pPlayerName );

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のサウンドプレイヤーを取得します。
    //!
    //! @param[in]    playerId    プレイヤーの ID です。
    //!
    //! @return   指定した ID のサウンドプレイヤーを返します。
    //!
    //! @see SoundPlayer クラス
    //!
    //! @date 2012/02/08 初版
    //---------------------------------------------------------------------------
    const SoundPlayer& GetSoundPlayer( SoundArchive::ItemId playerId ) const;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のサウンドプレイヤーを取得します。
    //!
    //! @param[in]    pPlayerName     プレイヤーのラベル文字列です。
    //!
    //! @return   指定した ID のサウンドプレイヤーを返します。
    //!
    //! @see SoundPlayer クラス
    //!
    //! @date 2012/02/08 初版
    //---------------------------------------------------------------------------
    const SoundPlayer& GetSoundPlayer( const char* pPlayerName ) const;

    //---------------------------------------------------------------------------
    //! @brief  各サウンドインスタンスが保持するユーザーパラメータのサイズを取得します。
    //!
    //!         ユーザーパラメータについては、
    //!         @ref Initialize の説明を参照してください。
    //!
    //!         @ref Initialize 関数に渡した userParamSizePerSound を 4 の倍数境界まで
    //!         切り上げた値が返ります。
    //!
    //! @return ユーザーパラメータのサイズを返します。
    //!
    //! @see Initialize
    //!
    //! @date 2012/07/24 初版
    //---------------------------------------------------------------------------
    size_t GetSoundUserParamSize() const { return m_SoundUserParamSize; }
    //@}

    //----------------------------------------
    //! @name インスタンス数取得

    //---------------------------------------------------------------------------
    //! @brief    サウンドプレイヤーの個数を取得します。
    //!
    //! @return   サウンドプレイヤーの個数を返します。
    //!
    //! @see SoundPlayer クラス
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    u32 GetSoundPlayerCount() const { return m_SoundPlayerCount; }

    //---------------------------------------------------------------------------
    //! @brief    再生可能なシーケンスサウンドの残数を取得します。
    //!
    //!           残数が 0 のときに新たにシーケンスサウンドを再生すると、
    //!           既に再生中の最もプレイヤープライオリティの低いサウンドが停止します。
    //!           ただし、新たに再生するサウンドのほうがプレイヤープライオリティが低い場合は、
    //!           再生に失敗します。
    //!
    //!           再生可能なシーケンスサウンドの総数は、
    //!           サウンドアーカイブに記述されていて、
    //!           @ref Initialize でサウンドアーカイブプレイヤーに設定されます。
    //!
    //! @return   再生可能なシーケンスサウンドの残数を返します。
    //!
    //! @see Initialize
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    int GetFreeSequenceSoundCount() const
    {
        return m_SequenceSoundInstanceManager.GetFreeCount();
    }

    //---------------------------------------------------------------------------
    //! @brief    再生可能なウェーブサウンドの残数を取得します。
    //!
    //!           残数が 0 のときに新たにウェーブサウンドを再生すると、
    //!           既に再生中の最もプレイヤープライオリティの低いサウンドが停止します。
    //!           ただし、新たに再生するサウンドのほうがプレイヤープライオリティが低い場合は、
    //!           再生に失敗します。
    //!
    //!           再生可能なウェーブサウンドの総数は、
    //!           サウンドアーカイブに記述されていて、
    //!           @ref Initialize でサウンドアーカイブプレイヤーに設定されます。
    //!
    //! @return   再生可能なウェーブサウンドの残数を返します。
    //!
    //! @see Initialize
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    int GetFreeWaveSoundCount() const
    {
        return m_WaveSoundInstanceManager.GetFreeCount();
    }

    //---------------------------------------------------------------------------
    //! @brief    再生可能なストリームサウンドの残数を取得します。
    //!
    //!           残数が 0 のときに新たにストリームサウンドを再生すると、
    //!           既に再生中の最もプレイヤープライオリティの低いサウンドが停止します。
    //!           ただし、新たに再生するサウンドのほうがプレイヤープライオリティが低い場合は、
    //!           再生に失敗します。
    //!
    //!           再生可能なストリームサウンドの総数は、
    //!           サウンドアーカイブに記述されていて、
    //!           @ref Initialize でサウンドアーカイブプレイヤーに設定されます。
    //!
    //! @return   再生可能なストリームサウンドの残数を返します。
    //!
    //! @see Initialize
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    int GetFreeStreamSoundCount() const
    {
        return m_StreamSoundInstanceManager.GetFreeCount();
    }
    //@}

    //----------------------------------------
    //! @name シーケンスサウンドカスタマイズ

    //---------------------------------------------------------------------------
    //! @brief  シーケンスコマンド 'userproc' で呼び出される
    //!         コールバック関数を登録します。
    //!
    //!         ここで登録したコールバックは、シーケンスデータ側のシーケンスコマンド
    //!         'userproc' が処理されたフレームで呼び出されます。
    //!         コールバック関数はサウンドスレッド呼び出されます。
    //!
    //!         'userproc' コマンドの詳細については、
    //!         シーケンスデータマニュアルを参照してください。
    //!
    //!         ここで設定したコールバック関数はサウンドスレッドから呼び出されます。
    //!         排他制御が必要な場合は、
    //!         @ref SoundSystem::SoundThreadScopedLock クラスあるいは、
    //!         @ref SoundSystem::LockSoundThread / @ref SoundSystem::UnlockSoundThread 関数を利用する必要があります。
    //!
    //!         コールバック関数はすみやかに処理を終える必要があります。
    //!         処理が長引くと、ノイズが発生する可能性が高くなります。
    //!         たとえば、コールバック関数内でブロックする可能性のある API
    //!         (クリティカルセクションなど) を呼び出すと、
    //!         処理を終えるのに時間がかかる可能性があります。
    //!
    //! @param[in] callback       登録するコールバック関数です。
    //! @param[in] callbackArg    コールバック関数に渡されるユーザー引数です。
    //!
    //! @see SequenceUserprocCallback
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetSequenceUserprocCallback(
            SequenceUserprocCallback callback, void* callbackArg );

    //---------------------------------------------------------------------------
    //! @brief  シーケンスサウンドの途中再生におけるスキップ処理の空回し幅を指定します。
    //!
    //!         @ref SoundStartable::StartInfo 構造体の startOffset で開始オフセットを指定した上で、
    //!         @ref StartSound を呼び出すと、シーケンスサウンドを途中から再生することができますが、
    //!         シーケンスの空回しのために処理がかかります。
    //!
    //!         デフォルトでは、1 サウンドフレーム (約 5ms) あたり最大
    //!         48 * 4 * 4 tick (シーケンスデータ上の tick です。nn::os::Tick の tick ではありません)
    //!         だけ空回ししますが、この結果サウンドスレッドの処理負荷が上がり、
    //!         音途切れが生じることがあります。
    //!
    //!         この場合、48 * 4 * 4 より小さい値を本関数で設定すると、
    //!         音途切れが緩和する可能性があります。
    //!
    //! @param[in] intervalTick     1 サウンドフレームあたりの最大空回し幅 (単位は tick) です。
    //!
    //! @see GetSequenceSkipIntervalTick
    //!
    //! @date 2011/10/13 初版
    //---------------------------------------------------------------------------
    static void SetSequenceSkipIntervalTick( s32 intervalTick );

    //---------------------------------------------------------------------------
    //! @brief  シーケンスサウンドの途中再生における、スキップ処理の空回し幅を取得します。
    //!
    //! @return シーケンスサウンドの途中再生における、スキップ処理の空回し幅を返します。
    //!
    //! @see SetSequenceSkipIntervalTick
    //!
    //! @date 2011/10/13 初版
    //---------------------------------------------------------------------------
    static s32 GetSequenceSkipIntervalTick();
    //@}


    // StartSound実装
    //! @briefprivate
    //! @param handle :private
    //! @param soundId :private
    //! @param ambientArgInfo :private
    //! @param actor :private
    //! @param holdFlag :private
    //! @param startInfo :private
    //! @return :private
    StartResult detail_SetupSoundImpl(
        SoundHandle* handle,
        u32 soundId,
        internal::BasicSound::AmbientInfo* ambientArgInfo,
        SoundActor* actor,
        bool holdFlag,
        const StartInfo* startInfo
    );

    //! @briefprivate
    //! @param pString :private
    //! @return :private
    virtual SoundArchive::ItemId detail_GetItemId( const char* pString )
    {
        NW_NULL_ASSERT( m_pSoundArchive );
        return m_pSoundArchive->GetItemId( pString );
    }

    //! @briefprivate
    //! @param isEnable :private
    //! @return :private
    void detail_EnableWarningPrint(bool isEnable)
    {
        m_IsEnableWarningPrint = isEnable;
    }
    //! @briefprivate
    //! @return :private
    bool detail_IsEnableWarningPrint() const
    {
        return m_IsEnableWarningPrint;
    }


    // 非公開関数
    //! @briefprivate
    //! @param parser :private
    void detail_SetMmlParser( internal::driver::MmlParser* parser )
    {
        if ( parser == NULL ) m_MmlSequenceTrackAllocator.SetMmlParser( &m_MmlParser );
        else m_MmlSequenceTrackAllocator.SetMmlParser( parser );
    }
    //! @briefprivate
    //! @param fileId :private
    //! @return :private
    const void* detail_GetFileAddress( SoundArchive::FileId fileId ) const;

    //! @briefprivate
    //! @return :private
    internal::SoundArchiveFilesHook* detail_GetSoundArchiveFilesHook() const
    {
        return m_pSoundArchiveFilesHook;
    }

    //! @briefprivate
    //! @param filesHook :private
    void detail_SetSoundArchiveFilesHook(internal::SoundArchiveFilesHook* filesHook)
    {
        m_pSoundArchiveFilesHook = filesHook;
    }

    const internal::SequenceSoundInstanceManager& detail_GetSequenceSoundInstanceManager() const
    {
            return m_SequenceSoundInstanceManager;
    }
    const internal::WaveSoundInstanceManager& detail_GetWaveSoundInstanceManager() const
    {
            return m_WaveSoundInstanceManager;
    }
    const internal::StreamSoundInstanceManager& detail_GetStreamSoundInstanceManager() const
    {
            return m_StreamSoundInstanceManager;
    }

protected:
    virtual StartResult detail_SetupSound(
        SoundHandle* handle,
        u32 soundId,
        bool holdFlag,
        const StartInfo* startInfo
    );

private:

    template< typename Sound >
    Sound* AllocSound(
        internal::SoundInstanceManager< Sound >* manager,
        SoundArchive::ItemId soundId,
        int priority,
        int ambientPriority,
        internal::BasicSound::AmbientInfo* ambientArgInfo
    );

    StartResult PrepareSequenceSoundImpl(
        internal::SequenceSound* sound,
        const SoundArchive::SoundInfo* commonInfo,
        const SoundArchive::SequenceSoundInfo* info,
        SoundStartable::StartInfo::StartOffsetType startOffsetType,
        int startOffset,
        int delayTime,
        const StartInfo::SeqSoundInfo* externalSeqInfo
    );

    StartResult PrepareStreamSoundImpl(
        internal::StreamSound* sound,
        const SoundArchive::SoundInfo* commonInfo,
        const SoundArchive::StreamSoundInfo* info,
        const SoundArchive::StreamSoundInfo2* info2,
        SoundStartable::StartInfo::StartOffsetType startOffsetType,
        int startOffset,
        int delayTime,
        const StartInfo::StreamSoundInfo* externalStrmInfo
    );

    StartResult PrepareWaveSoundImpl(
        internal::WaveSound* sound,
        const SoundArchive::SoundInfo* commonInfo,
        const SoundArchive::WaveSoundInfo* info,
        SoundStartable::StartInfo::StartOffsetType startOffsetType,
        int startOffset,
        int delayTime,
        const StartInfo::WaveSoundInfo* externalWsdInfo
    );

    bool IsSoundArchiveFileHooksEnabled() const;
    void LockSoundArchiveFileHooks();
    void UnlockSoundArchiveFileHooks();

    bool IsSequenceSoundEdited(const char* soundName) const;
    bool IsStreamSoundEdited(const char* soundName) const;
    bool IsWaveSoundEdited(const char* soundName) const;

    internal::PlayerHeap* CreatePlayerHeap(
            void** ppBuffer, const void* pEndAddress, size_t heapSize );

    // Setup
    bool SetupMram( const SoundArchive* arc, void* buffer, size_t size, size_t userParamSizePerSound );
    bool SetupSoundPlayer( const SoundArchive* arc, void** buffer, const void* endp );
    bool SetupSequenceSound( int numSounds, void** buffer, const void* endp );
    bool SetupSequenceTrack( int numTracks, void** buffer, const void* endp );
    bool SetupWaveSound( int numSounds, void** buffer, const void* endp );
    bool SetupStreamSound( int numSounds, void** buffer, const void* endp );
    bool SetupStreamBuffer( const SoundArchive* arc, void* buffer, size_t size );
    bool SetupUserParamForBasicSound(
            const SoundArchive::SoundArchivePlayerInfo& info, void** buffer, const void* endp,
            size_t userParamSizePerSound);

    void SetCommonSoundParam(
            internal::BasicSound* sound, const SoundArchive::SoundInfo* commonInfo );

    bool IsLoadWaveArchive( const void* bankFile, internal::LoadItemInfo* info ) const;

    const SoundArchive* m_pSoundArchive;

    // コールバック
    SequenceNoteOnCallback m_SequenceCallback;
    SequenceUserprocCallback m_SequenceUserprocCallback;
    void* m_pSequenceUserprocCallbackArg;

    // サウンドプレイヤー
    u32 m_SoundPlayerCount;
    SoundPlayer* m_pSoundPlayers;

    // マネージャー・アロケータ
    internal::SequenceSoundInstanceManager m_SequenceSoundInstanceManager;
    internal::driver::SequenceSoundLoaderManager m_SequenceSoundLoaderManager;
    internal::WaveSoundInstanceManager m_WaveSoundInstanceManager;
    internal::driver::WaveSoundLoaderManager m_WaveSoundLoaderManager;
    internal::StreamSoundInstanceManager m_StreamSoundInstanceManager;
    internal::driver::StreamSoundLoaderManager m_StreamSoundLoaderManager;

    internal::driver::SequenceTrackAllocator* m_pSequenceTrackAllocator;
    internal::driver::MmlSequenceTrackAllocator m_MmlSequenceTrackAllocator; // デフォルトのシーケンストラックアロケータ

    internal::driver::StreamBufferPool m_StreamBufferPool;

    // デフォルトのシーケンスパーサ
    internal::driver::MmlParser m_MmlParser;

    // バッファ情報記憶
    size_t m_SoundUserParamSize;

    // データマネージャ
    const SoundDataManager* m_pSoundDataManager;

    // フック
    internal::SoundArchiveFilesHook* m_pSoundArchiveFilesHook;

    bool m_IsEnableWarningPrint;
    bool m_IsInitialized;
    u8 padding;
};

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SOUND_ARCHIVE_PLAYER_H_ */

