﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SOUND_ARCHIVE_FILE_READER_H_
#define NW_SND_SOUND_ARCHIVE_FILE_READER_H_

#include <nw/snd/snd_SoundArchiveFile.h>
#include <nw/snd/snd_SoundArchive.h>

namespace nw {
namespace snd {
namespace internal {

class SoundArchiveFileReader
{
public:
    static const u32 SIGNATURE_FILE = NW_UT_MAKE_SIGWORD( 'F', 'S', 'A', 'R' );

    SoundArchiveFileReader();
    ~SoundArchiveFileReader() {}

    // ------------------------------------------------------------------------
    //  初期化
    void Initialize( const void* soundArchiveData );
    void Finalize();
    bool IsStreamSendAvailable() const;
    bool IsFilterSupportedVersion() const;
    bool IsStreamPrefetchAvailable() const;

    // ブロック情報設定
    void SetStringBlock( const void* stringBlock );
    void SetInfoBlock( const void* infoBlock );

    // ブロック情報取得 (Initialize 後に有効)
    inline u32 GetStringBlockSize() const   { return m_Header.GetStringBlockSize(); }
    inline s32 GetStringBlockOffset() const { return m_Header.GetStringBlockOffset(); }
    inline u32 GetInfoBlockSize() const     { return m_Header.GetInfoBlockSize(); }
    inline s32 GetInfoBlockOffset() const   { return m_Header.GetInfoBlockOffset(); }
    inline u32 GetFileBlockSize() const     { return m_Header.GetFileBlockSize(); }
    inline s32 GetFileBlockOffset() const   { return m_Header.GetFileBlockOffset(); }

    // ------------------------------------------------------------------------
    //  文字列ブロックへのアクセス

    u32 GetStringCount() const;
    SoundArchive::ItemId GetItemId( const char* pStr ) const;
    const char* GetItemLabel( SoundArchive::ItemId id ) const;
    SoundArchive::FileId GetItemFileId( SoundArchive::ItemId id ) const;
    SoundArchive::FileId GetItemPrefetchFileId( SoundArchive::ItemId id ) const;

    // デバッグ
    void DumpTree() const;
    const char* GetString( SoundArchive::ItemId stringId ) const;    // 本当は private


    // ------------------------------------------------------------------------
    //  情報ブロックへのアクセス

    // 個数取得
    u32 GetSoundCount() const;
    u32 GetBankCount() const;
    u32 GetPlayerCount() const;
    u32 GetSoundGroupCount() const;
    u32 GetGroupCount() const;
    u32 GetWaveArchiveCount() const;
    u32 GetFileCount() const;

    // 情報取得
    bool ReadSoundInfo(
            SoundArchive::ItemId soundId,
            SoundArchive::SoundInfo* info ) const;
    bool ReadBankInfo(
            SoundArchive::ItemId bankId,
            SoundArchive::BankInfo* info ) const;
    bool ReadPlayerInfo(
            SoundArchive::ItemId playerId,
            SoundArchive::PlayerInfo* info ) const;
    bool ReadSoundGroupInfo(
            SoundArchive::ItemId soundGroupId,
            SoundArchive::SoundGroupInfo* info ) const;
    bool ReadGroupInfo(
            SoundArchive::ItemId groupId,
            SoundArchive::GroupInfo* info ) const;
    bool ReadWaveArchiveInfo(
            SoundArchive::ItemId warcId,
            SoundArchive::WaveArchiveInfo* info ) const;
    bool ReadFileInfo(
            SoundArchive::FileId id,
            SoundArchive::FileInfo* info, int index=0 ) const;
    bool ReadSoundArchivePlayerInfo(
            SoundArchive::SoundArchivePlayerInfo* info ) const;

    // 以下は、SoundInfo から直接アクセスする
    SoundArchive::SoundType GetSoundType( SoundArchive::ItemId soundId ) const;
    u32 GetSoundUserParam( SoundArchive::ItemId soundId ) const;
    bool ReadSoundUserParam( SoundArchive::ItemId soundId, int index, u32& value ) const;
    bool ReadSound3DInfo(
            SoundArchive::ItemId soundId,
            SoundArchive::Sound3DInfo* info ) const;
    bool ReadSequenceSoundInfo(
            SoundArchive::ItemId soundId,
            SoundArchive::SequenceSoundInfo* info ) const;
    bool ReadStreamSoundInfo(
            SoundArchive::ItemId soundId,
            SoundArchive::StreamSoundInfo* info ) const;
    bool ReadStreamSoundInfo2(
            SoundArchive::ItemId soundId,
            SoundArchive::StreamSoundInfo2* info ) const;
    bool ReadWaveSoundInfo(
            SoundArchive::ItemId soundId,
            SoundArchive::WaveSoundInfo* info ) const;

#if defined(NW_PLATFORM_CAFE)
    const Util::Table<ut::ResU32>* GetWaveArchiveIdTable(
#else
    const Util::Table<u32>* GetWaveArchiveIdTable(
#endif
            SoundArchive::ItemId id ) const;

#if defined(NW_PLATFORM_CAFE)
    const Util::Table<ut::ResU32>* GetAttachedGroupTable(
#else
    const Util::Table<u32>* GetAttachedGroupTable(
#endif
            SoundArchive::FileId id ) const;

    // TODO: 以下は、ファイルブロック周りで、対応がまだ
    // bool ReadFilePos( SoundArchive::FileId fileId, unsigned long index, SoundArchive::FilePos* info ) const;
    // bool ReadGroupItemInfo( SoundArchive::GroupId groupId, unsigned long index, SoundArchive::GroupItemInfo* info ) const;
    // const char* GetExtFilePath( u32 offset ) const;

private:
    SoundArchiveFile::FileHeader m_Header;
    const SoundArchiveFile::StringBlockBody* m_pStringBlockBody;
    const SoundArchiveFile::InfoBlockBody* m_pInfoBlockBody;
    const SoundArchiveFile::FileBlockBody* m_pFileBlockBody;
};

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

#endif /* NW_SND_SOUND_ARCHIVE_FILE_READER_H_ */

