﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_SoundArchive.h
 *
 * @file snd_SoundArchive.h
 */

#ifndef NW_SND_SOUND_ARCHIVE_H_
#define NW_SND_SOUND_ARCHIVE_H_

#include <nw/config.h>
#include <nw/types.h>
#include <nw/ut.h>
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_ItemType.h>
#include <nw/snd/snd_Config.h>
#include <nw/snd/snd_FsFileStream.h>

namespace nw {

namespace snd {

namespace internal {
class SoundArchiveFileReader;
class SoundArchiveParametersHook;
} // namespace nw::snd::internal


//---------------------------------------------------------------------------
//! @brief        サウンドアーカイブを扱うクラスです。
//!
//!               ユーザーがこのクラスを直接扱うことはできません。
//!               このクラスを継承した @ref FsSoundArchive クラス、
//!               または @ref MemorySoundArchive クラスを使用してください。
//!
//! @see FsSoundArchive
//! @see MemorySoundArchive
//!
//! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
class SoundArchive
{
public:
    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブの中でのアイテム ID を示す型です。
    //!
    //!           サウンド、バンク、プレイヤー、グループ、波形アーカイブに、
    //!           重複しないアイテム ID が割り振られています。
    //!
    //! @see GetItemLabel
    //! @see GetItemId
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    typedef u32 ItemId;

    //---------------------------------------------------------------------------
    //! @brief    無効な ID を表す値です。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static const ItemId INVALID_ID = 0xffffffff;

    //---------------------------------------------------------------------------
    //! @brief  サウンドのユーザーパラメータの最大数です。
    //!
    //!         @ref ReadSoundUserParam の index として渡せる最大値として利用できます。
    //!
    //! @ref ReadSoundUserParam
    //! @date 2011/10/03 初版
    //---------------------------------------------------------------------------
    static const int USER_PARAM_INDEX_MAX = 3;
        // snd_SoundArchiveFile.cpp の "USER_PARAM_COUNT - 1" が該当する

    //---------------------------------------------------------------------------
    //! @brief    サウンドの種類を表す型です。
    //!
    //! @see GetSoundType
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    enum SoundType
    {
        SOUND_TYPE_INVALID = 0,     //!< 不正な種類のサウンドです。

        SOUND_TYPE_SEQ,             //!< シーケンスサウンドです。
        SOUND_TYPE_STRM,            //!< ストリームサウンドです。
        SOUND_TYPE_WAVE             //!< ウェーブサウンドです。
    };

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief    ストリームファイルの種類を表す型です。
    //---------------------------------------------------------------------------
    enum StreamFileType
    {
        STREAM_FILE_TYPE_INVALID = 0,       //!< 不正なファイルです。

        STREAM_FILE_TYPE_NW_STREAM_BINARY,  //!< NintendoWare のストリームバイナリファイルです。

        STREAM_FILE_TYPE_ADTS               //!< ADTS形式 のストリームバイナリファイルです。
    };

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブの中でのファイル ID を示す型です。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    typedef ItemId FileId;

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブの中での文字列 ID を示す型です。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    typedef ItemId StringId;

    //---------------------------------------------------------------------------
    //! @brief  サウンドの設定情報の構造体です。
    //!
    //!         isFrontBypass が true で、@ref SoundSystem::SetOutputMode
    //!         で設定する出力モードが OUTPUT_MODE_SURROUND の場合、
    //!         当該サウンドはフロントチャンネルをバイパスします。
    //!
    //! @see ReadSoundInfo
    //! @see FileId
    //! @see ItemId
    //! @see PanMode
    //! @see PanCurve
    //!
    //! @date 2012/07/11 isFrontBypass 追加
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    struct SoundInfo
    {
        FileId fileId;      //!< サウンドアーカイブ内でのファイル ID です。
        ItemId playerId;    //!< サウンドを再生するときに使用されるプレイヤーの ID です。

        //! アクターでサウンドを再生するときに使用されるアクタープレイヤーの番号です。
        u8 actorPlayerId;
        u8 playerPriority;  //!< サウンドを再生するときのプレイヤープライオリティの値です。
        u8 volume;          //!< サウンドを再生するときの音量です。
        u8 remoteFilter;    //!< リモコンフィルタの値です。
        PanMode panMode;    //!< パンモードの種類です。
        PanCurve panCurve;  //!< パンのカーブの種類です。

        bool isFrontBypass; //!< フロントバイパスするか否かを示すフラグです。
    };

    //---------------------------------------------------------------------------
    //! @brief    サウンドの 3D サウンド設定情報の構造体です。
    //!
    //! @see ReadSound3DInfo
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    struct Sound3DInfo
    {
        //---------------------------------------------------------------------------
        //! @brief    3D 減衰カーブの形状を示す列挙体です。
        //!
        //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        enum DecayCurve
        {
            DECAY_CURVE_LOG    = 1, //!< 対数カーブです。
            DECAY_CURVE_LINEAR = 2  //!< リニア (直線) です。
        };

        //---------------------------------------------------------------------------
        //! @brief    3D サウンドで各パラメータを計算するかどうかを示すフラグです。
        //!
        //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        enum FlagCtrl
        {
            FLAG_CTRL_VOLUME   = (1 << 0), //!< 音量を計算するかどうかを示すフラグです。
            FLAG_CTRL_PRIORITY = (1 << 1), //!< プレイヤープライオリティを計算するかどうかを示すフラグです。
            FLAG_CTRL_PAN      = (1 << 2), //!< パンを計算するかどうかを示すフラグです。
            FLAG_CTRL_SPAN     = (1 << 3), //!< サラウンドパンを計算するかどうかを示すフラグです。
            FLAG_CTRL_FILTER   = (1 << 4)  //!< biquad フィルタを計算するかどうかを示すフラグです。
        };

        //---------------------------------------------------------------------------
        //! @brief    3D 計算で処理するパラメータを表現するビットフラグです。
        //!           FLAG_CTRL_*** の論理和で表現されます。
        //---------------------------------------------------------------------------
        u32 flags;
        f32 decayRatio;     //!< 3D 減衰率です。
        u8 decayCurve;      //!< 3D 減衰カーブの形状です。@ref DecayCurve が格納されます。
        u8 dopplerFactor;   //!< 3D ドップラーファクターです。
    };

    //---------------------------------------------------------------------------
    //! @brief    シーケンスサウンドに関連付けられるバンクの最大数です。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static const u32 SEQ_BANK_MAX = nw::snd::SEQ_BANK_MAX;

    //---------------------------------------------------------------------------
    //! @brief    シーケンスサウンドの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           シーケンスサウンドの情報を取得するための構造体です。
    //!
    //! @see ReadSequenceSoundInfo
    //!
    //! @date 2011/09/12 allocateTrackFlags について説明を追記
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    struct SequenceSoundInfo
    {
        u32 startOffset;                //!< 非公開です。ライブラリ内部で使用されています。
        u32 bankIds[ SEQ_BANK_MAX ];    //!< シーケンスサウンドで使用されるバンク ID です。

        //---------------------------------------------------------------------------
        //! @brief  シーケンスサウンドに含まれるトラックのビットフラグです。
        //!         下位ビットから順にトラック 0, トラック 1, トラック 2,... を表します。
        //!         有効なビットのトラックを操作することができます。
        //---------------------------------------------------------------------------
        u32 allocateTrackFlags;

        //! シーケンスサウンドで発音される音のチャンネルプライオリティの値です。
        u8  channelPriority;

        //---------------------------------------------------------------------------
        //! @brief    シーケンスサウンドで発音される音が、
        //!           リリース時にチャンネルプライオリティを固定されるかどうかのフラグです。
        //!           固定しない場合はリリース時にチャンネルプライオリティが下がります。
        //---------------------------------------------------------------------------
        bool isReleasePriorityFix;

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        SequenceSoundInfo()
        : startOffset( 0 ),
          allocateTrackFlags( 0 ),
          channelPriority( 0 ),
          isReleasePriorityFix( false )
        {
            for ( u32 i = 0; i < SEQ_BANK_MAX; i++ )
            {
                bankIds[ i ] = INVALID_ID;
            }
        }
    };


    //! @brief  ストリームサウンドの最大トラック数です。
    //! @date 2012/06/18    初版
    static const u32 STRM_TRACK_NUM = internal::STRM_TRACK_NUM;

    //! @brief  ストリームサウンドの各トラックの情報です。
    //! @date 2012/07/12    フロントバイパス対応、サラウンドパン対応
    //! @date 2012/06/18    初版
    struct StreamTrackInfo
    {
        u8 volume;  //!< トラックの音量です。
        u8 pan;     //!< トラックのパンです。
        u8 span;    //!< トラックのサラウンドパンです。
        u8 flags;   //!< トラックのサラウンドフラグです。
        u8 mainSend;            //!< メインバスのセンド量です。
        u8 fxSend[AUX_BUS_NUM]; //!< 補助バスのセンド量です。
        u8 lpfFreq;             //!< ローパスフィルタの値です。
        u8 biquadType;          //!< biquadフィルタの種類です。
        u8 biquadValue;         //!< biquadフィルタの値です。
        u8 channelCount; //!< トラックのチャンネル数です
        s8 globalChannelIndex[WAVE_CHANNEL_MAX]; //!< @briefprivate

        //! @brief コンストラクタです。
        //! @date 2012/06/18 初版
        StreamTrackInfo() : volume(0), pan(0), span(0), channelCount(0), mainSend(127), lpfFreq(64), biquadType(0), biquadValue(0)
        {
            std::memset(globalChannelIndex, -1, sizeof(s8)*WAVE_CHANNEL_MAX);
            std::memset(fxSend, 0, sizeof(u8)*AUX_BUS_NUM);
        }
    };

    //---------------------------------------------------------------------------
    //! @brief    ストリームサウンドの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           ストリームサウンドの情報を取得するための構造体です。
    //!
    //! @see ReadStreamSoundInfo
    //!
    //! @date 2011/09/12 初版
    //---------------------------------------------------------------------------
    struct StreamSoundInfo
    {
        //---------------------------------------------------------------------------
        //! @brief  ストリームサウンドに含まれるトラックのビットフラグです。
        //!         下位ビットから順にトラック 0, トラック 1, トラック 2, … を表します。
        //!         有効なビットのトラックを操作することができます。
        //---------------------------------------------------------------------------
        u16 allocateTrackFlags;

        //! ストリームサウンドに含まれるチャンネルの数です。
        u16 allocateChannelCount;

        //! ピッチです。
        f32 pitch;

        //! メインバスのセンド量です。
        u8  mainSend;

        //! 補助バスのセンド量です。
        u8  fxSend[AUX_BUS_NUM];

        //! ストリームサウンドのトラック情報です。
        StreamTrackInfo trackInfo[STRM_TRACK_NUM];

        //---------------------------------------------------------------------------
        //! @briefprivate
        //!
        //! @brief  ストリームファイルの種類です。
        //---------------------------------------------------------------------------
        StreamFileType streamFileType;

        //---------------------------------------------------------------------------
        //! @briefprivate
        //!
        //! @brief  プリフェッチストリームファイルのサウンドアーカイブ内でのファイル ID です。
        //---------------------------------------------------------------------------
        FileId prefetchFileId;

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //! @date 2011/09/12 初版
        //---------------------------------------------------------------------------
        StreamSoundInfo()
        : allocateTrackFlags( 0 ),
          allocateChannelCount( 0 ),
          pitch( 1.0f ),
          mainSend( 127 ),
          streamFileType( STREAM_FILE_TYPE_INVALID ),
          prefetchFileId( INVALID_ID )
        {
            std::memset(fxSend, 0, sizeof(u8)*AUX_BUS_NUM);
        }

        //---------------------------------------------------------------------------
        //! @brief 内部でトラックに対してチャンネルを割り当てるための計算などを行います。
        //!
        //!        トラック情報のチャンネル数を変更したときには必ず実行する必要があります。
        //!        詳しくは、externalFile デモをご確認ください。
        //!
        //! @date 2014/05/23 初版
        //---------------------------------------------------------------------------
        void Setup();
    };

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief    ストリームサウンド情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           ストリームサウンドの追加情報を取得するための構造体です。
    //!
    //! @see ReadStreamSoundInfo2
    //---------------------------------------------------------------------------
    struct StreamSoundInfo2
    {
        //---------------------------------------------------------------------------
        //! @brief  ループの有無です。
        //---------------------------------------------------------------------------
        bool isLoop;

        //---------------------------------------------------------------------------
        //! @brief  ループスタートフレームです。
        //---------------------------------------------------------------------------
        u32 loopStartFrame;

        //---------------------------------------------------------------------------
        //! @brief  ループエンドフレームです。
        //---------------------------------------------------------------------------
        u32 loopEndFrame;

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //---------------------------------------------------------------------------
        StreamSoundInfo2()
            : isLoop( false ),
            loopStartFrame( 0 ),
            loopEndFrame( 0 )
        {}
    };

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief    ウェーブサウンドの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           ウェーブサウンドの情報を取得するための構造体です。
    //!
    //! @see detail_ReadWaveSoundInfo
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    struct WaveSoundInfo
    {
        u32     index;              // ウェーブサウンドセットファイル (bcwsd) 内で何番目か
        u32     allocateTrackCount; // 現状は 1。今後マルチトラック化した場合、ほかの値が入る
        u8      channelPriority;
        bool    isReleasePriorityFix;

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        WaveSoundInfo()
        : allocateTrackCount( 0 ),
          channelPriority( 0 ),
          isReleasePriorityFix( false ) {}
    };

    //---------------------------------------------------------------------------
    //! @brief    バンクの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           バンクの情報を取得するための構造体です。
    //!
    //! @see ReadBankInfo
    //! @see FileId
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    struct BankInfo
    {
        FileId fileId;              //!< サウンドアーカイブ内でのファイル ID です。

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        BankInfo() : fileId( INVALID_ID ) {}
    };

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief    波形アーカイブの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           波形アーカイブの情報を取得するための構造体です。
    //!
    //! @see ReadWaveArchiveInfo
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    struct WaveArchiveInfo
    {
        u32  fileId;
        u32  waveCount;
        bool isLoadIndividual;
        u8   padding1;
        u16  padding2;

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        WaveArchiveInfo() : fileId( INVALID_ID ), isLoadIndividual( false ) {}
    };

    //---------------------------------------------------------------------------
    //! @brief    プレイヤーの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           プレイヤーの情報を取得するための構造体です。
    //!
    //! @see ReadPlayerInfo
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    struct PlayerInfo
    {
        int playableSoundMax;   //!< プレイヤーで同時に再生できるサウンドの数です。
        u32 playerHeapSize;     //!< プレイヤーに割り当てられるプレイヤーヒープのサイズです。

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        PlayerInfo()
        : playableSoundMax( 0 ),
          playerHeapSize( 0 ) {}
    };

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief    ウェーブサウンドセット・シーケンスサウンドセットの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           ウェーブサウンドセット・シーケンスサウンドセットの情報を
    //!           取得するための構造体です。
    //!
    //! @see detail_ReadSoundGroupInfo
    //!
    //! @date 2010/01/15 初版
    //---------------------------------------------------------------------------
    struct SoundGroupInfo
    {
        ItemId startId;
        ItemId endId;
#if defined(NW_PLATFORM_CAFE)
        const internal::Util::Table<ut::ResU32>* fileIdTable;
#else
        const internal::Util::Table<u32>* fileIdTable;
#endif

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        SoundGroupInfo()
        : startId( INVALID_ID ),
          endId( INVALID_ID ),
          fileIdTable( NULL ) {}
    };

    //---------------------------------------------------------------------------
    //! @brief  グループの設定情報の構造体です。
    //!
    //!         サウンドアーカイブで定義されている、
    //!         グループの情報を取得するための構造体です。
    //!
    //!         groupFileSize には、サウンドアーカイブ内に格納される「グループファイル」
    //!         のファイルサイズが含まれています。
    //!
    //!         SoundMaker 上で設定するグループの出力方式が「埋め込み」の場合は、
    //!         当該グループを SoundDataManager::LoadData
    //!         でロードする際に必要なメモリサイズに相当します。
    //!
    //!         一方、出力方式が「リンク」の場合は、
    //!         グループファイルには「当該グループに含まれるアイテム情報」
    //!         しか含まれていない (＝実データは含まれていない) ため、
    //!         LoadData の際に必要なメモリサイズとは異なります。
    //!
    //! @see ReadGroupInfo
    //!
    //! @date 2012/02/23 初版
    //---------------------------------------------------------------------------
    struct GroupInfo
    {
        FileId fileId;      //!< サウンドアーカイブ内でのファイル ID です。
        u32 groupFileSize;  //!< 上記 fileId のファイルサイズです。

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //! @date 2012/02/23 初版
        //---------------------------------------------------------------------------
        GroupInfo() : fileId( INVALID_ID ), groupFileSize(0) {}
    };

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブプレイヤーの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           サウンドアーカイブプレイヤーの情報を取得するための構造体です。
    //!
    //! @see ReadSoundArchivePlayerInfo
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    struct SoundArchivePlayerInfo
    {
        int sequenceSoundMax;   //!< 確保するシーケンスサウンドインスタンスの数です。
        int sequenceTrackMax;   //!< 確保するシーケンストラックインスタンスの数です。
        int streamSoundMax;     //!< 確保するストリームサウンドインスタンスの数です。
        int streamTrackMax;     //!< 確保するストリームトラックインスンスの数です。
        int streamChannelMax;   //!< 確保するストリームチャンネルインスタンスの数です。
        int waveSoundMax;       //!< 確保するウェーブサウンドインスタンスの数です。
        int waveTrackMax;       //!< 確保するウェーブサウンドトラックの数です。
        int streamBufferTimes;  //!< 確保するストリームサウンドのバッファサイズの倍率です。
    };

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief    ファイルの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           ファイルの情報を取得するための構造体です。
    //!
    //! @see detail_ReadFileInfo
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    struct FileInfo
    {
        static const u32 INVALID_OFFSET = 0xffffffff;
        static const u32 INVALID_SIZE = 0xffffffff;

        u32 fileSize;
        u32 offsetFromFileBlockHead;
        const char* externalFilePath;


        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        FileInfo()
        : fileSize( INVALID_SIZE ),
          offsetFromFileBlockHead( INVALID_OFFSET ),
          externalFilePath( NULL )
        {}
    };

    //----------------------------------------
    //! @name コンストラクタ / デストラクタ
    //@{
protected:
    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    SoundArchive();

public:
    //---------------------------------------------------------------------------
    //! @brief    デストラクタです。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    virtual ~SoundArchive();

    //@}

public:
    //----------------------------------------
    //! @name 状態取得
    //@{
    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブ中のデータが利用可能な状態か調べます。
    //!
    //! @return   サウンドアーカイブ中のデータが利用可能な状態なら true を、
    //!           そうでなければ false を返します。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool IsAvailable() const;
    //@}

    //----------------------------------------
    //! @name ID / ラベル文字列
    //@{
    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブ中のサウンドの個数を取得します。
    //!
    //! @return   サウンドアーカイブ中のサウンドの個数を返します。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    u32 GetSoundCount() const;

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブ中のグループの個数を取得します。
    //!
    //! @return   サウンドアーカイブ中のグループの個数を返します。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    u32 GetGroupCount() const;

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブ中のプレイヤーの個数を取得します。
    //!
    //! @return   サウンドアーカイブ中のプレイヤーの個数を返します。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    u32 GetPlayerCount() const;

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief    サウンドアーカイブ中のウェーブサウンドセット・
    //!           シーケンスサウンドセットの個数を取得します。
    //!
    //! @return   サウンドアーカイブ中のウェーブサウンドセット・
    //!           シーケンスサウンドセットの個数を返します。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    u32 GetSoundGroupCount() const;

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブ中のバンクの個数を取得します。
    //!
    //! @return   サウンドアーカイブ中のバンクの個数を返します。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    u32 GetBankCount() const;

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブ中の波形アーカイブの個数を取得します。
    //!
    //! @return   サウンドアーカイブ中の波形アーカイブの個数を返します。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    u32 GetWaveArchiveCount() const;

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief    サウンドアーカイブ中のファイルの個数を取得します。
    //!
    //! @return   サウンドアーカイブ中のファイルの個数を返します。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    u32 detail_GetFileCount() const;

    //---------------------------------------------------------------------------
    //! @brief  アイテム ID から、対応するラベル文字列を取得します。
    //!
    //!         指定したアイテム ID が存在しない場合は、NULL を返します。
    //!
    //!         @ref MemorySoundArchive 、@ref FsSoundArchive 問わず、
    //!         文字列情報を利用する場合は、
    //!         サウンドアーカイブファイルに文字列テーブルが含まれている必要があります。
    //!
    //!         サウンドアーカイブファイルに文字列テーブルを含めるには、
    //!         SoundMaker の [プロジェクト設定] - [サウンドアーカイブ] タブ -
    //!         [文字列テーブルを出力する] にチェックをつけておいてください。
    //!
    //!         @ref FsSoundArchive 使用時は、あらかじめ
    //!         @ref FsSoundArchive::LoadLabelStringData でラベル文字列データを
    //!         ロードしておく必要があります。
    //!         文字列データをロードしていない無い場合は、NULL を返します。
    //!
    //!         @ref MemorySoundArchive に文字列テーブルが含まれていない場合は、
    //!         NULL を返します。
    //!
    //! @param[in]    id      サウンドアーカイブ内のアイテム ID です。
    //!
    //! @return   指定した ID に対応したラベル文字列を返します。
    //!
    //! @see ItemId
    //! @see GetItemId
    //! @see FsSoundArchive::LoadLabelStringData
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    const char* GetItemLabel( ItemId id ) const;

    //---------------------------------------------------------------------------
    //! @brief  ラベル文字列から、対応するアイテム ID に変換します。
    //!
    //!         指定したラベル文字列が存在しない場合は、
    //!         @ref INVALID_ID を返します。
    //!
    //!         @ref MemorySoundArchive 、@ref FsSoundArchive 問わず、
    //!         文字列情報を利用する場合は、
    //!         サウンドアーカイブファイルに文字列テーブルが含まれている必要があります。
    //!
    //!         サウンドアーカイブファイルに文字列テーブルを含めるには、
    //!         SoundMaker の [プロジェクト設定] - [サウンドアーカイブ] タブ -
    //!         [文字列テーブルを出力する] にチェックをつけておいてください。
    //!
    //!         @ref FsSoundArchive 使用時は、あらかじめ
    //!         @ref FsSoundArchive::LoadLabelStringData でラベル文字列データを
    //!         ロードしておく必要があります。
    //!
    //! @param[in]    pLabel  ラベル文字列です。
    //!
    //! @return   指定したラベル文字列に対応したアイテム ID を返します。
    //!
    //! @see GetItemLabel
    //! @see FsSoundArchive::LoadLabelStringData
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    ItemId GetItemId( const char* pLabel ) const;

    //---------------------------------------------------------------------------
    //! @brief  アイテム ID から、対応するファイル ID を取得します。
    //!
    //!         変換できるアイテム ID の種類は、
    //!         サウンド (STRM / WSD / SEQ) および、
    //!         ウェーブサウンドセット、シーケンスサウンドセット、
    //!         バンク、波形アーカイブ、グループです。
    //!
    //!         ウェーブサウンドセットの ID を指定した場合と、
    //!         当該ウェーブサウンドセットに含まれる任意のウェーブサウンドの ID
    //!         を指定した場合では、同じファイル ID が返ります。
    //!
    //!         シーケンスサウンドセットの ID を指定した場合と、
    //!         当該シーケンスサウンドセットに含まれるシーケンスサウンドのうち、
    //!         一番若い ID のシーケンスサウンドを指定した場合では、
    //!         同じファイル ID が返ります。
    //!
    //!         プレイヤーの ID を指定すると、
    //!         SoundArchive::INVALID_ID が返ります。
    //!
    //!         本関数で取得したファイル ID は、@ref SoundDataManager::SetFileAddress
    //!         関数で利用することができます。
    //!
    //! @param[in]    id      サウンドアーカイブ内のアイテム ID です。
    //!
    //! @return   指定したアイテム ID に対応したファイル ID を返します。
    //!
    //! @see FileId
    //! @see ItemId
    //! @see SoundDataManager::SetFileAddress
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    FileId GetItemFileId( ItemId id ) const;

    //---------------------------------------------------------------------------
    //! @brief  アイテム ID から、対応するプリフェッチデータのファイル ID を取得します。
    //!
    //!         変換できるアイテム ID の種類は、
    //!         プリフェッチデータを持つストリームサウンドのみです。
    //!
    //!         それ以外のプレイヤーの ID を指定すると、
    //!         SoundArchive::INVALID_ID が返ります。
    //!
    //!         本関数で取得したファイル ID は、@ref SoundDataManager::SetFileAddress
    //!         関数で利用することができます。
    //!
    //! @param[in]    id      サウンドアーカイブ内のアイテム ID です。
    //!
    //! @return   指定したアイテム ID に対応したプリフェッチデータのファイル ID を返します。
    //!
    //! @see FileId
    //! @see ItemId
    //! @see SoundDataManager::SetFileAddress
    //!
    //! @date 2013/09/10 初版
    //---------------------------------------------------------------------------
    FileId GetItemPrefetchFileId( ItemId id ) const;

    //---------------------------------------------------------------------------
    //! @brief      サウンドの通し番号をサウンド ID に変換します。
    //!
    //!             index に 0 以上 @ref GetSoundCount 未満の値を渡すと、
    //!             全サウンドの ID を取得することができます。
    //!
    //!             最大値は範囲チェックされませんので、
    //!             取得値を利用する場合は、アプリケーション側で注意してください。
    //!
    //! @param[in]  index   サウンドの通し番号です。
    //!
    //! @return     通し番号に対応したサウンド ID を返します。
    //!
    //! @see ItemId
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static ItemId GetSoundIdFromIndex( u32 index )
    {
        return internal::Util::GetMaskedItemId( index, internal::ItemType_Sound );
    }
    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief      サウンドグループ (ウェーブサウンドセット、シーケンスサウンドセット)
    //!             の通し番号をサウンド ID に変換します。
    //!
    //!             index に 0 以上 @ref GetSoundGroupCount 未満の値を渡すと、
    //!             全サウンドグループの ID を取得することができます。
    //!
    //!             最大値は範囲チェックされませんので、
    //!             取得値を利用する場合は、アプリケーション側で注意してください。
    //!
    //! @param[in]  index   サウンドグループの通し番号です。
    //!
    //! @return     通し番号に対応したサウンドグループ ID を返します。
    //!
    //! @see ItemId
    //! @see GetSoundGroupCount
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static ItemId GetSoundGroupIdFromIndex( u32 index )
    {
        return internal::Util::GetMaskedItemId( index, internal::ItemType_SoundGroup );
    }
    //---------------------------------------------------------------------------
    //! @brief      バンクの通し番号をバンク ID に変換します。
    //!
    //!             index に 0 以上 @ref GetBankCount 未満の値を渡すと、
    //!             全バンクの ID を取得することができます。
    //!
    //!             最大値は範囲チェックされませんので、
    //!             取得値を利用する場合は、アプリケーション側で注意してください。
    //!
    //! @param[in]  index   バンクの通し番号です。
    //!
    //! @return     通し番号に対応したバンク ID を返します。
    //!
    //! @see ItemId
    //! @see GetBankCount
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static ItemId GetBankIdFromIndex( u32 index )
    {
        return internal::Util::GetMaskedItemId( index, internal::ItemType_Bank );
    }
    //---------------------------------------------------------------------------
    //! @brief      プレイヤーの通し番号をプレイヤー ID に変換します。
    //!
    //!             index に 0 以上 @ref GetPlayerCount 未満の値を渡すと、
    //!             全プレイヤーの ID を取得することができます。
    //!
    //!             最大値は範囲チェックされませんので、
    //!             取得値を利用する場合は、アプリケーション側で注意してください。
    //!
    //! @param[in]  index   プレイヤーの通し番号です。
    //!
    //! @return     通し番号に対応したプレイヤー ID を返します。
    //!
    //! @see ItemId
    //! @see GetPlayerCount
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static ItemId GetPlayerIdFromIndex( u32 index )
    {
        return internal::Util::GetMaskedItemId( index, internal::ItemType_Player );
    }
    //---------------------------------------------------------------------------
    //! @brief      波形アーカイブの通し番号を波形アーカイブ ID に変換します。
    //!
    //!             index に 0 以上 @ref GetWaveArchiveCount 未満の値を渡すと、
    //!             全波形アーカイブの ID を取得することができます。
    //!
    //!             最大値は範囲チェックされませんので、
    //!             取得値を利用する場合は、アプリケーション側で注意してください。
    //!
    //! @param[in]  index   波形アーカイブの通し番号です。
    //!
    //! @return     通し番号に対応した波形アーカイブ ID を返します。
    //!
    //! @see ItemId
    //! @see GetWaveArchiveCount
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static ItemId GetWaveArchiveIdFromIndex( u32 index )
    {
        return internal::Util::GetMaskedItemId( index, internal::ItemType_WaveArchive );
    }
    //---------------------------------------------------------------------------
    //! @brief      グループの通し番号をグループ ID に変換します。
    //!
    //!             index に 0 以上 @ref GetGroupCount 未満の値を渡すと、
    //!             全グループの ID を取得することができます。
    //!
    //!             最大値は範囲チェックされませんので、
    //!             取得値を利用する場合は、アプリケーション側で注意してください。
    //!
    //! @param[in]  index   グループの通し番号です。
    //!
    //! @return     通し番号に対応したグループ ID を返します。
    //!
    //! @see ItemId
    //! @see GetGroupCount
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static ItemId GetGroupIdFromIndex( u32 index )
    {
        return internal::Util::GetMaskedItemId( index, internal::ItemType_Group );
    }
    //@}


    //----------------------------------------
    //! @name 情報取得
    //@{
    //---------------------------------------------------------------------------
    //! @brief  指定した ID のサウンドのユーザーパラメータ (無印) を取得します。
    //!
    //!         指定した ID のサウンドが存在しない場合は、
    //!         0 を返します。
    //!
    //!         SoundMaker でユーザーパラメータ (無印) を無効にした場合、
    //!         無効値 (0xffffffff) を返します。
    //!
    //!         本関数では、ユーザーパラメータ (無印) のデータしか取得できません。
    //!         ユーザーパラメータ 1 ～ 3 を取得したい場合は、
    //!         @ref ReadSoundUserParam をご利用ください。
    //!
    //! @param[in]    soundId     サウンドアーカイブ内のサウンド ID です。
    //!
    //! @return   サウンドのユーザーパラメータを返します。
    //!
    //! @see    ItemId
    //! @see    ReadSoundUserParam
    //!
    //! @date 2012/03/15 ユーザーパラメータ無効時の説明を追加
    //! @date 2012/02/09 参照に ReadSoundUserParam を追加
    //! @date 2011/10/03 ユーザーパラメータの数が 0 の場合の説明を追記
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    u32 GetSoundUserParam( ItemId soundId ) const;

    //---------------------------------------------------------------------------
    //! @brief  指定した ID のサウンドのユーザーパラメータを取得します。
    //!
    //!         index は 0 ～ USER_PARAM_INDEX_MAX まで指定することができます。
    //!         SoundMaker の [プロジェクトの設定] - [ユーザーパラメータ] タブで有効にした
    //!         ユーザーパラメータの番号と異なる番号を index として渡した場合は、
    //!         取得に失敗し false を返します。
    //!
    //!         SoundMaker 上の「ユーザーパラメータ (無印)」は、index = 0 に相当します。
    //!         「ユーザーパラメータ 1 ～ 3」は、index = 1 ～ 3 に相当します。
    //!
    //!         soundId として 不正なサウンド ID を渡した場合も取得に失敗し、
    //!         false を返します。
    //!
    //!         取得に成功すると、value に取得できたパラメータが格納され、
    //!         true を返します。
    //!
    //!         index に 0 を渡して取得できた値は、
    //!         @ref GetSoundUserParam で得られる値と同じです。
    //!         (@ref GetSoundUserParam は過去との互換性のために残されています)
    //!
    //! @param[in]    soundId   サウンドアーカイブ内のサウンド ID です。
    //! @param[in]    index     ユーザーパラメータの番号です。
    //! @param[in]    value     ユーザーパラメータの格納先です。
    //!
    //! @return   サウンドのユーザーパラメータが取得できた場合は true を、
    //!           取得できなかった場合は false を返します。
    //!
    //! @see    ItemId
    //! @see    USER_PARAM_INDEX_MAX
    //! @see    GetSoundUserParam
    //!
    //! @date 2012/03/15 SoundMaker 上のユーザーパラメータと index の対応関係について追記
    //! @date 2012/02/09 全体的に説明を修正
    //! @date 2011/10/03 初版
    //---------------------------------------------------------------------------
    bool ReadSoundUserParam( ItemId soundId, int index, u32& value ) const;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のサウンドの種類を取得します。
    //!
    //!           指定した ID のサウンドが存在しない場合は、
    //!           @ref SOUND_TYPE_INVALID を返します。
    //!
    //! @param[in]    soundId     サウンドアーカイブ内のサウンド ID です。
    //!
    //! @return   サウンドの種類を返します。
    //!
    //! @see SoundType
    //! @see ItemId
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    SoundType GetSoundType( ItemId soundId ) const;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のサウンドの情報を取得します。
    //!
    //!           指定した ID のサウンドが存在しない場合は読み取りは失敗します。
    //!
    //! @param[in]    soundId     サウンドアーカイブ内のサウンド ID です。
    //! @param[out]   info        サウンド情報を格納するための構造体です。
    //!
    //! @return   サウンド情報の取得に成功したら true を、失敗したら false を返します。
    //!
    //! @see ItemId
    //! @see SoundInfo
    //!
    //! @date 2012/02/23 文言微調整 (情報を取得するための～ → 情報を格納するための～)
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool ReadSoundInfo( ItemId soundId, SoundInfo* info ) const;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のシーケンスサウンドの情報を取得します。
    //!
    //!           指定した ID のシーケンスサウンドが存在しない場合は読み取りは失敗します。
    //!
    //! @param[in]    soundId     サウンドアーカイブ内のサウンド ID です。
    //! @param[out]   info        シーケンスサウンド情報を格納するための構造体です。
    //!
    //! @return   シーケンスサウンド情報の取得に成功したら true を、失敗したら false を返します。
    //!
    //! @see ItemId
    //! @see SequenceSoundInfo
    //!
    //! @date 2012/02/23 文言微調整 (情報を取得するための～ → 情報を格納するための～)
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool ReadSequenceSoundInfo( ItemId soundId, SequenceSoundInfo* info ) const;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のバンクの情報を取得します。
    //!
    //!           指定した ID のバンクが存在しない場合は読み取りは失敗します。
    //!
    //! @param[in]    bankId      サウンドアーカイブ内のバンク ID です。
    //! @param[out]   info        バンク情報を格納するための構造体です。
    //!
    //! @return   バンク情報の取得に成功したら true を、失敗したら false を返します。
    //!
    //! @see ItemId
    //! @see PlayerInfo
    //!
    //! @date 2012/02/23 文言微調整 (情報を取得するための～ → 情報を格納するための～)
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool ReadBankInfo( ItemId bankId, BankInfo* info ) const;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のプレイヤーの情報を取得します。
    //!
    //!           指定した ID のプレイヤーが存在しない場合は読み取りは失敗します。
    //!
    //! @param[in]    playerId    サウンドアーカイブ内のプレイヤー ID です。
    //! @param[out]   info        プレイヤー情報を格納するための構造体です。
    //!
    //! @return   プレイヤー情報の取得に成功したら true を、失敗したら false を返します。
    //!
    //! @see ItemId
    //! @see PlayerInfo
    //!
    //! @date 2012/02/23 文言微調整 (情報を取得するための～ → 情報を格納するための～)
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool ReadPlayerInfo( ItemId playerId, PlayerInfo* info ) const;

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブプレイヤーの情報を取得します。
    //!
    //! @param[out]   info    サウンドアーカイブプレイヤー情報を格納するための構造体です。
    //!
    //! @return   サウンドアーカイブプレイヤー情報の取得に成功したら true を、
    //!           失敗したら false を返します。
    //!
    //! @see SoundArchivePlayerInfo
    //!
    //! @date 2012/02/23 文言微調整 (情報を取得するための～ → 情報を格納するための～)
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool ReadSoundArchivePlayerInfo( SoundArchivePlayerInfo* info ) const;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のストリームサウンドの情報を取得します。
    //!
    //!           指定した ID のストリームサウンドが存在しない場合は読み取りは失敗します。
    //!
    //! @param[in]    soundId     サウンドアーカイブ内のサウンド ID です。
    //! @param[out]   info        ストリームサウンド情報を格納するための構造体です。
    //!
    //! @return   ストリームサウンド情報の取得に成功したら true を、失敗したら false を返します。
    //!
    //! @see ItemId
    //! @see StreamSoundInfo
    //!
    //! @date 2012/02/23 文言微調整 (情報を取得するための～ → 情報を格納するための～)
    //! @date 2011/09/12 初版
    //---------------------------------------------------------------------------
    bool ReadStreamSoundInfo( ItemId soundId, StreamSoundInfo* info ) const;

    //! @briefprivate
    //! @param soundId :private
    //! @param info :private
    //! @return :private
    bool detail_ReadStreamSoundInfo2( ItemId soundId, StreamSoundInfo2* info ) const;

    //! @briefprivate
    //! @param soundId :private
    //! @param info :private
    //! @return :private
    bool detail_ReadWaveSoundInfo( ItemId soundId, WaveSoundInfo* info ) const;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID の 3D サウンド情報を取得します。
    //!
    //!           指定した ID のサウンドが存在しない場合は読み取りに失敗します。
    //!
    //! @param[in]    soundId サウンドアーカイブ内のサウンド ID です。
    //! @param[out]   info    サウンドアーカイブプレイヤー情報を格納するための構造体です。
    //!
    //! @return   3D サウンド情報の取得に成功したら true を、
    //!           失敗したら false を返します。
    //!
    //! @see Sound3DInfo
    //!
    //! @date 2012/02/23 文言微調整 (情報を取得するための～ → 情報を格納するための～)
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool ReadSound3DInfo( ItemId soundId, Sound3DInfo* info ) const;

    //! @briefprivate
    //! @param warcId :private
    //! @param info :private
    //! @return :private
    bool ReadWaveArchiveInfo( ItemId warcId, WaveArchiveInfo* info ) const;

    //! @briefprivate
    //! @param groupId :private
    //! @param info :private
    //! @return :private
    bool detail_ReadSoundGroupInfo( ItemId groupId, SoundGroupInfo* info ) const;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のグループ情報を取得します。
    //!
    //!           指定した ID のグループが存在しない場合は読み取りに失敗します。
    //!
    //! @param[in]    groupId サウンドアーカイブ内のグループ ID です。
    //! @param[out]   info    グループ情報を格納するための構造体です。
    //!
    //! @return   グループ情報の取得に成功したら true を、
    //!           失敗したら false を返します。
    //!
    //! @see GroupInfo
    //!
    //! @date 2012/02/23 初版
    //---------------------------------------------------------------------------
    bool ReadGroupInfo( ItemId groupId, GroupInfo* info ) const;

    //! @briefprivate
    //! @param fileId :private
    //! @param info :private
    //! @return :private
    bool detail_ReadFileInfo( FileId fileId, FileInfo* info ) const;

    // TODO: bool detail_ReadGroupItemInfo( GroupId groupId, unsigned long index, GroupItemInfo* info ) const;

    // TODO: bool detail_ReadFilePos( FileId fileId, unsigned long index, FilePos* filePos ) const;

    //! @briefprivate
    //! @param id :private
    //! @return :private
#if defined(NW_PLATFORM_CAFE)
    const internal::Util::Table<ut::ResU32>* detail_GetWaveArchiveIdTable( ItemId id ) const;
#else
    const internal::Util::Table<u32>* detail_GetWaveArchiveIdTable( ItemId id ) const;
#endif

    // ファイルアクセス
    //! @briefprivate
    //! @param fileId :private
    //! @return :private
    virtual const void* detail_GetFileAddress( FileId fileId ) const = 0;
    //! @briefprivate
    //! @return :private
    virtual size_t detail_GetRequiredStreamBufferSize() const = 0;

    //! @briefprivate
    //! @param fileId :private
    //! @param buffer :private
    //! @param size :private
    //! @param cacheBuffer :private
    //! @param cacheSize :private
    //! @return :private
    ut::FileStream* detail_OpenFileStream(
            FileId fileId, void* buffer, int size, void* cacheBuffer, size_t cacheSize ) const;

    //! @briefprivate
    //! @param fileId :private
    //! @return :private
#if defined(NW_PLATFORM_CAFE)
    const internal::Util::Table<ut::ResU32>*
#else
    const internal::Util::Table<u32>*
#endif
        detail_GetAttachedGroupTable( FileId fileId ) const;

    //! @briefprivate
    //! @return :private
    internal::SoundArchiveParametersHook* detail_GetParametersHook() const
    {
        return m_pParametersHook;
    }

    //! @briefprivate
    //! @param parametersHook :private
    void detail_SetParametersHook(internal::SoundArchiveParametersHook* parametersHook)
    {
        m_pParametersHook = parametersHook;
    }
    //@}

    //----------------------------------------
    //! @name ファイルパス
    //@{
    //---------------------------------------------------------------------------
    //! @brief    外部参照ファイルのルートディレクトリを設定します。
    //!
    //!           サウンドアーカイブの外部参照ファイルのファイルパスが、
    //!           相対パスで記述されている場合には、
    //!           この関数で設定したルートディレクトリからの相対パスとして解釈されます。
    //!
    //!           初期状態はルートディレクトリに設定されています。
    //!
    //!           この関数は @ref nw::snd::FsSoundArchive::Open から呼び出されています。
    //!
    //! @param[in] extFileRoot    ルートディレクトリに設定するディレクトリパスです。
    //!
    //! @see nw::snd::FsSoundArchive::Open
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetExternalFileRoot( const char* extFileRoot );
    //@}

    //! @briefprivate
    virtual void FileAccessBegin() const {}

    //! @briefprivate
    virtual void FileAccessEnd() const {}

#if defined(NW_PLATFORM_CAFE)
    // ストリームサウンド用
    //! @briefprivate
    FSClient* detail_GetFsClient() const { return m_pFsClient; }

    //---------------------------------------------------------------------------
    //! @brief    内部で使用するFSClientを設定します。
    //!
    //! @param[in] pFsClient 使用するFSClientです。
    //!
    //! @date 2014/01/17 初版
    //---------------------------------------------------------------------------
    void SetFsClient(FSClient* pFsClient) { m_pFsClient = pFsClient; }

    //---------------------------------------------------------------------------
    //! @brief    CreateFsCommandBlockBufferPool を実行するために
    //!           必要なバッファのサイズを取得します。
    //!
    //! @return   必要なメモリサイズを返します。
    //!
    //! @date 2014/01/17 初版
    //---------------------------------------------------------------------------
    size_t GetRequiredFsCommandBlockPoolMemSize() const;

    //! @briefprivate
    internal::FsFileStream::FsCommandBlockPool* detail_GetFsCommandBlockPool() const
    {
        return &m_FsCmdBlockPool;
    }

    //---------------------------------------------------------------------------
    //! @brief    内部で使用するFSCmdBlockのプールを作成します。
    //!
    //! @param[in] buffer     プールの作成に使用するバッファです。
    //! @param[in] bufferSize バッファのサイズです。
    //!
    //! @date 2014/01/17 初版
    //---------------------------------------------------------------------------
    void CreateFsCommandBlockBufferPool(void* buffer, size_t bufferSize);

    //---------------------------------------------------------------------------
    //! @brief    内部で使用するFSCmdBlockのプールを破棄します。
    //!
    //! @date 2014/01/17 初版
    //---------------------------------------------------------------------------
    void DestroyFsCommandBlockBufferPool();

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブでの FS コマンドに使用する優先度を取得します。
    //!
    //! @return   サウンドアーカイブでの FS コマンドに使用する優先度です。
    //!
    //! @date 2013/01/15 初版
    //---------------------------------------------------------------------------
    virtual u8 GetFsPriority() const { return FS_PRIORITY_DEFAULT; }

    //---------------------------------------------------------------------------
    //! @brief    ストリームサウンドのファイル入出力に用いる FS コマンドの優先度を取得します。
    //!
    //! @return   ストリームサウンドのファイル入出力に用いる FS コマンドの優先度です。
    //!
    //! @date 2013/01/15 初版
    //---------------------------------------------------------------------------
    virtual u8 GetFsStreamPriority() const { return FS_PRIORITY_DEFAULT; }
#endif

    //! @briefprivate
    //! @param path :private
    //! @param buf :private
    //! @param bufSize :private
    //! @return :private
    const char* detail_GetExternalFileFullPath(const char* path, char* buf, size_t bufSize) const;

protected:
    static const int FILE_PATH_MAX  = internal::FILE_PATH_MAX;

    //! @briefprivate
    //! @param fileReader :private
    void Initialize( internal::SoundArchiveFileReader* fileReader );

    //! @briefprivate
    void Finalize();

    //! @briefprivate
    //! @param buffer :private
    //! @param size :private
    //! @param begin :private
    //! @param length :private
    //! @return :private
    virtual ut::FileStream* OpenStream(
        void* buffer,
        int size,
        u32 begin,
        u32 length
    ) const = 0;

    //! @briefprivate
    //! @param buffer :private
    //! @param size :private
    //! @param extFilePath :private
    //! @param cacheBuffer :private
    //! @param cacheSize :private
    //! @return :private
    virtual ut::FileStream* OpenExtStream(
        void* buffer,
        int size,
        const char* extFilePath,
        void* cacheBuffer,
        size_t cacheSize
    ) const = 0;

private:
    ut::FileStream* OpenExtStreamImpl(
        void* buffer,
        int size,
        const char* extFilePath,
        void* cacheBuffer,
        size_t cacheSize
    ) const;

    internal::SoundArchiveFileReader* m_pFileReader;
    internal::SoundArchiveParametersHook* m_pParametersHook;

    char m_ExtFileRoot[ FILE_PATH_MAX ];

    // キャッシュ (頻繁にアクセスする)
    u32 m_FileBlockOffset;

#if defined(NW_PLATFORM_CAFE)
    mutable FSClient* m_pFsClient;
    mutable internal::FsFileStream::FsCommandBlockPool m_FsCmdBlockPool;
#endif
};

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SOUND_ARCHIVE_H_ */

