﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_Sound3DManager.h
 *
 * @file snd_Sound3DManager.h
 */

#ifndef NW_SND_SOUND_3D_MANAGER_H_
#define NW_SND_SOUND_3D_MANAGER_H_

#include <nw/math/math_Types.h>     // nw::math::VEC3
#include <nw/ut/ut_LinkList.h>
#include <nw/snd/snd_BasicSound.h>
#include <nw/snd/snd_InstancePool.h>
#include <nw/snd/snd_Sound3DListener.h>

namespace nw {
namespace snd {

/* ========================================================================
        typename declaration
   ======================================================================== */

class SoundArchive;
class Sound3DManager;
class Sound3DEngine;

/* ========================================================================
        struct difinition
   ======================================================================== */

//---------------------------------------------------------------------------
//! @brief    3Dサウンドのパラメータです。
//!
//!           このパラメータを用いて 3D サウンドの計算を行います。
//!
//!           ctrlFlag, decayCurve, decayRatio, dopplerFactor, soundUserParam
//!           は SoundMaker で各サウンドに設定した値が入ります。
//!
//!           actorUserParam は、@ref Sound3DActor::SetUserParam で設定された値です。
//!
//!           dopplerFactor は、ドップラー効果のかかりり具合を表します。
//!           0 のときは音が変化せず、値が大きくなるほど変化が大きくなります。
//!
//! @see Sound3DEngine::UpdateAmbientParam
//! @see Sound3DActor::SetUserParam
//! @see DecayCurve
//! @see nw::math::VEC3
//!
//! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
struct Sound3DParam
{
    //! 3D サウンドアクターの現在位置です。
    math::VEC3 position;

    //! 3D サウンドアクターの速度です。
    math::VEC3 velocity;

    //! パラメータのコントロールフラグです。サウンドライブラリ内部で使用されます。
    u32 ctrlFlag;

    //! アクターに設定されたユーザーパラメータです。
    u32 actorUserParam;

    //! サウンドに設定されたユーザーパラメータです。
    u32 soundUserParam;

    //! アクターとリスナーが音量減衰の単位距離分だけ離れているときの音量の減衰率です。
    f32 decayRatio;

    //! 音量減衰カーブの種類です。@ref DecayCurve の値を設定します。
    u8  decayCurve;

    //! ドップラーファクターです。
    u8  dopplerFactor;

    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    Sound3DParam();
};

/* ========================================================================
        class difinition
   ======================================================================== */

namespace internal {

class ISound3DEngine
{
public:
    virtual ~ISound3DEngine() {}

    virtual void detail_UpdateAmbientParam(
        const Sound3DManager* manager,
        const Sound3DParam* actorParam,
        u32 soundId,
        SoundAmbientParam* param
    ) = 0;

    virtual int GetAmbientPriority(
        const Sound3DManager* manager,
        const Sound3DParam* actorParam,
        u32 soundId
    ) = 0;
};

}

//---------------------------------------------------------------------------
//! @brief    3D サウンドのパラメータ演算と管理を行うクラスです。
//!
//!           @ref nw::snd::Sound3DListener クラスと
//!           @ref nw::snd::Sound3DActor クラスの情報を用いて、
//!           サウンドのパラメータ演算を行います。
//!
//!           Sound3DActor::Initialize() の引数で渡した Sound3DManager について、
//!           Sound3DActor で再生したサウンドが鳴り終わる前に破棄を行うと
//!           SoundArchivePlayer::Update() で例外で停止する可能性があります。
//!
//!           これは Sound3DActor で再生したサウンドが
//!           Sound3DManager をコールバックとして参照しているためです。
//!           Sound3DManager を破棄する場合は
//!           必ず参照しているサウンドが全て再生完了してから行ってください。
//!
//! @see Sound3DListener クラス
//! @see Sound3DActor クラス
//!
//! @date 2010/02/25 初版
//! @date 2013/06/28 Sound3DManager の破棄に関する注意点を追記
//---------------------------------------------------------------------------
class Sound3DManager
: public internal::BasicSound::AmbientParamUpdateCallback,
  public internal::BasicSound::AmbientArgAllocatorCallback
{
public:
    //! @briefprivate
    typedef internal::InstancePool<Sound3DParam> Sound3DParamPool;

    //---------------------------------------------------------------------------
    //! @brief    3D サウンドリスナーのリストを表す型です。
    //!
    //! @see Sound3DListener クラス
    //! @see ut::LinkList クラス
    //!
    //! @date 2010/03/12 初版
    //---------------------------------------------------------------------------
    typedef ut::LinkList< Sound3DListener, offsetof(Sound3DListener,m_LinkNode)> ListenerList;

    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */
public:

    //! @name コンストラクタ
    //@{
    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //!
    //!           初期化時はプレイヤープライオリティ最大減少量を 32 に設定します。
    //!
    //! @date 2010/02/25 初版
    //---------------------------------------------------------------------------
    Sound3DManager();
    //@}

    //! @name 初期化
    //@{
    //---------------------------------------------------------------------------
    //! @brief    初期化に必要なメモリのサイズを取得します。
    //!
    //! @param[in] archive    初期化のための情報取得に使用するサウンドアーカイブです。
    //!
    //! @return   初期化に必要なメモリのサイズを返します。
    //!
    //! @see Initialize
    //!
    //! @date 2010/02/25 初版
    //---------------------------------------------------------------------------
    size_t GetRequiredMemSize( const SoundArchive* archive );

    //---------------------------------------------------------------------------
    //! @brief    3D サウンドマネージャーの初期化を行います。
    //!
    //!           3D サウンドマネージャーを使用する前に初期化を行う必要があります。
    //!           3D サウンドを再生する際に、
    //!           3D マネージャでセットアップされたメモリ領域を利用します。
    //!
    //!           3D サウンドマネージャーが必要とするメモリのサイズは
    //!           @ref nw::snd::Sound3DManager::GetRequiredMemSize
    //!           で取得することができます。
    //!
    //! @param[in] archive  3D サウンドマネージャーで使用するサウンドアーカイブです。
    //! @param[in] buffer   バッファへのポインタです。
    //! @param[in] size     バッファサイズです。
    //!
    //! @return   初期化に成功したら true を、失敗したら false を返します。
    //!
    //! @see GetRequiredMemSize
    //!
    //! @date 2010/02/25 初版
    //---------------------------------------------------------------------------
    bool Initialize(
        const SoundArchive* archive,
        void* buffer,
        size_t size
    );

    //! @briefprivate
    //! @param archive :private
    //! @return :private
    bool InitializeWithMoreSoundArchive( const SoundArchive* archive );

    //---------------------------------------------------------------------------
    //! @brief    3D サウンドデータマネージャを破棄します。
    //!
    //! @return   成功なら true を、失敗なら false を返します。
    //!           Sound3DManager が未初期化のときのみ失敗します。
    //!
    //! @see Initialize
    //!
    //! @date 2010/10/12 初版
    //---------------------------------------------------------------------------
    bool Finalize();
    //@}

    //! @name 3D サウンドリスナー
    //@{
    //---------------------------------------------------------------------------
    //! @brief    3D サウンドリスナーを登録します。
    //!
    //!           3D サウンドを鳴らすためには 3D サウンドマネージャーに
    //!           3D サウンドリスナーを登録する必要があります。
    //!
    //!           3D サウンドリスナーは複数登録することができます。
    //!           複数のリスナーを登録した場合の詳細は、
    //!           プログラマーガイドの「マルチリスナー」を参照してください。
    //!
    //!           ただし、ひとつの 3D サウンドリスナーを複数の
    //!           3D サウンドマネージャーに登録することはできません。
    //!           また、ひとつの 3D サウンドリスナーをひとつの
    //!           3D サウンドマネージャーに複数回登録することもできません。
    //!
    //! @param[in] listener   登録する 3D サウンドリスナーへのポインタです。
    //!
    //! @see Sound3DListener クラス
    //! @see RemoveListener
    //!
    //! @date 2010/02/25 初版
    //---------------------------------------------------------------------------
    void AddListener( Sound3DListener* listener ) { m_ListenerList.PushBack( listener ); }

    //---------------------------------------------------------------------------
    //! @brief    指定した登録済みの 3D サウンドリスナーを登録解除します。
    //!
    //! @param[in] listener   登録を削除する 3D サウンドリスナーへのポインタです。
    //!
    //! @see Sound3DListener クラス
    //! @see AddListener
    //!
    //! @date 2010/03/12 初版
    //---------------------------------------------------------------------------
    void RemoveListener( Sound3DListener* listener ) { m_ListenerList.Erase( listener ); }

    //---------------------------------------------------------------------------
    //! @brief    登録されている 3D サウンドリスナーのリストを取得します。
    //!
    //!           取得したリスト内の 3D サウンドリスナーの並び順は、
    //!           @ref AddListener  で 3D サウンドマネージャーに登録された順番です。
    //!
    //! @return   3D サウンドリスナーのリストを返します。
    //!
    //! @see Sound3DListener クラス
    //! @see AddListener
    //! @see RemoveListener
    //!
    //! @date 2010/03/12 初版
    //---------------------------------------------------------------------------
    const ListenerList& GetListenerList() const { return m_ListenerList; }
    //@}

    //! @name 3D サウンドエンジン
    //@{
    //---------------------------------------------------------------------------
    //! @brief    3D サウンドエンジンを登録します。
    //!
    //!           3D サウンドエンジンは、3D サウンドのパラメータの計算処理が定義されたクラスです。
    //!           初期状態では、サウンドライブラリで用意されているデフォルトのエンジンクラスが
    //!           登録されていますので、この関数を呼び出さなくても
    //!           3D サウンドを使用することができます。
    //!
    //!           カスタマイズした 3D サウンドエンジンクラスを使用したい場合には、
    //!           この関数を呼び出して 3D サウンドエンジンを登録してください。
    //!
    //! @param[in] engine 登録する 3D サウンドエンジンへのポインタです。
    //!
    //! @see Sound3DEngine クラス
    //!
    //! @date 2010/03/12 初版
    //---------------------------------------------------------------------------
    void SetEngine( Sound3DEngine* engine );
    //@}

    //! @name パラメータ
    //@{
    //---------------------------------------------------------------------------
    //! @brief    最大プレイヤープライオリティ減少量を設定します。
    //!
    //!           3D サウンドのプレイヤープライオリティは、音量の減衰に比例して減少します。
    //!           この関数は音量が 0 になったときのプレイヤープライオリティの減少値を設定します。
    //!
    //! @param[in] maxPriorityReduction   最大プレイヤープライオリティ減少量です。
    //!
    //! @see GetMaxPriorityReduction
    //!
    //! @date 2010/03/12 初版
    //---------------------------------------------------------------------------
    void SetMaxPriorityReduction( int maxPriorityReduction )
    {
        m_MaxPriorityReduction = maxPriorityReduction;
    }

    //---------------------------------------------------------------------------
    //! @brief    現在設定されている最大プレイヤープライオリティ減少量を取得します。
    //!
    //! @return   現在設定されている最大プレイヤープライオリティ減少量を返します。
    //!
    //! @see SetMaxPriorityReduction
    //!
    //! @date 2010/03/12 初版
    //---------------------------------------------------------------------------
    int GetMaxPriorityReduction() const
    {
        return m_MaxPriorityReduction;
    }

    //---------------------------------------------------------------------------
    //! @brief    3D サウンドで設定されるパンの変化幅を設定します。
    //!
    //!           panRange に 1.0 を指定すると、定位の変化が最大になります。
    //!           1.0 より小さくすると、定位の変化幅を抑えることが出来ます。
    //!
    //!           panRange の初期値は 0.9 です。
    //!
    //! @param[in] panRange   パンの変化幅 ( 0.0 ～ 1.0 ) です。
    //!
    //! @see GetPanRange
    //!
    //! @date 2010/03/12 初版
    //---------------------------------------------------------------------------
    void SetPanRange( f32 panRange ) { m_PanRange = panRange; }

    //---------------------------------------------------------------------------
    //! @brief    現在設定されているパンの変化幅を取得します。
    //!
    //! @return   現在設定されているパンの変化幅を返します。
    //!
    //! @see SetPanRange
    //!
    //! @date 2010/03/12 初版
    //---------------------------------------------------------------------------
    f32 GetPanRange() const { return m_PanRange; }

    //---------------------------------------------------------------------------
    //! @brief    3D サウンドで設定される音速を設定します。
    //!
    //!           設定した音速は、ドップラー効果による音程変化の計算に使用されます。
    //!
    //!           設定する値の単位は 1 フレーム当たりの音の速さです。
    //!           音速は約 340m / 秒ですので、3D 空間の座標の単位系が 1.0f で 1m である場合、
    //!           60 フレームで動作しているとすると、340.0f / 60 が設定すべき値になります。
    //!
    //!           音速に 0.0f を設定すると、ドップラー効果が発生しなくなります。
    //!           デフォルト値は 0.0f です。
    //!
    //! @param[in] sonicVelocity  音速です。
    //!
    //! @see GetSonicVelocity
    //!
    //! @date 2010/03/12 初版
    //---------------------------------------------------------------------------
    void SetSonicVelocity( f32 sonicVelocity ) { m_SonicVelocity = sonicVelocity; }

    //---------------------------------------------------------------------------
    //! @brief    現在設定されている音速を取得します。
    //!
    //! @return   現在設定されている音速を返します。
    //!
    //! @see SetSonicVelocity
    //!
    //! @date 2010/03/12 初版
    //---------------------------------------------------------------------------
    f32 GetSonicVelocity() const { return m_SonicVelocity; }

    //---------------------------------------------------------------------------
    //! @brief  3D サウンドで設定される biquad フィルタの種類を設定します。
    //!
    //!         biquad フィルタは複数の箇所での設定が重ね合わされず、
    //!         以下の優先度に従って設定されます。
    //!         優先度が高い箇所でパラメータの設定がされた場合、
    //!         それより下位の設定は上書きされます。
    //!
    //!         -# サウンドハンドルでの設定
    //!         -# サウンドプレイヤーでの設定
    //!         -# アンビエントパラメータ構造体での設定
    //!         -# シーケンスデータでの設定
    //!
    //!         フィルタの種類 type は @ref BiquadFilterType の値を使用します。
    //!         プリセットで用意されているフィルタの種類の他、
    //!         ユーザーが登録したフィルタを選択することができます。
    //!
    //!         type には BIQUAD_FILTER_TYPE_MIN ～ BIQUAD_FILTER_TYPE_MAX の値を設定します。
    //!         上記の範囲外の値を入れると、アサートで停止します
    //!         (Debug/Development 版のみ。Release 版は無視され値が設定されますが、
    //!         正常な動作は保証されません)。
    //!
    //! @param[in] type   biquad フィルタの種類。
    //!
    //! @see BiquadFilterType
    //! @see GetBiquadFilterType
    //!
    //! @date 2010/11/30 初版
    //---------------------------------------------------------------------------
    void SetBiquadFilterType( int type );

        // TODO: 以下、要修正。RVL と比べて軽い実装になっている？
        // biquad フィルタは従来の LPF に比べ 3 倍強の DSP 負荷がかかります。
        // これは、AUX バス1本分のミキサーの負荷とほぼ同等です。

    //---------------------------------------------------------------------------
    //! @brief    現在設定されている biquad フィルタの種類を取得します。
    //!
    //! @return   現在設定されている biquad フィルタの種類を返します。
    //!
    //! @see BiquadFilterType
    //! @see SetBiquadFilterType
    //!
    //! @date 2010/11/30 初版
    //---------------------------------------------------------------------------
    int GetBiquadFilterType() const { return m_BiquadFilterType; }
    //@}

private:
    virtual void detail_UpdateAmbientParam(
        const void* arg,
        u32 soundId,
        SoundAmbientParam* param
    );
    virtual int detail_GetAmbientPriority(
        const void* arg,
        u32 soundId
    );
    virtual void* detail_AllocAmbientArg( size_t argSize );
    virtual void detail_FreeAmbientArg(
        void* arg,
        const internal::BasicSound* sound
    );

    Sound3DParamPool m_ParamPool;
    ListenerList m_ListenerList;
    internal::ISound3DEngine* m_pSound3DEngine;

    s32 m_MaxPriorityReduction;
    f32 m_PanRange;
    f32 m_SonicVelocity;
    s32 m_BiquadFilterType;

    s32 m_FreeMemSizeAfterCheking;
    bool m_IsInitialized;
    u8 padding[3];
};

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SOUND_3D_MANAGER_H_ */

