﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_Sound3DListener.h
 *
 * @file snd_Sound3DListener.h
 */

#ifndef NW_SND_SOUND_3D_LISTENER_H_
#define NW_SND_SOUND_3D_LISTENER_H_

#include <nw/math/math_Types.h>     // nw::math::MTX34, VEC3
#include <nw/ut/ut_LinkList.h>

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief    3D 空間内でのリスニングポイントを表す 3D サウンドリスナークラスです。
//!
//!           リスナーは @ref Sound3DManager クラスに登録して使用します。
//!           リスナーに設定されたリスナー行列の情報を用いて、
//!           サウンドのパラメータを計算して設定します。
//!
//! @see Sound3DManager クラス
//! @see Sound3DManager::AddListener
//!
//! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
class Sound3DListener
{
public:

    //---------------------------------------------------------------------------
    //! @brief  出力先属性をあらわします。
    //!
    //!         今後この列挙体は削除されます。互換性維持のため、残されています。
    //!         以後は、@ref OutputType をご利用ください。
    //!
    //! @see SetListenerOutputType
    //! @see GetListenerOutputType
    //!
    //! @date 2011/11/14 今後削除される旨、追記
    //! @date 2011/07/21 初版
    //---------------------------------------------------------------------------
    enum ListenerOutputType
    {
        //! TV 出力用の計算をするリスナーであることをあらわします。
        LISTENER_OUTPUT_TYPE_TV  = (1 << 0),

        //! DRC 出力用の計算をするリスナーであることをあらわします。
        LISTENER_OUTPUT_TYPE_DRC = (1 << 1),

        //! TV / DRC 両出力用の計算をするリスナーであることをあらわします。
        LISTENER_OUTPUT_TYPE_BOTH = (LISTENER_OUTPUT_TYPE_TV | LISTENER_OUTPUT_TYPE_DRC)
    };

    //---------------------------------------------------------------------------
    //! @brief  出力先属性をあらわします。
    //!
    //!         OUTPUT_TYPE_DRC1 を有効にしても効果はありません。
    //!         このフラグは、将来的に Cafe SDK にて 2 台目の
    //!         DRC への接続がサポートされた場合に備えて用意されたものです。
    //!
    //! @see SetOutputTypeFlag
    //! @see GetOutputTypeFlag
    //!
    //! @date 2011/12/28 誤植修正 (OTUPUT_TYPE_ALL → OUTPUT_TYPE_ALL)
    //! @date 2011/11/14 初版
    //---------------------------------------------------------------------------
    enum OutputType
    {
        //! TV 出力用の計算をするリスナーであることをあらわします。
        OUTPUT_TYPE_TV = (1<<0),

        //! DRC 0 出力用の計算をするリスナーであることをあらわします。
        OUTPUT_TYPE_DRC0 = (1<<1),

        //! DRC 1 出力用の計算をするリスナーであることをあらわします。
        OUTPUT_TYPE_DRC1 = (1<<2),

        //! TV / DRC すべての出力に対して計算をするリスナーであることをあらわします。
        OUTPUT_TYPE_ALL = 0xffffffff
    };

    //! @name コンストラクタ
    //@{
    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //!
    //!           インテリアサイズ、最大音量範囲、減衰単位距離はそれぞれ
    //!           1.0 で初期化されます。
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    Sound3DListener();
    //@}

    //! @name リスナー行列
    //@{
    //---------------------------------------------------------------------------
    //! @brief    リスナー行列を設定します。
    //!
    //!           3D リスナーのリスナー行列を設定します。
    //!           設定された行列は、3D サウンドのパラメータ計算で使用されます。
    //!
    //!           新しくリスナー行列を設定した際、前回設定されたリスナー行列との差分を計算し、
    //!           リスナーの速度が自動的に設定されます。
    //!           リスナーの速度は、ドップラー効果による音程変化に反映されます。
    //!           速度は @ref SetVelocity で明示的に設定することも可能です。
    //!
    //!           座標が別の位置に飛んだ場合、飛んだ距離の差分で速度が計算されてしまうため、
    //!           極めて高速で移動したように処理され、急激な音程変化が発生します。
    //!           このような場合、この関数を呼び出した後で、
    //!           @ref SetVelocity を呼び出し、
    //!           速度を明示的に設定する必要があることに注意してください。
    //!
    //! @param[in] mtx    リスナー行列です。
    //!
    //! @see GetMatrix
    //! @see SetVelocity
    //! @see nw::math::MTX34 構造体
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetMatrix( const nw::math::MTX34& mtx );

    //---------------------------------------------------------------------------
    //! @brief    現在設定されているリスナー行列を取得します。
    //!
    //! @return   現在設定されているリスナー行列を返します。
    //!
    //! @see SetMatrix
    //! @see nw::math::MTX34 構造体
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    const nw::math::MTX34& GetMatrix() const { return m_Mtx; }

    //---------------------------------------------------------------------------
    //! @brief    リスナー行列をリセットします。
    //!
    //!           3D サウンドリスナーのリスナー行列と速度をゼロクリアします。
    //!
    //!           この関数でリセットした直後の @ref SetMatrix の呼び出しでは、
    //!           3D サウンドリスナーの速度は設定されません。
    //!
    //! @see SetMatrix
    //! @see SetVelocity
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void ResetMatrix();

    //---------------------------------------------------------------------------
    //! @brief    設定されているリスナーの位置情報を取得します。
    //!
    //!           この情報は、@ref SetMatrix で設定されたリスナー行列を元に計算されます。
    //!
    //! @return   現在設定されているリスナーの位置情報を返します。
    //!
    //! @see SetMatrix
    //! @see nw::math::VEC3 構造体
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetPosition() const { return m_Position; }

    //---------------------------------------------------------------------------
    //! @brief    リスナーの速度を設定します。
    //!
    //!           設定された速度は、ドップラー効果による音程変化に反映されます。
    //!
    //!           3D サウンドリスナーの速度は @ref SetMatrix で座標を設定した際、
    //!           前回の登録座標との差分から自動的に設定されますが、
    //!           この関数を使用すると 3D サウンドリスナーの速度を手動で設定することができます。
    //!
    //! @param[in] velocity   リスナーの速度です。
    //!
    //! @see GetVelocity
    //! @see SetMatrix
    //! @see nw::math::VEC3 構造体
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetVelocity( const nw::math::VEC3& velocity );

    //---------------------------------------------------------------------------
    //! @brief    設定されているリスナーの速度を取得します。
    //!
    //!           通常、@ref SetMatrix で座標を設定した際、
    //!           前回の登録座標との差分から自動的に設定されますが、
    //!           @ref SetVelocity を使用して手動で設定することも可能です。
    //!
    //! @return   現在設定されているリスナーの速度を返します。
    //!
    //! @see SetVelocity
    //! @see SetMatrix
    //! @see nw::math::VEC3 構造体
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetVelocity() const { return m_Velocity; }
    //@}

    //! @name パラメータ
    //@{
    //---------------------------------------------------------------------------
    //! @brief    インテリアサイズを設定します。
    //!
    //!           インテリアサイズとは、パンが変化するエリアの大きさで、
    //!           リスナーからの距離（半径）で設定します。
    //!           インテリアサイズを大きくすると、パン変化が緩やかになります。
    //!           逆にインテリアサイズを小さくすると、パン変化が急になります。
    //!
    //!           リスナーからの距離が離れると音量は減衰していきますが、
    //!           減衰量は最大音量範囲の設定 (@ref SetMaxVolumeDistance で設定します)
    //!           に依存します。
    //!
    //! @param[in] interiorSize   インテリアサイズです。
    //!
    //! @see GetInteriorSize
    //! @see SetMaxVolumeDistance
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetInteriorSize( f32 interiorSize );

    //---------------------------------------------------------------------------
    //! @brief    現在設定されているインテリアサイズを取得します。
    //!
    //! @return   現在設定されているインテリアサイズを返します。
    //!
    //! @see SetInteriorSize
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    f32 GetInteriorSize() const { return m_InteriorSize; }

    //---------------------------------------------------------------------------
    //! @brief    最大音量範囲を設定します。
    //!
    //!           リスナーとアクターの距離が最大音量範囲よりも小さければ、
    //!           そのアクターが再生するサウンドは音量の減衰無く再生されます。
    //!           最大音量範囲を超えると、その距離に応じて再生するサウンドの音量が減衰されます。
    //!
    //! @param[in] maxVolumeDistance  最大音量範囲です。
    //!
    //! @see GetMaxVolumeDistance
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetMaxVolumeDistance( f32 maxVolumeDistance );

    //---------------------------------------------------------------------------
    //! @brief    現在設定されている最大音量範囲を取得します。
    //!
    //! @return   現在設定されている最大音量範囲を返します。
    //!
    //! @see SetMaxVolumeDistance
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    f32 GetMaxVolumeDistance() const { return m_MaxVolumeDistance; }

    //---------------------------------------------------------------------------
    //! @brief    音量減衰の単位距離を設定します。
    //!
    //!           一般的に 3D サウンドアクターと 3D サウンドリスナーの距離が離れると、
    //!           サウンドの音量は減衰していきます。
    //!           減衰の速度は、サウンドごとに設定される減衰係数と、
    //!           この単位距離によって定まります。
    //!
    //!           サウンドが単位距離だけリスナーから離れるたびに、
    //!           サウンドの音量が減衰係数に応じて下がります。
    //!           減衰の仕方はサウンドごとに設定される減衰曲線に従います。
    //!
    //! @param[in] unitDistance   音量減衰の単位距離です。
    //!
    //! @see GetUnitDistance
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetUnitDistance( f32 unitDistance );

    //---------------------------------------------------------------------------
    //! @brief    現在設定されている音量減衰の単位距離を取得します。
    //!
    //! @return   現在設定されている音量減衰の単位距離を返します。
    //!
    //! @see SetUnitDistance
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    f32 GetUnitDistance() const { return m_UnitDistance; }

    //---------------------------------------------------------------------------
    //! @brief    リスナーにユーザーパラメータを設定します。
    //!
    //! @param[in] param ユーザーパラメータです。
    //!
    //! @see GetUserParam
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetUserParam( u32 param ) { m_UserParam = param; }

    //---------------------------------------------------------------------------
    //! @brief    設定されているユーザーパラメータを取得します。
    //!
    //! @return   現在設定されているリスナーのユーザーパラメータを返します。
    //!
    //! @see SetUserParam
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    u32 GetUserParam() const { return m_UserParam; }

    //---------------------------------------------------------------------------
    //! @brief  biquad フィルタのかかり具合を設定します。
    //!
    //!         本関数で設定された値は、3D サウンド計算のデフォルト実装
    //!         (@ref Sound3DCalculator::CalcBiquadFilterValue)
    //!         では、下記計算式における「かかり具合」として利用されています。
    //!
    //! @code
    //! biquad フィルタ値 ＝
    //!     { ( アクターとリスナーとの距離 － 最大音量範囲 ) ÷ 音量減衰単位距離 }
    //!     × かかり具合
    //! @endcode
    //!
    //!         (デフォルト実装 (@ref Sound3DCalculator::CalcBiquadFilterValue) では、
    //!         上記で求められた biquad フィルタ値が @ref SetMaxBiquadFilterValue
    //!         で設定した値より大きい場合は、@ref SetMaxBiquadFilterValue
    //!         で設定した値に丸められます。)
    //!
    //!         デフォルトでは 0.5f が設定されています。
    //!
    //! @param[in] value    biquad フィルタのかかり具合です。
    //!
    //! @see Sound3DCalculator::CalcBiquadFilterValue
    //! @see GetUnitBiquadFilterValue
    //! @see SetMaxBiquadFilterValue
    //!
    //! @date 2012/03/09 説明修正 (「かかり具合に関する単位距離」→「かかり具合」)、
    //!                  デフォルト値修正 (1.0f → 0.5f)
    //! @date 2011/07/21 初版
    //---------------------------------------------------------------------------
    void SetUnitBiquadFilterValue( f32 value );

    //---------------------------------------------------------------------------
    //! @brief  biquad フィルタのかかり具合を取得します。
    //!
    //!         3D サウンド計算のデフォルト実装 (@ref Sound3DCalculator::CalcBiquadFilterValue)
    //!         で呼び出されます。
    //!
    //!         また 3D サウンド計算をカスタマイズする際に本関数を呼び出すことで、
    //!         リスナー別の「biquad フィルタのかかり具合」として利用することができます。
    //!
    //! @return biquad フィルタのかかり具合を返します。
    //!
    //! @see SetUnitBiquadFilterValue
    //!
    //! @date 2012/03/09 説明修正 (「かかり具合に関する単位距離」→「かかり具合」)
    //! @date 2011/07/21 初版
    //---------------------------------------------------------------------------
    f32 GetUnitBiquadFilterValue() const { return m_UnitBiquadFilterValue; }

    //---------------------------------------------------------------------------
    //! @brief  biquad フィルタのかかり具合の最大値を設定します。
    //!
    //!         デフォルトでは 1.0f が設定されています。
    //!
    //!         本関数で設定された値が、
    //!         3D サウンド計算のデフォルト実装内でどのようにの利用されているかについては、
    //!         @ref SetUnitBiquadFilterValue 関数のリファレンスを参照しください。
    //!
    //! @param[in] value    biquad フィルタのかかり具合の最大値です。
    //!
    //! @see SetUnitBiquadFilterValue
    //! @see GetMaxBiquadFilterValue
    //!
    //! @date 2011/07/21 初版
    //---------------------------------------------------------------------------
    void SetMaxBiquadFilterValue( f32 value );

    //---------------------------------------------------------------------------
    //! @brief  biquad フィルタのかかり具合の最大値を取得します。
    //!
    //!         3D サウンド計算のデフォルト実装 (@ref Sound3DCalculator::CalcBiquadFilterValue)
    //!         で呼び出されます。
    //!
    //!         また 3D サウンド計算をカスタマイズする際に本関数を呼び出すことで、
    //!         リスナー別の「biquad フィルタのかかり具合の最大値」
    //!         として利用することができます。
    //!
    //! @return  biquad フィルタのかかり具合の最大値を返します。
    //!
    //! @date 2011/07/21 初版
    //---------------------------------------------------------------------------
    f32 GetMaxBiquadFilterValue() const { return m_MaxBiquadFilterValue; }

    //---------------------------------------------------------------------------
    //! @brief  出力先属性を設定します。
    //!
    //!         @ref Sound3DEngine::UpdateAmbientParam をオーバーライドして
    //!         3D サウンド計算をカスタマイズする際、ここで設定した出力先属性を
    //!         @ref GetListenerOutputType で参照することで、
    //!         出力先別に処理を書き分けることができます。
    //!
    //!         デフォルトでは LISTENER_OUTPUT_TYPE_TV が設定されています。
    //!
    //!         今後この関数は削除されます。互換性維持のため、残されています。
    //!         以後は、@ref SetOutputTypeFlag をご利用ください。
    //!
    //! @param[in] type     設定する出力先属性です。
    //!
    //! @see ListenerOutputType
    //! @see GetListenerOutputType
    //!
    //! @date 2011/11/14 今後削除される旨、追記
    //! @date 2011/07/21 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION(void SetListenerOutputType( ListenerOutputType type ));

    //---------------------------------------------------------------------------
    //! @brief  出力先属性を取得します。
    //!
    //!         @ref Sound3DEngine::UpdateAmbientParam をオーバーライドして
    //!         3D サウンド計算をカスタマイズする際、@ref SetListenerOutputType
    //!         で設定した出力先属性を本関数経由で参照することで、
    //!         出力先別に処理を書き分けることができます。
    //!
    //!         今後この関数は削除されます。互換性維持のため、残されています。
    //!         以後は、@ref GetOutputTypeFlag をご利用ください。
    //!
    //! @return     設定されている出力先属性を返します。
    //!
    //! @see ListenerOutputType
    //! @see SetListenerOutputType
    //!
    //! @date 2011/11/14 今後削除される旨、追記
    //! @date 2011/07/21 初版
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION(ListenerOutputType GetListenerOutputType() const);

    //---------------------------------------------------------------------------
    //! @brief  出力先属性を設定します。
    //!
    //!         @ref Sound3DEngine::UpdateAmbientParam をオーバーライドして
    //!         3D サウンド計算をカスタマイズする際、ここで設定した出力先属性を
    //!         @ref GetOutputTypeFlag で参照することで、
    //!         出力先別に処理を書き分けることができます。
    //!
    //!         type には @ref OutputType の論理和を設定します。
    //!         デフォルトでは OUTPUT_TYPE_TV が設定されています。
    //!
    //! @param[in] outputTypeFlag  設定する出力先属性です。
    //!
    //! @see OutputType
    //! @see GetOutputTypeFlag
    //!
    //! @date 2011/11/14 初版
    //---------------------------------------------------------------------------
    void SetOutputTypeFlag( u32 outputTypeFlag ) { m_OutputTypeFlag = outputTypeFlag; }

    //---------------------------------------------------------------------------
    //! @brief  出力先属性を取得します。
    //!
    //!         @ref Sound3DEngine::UpdateAmbientParam をオーバーライドして
    //!         3D サウンド計算をカスタマイズする際、@ref SetOutputTypeFlag
    //!         で設定した出力先属性を本関数経由で参照することで、
    //!         出力先別に処理を書き分けることができます。
    //!
    //!         返り値として @ref OutputType の論理和を取得することができます。
    //!
    //! @return     設定されている出力先属性を返します。
    //!
    //! @see OutputType
    //! @see SetOutputTypeFlag
    //!
    //! @date 2011/11/14 初版
    //---------------------------------------------------------------------------
    u32 GetOutputTypeFlag() const { return m_OutputTypeFlag; }
    //@}

private:
    void CalcPositionFromMatrix( const nw::math::MTX34& mtx, nw::math::VEC3* pos );

    nw::math::MTX34 m_Mtx;
    nw::math::VEC3 m_Position;
    nw::math::VEC3 m_Velocity;
    f32 m_InteriorSize;
    f32 m_MaxVolumeDistance;
    f32 m_UnitDistance;
    u32 m_UserParam;
    f32 m_UnitBiquadFilterValue;
    f32 m_MaxBiquadFilterValue;
    ListenerOutputType m_ListenerOutputType;
    u32 m_OutputTypeFlag;
    bool m_ResetMatrixFlag;

public:
    //! @briefprivate
    ut::LinkListNode m_LinkNode;
};

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SOUND_3D_LISTENER_H_ */

