﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_Sound3DCalculator.h
 *
 * @file snd_Sound3DCalculator.h
 */


#ifndef NW_SND_SOUND_3D_CALCULATOR_H_
#define NW_SND_SOUND_3D_CALCULATOR_H_

#include <nw/math/math_Constant.h>      // nw::math::F_PI
#include <nw/math/math_Types.h>         // nw::math::VEC3
#include <nw/snd/snd_SoundArchive.h>    // Sound3DInfo::DecayCurve

namespace nw {
namespace snd {

class Sound3DManager;
class Sound3DListener;
struct Sound3DParam;

//---------------------------------------------------------------------------
//! @brief    3D サウンドの標準的なパラメータ計算処理をまとめたクラスです。
//!
//!           @ref Sound3DEngine クラスから呼び出されています。
//!
//!           カスタマイズして独自の 3D サウンドエンジンを作成する場合には、
//!           このクラスの関数を利用することができます。
//!
//! @see Sound3DEngine クラス
//!
//! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
class Sound3DCalculator
{
public:
    //---------------------------------------------------------------------------
    //! @brief  パン計算ためのパラメータ構造体です。
    //!
    //!         @ref CalcPan でパン計算を行う際に参照するパラメータ構造体です。
    //!
    //!         stereoSpeakerAngle は、パン計算を行う際に使用するスピーカーの位置を設定します。
    //!         スピーカー配置は左右対称で、
    //!         左右それぞれのスピーカーの正面からの角度をラジアンで指定します。
    //!         ステレオモードのスピーカー配置は 45 度です。
    //!
    //!         surroundPanOffset はサラウンドパンに対するオフセット値を設定します。
    //!         設定する値は、通常のサラウンドパンの値と同じスケールで、-1.0f
    //!         を設定すると中央に位置していたサラウンドパンが最前方から発音されます。
    //!         1.0f を設定すると中央に位置していたサラウンドパンが最後方から発音されます。
    //!
    //! @see CalcPan
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    struct CalcPanParam
    {
        //! ステレオモードのときのスピーカー位置の中央からの角度（ラジアン）です。
        f32 stereoSpeakerAngle;

        //! サラウンドモードのときのフロントスピーカー位置の中央からの角度（ラジアン）です。
        f32 surroundSpeakerFrontAngle;

        //! モードのときのリアスピーカー位置の中央からの角度（ラジアン）です。
        f32 surroundSpeakerRearAngle;

        //! サラウンドパンのオフセット値です。
        f32 surroundPanOffset;

        //---------------------------------------------------------------------------
        //! @brief    コンストラクタです。
        //!
        //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        CalcPanParam()
        : stereoSpeakerAngle( nw::math::F_PI / 4.0f ),              // 45度
          surroundSpeakerFrontAngle( nw::math::F_PI / 6.0f ),       // 30度
          surroundSpeakerRearAngle( nw::math::F_PI * 2.0f / 3.0f ), // 120度
          surroundPanOffset( 0.0f )
        {}
    };

    //! @name パラメータ計算
    //@{
    //---------------------------------------------------------------------------
    //! @brief  3D サウンドの音量と優先度の標準的なパラメータ計算を行います。
    //!
    //!         3D サウンドでは音量と優先度が連動しますので、
    //!         この関数でまとめて計算を行います。
    //!
    //!         この関数は、過去の互換性維持のために残されています。
    //!         現在の nw::snd ライブラリでは、計算処理削減のため、
    //!         当関数の代わりに actorDistance
    //!         引数付きのオーバーロード関数を呼び出しています。
    //!
    //! @param[in] manager            計算に使用する 3D サウンドマネージャです。
    //! @param[in] listener           計算に使用する 3D サウンドリスナーです。
    //! @param[in] actorParam         計算に使用する 3D サウンドパラメータです。
    //! @param[out] volumePtr         音量の計算結果を格納するポインタです。
    //! @param[out] priorityPtr       優先度の計算結果を格納するポインタです。
    //!
    //! @see Sound3DManager クラス
    //! @see Sound3DListener クラス
    //! @see Sound3DParam 構造体
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void CalcVolumeAndPriority(
        const Sound3DManager& manager,
        const Sound3DListener& listener,
        const Sound3DParam& actorParam,
        f32* volumePtr,
        int* priorityPtr
    );

    //---------------------------------------------------------------------------
    //! @brief  3D サウンドの音量と優先度の標準的なパラメータ計算を行います。
    //!
    //!         3D サウンドでは音量と優先度が連動しますので、
    //!         この関数でまとめて計算を行います。
    //!
    //!         現在の nw::snd ライブラリでは、計算処理削減のため、
    //!         本関数の外側であらかじめ actorDistanceを計算しておき、
    //!             - 本関数
    //!             - @ref CalcPitch
    //!             - @ref CalcBiquadFilterValue
    //!         に計算済みの actorDistance を渡すような処理になっています。
    //!
    //!         詳しくは @ref Sound3DEngine::UpdateAmbientParam の実装をごらんください。
    //!
    //! @param[in] manager          計算に使用する 3D サウンドマネージャです。
    //! @param[in] listener         計算に使用する 3D サウンドリスナーです。
    //! @param[in] actorParam       計算に使用する 3D サウンドパラメータです。
    //! @param[in] actorDistance    アクターとリスナーの距離です。
    //! @param[out] volumePtr       音量の計算結果を格納するポインタです。
    //! @param[out] priorityPtr     優先度の計算結果を格納するポインタです。
    //!
    //! @see Sound3DManager クラス
    //! @see Sound3DListener クラス
    //! @see Sound3DParam 構造体
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void CalcVolumeAndPriority(
        const Sound3DManager& manager,
        const Sound3DListener& listener,
        const Sound3DParam& actorParam,
        f32  actorDistance,
        f32* volumePtr,
        int* priorityPtr
    );

    //---------------------------------------------------------------------------
    //! @brief    3D サウンドのパンの標準的なパラメータ計算を行います。
    //!
    //! @param[in] manager      計算に使用する 3D サウンドマネージャです。
    //! @param[in] listener     計算に使用する 3D サウンドリスナーです。
    //! @param[in] actorParam   計算に使用する 3D サウンドパラメータです。
    //! @param[in] calcPanParam パン計算に使用するパラメータです。
    //! @param[out] panPtr      パンの計算結果を格納するポインタです。
    //! @param[out] spanPtr     サラウンドパンの計算結果を格納するポインタです。
    //!
    //! @see Sound3DManager クラス
    //! @see Sound3DListener クラス
    //! @see Sound3DParam 構造体
    //! @see CalcPanParam 構造体
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void CalcPan(
        const Sound3DManager& manager,
        const Sound3DListener& listener,
        const Sound3DParam& actorParam,
        const CalcPanParam& calcPanParam,
        f32* panPtr,
        f32* spanPtr
    );

    //---------------------------------------------------------------------------
    //! @brief  3D サウンドの音程の標準的なパラメータ計算を行います。
    //!
    //!         この関数は、過去の互換性維持のために残されています。
    //!         現在の nw::snd ライブラリは、計算処理削減のため、
    //!         当関数の代わりに actorPosition および actorDistance
    //!         引数付きのオーバーロード関数を呼び出しています。
    //!
    //! @param[in] manager            計算に使用する 3D サウンドマネージャです。
    //! @param[in] listener           計算に使用する 3D サウンドリスナーです。
    //! @param[in] actorParam         計算に使用する 3D サウンドパラメータです。
    //! @param[out] pitchPtr          音程の計算結果を格納するポインタです。
    //!
    //! @see Sound3DManager クラス
    //! @see Sound3DListener クラス
    //! @see Sound3DParam 構造体
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void CalcPitch(
        const Sound3DManager& manager,
        const Sound3DListener& listener,
        const Sound3DParam& actorParam,
        f32* pitchPtr
    );

    //---------------------------------------------------------------------------
    //! @brief  3D サウンドの音程の標準的なパラメータ計算を行います。
    //!
    //!         現在の nw::snd ライブラリでは、計算処理削減のため、
    //!         本関数の外側であらかじめ actorDistanceを計算しておき、
    //!             - 本関数
    //!             - @ref CalcVolumeAndPriority
    //!             - @ref CalcBiquadFilterValue
    //!         に計算済みの actorDistance を渡すような処理になっています。
    //!
    //!         詳しくは @ref Sound3DEngine::UpdateAmbientParam の実装をごらんください。
    //!
    //! @param[in] manager          計算に使用する 3D サウンドマネージャです。
    //! @param[in] listener         計算に使用する 3D サウンドリスナーです。
    //! @param[in] actorParam       計算に使用する 3D サウンドパラメータです。
    //! @param[in] actorPosition    計算に使用する 3D サウンドアクターの座標です。
    //!                             リスナー座標との差分を渡します。
    //!                             詳しくは、@ref Sound3DEngine::UpdateAmbientParam
    //!                             の実装をごらんください。
    //! @param[in] actorDistance    アクターとリスナーの距離です。
    //! @param[out] pitchPtr        音程の計算結果を格納するポインタです。
    //!
    //! @see Sound3DManager クラス
    //! @see Sound3DListener クラス
    //! @see Sound3DParam 構造体
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void CalcPitch(
        const Sound3DManager& manager,
        const Sound3DListener& listener,
        const Sound3DParam& actorParam,
        const nw::math::VEC3& actorPosition,
        f32  actorDistance,
        f32* pitchPtr
    );

    //---------------------------------------------------------------------------
    //! @brief  3D サウンドの biquad フィルタのかかり具合の標準的なパラメータ計算を行います。
    //!
    //!         この関数は、過去の互換性維持のために残されています。
    //!         現在の nw::snd ライブラリは、計算処理削減のため、
    //!         当関数の代わりに actorDistance
    //!         引数付きのオーバーロード関数を呼び出しています。
    //!
    //! @param[in]  manager              計算に使用する 3D サウンドマネージャです。
    //! @param[in]  listener             計算に使用する 3D サウンドリスナーです。
    //! @param[in]  actorParam           計算に使用する 3D サウンドパラメータです。
    //! @param[out] biquadFilterValuePtr biquad フィルタのかかり具合の計算結果を格納するポインタです。
    //!
    //! @see Sound3DManager クラス
    //! @see Sound3DListener クラス
    //! @see Sound3DParam 構造体
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void CalcBiquadFilterValue(
        const Sound3DManager& manager,
        const Sound3DListener& listener,
        const Sound3DParam& actorParam,
        f32* biquadFilterValuePtr
    );

    //---------------------------------------------------------------------------
    //! @brief  3D サウンドの biquad フィルタのかかり具合の標準的なパラメータ計算を行います。
    //!
    //!         現在の nw::snd ライブラリでは、計算処理削減のため、
    //!         本関数の外側であらかじめ actorDistanceを計算しておき、
    //!             - 本関数
    //!             - CalcVolumeAndPriorityImpl (ライブラリの内部関数です。詳しくは実装をご覧ください)
    //!             - @ref CalcPitch
    //!         に計算済みの actorDistance を渡すような処理になっています。
    //!         詳しくは @ref Sound3DEngine::UpdateAmbientParam の実装をごらんください。
    //!
    //! @param[in]  manager              計算に使用する 3D サウンドマネージャです。
    //! @param[in]  listener             計算に使用する 3D サウンドリスナーです。
    //! @param[in]  actorParam           計算に使用する 3D サウンドパラメータです。
    //! @param[in]  actorDistance        アクターとリスナーの距離です。
    //! @param[out] biquadFilterValuePtr biquad フィルタのかかり具合の計算結果を格納するポインタです。
    //!
    //! @see Sound3DManager クラス
    //! @see Sound3DListener クラス
    //! @see Sound3DParam 構造体
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    static void CalcBiquadFilterValue(
        const Sound3DManager& manager,
        const Sound3DListener& listener,
        const Sound3DParam& actorParam,
        f32  actorDistance,
        f32* biquadFilterValuePtr
    );
    //@}

private:
    static void CalcVolumeAndPriorityImpl(
        f32 actorDistance,
        SoundArchive::Sound3DInfo::DecayCurve decayCurve,
        f32 decayRatio,
        int maxPriorityReduction,
        f32 maxVolumeDistance,
        f32 unitDistance,
        f32* volumePtr,
        int* priorityPtr
    );
    static void CalcPanImpl(
        const nw::math::VEC3& pos,
        f32 interiorSize,
        f32 actorDistance,
        f32 panRange,
        f32 stereoSpeakerAngle,
        f32 surroundSpeakerFrontAngle,
        f32 surroundSpeakerRearAngle,
        f32 surroundPanOffset,
        f32* panPtr,
        f32* surroundPanPtr
    );
    static void CalcPanSurround(
        const nw::math::VEC3& pos,
        f32 interiorSize,
        f32 actorDistance,
        f32 panRange,
        f32 surroundSpeakerFrontAngle,
        f32 surroundSpeakerRearAngle,
        f32 surroundPanOffset,
        f32* panPtr,
        f32* surroundPanPtr
    );
#if 0
    static void CalcPanStereo(
        const nw::math::VEC3& pos,
        f32 interiorSize,
        f32 actorDistance,
        f32 panRange,
        f32 stereoSpeakerAngle,
        f32* panPtr,
        f32* surroundPanPtr
    );
#endif
    static void CalcAngleAndDistance(
        const nw::math::VEC3& pos,
        f32 actorDistance,
        f32 interiorSize,
        f32* anglePtr,
        f32* distancePtr
    );
};

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SOUND_3D_CALCULATOR_H_ */

