﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_RemoteSpeaker.h
 *
 * @file snd_RemoteSpeaker.h
 */

#ifndef NW_SND_REMOTE_SPEAKER_H_
#define NW_SND_REMOTE_SPEAKER_H_

#include <nw/types.h>

#if defined( NW_PLATFORM_CAFE )
  #include <cafe/os.h>
  #include <cafe/ax.h>
  #include <cafe/wenc.h>
#endif

namespace nw {
namespace snd {

namespace internal {
class RemoteSpeakerManager;
}

//---------------------------------------------------------------------------
//! @briefprivate
//! @brief  Cafe リモコンスピーカーを制御するためのクラスです。
//!
//!         本クラスのインスタンスを作成することはできません。
//!         @ref SoundSystem::GetRemoteSpeaker を呼び出してインスタンスを取得してください。
//!
//!         連続 8 分以上 Wii リモコンのスピーカーから出力し続けると、
//!         ノイズが発生する可能性があります。
//!         8 分以内に最低 1 秒間の無音期間を設けることでこれを回避できます。
//!         本クラスでは、@ref EnableOutput で出力が有効に設定されていても、
//!         出力データが無音の場合は自動的に出力を停止します。
//!
//!         詳しくは、Demo/snd/remote デモをご覧ください。
//!
//! @see SoundSystem::GetRemoteSpeaker
//!
//! @date 2011/08/10 初版 (8 分制約については、Cafe でも同条件かどうか確認中)
//---------------------------------------------------------------------------
class RemoteSpeaker
{
    friend class internal::RemoteSpeakerManager;

public:

    //---------------------------------------------------------------------------
    //! @brief 初期化や終了処理が完了した時に呼び出されるコールバック関数の型定義です。
    //!
    //!        Cafe SDK の WPADCallback と同じ引数・返り値になっています。
    //!
    //! @param[in] chan     WPAD_CHAN? のいずれかです。
    //! @param[in] result   WPAD_SPEAKER_??? のいずれかです。
    //!
    //! @see Initialize
    //! @see Finalize
    //!
    //! @date 2012/03/30 初版
    //---------------------------------------------------------------------------
    typedef void (*WpadCallback)( s32 ch, s32 result );

    //! @name 初期化
    //@{
    //---------------------------------------------------------------------------
    //! @brief  リモコンスピーカーの初期化を行います。
    //!
    //!         Cafe リモコンのスピーカーでサウンドを出力するための初期化を行います。
    //!         初期化を行うためには、 WPADInit 関数を呼び出して WPAD ライブラリの
    //!         初期化が完了していなければなりません。
    //!
    //!         この関数は、内部で WPADControlSpeaker 関数を呼び出し、
    //!         Cafe リモコンのスピーカーの制御コマンド WPAD_SPEAKER_ON と
    //!         WPAD_SPEAKER_PLAY を順に登録し、スピーカーからサウンドを再生可能な状態にします。
    //!
    //!         初期化処理は非同期で行われます。
    //!         セットアップが完了したときにコールバック関数 callback が呼び出されます。
    //!         コールバック関数は返り値にかかわらず呼び出されます。
    //!
    //! @param[in] callback 初期化完了時に呼び出されるコールバック関数です。
    //!
    //! @return 初期化の非同期処理開始前に失敗した場合は false を返します。
    //!         true を返す場合でも、非同期の初期化処理に失敗する可能性があります。
    //!
    //! @see Finalize
    //!
    //! @date 2012/03/30 引数の型を WPADCallback から RemoteSpeaker::WpadCallback に変更
    //! @date 2011/08/10 初版
    //---------------------------------------------------------------------------
#if defined(NW_PLATFORM_CAFE)
    bool Initialize( WpadCallback callback = NULL );
#else
    NW_DEPRECATED_FUNCTION_MSG( bool Initialize( WpadCallback callback ) { NW_UNUSED_VARIABLE(callback); return false; }, "This API is only supported for Cafe.");
#endif


    //---------------------------------------------------------------------------
    //! @brief  リモコンスピーカーの終了処理を行います。
    //!
    //!         リモコンが切断されたときには、この関数を呼び出してください。
    //!         この関数は、 内部で WPADControlSpeaker 関数を呼び出し、
    //!         Cafe リモコンのスピーカーの制御コマンド WPAD_SPEAKER_OFF を登録し、
    //!         スピーカーからサウンドを再生しない状態に設定します。
    //!
    //!         callback は、終了処理が完了したときに呼び出されるコールバック関数です。
    //!
    //! @param[in] callback 終了処理完了時に呼び出されるコールバック関数です。
    //!
    //! @see Initialize
    //!
    //! @date 2012/03/30 引数の型を WPADCallback から RemoteSpeaker::WpadCallback に変更
    //! @date 2011/08/10 初版
    //---------------------------------------------------------------------------
#if defined(NW_PLATFORM_CAFE)
    void Finalize( WpadCallback callback = NULL );
#else
    NW_DEPRECATED_FUNCTION_MSG( void Finalize( WpadCallback callback ) { NW_UNUSED_VARIABLE(callback); }, "This API is only supported for Cafe.");
#endif

    //---------------------------------------------------------------------------
    //! @brief  リモコンスピーカーの初期化が完了しているかどうかを取得します。
    //!
    //!         @ref Initialize の呼び出しが成功し、
    //!         Cafe リモコンのスピーカーへの制御コマンドの処理も完了した場合に
    //!         true を返します。
    //!
    //! @return リモコンスピーカーのセットアップが完了していれば true を、
    //!         そうでなければ false を返します。
    //!
    //! @see Initialize
    //!
    //! @date 2011/08/10 初版
    //---------------------------------------------------------------------------
#if defined(NW_PLATFORM_CAFE)
    bool IsAvailable() const { return m_State == STATE_SPEAKER_PLAY; }
#else
    NW_DEPRECATED_FUNCTION_MSG( bool IsAvailable() const { return false; }, "This API is only supported for Cafe.");
#endif
    //@}

    //! @name 出力
    //@{
    //---------------------------------------------------------------------------
    //! @brief  リモコンスピーカーの出力を有効にするかどうかを設定します。
    //!
    //!         初期状態では true が設定されています。
    //!
    //!         リモコンスピーカーの出力が有効になっている場合でも、
    //!         データが無音の場合は出力処理を行いません。
    //!
    //!         連続 8 分以上 Cafe リモコンのスピーカーから出力すると正常に再生されません。
    //!         そのため、8 分以内に最低 1 秒間の無音期間を作ってください。
    //!         なお、連続 8 分以上の出力が続いた場合は強制的に 1 秒間出力が停止されます。
    //!
    //! @param[in] enable   true は有効に、false は無効に設定します。
    //!
    //! @return 設定に成功したら true を、失敗したら false を返します。
    //!
    //! @see IsEnabledOutput
    //!
    //! @date 2011/08/10 初版
    //---------------------------------------------------------------------------
#if defined(NW_PLATFORM_CAFE)
    bool EnableOutput( bool enable );
#else
    NW_DEPRECATED_FUNCTION_MSG( bool EnableOutput( bool enable ) { NW_UNUSED_VARIABLE(enable); return false; }, "This API is only supported for Cafe.");
#endif

    //---------------------------------------------------------------------------
    //! @brief  リモコンスピーカーの出力が有効かどうかを調べます。
    //!
    //!         @ref EnableOutput で設定された状態を取得します。
    //!         @ref Initialize が呼ばれていない場合は false を返します。
    //!
    //! @return リモコンスピーカーの出力が有効であれば true を、
    //!         無効であれば false を返します。
    //!
    //! @see EnableOutput
    //!
    //! @date 2011/08/10 初版
    //---------------------------------------------------------------------------
#if defined(NW_PLATFORM_CAFE)
    bool IsEnabledOutput() const;
#else
    NW_DEPRECATED_FUNCTION_MSG( bool IsEnabledOutput() const { return false; }, "This API is only supported for Cafe.");
#endif

    //---------------------------------------------------------------------------
    //! @brief  リモコンスピーカーへ出力されているかどうかを取得します。
    //!
    //!         @ref IsEnabledOutput が true を返す場合でも、
    //!         データが無音の場合は実際には出力が行われません。
    //!         この関数では、実際にデータが出力されている場合のみ true を返します。
    //!
    //! @return リモコンスピーカーに出力されていれば true を、
    //!         そうでなければ false を返します。
    //!
    //! @see IsEnabledOutput
    //!
    //! @date 2011/08/10 初版
    //---------------------------------------------------------------------------
#if defined(NW_PLATFORM_CAFE)
    bool IsPlaying() const;
#else
    NW_DEPRECATED_FUNCTION_MSG( bool IsPlaying() const { return false; }, "This API is only supported for Cafe.");
#endif

    //---------------------------------------------------------------------------
    //! @brief  リモコンスピーカーへ連続出力している時間を取得します。
    //!
    //! @return 連続出力している時間をミリ秒単位で返します。
    //!
    //! @date 2011/08/10 初版
    //---------------------------------------------------------------------------
#if defined(NW_PLATFORM_CAFE)
    u32 GetContinuousPlayTime() const;
#else
    NW_DEPRECATED_FUNCTION_MSG( u32 GetContinuousPlayTime() const { return 0; }, "This API is only supported for Cafe.");
#endif
    //@}

    //! @briefprivate
    //! @return :private
    int detail_GetState() const { return m_State; }

private:
    static const int CONTINUOUS_PLAY_MINUTES = 8 * 60; // 連続再生可能な時間=8分
    static const int CONTINUOUS_PLAY_INTERVAL_MINUTES = 1; // 連続再生に必要な間隔=1秒
    static const int SAMPLES_PER_AUDIO_PACKET = 40;

    enum SpeakerState
    {
        STATE_INVALID,
        STATE_EXEC_SPEAKER_ON,
        STATE_SPEAKER_ON,
        STATE_EXEC_SPEAKER_PLAY,
        STATE_SPEAKER_PLAY,
        STATE_EXEC_SPEAKER_OFF,
        STATE_SPEAKER_OFF
    };
    enum SpeakerCommand
    {
        COMMAND_NONE,
        COMMAND_SPEAKER_ON,
        COMMAND_SPEAKER_PLAY,
        COMMAND_SPEAKER_OFF
    };

    RemoteSpeaker();
#if defined( NW_PLATFORM_CAFE )
    bool IsSetupBusy() const
    {
        return
            ( m_State == STATE_EXEC_SPEAKER_ON ) ||
            ( m_State == STATE_SPEAKER_ON ) ||
            ( m_State == STATE_EXEC_SPEAKER_PLAY )
            ;
    }
    void SetChannelIndex( int channelIndex ) { m_ChannelIndex = channelIndex; }
    void UpdateStreamData( const s16* axRemoteSamples );
    void Update();

    void InitParam();
    void ClearParam();
    void ExecCommand( SpeakerCommand cmd );
    bool IsAllSampleZero( const s16* axRemoteSamples );
    void NotifyCallback( s32 channel, s32 result );

    static void SpeakerOnCallback( s32 channel, s32 result );
    static void SpeakerPlayCallback( s32 channel, s32 result );
    static void SpeakerOffCallback( s32 channel, s32 result );

    static void ContinueAlarmHandler( OSAlarm* alarm, OSContext* context );
    static void IntervalAlarmHandler( OSAlarm* alarm, OSContext* context );

    bool m_InitFlag;
    bool m_PlayFlag;
    bool m_EnableFlag;
    bool m_FirstEncodeFlag;
    bool m_ValidCallbackFlag;
    bool m_CommandBusyFlag;
    volatile bool m_ContinueFlag;
    volatile bool m_IntervalFlag;

#endif // defined( NW_PLATFORM_CAFE )
    SpeakerState m_State;
#if defined( NW_PLATFORM_CAFE )
    SpeakerCommand m_UserCommand;
    SpeakerCommand m_InternalCommand;

    WENCInfo m_EncodeInfo;
    int m_ChannelIndex;

    WpadCallback m_WpadCallback;

    OSAlarm m_ContinueAlarm;
    OSAlarm m_InvervalAlarm;
    OSTime m_ContinueBeginTime;
#endif // defined( NW_PLATFORM_CAFE )
};

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_REMOTE_SPEAKER_H_ */

