﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_MULTI_VOICE_H_
#define NW_SND_MULTI_VOICE_H_

#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_Adpcm.h>
#include <nw/snd/snd_Voice.h>
#include <nw/ut/ut_LinkList.h>          // for ut::LinkListNode

namespace nw {
namespace snd {
namespace internal {
namespace driver {

class MultiVoiceManager;

class MultiVoice
{
private:
    NW_DISALLOW_COPY_AND_ASSIGN(MultiVoice);

    friend class nw::snd::internal::driver::MultiVoiceManager;

public:
    // ------------------------------------------------------------------------
    // 更新フラグ
    static const int UPDATE_START           = 0x0001 << 0;
    static const int UPDATE_PAUSE           = 0x0001 << 1;
    static const int UPDATE_SRC             = 0x0001 << 2;
    static const int UPDATE_MIX             = 0x0001 << 3;
    static const int UPDATE_LPF             = 0x0001 << 4;
    static const int UPDATE_BIQUAD          = 0x0001 << 5;
    static const int UPDATE_VE              = 0x0001 << 6;
    static const int UPDATE_REMOTE_FILTER   = 0x0001 << 7;

    // ------------------------------------------------------------------------
    // コールバック関数
    enum VoiceCallbackStatus
    {
        CALLBACK_STATUS_FINISH_WAVE,
        CALLBACK_STATUS_CANCEL,
        CALLBACK_STATUS_DROP_VOICE,
        CALLBACK_STATUS_DROP_DSP
    };

    typedef void (*VoiceCallback)(
        MultiVoice* voice,
        VoiceCallbackStatus status,
        void* callbackData
    );

    // ------------------------------------------------------------------------
    // パラメータ境界
    static const f32 VOLUME_MIN;
    static const f32 VOLUME_DEFAULT;
    static const f32 VOLUME_MAX;
    static const f32 PAN_LEFT;
    static const f32 PAN_CENTER;
    static const f32 PAN_RIGHT;
    static const f32 SPAN_FRONT;
    static const f32 SPAN_CENTER;
    static const f32 SPAN_REAR;
    static const f32 CUTOFF_FREQ_MIN;
    static const f32 CUTOFF_FREQ_MAX;
    static const f32 BIQUAD_VALUE_MIN;
    static const f32 BIQUAD_VALUE_MAX;
    static const f32 SEND_MIN;
    static const f32 SEND_MAX;
    static const u8 REMOTE_FILTER_MIN = 0;
    static const u8 REMOTE_FILTER_MAX = 127;

    static const u32 PRIORITY_NODROP = Voice::PRIORITY_NODROP;

    MultiVoice();
    ~MultiVoice();

    // ------------------------------------------------------------------------
    // 確保と解放
    bool Alloc(
        int channelCount,       // TODO: voiceOutCount を追加する
        int priority,
        MultiVoice::VoiceCallback callback,
        void* callbackData,
        VoiceRendererType rendererType
    );
    void Free();

    // ------------------------------------------------------------------------
    // 動作
    void Start();
    void Stop();
    void UpdateVoiceStatus();
    void Pause( bool flag );

    void Calc();
    void Update();

    // ------------------------------------------------------------------------
    // 動作状態
    bool IsActive() const   { return m_ChannelCount > 0; }
    bool IsRun()    const;
    bool IsPause()  const   { return m_IsPause == true; }
    bool IsPlayFinished() const;

    // ------------------------------------------------------------------------
    // パラメータ
    void SetSampleFormat( SampleFormat format );
    void SetSampleRate( int sampleRate );

    f32 GetVolume() const { return m_Volume; }
    void SetVolume( f32 volume );

    f32 GetPitch() const { return m_Pitch; }
    void SetPitch( f32 pitch );

    void SetPanMode( PanMode panMode );
    void SetPanCurve( PanCurve panCurve );


    f32 GetLpfFreq() const { return m_LpfFreq; }
    void SetLpfFreq( f32 lpfFreq );

    int GetBiquadType() const { return m_BiquadType; }
    f32 GetBiquadValue() const { return m_BiquadValue; }
    void SetBiquadFilter( int type, f32 value );

    int GetPriority() const { return m_Priority; }
    void SetPriority( int priority );

    void SetFrontBypass( bool isFrontBypass );

    void SetInterpolationType( u8 interpolationType );

    void SetRemoteFilter( u8 filter );
    u8 GetRemoteFilter() const { return m_RemoteFilter; }

    // ------------------------------------------------------------------------
    // ボイスパラメータ
    int GetSdkVoiceCount() const { return m_ChannelCount; }
    const Voice& detail_GetSdkVoice( int index ) const;

    // ------------------------------------------------------------------------
    // 出力パラメータ
    void SetOutputLine( u32 lineFlag );
    u32 GetOutputLine() const { return m_OutputLineFlag; }

    void SetTvParam( const OutputParam& param );
    const OutputParam& GetTvParam() const { return m_TvParam; }

    void SetDrcParam( u32 drcIndex, const OutputParam& param );
    const OutputParam& GetDrcParam( u32 drcIndex ) const
    {
        NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
        return m_DrcParam[drcIndex];
    }

    void SetRemoteParam( u32 remoteIndex, const RemoteOutputParam& param );
    const RemoteOutputParam& GetRemoteParam( u32 remoteIndex )
    {
        NW_ASSERT_MAXLT( remoteIndex, DRC_OUT_COUNT );
        return m_RemoteParam[remoteIndex];
    }

    // ------------------------------------------------------------------------
    // サンプルの調整
    u32 GetCurrentPlayingSample() const;

    nw::snd::SampleFormat GetFormat() const;

    static u32 FrameToByte( u32 sample, SampleFormat format );
    static void CalcOffsetAdpcmParam(
        nw::snd::AdpcmContext* context, // in/out
        const nw::snd::AdpcmParam& param,
        u32 offsetSamples,
        const void* dataAddress );

    void AppendWaveBuffer(
        int channel,
        /*int voiceOut,*/
        WaveBuffer* waveBuffer,
        bool lastFlag
    );

    void SetAdpcmParam(
        int channelIndex,
        const AdpcmParam& param
    );



private:
    struct PreMixVolume
    {
        f32 volume[CHANNEL_INDEX_NUM];
        f32 lrMixedVolume;
    };

    void InitParam(
        MultiVoice::VoiceCallback callback,
        void* callbackData
    );

    // Update関数
    void CalcSrc( bool initialUpdate );
    void CalcVe();
    void CalcMix();
    void CalcLpf();
    void CalcBiquadFilter();
    void CalcRemoteFilter();

    void CalcPreMixVolume(
            PreMixVolume* mix, const OutputParam& param, int channelIndex,
            OutputDevice device );
    void CalcTvMix( OutputMix* mix, const PreMixVolume& preMix );
    void CalcDrcMix( u32 drcIndex, OutputMix* mix, const PreMixVolume& preMix );
    void CalcRemoteMix( RemoteOutputMix* mix, const PreMixVolume& preMix );
    void CalcMixImpl(
            OutputMix* mix,
            u32 outputDeviceIndex,
            const OutputParam& param,
            const PreMixVolume& pre );

    void RunAllSdkVoice();
    void StopAllSdkVoice();
    void PauseAllSdkVoice();

    void SetOutputParamImpl( const OutputParam& in, OutputParam& out );

private:
    Voice    m_Voice[ WAVE_CHANNEL_MAX ];

    int                 m_ChannelCount;

    VoiceCallback       m_Callback;
    void*               m_pCallbackData;

    bool                m_IsActive;
    bool                m_IsStart;
    bool                m_IsStarted;
    bool                m_IsPause;
    bool                m_IsPausing;
    bool                m_IsInitialized;

    WaveBuffer* m_pLastWaveBuffer;

    u16                 m_SyncFlag;

    u8                  m_BiquadType;
    u8                  m_RemoteFilter; // 0(none) - 127(max)

    bool                m_IsEnableFrontBypass;

    f32                 m_Volume;
    f32                 m_Pitch;
    PanMode             m_PanMode;
    PanCurve            m_PanCurve;
    f32                 m_LpfFreq;
    f32                 m_BiquadValue;
    int                 m_Priority;

    u32                 m_OutputLineFlag;

    // L/R: -1.0f(left) - 0.0f(center) - 1.0f(right)
    // F/R: 0.0f(front) - 2.0f(rear)
    OutputParam m_TvParam;
    OutputParam m_DrcParam[DRC_OUT_COUNT];
    RemoteOutputParam m_RemoteParam[REMOTE_OUT_COUNT];

    nw::snd::SampleFormat m_Format;

    uptr m_VoiceUser;    // Channel or StreamTrack

public:
    ut::LinkListNode    m_LinkNode;
};

} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

#endif /* NW_SND_MULTI_VOICE_H_ */

