﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_HARDWARE_MANAGER_H_
#define NW_SND_HARDWARE_MANAGER_H_

#include <nw/types.h>
#include <nw/ut/ut_LinkList.h>
#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_MoveValue.h>
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_FinalMixCallback.h>
#include <nw/snd/snd_FxBase.h>
#include <nw/snd/snd_BiquadFilterCallback.h>
#include <nw/snd/snd_BiquadFilterPresets.h>

#if defined( NW_PLATFORM_CAFE )
  #include <cafe/ax.h>
#elif defined( NW_PLATFORM_WIN32 )
  #include <winext/cafe/ax.h>
#elif defined( NW_USE_NINTENDO_SDK )
  // TODO: nn_audio
  #include <winext/cafe/ax.h>
#endif

namespace nw {
namespace snd {
namespace internal {
namespace driver {

class HardwareManager : public Util::Singleton<HardwareManager>
{
public:
    typedef void (*HardwareCallback)();

public:
    static const u32 SOUND_FRAME_INTERVAL_MSEC = AX_MS_PER_FRAME;
    static const u32 SOUND_FRAME_INTERVAL_USEC = AX_MS_PER_FRAME * 1000;

    // 何ボイスあるか？
    static const u32 MAX_VOICE_COUNT = AX_MAX_VOICES;

    // ------------------------------------------------------------------------
    // 初期化
    void Initialize();
    bool IsInitialized() { return m_IsInitialized != 0; }
    void Finalize();

    void Update();

    static void FlushDataCache( void* address, u32 length );

    // ------------------------------------------------------------------------
    // 出力モード
    void SetOutputMode( OutputMode mode, OutputDevice device = OUTPUT_DEVICE_MAIN );
    OutputMode GetOutputMode( OutputDevice device = OUTPUT_DEVICE_MAIN ) const
    {
        return m_OutputMode[device];
    }
    OutputMode GetEndUserOutputMode(OutputDevice device = OUTPUT_DEVICE_MAIN ) const
    {
        return m_EndUserOutputMode[device];
    }
    void UpdateEndUserOutputMode();

    // ------------------------------------------------------------------------
    // 出力先デバイス
    void SetOutputDeviceFlag( u32 outputLineIndex, u8 outputDeviceFlag );
    u8 GetOutputDeviceFlag( u32 outputLineIndex ) const
    {
        if ( outputLineIndex < OUTPUT_LINE_INDEX_MAX )
        {
            return m_OutputDeviceFlag[outputLineIndex];
        }

        return 0;
    }

    // ------------------------------------------------------------------------
    // エフェクト

    // エフェクト管理
    typedef ut::LinkList< FxBase, offsetof(FxBase, m_Link) > FxList;
    static const SampleFormat FX_SAMPLE_FORMAT = SAMPLE_FORMAT_PCM_S32;
    static const int FX_SAMPLE_RATE = 32000;
    static const int FX_BUFFER_SIZE = 384;

    // エフェクト登録、削除、削除確認
    bool AppendEffect( AuxBus bus, FxBase* pFx, OutputDevice device );
    void ClearEffect( AuxBus bus, int fadeTimes, OutputDevice device );
    bool IsFinishedClearEffect( AuxBus bus, OutputDevice device ) const
    {
        NW_MINMAXLT_ASSERT( bus, AUX_BUS_A, AUX_BUS_A + AUX_BUS_NUM );
        NW_MINMAXLT_ASSERT( device, 0, OUTPUT_DEVICE_COUNT );
        const FxList& list = GetEffectList(bus, device);
        if ( list.IsEmpty() )
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    // リターンボリューム
    void SetAuxReturnVolume( AuxBus bus, f32 volume, int fadeTimes, OutputDevice device )
    {
        NW_MINMAXLT_ASSERT( bus, AUX_BUS_A, AUX_BUS_A + AUX_BUS_NUM );
        NW_MINMAXLT_ASSERT( device, 0, OUTPUT_DEVICE_COUNT );
        m_AuxUserVolume[ device ][ bus ].SetTarget( volume, fadeTimes );
    }
    f32 GetAuxReturnVolume( AuxBus bus, OutputDevice device ) const
    {
        NW_MINMAXLT_ASSERT( bus, AUX_BUS_A, AUX_BUS_A + AUX_BUS_NUM );
        NW_MINMAXLT_ASSERT( device, 0, OUTPUT_DEVICE_COUNT );
        return m_AuxUserVolume[ device ][ bus ].GetValue();
    }

#ifdef NW_PLATFORM_CTR
    // エフェクト負荷計測
    nn::os::Tick GetEffectProcessTick( AuxBus bus ) const
    {
        return m_EffectProcessTick[ bus ];
    }
#endif


    // ------------------------------------------------------------------------
    f32 GetOutputVolume() const;

    // ------------------------------------------------------------------------
    // マスターボリューム
    void SetMasterVolume( float volume, int fadeTimes );
    f32 GetMasterVolume() const { return m_MasterVolume.GetValue(); }

    // ------------------------------------------------------------------------
    // SRC タイプ
    void SetSrcType( SrcType type );
    SrcType GetSrcType() const { return m_SrcType; }

    // ------------------------------------------------------------------------
    // Biquad フィルタ
    void SetBiquadFilterCallback( int type, const BiquadFilterCallback* cb );
    const BiquadFilterCallback* GetBiquadFilterCallback( int type )
    {
        NW_MINMAX_ASSERT( type, BIQUAD_FILTER_TYPE_DATA_MIN, BIQUAD_FILTER_TYPE_MAX );
        return m_BiquadFilterCallbackTable[ type ];
    }

    // ------------------------------------------------------------------------
    // リセット前準備
    void PrepareReset();
    bool IsResetReady() const;

    void AppendFinalMixCallback( FinalMixCallback* userCallback );
    void PrependFinalMixCallback( FinalMixCallback* userCallback );
    void EraseFinalMixCallback( FinalMixCallback* userCallback );

    void AppendReadOnlyFinalMixCallback( ReadOnlyFinalMixCallback* userCallback );
    void EraseReadOnlyFinalMixCallback( ReadOnlyFinalMixCallback* userCallback );

    HardwareManager();

private:
#if defined( NW_PLATFORM_WIN32 )
    typedef nw::internal::winext::AX_FINAL_MIX_CB_STRUCT AX_FINAL_MIX_CB_STRUCT;
    typedef nw::internal::winext::AXUserFinalMixCallback AXUserFinalMixCallback;
    typedef nw::internal::winext::AXAUXCBSTRUCT AXAUXCBSTRUCT;
    static const u32 AX_MAX_NUM_DEVICES = nw::internal::winext::AX_MAX_NUM_DEVICES;
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
    typedef nw::internal::winext::AX_FINAL_MIX_CB_STRUCT AX_FINAL_MIX_CB_STRUCT;
    typedef nw::internal::winext::AXUserFinalMixCallback AXUserFinalMixCallback;
    typedef nw::internal::winext::AXAUXCBSTRUCT AXAUXCBSTRUCT;
    static const u32 AX_MAX_NUM_DEVICES = nw::internal::winext::AX_MAX_NUM_DEVICES;
#elif defined( NW_USE_NINTENDO_SDK )
    // TODO: nn_audio
    typedef nw::internal::winext::AX_FINAL_MIX_CB_STRUCT AX_FINAL_MIX_CB_STRUCT;
    typedef nw::internal::winext::AXUserFinalMixCallback AXUserFinalMixCallback;
    typedef nw::internal::winext::AXAUXCBSTRUCT AXAUXCBSTRUCT;
    static const u32 AX_MAX_NUM_DEVICES = nw::internal::winext::AX_MAX_NUM_DEVICES;
#endif

    static void MainFinalMixCallbackFunc( AX_FINAL_MIX_CB_STRUCT* info );
    static void DrcFinalMixCallbackFunc( AX_FINAL_MIX_CB_STRUCT* info );
    void OnMainFinalMixCallback( AX_FINAL_MIX_CB_STRUCT* info );
    void OnDrcFinalMixCallback( AX_FINAL_MIX_CB_STRUCT* info );

private:
    static const u8 AUX_CALLBACK_WAIT_FRAME = 2;
    static const u16 AUX_RETURN_VOLUME_MAX = 0x8000;

    static void AuxCallbackFunc( s32** data, void* context, AXAUXCBSTRUCT* info );
    typedef void (*AuxCallback)( s32** data, void* context, AXAUXCBSTRUCT* info );

    static const BiquadFilterLpf        BIQUAD_FILTER_LPF;
    static const BiquadFilterHpf        BIQUAD_FILTER_HPF;
    static const BiquadFilterBpf512     BIQUAD_FILTER_BPF_512;
    static const BiquadFilterBpf1024    BIQUAD_FILTER_BPF_1024;
    static const BiquadFilterBpf2048    BIQUAD_FILTER_BPF_2048;

    void FinalizeEffect( AuxBus bus, OutputDevice device );
    FxList& GetEffectList( AuxBus bus, OutputDevice device )
    {
        NW_MINMAXLT_ASSERT( bus, AUX_BUS_A, AUX_BUS_A + AUX_BUS_NUM );
        NW_MINMAXLT_ASSERT( device, 0, OUTPUT_DEVICE_COUNT );
        return m_FxList[ device ][ bus ];
    }
    const FxList& GetEffectList( AuxBus bus, OutputDevice device ) const
    {
        NW_MINMAXLT_ASSERT( bus, AUX_BUS_A, AUX_BUS_A + AUX_BUS_NUM );
        NW_MINMAXLT_ASSERT( device, 0, OUTPUT_DEVICE_COUNT );
        return m_FxList[ device ][ bus ];
    }

    bool                m_IsInitialized;

    OutputMode          m_OutputMode[OUTPUT_DEVICE_COUNT];
    OutputMode          m_EndUserOutputMode[OUTPUT_DEVICE_COUNT];   // Cafe メニュー上での設定
    SrcType             m_SrcType;

    MoveValue<f32,int>  m_MasterVolume;
    MoveValue<f32,int>  m_VolumeForReset;
    MoveValue<f32,int>  m_AuxFadeVolume[ OUTPUT_DEVICE_COUNT ][ AUX_BUS_NUM ];
    MoveValue<f32,int>  m_AuxUserVolume[ OUTPUT_DEVICE_COUNT ][ AUX_BUS_NUM ];
    FxList              m_FxList[ OUTPUT_DEVICE_COUNT ][ AUX_BUS_NUM ];
    AuxCallback         m_AuxCallback[ AX_MAX_NUM_DEVICES ][ AUX_BUS_NUM ];    // Initialize 前の登録 CB

    typedef ut::LinkList< FinalMixCallback, offsetof(FinalMixCallback, m_Link) > FinalMixCallbackList;
    typedef ut::LinkList< ReadOnlyFinalMixCallback, offsetof(ReadOnlyFinalMixCallback, m_Link) > ReadOnlyFinalMixCallbackList;

    FinalMixCallbackList             m_FinalMixCallbackList;
    ReadOnlyFinalMixCallbackList     m_ReadOnlyFinalMixCallbackList;
    AXUserFinalMixCallback           m_LastMainFinalMixCallback;
    AXUserFinalMixCallback           m_LastDrcFinalMixCallback;
    ut::CriticalSection              m_CriticalSection;

    const BiquadFilterCallback* m_BiquadFilterCallbackTable[BIQUAD_FILTER_TYPE_MAX+1];

    void*               m_AuxCallbackContext[ AX_MAX_NUM_DEVICES ][ AUX_BUS_NUM ];
    nw::ut::Tick        m_EffectProcessTick[ OUTPUT_DEVICE_COUNT ][ AUX_BUS_NUM ];

    // コールバック登録後、バッファがクリアになるまで待つ
    // (AUX_CALLBACK_WAIT_FRAME [=2] フレーム)
    u8 m_AuxCallbackWaitCounter[ OUTPUT_DEVICE_COUNT ][ AUX_BUS_NUM ];

    u8 m_OutputDeviceFlag[ OUTPUT_LINE_INDEX_MAX ];

public:
    struct CallbackListNode
    {
        ut::LinkListNode link;
        HardwareCallback callback;
    };

    typedef ut::LinkList< CallbackListNode, offsetof(CallbackListNode,link)> CallbackList;
};

} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_HARDWARE_MANAGER_H_ */

