﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_Global.h
 *
 * @file snd_Global.h
 */

#ifndef NW_SND_GLOBAL_H_
#define NW_SND_GLOBAL_H_

#include <nw/types.h>
#include <nw/config.h>
#include <nw/ut/res/ut_ResTypes.h>
#include <nw/ut.h>
#include <nw/snd/snd_Adpcm.h>

#if defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
#include <limits.h>
#include <stdlib.h>
#endif

#if defined( NW_PLATFORM_CAFE )
  #include <cafe/pad.h>   // PAD_MAX_CONTROLLERS
  #include <cafe/vpad.h>  // VPADBASE_MAX_CONTROLLERS
#else
  #define PAD_MAX_CONTROLLERS      4
  #define VPADBASE_MAX_CONTROLLERS 2
#endif

namespace nw {
namespace snd {

/* ========================================================================
        sndライブラリ共通のグローバル定義
   ======================================================================== */

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @brief    サウンドフレーム処理から呼ばれるユーザーコールバック関数の型定義です。
//!
//!           @ref SoundSystem::SetSoundFrameUserCallback で登録することができます。
//!
//! @param[in] arg    コールバック関数のユーザー引数です。
//!
//! @see SoundSystem::SetSoundFrameUserCallback
//!
//! @date 2011/07/21 初版
//---------------------------------------------------------------------------
typedef void(*SoundFrameUserCallback)( uptr arg );

//---------------------------------------------------------------------------
//! @brief    サウンドの再生を停止するときに呼ばれるコールバック関数の型定義です。
//!
//!           @ref SoundStartable::StartInfo で登録することができます。
//!
//! @see SoundStartable::StartInfo
//!
//! @date 2013/04/05 初版
//---------------------------------------------------------------------------
typedef void(*SoundStopCallback)();

//---------------------------------------------------------------------------
//! @brief    サウンドの出力モードの定義です。
//!
//! @see      SoundSystem::SetOutputMode
//! @see      SoundSystem::GetOutputMode
//!
//! @date   2011/11/25 OUTPUT_MODE_SURROUND を追加
//! @date   2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
enum OutputMode
{
    OUTPUT_MODE_MONO,       //!< モノラルモードです。パンは効果がありません。
    OUTPUT_MODE_STEREO,     //!< ステレオモードです。
    OUTPUT_MODE_SURROUND,   //!< サラウンドモードです。
    OUTPUT_MODE_DPL2,       //!< @briefprivate
    OUTPUT_MODE_NUM         //!< 出力モードの総数です。
};


// ------------------------------------------------------------------------
// スピーカー

//---------------------------------------------------------------------------
//! @brief  サウンドの出力先デバイスを表す定数です。
//!
//! @see    SoundSystem::SetOutputDeviceFlag
//! @see    SoundSystem::GetOutputDeviceFlag
//!
//! @date 2013/05/20 初版
//---------------------------------------------------------------------------
enum OutputDeviceIndex
{
    OUTPUT_DEVICE_INDEX_MAIN,    //!< メイン出力 (TV 出力) です。
    OUTPUT_DEVICE_INDEX_REMOTE0, //!< チャンネル 0 のリモコンスピーカーです。
    OUTPUT_DEVICE_INDEX_REMOTE1, //!< チャンネル 1 のリモコンスピーカーです。
    OUTPUT_DEVICE_INDEX_REMOTE2, //!< チャンネル 2 のリモコンスピーカーです。
    OUTPUT_DEVICE_INDEX_REMOTE3, //!< チャンネル 3 のリモコンスピーカーです。
    OUTPUT_DEVICE_INDEX_DRC0,    //!< DRC 0 のスピーカーです。
    OUTPUT_DEVICE_INDEX_DRC1,    //!< DRC 1 のスピーカーです。
    OUTPUT_DEVICE_INDEX_COUNT    //!< 出力先デバイスの数です。
};

//---------------------------------------------------------------------------
//! @brief  サウンドの出力先を表すインデックスです。
//!
//! @date 2013/05/20 初版
//---------------------------------------------------------------------------
enum OutputLineIndex
{
    OUTPUT_LINE_INDEX_MAIN,         //!< メイン出力 (TV への出力) を表します。
    OUTPUT_LINE_INDEX_REMOTE0,      //!< チャンネル 0 のリモコンスピーカーへの出力を表します。
    OUTPUT_LINE_INDEX_REMOTE1,      //!< チャンネル 1 のリモコンスピーカーへの出力を表します。
    OUTPUT_LINE_INDEX_REMOTE2,      //!< チャンネル 2 のリモコンスピーカーへの出力を表します。
    OUTPUT_LINE_INDEX_REMOTE3,      //!< チャンネル 3 のリモコンスピーカーへの出力を表します。
    OUTPUT_LINE_INDEX_DRC0,         //!< DRC 0 スピーカーへの出力を表します。
    OUTPUT_LINE_INDEX_DRC1,         //!< DRC 1 スピーカーへの出力を表します。
    OUTPUT_LINE_INDEX_RESERVED_MAX, //!< 予約済みの出力先の最大数です。これより前の出力先は設定変更ができません。
    OUTPUT_LINE_INDEX_USER_0 = 16,  //!< ユーザ定義の出力 0 を表します。
    OUTPUT_LINE_INDEX_USER_1,       //!< ユーザ定義の出力 1 を表します。
    OUTPUT_LINE_INDEX_USER_2,       //!< ユーザ定義の出力 2 を表します。
    OUTPUT_LINE_INDEX_USER_3,       //!< ユーザ定義の出力 3 を表します。
    OUTPUT_LINE_INDEX_MAX = 32      //!< 出力先の最大数です。
};

//---------------------------------------------------------------------------
//! @brief  サウンドの出力先を表すビットフラグです。
//!
//! @see SoundHandle::SetOutputLine
//! @see SoundPlayer::SetDefaultOutputLine
//! @see SequenceSoundHandle::SetTrackOutputLine
//! @see StreameSoundHandle::SetTrackOutputLine
//!
//! @date 2011/11/25 @ref SequenceSoundHandle および @ref StreamSoundHandle への参照を追加
//! @date 2011/08/19 リモコンスピーカー対応にともない、記述を調整
//! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
enum OutputLine
{
    OUTPUT_LINE_MAIN         = 1 << OUTPUT_LINE_INDEX_MAIN,         //!< メイン出力 (TV への出力) を表します。
    OUTPUT_LINE_REMOTE0      = 1 << OUTPUT_LINE_INDEX_REMOTE0,      //!< チャンネル 0 のリモコンスピーカーへの出力を表します。
    OUTPUT_LINE_REMOTE1      = 1 << OUTPUT_LINE_INDEX_REMOTE1,      //!< チャンネル 1 のリモコンスピーカーへの出力を表します。
    OUTPUT_LINE_REMOTE2      = 1 << OUTPUT_LINE_INDEX_REMOTE2,      //!< チャンネル 2 のリモコンスピーカーへの出力を表します。
    OUTPUT_LINE_REMOTE3      = 1 << OUTPUT_LINE_INDEX_REMOTE3,      //!< チャンネル 3 のリモコンスピーカーへの出力を表します。
    OUTPUT_LINE_DRC0         = 1 << OUTPUT_LINE_INDEX_DRC0,         //!< DRC 0 スピーカーへの出力を表します。
    OUTPUT_LINE_DRC1         = 1 << OUTPUT_LINE_INDEX_DRC1,         //!< DRC 1 スピーカーへの出力を表します。
    OUTPUT_LINE_RESERVED_MAX = 1 << OUTPUT_LINE_INDEX_RESERVED_MAX, //!< 予約済みの出力先の最大数です。これより前の出力先は設定変更ができません。
    OUTPUT_LINE_USER_0       = 1 << OUTPUT_LINE_INDEX_USER_0,       //!< ユーザ定義の出力 0 を表します。
    OUTPUT_LINE_USER_1       = 1 << OUTPUT_LINE_INDEX_USER_1,       //!< ユーザ定義の出力 1 を表します。
    OUTPUT_LINE_USER_2       = 1 << OUTPUT_LINE_INDEX_USER_2,       //!< ユーザ定義の出力 2 を表します。
    OUTPUT_LINE_USER_3       = 1 << OUTPUT_LINE_INDEX_USER_3        //!< ユーザ定義の出力 3 を表します。
};

// ------------------------------------------------------------------------
// エフェクトバス

//---------------------------------------------------------------------------
//! @brief    AUX バス ID の定義です。
//!
//! @see SoundSystem::AppendEffect
//! @see SoundHandle::SetFxSend
//! @see SequenceSoundHandle::SetFxSend
//! @see StreamSoundHandle::SetFxSend
//! @see WaveSoundHandle::SetFxSend
//! @see SoundPlayer::SetFxSend
//! @see OutputAmbientParam
//!
//! @date 2011/11/25    @ref OutputAmbientParam への参照を追加
//! @date 2011/07/07    NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
enum AuxBus
{
    AUX_BUS_A,      //!< AUX バス A です。
    AUX_BUS_B,      //!< AUX バス B です。
    AUX_BUS_C,      //!< AUX バス C です。
    AUX_BUS_NUM     //!< AUX バスの本数です。
};

// ------------------------------------------------------------------------
// サンプルフォーマット

//---------------------------------------------------------------------------
//! @brief    サンプルフォーマットです。
//!
//!           @ref FxBase::UpdateBuffer では、
//!           常に SAMPLE_FORMAT_PCM_S32 が入ります。
//!
//! @see FxBase::UpdateBuffer
//!
//! @date 2011/07/07    NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
enum SampleFormat
{
    SAMPLE_FORMAT_PCM_S8,       //!< 符号付き 8 ビットフォーマットです。
    SAMPLE_FORMAT_PCM_S16,      //!< 符号付き 16 ビットフォーマットです。
    SAMPLE_FORMAT_DSP_ADPCM,    //!< DSP ADPCM フォーマットです。
    SAMPLE_FORMAT_PCM_S32       //!< 符号付き 32 ビットフォーマットです。
};

// ------------------------------------------------------------------------
// シーケンスミュート

//---------------------------------------------------------------------------
//! @brief    シーケンストラックのミュート、及びミュート解除の定義です。
//!
//! @see SequenceSoundHandle::SetTrackMute
//!
//! @date 2011/07/07    NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
enum SeqMute
{
    //---------------------------------------------------------------------------
    //! @brief    ミュート状態を解除します。
    //---------------------------------------------------------------------------
    SEQ_MUTE_OFF,

    //---------------------------------------------------------------------------
    //! @brief    シーケンストラックをミュート状態にします。
    //!           そのトラックで発音中の音は変化しません。
    //---------------------------------------------------------------------------
    SEQ_MUTE_NO_STOP,

    //---------------------------------------------------------------------------
    //! @brief    シーケンストラックをミュート状態にします。
    //!           同時に、そのトラックで発音中の音をエンベロープのリリースを発音してから、
    //!           緩やかに停止させます。
    //---------------------------------------------------------------------------
    SEQ_MUTE_RELEASE,

    //---------------------------------------------------------------------------
    //! @brief    シーケンストラックをミュート状態にします。
    //!           同時に、そのトラックで発音中の音を直ちに停止します。
    //---------------------------------------------------------------------------
    SEQ_MUTE_STOP
};

//---------------------------------------------------------------------------
//! @brief  DRC 出力の最大個数です。
//!
//! @date 2013/04/18 値が 1 固定であるという記述を削除
//! @date 2011/11/25 初版
//---------------------------------------------------------------------------
static const int DRC_OUT_COUNT = VPADBASE_MAX_CONTROLLERS;

//! @brief 出力先デバイスの定義です
enum OutputDevice
{
    OUTPUT_DEVICE_MAIN,                     //!< メイン出力 (テレビ出力) です。
    OUTPUT_DEVICE_DRC,                      //!< DRC 出力です。
    OUTPUT_DEVICE_DRC0 = OUTPUT_DEVICE_DRC, //!< DRC0 出力です。OUTPUT_DEVICE_DRC と同じです。
    OUTPUT_DEVICE_DRC1,                     //!< DRC1 出力です。
    OUTPUT_DEVICE_COUNT                     //! デバイスの数を表します。
};

/*!
  @briefprivate
  @brief    チャンネルの位置を表す列挙型です。
  @date 2011/07/07    NW4F 1.0.0 PR 公開に向けた調整
*/
enum ChannelIndex
{
    CHANNEL_INDEX_FRONT_LEFT  = 0,  //!< 左前チャンネルを表します。
    CHANNEL_INDEX_FRONT_RIGHT = 1,  //!< 右前チャンネルを表します。
    CHANNEL_INDEX_REAR_LEFT   = 2,  //!< 左後チャンネルを表します。
    CHANNEL_INDEX_REAR_RIGHT  = 3,  //!< 右後チャンネルを表します。
    CHANNEL_INDEX_FRONT_CENTER,
    CHANNEL_INDEX_LFE,
    CHANNEL_INDEX_NUM
};

//! @briefprivate
enum DrcOutChannelIndex
{
    DRC_OUT_CHANNEL_INDEX_LEFT = 0,
    DRC_OUT_CHANNEL_INDEX_RIGHT = 1,
    DRC_OUT_CHANNEL_INDEX_SURROUND = 2, // 対応されるかわからないが念のため
    DRC_OUT_CHANNEL_INDEX_NUM
};

// ------------------------------------------------------------------------
// 波形データ
// 2ch [ステレオ] 波形までサポートします。
//! @briefprivate
static const u32 WAVE_CHANNEL_MAX = 2;

//! @briefprivate
static const u32 VOICE_OUT_MAX = 4;

//! @briefprivate
static const u32 SEQ_BANK_MAX = 4;

// ------------------------------------------------------------------------
// ステレオ時のパン動作モード
//! @briefprivate
static const int PAN_MODE_NUM = 2;

//---------------------------------------------------------------------------
//! @brief    パンの動作モードの定義です。
//!
//! @see      SoundArchive::SoundInfo
//!
//! @date 2011/07/07    NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
enum PanMode
{
    PAN_MODE_DUAL,      //!< ステレオを２本のモノラルとしてそれぞれに定位処理を行います。
    PAN_MODE_BALANCE,   //!< 左右チャンネルの音量バランスを処理します。
    PAN_MODE_INVALID    //!< 不正な種類のパンモードです。
};

// ------------------------------------------------------------------------
// パン計算の曲線

//! @briefprivate
static const int PAN_CURVE_NUM = 9;

//---------------------------------------------------------------------------
//! @brief    パンカーブの定義です。
//!
//!           パンのカーブは以下の項目の組み合わせで定義されています。
//!
//!           - 曲線の種類（平方根、三角関数、線形）
//!           - 曲線のスケール（パンが中央のときに 0dB とするか、
//!             パンが両端に振るられたときに 0dB とするか）
//!           - パンの曲線が 0dB を超えたときに、 0dB にクランプするかどうか
//!
//! @see      SoundArchive::SoundInfo
//!
//! @date 2011/07/07    NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
enum PanCurve
{
    PAN_CURVE_SQRT,             //!< 平方根のカーブ。中央で-3dB、両端に振り切ると0dBとなります。
    PAN_CURVE_SQRT_0DB,         //!< 平方根のカーブ。中央で0dB、両端に振り切ると+3dBとなります。
    PAN_CURVE_SQRT_0DB_CLAMP,   //!< 平方根のカーブ。中央で0dB、両端に振り切ると0dBとなります。
    PAN_CURVE_SINCOS,           //!< 三角関数のカーブ。中央で-3dB、両端に振り切ると0dBとなります。
    PAN_CURVE_SINCOS_0DB,       //!< 三角関数のカーブ。中央で0dB、両端に振り切ると+3dBとなります。
    PAN_CURVE_SINCOS_0DB_CLAMP, //!< 三角関数のカーブ。中央で0dB、両端に振り切ると0dBとなります。
    PAN_CURVE_LINEAR,           //!< 線形のカーブ。中央で-6dB、両端に振り切ると0dBとなります。
    PAN_CURVE_LINEAR_0DB,       //!< 線形のカーブ。中央で0dB、両端に振り切ると+6dBとなります。
    PAN_CURVE_LINEAR_0DB_CLAMP, //!< 線形のカーブ。中央で0dB、両端に振り切ると0dBとなります。
    PAN_CURVE_INVALID           //!< 不正な種類のパンカーブです。
};

//---------------------------------------------------------------------------
//! @brief  ウェーブサウンドが参照する波形の種類を示す列挙型です。
//!
//! @date 2014/05/20 初版
//---------------------------------------------------------------------------
enum WaveType
{
    WAVE_TYPE_NWWAV,        //!< bfwav 形式の波形です。ループに対応しています。
    WAVE_TYPE_DSPADPCM,     //!< SDK の dspadpcm.exe で出力した波形です。ループには対応していません。
    WAVE_TYPE_INVALID = -1  //!< 無効な種類の波形です。
};

//---------------------------------------------------------------------------
//! @brief    音量減衰カーブの列挙型です。
//!
//!           音量減衰のカーブは 3D サウンドで使用されています。
//!           @ref Sound3DParam  を参照してください。
//!
//! @see Sound3DParam 構造体
//!
//! @date 2011/07/07    NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
enum DecayCurve
{
    DECAY_CURVE_LOG    = 1, //!< 対数のカーブです。
    DECAY_CURVE_LINEAR = 2  //!< 線形のカーブです。
};

//! @briefprivate
struct DspAdpcmParam
{
#if defined(NW_PLATFORM_CAFE)
    ut::ResU16 coef[8][2];
    ut::ResU16 predScale;
    ut::ResU16 yn1;
    ut::ResU16 yn2;
#else
    u16 coef[8][2];
    u16 predScale;
    u16 yn1;
    u16 yn2;
#endif
};

// ADSHR カーブ
//! @briefprivate
struct AdshrCurve
{
    u8 attack;
    u8 decay;
    u8 sustain;
    u8 hold;
    u8 release;
    // u8 padding;
    // u16 padding2;

    AdshrCurve( u8 a = 0, u8 d = 0, u8 s = 0, u8 h = 0, u8 r = 0 )
    : attack( a ),
      decay( d ),
      sustain( s ),
      hold( h ),
      release( r )
    {}
};

// ------------------------------------------------------------------------
// Biquadフィルタの種類

//---------------------------------------------------------------------------
//! @brief  biquad フィルタの種類を示す列挙体です。
//!
//!         biquad フィルタでは、フィルタの係数を変更することにより
//!         様々な種類のフィルタを使用することができます。
//!
//!         ライブラリで用意されているプリセットの係数を利用するには、
//!         BIQUAD_FILTER_TYPE_LPF ～ BIQUAD_FILTER_TYPE_BPF2048 を指定してください。
//!
//!         biquad フィルタを使用しない場合は BIQUAD_FILTER_TYPE_NONE を指定してください。
//!
//!         また、ユーザーが設計したフィルタ係数を登録する際のタイプ番号として
//!         BIQUAD_FILTER_TYPE_USER_MIN ～ BIQUAD_FILTER_TYPE_USER_MAX
//!         までの値をご利用ください。
//!
//!         biquad フィルタは @ref SoundHandle::SetBiquadFilter
//!         関数から利用することができます。
//!
//! @see SoundHandle::SetBiquadFilter
//!
//! @date 2011/07/07    NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
enum BiquadFilterType
{
    BIQUAD_FILTER_TYPE_INHERIT = -1, //!< バイナリデータの値を使用します。

    BIQUAD_FILTER_TYPE_NONE    = 0,  //!< biquad フィルタを使用しません。
    BIQUAD_FILTER_TYPE_LPF     = 1,  //!< プリセットのローパスフィルタを使用します。
    BIQUAD_FILTER_TYPE_HPF     = 2,  //!< プリセットのハイパスフィルタを使用します。

    //! プリセットのバンドパスフィルタ (中心周波数 512Hz) を使用します。
    BIQUAD_FILTER_TYPE_BPF512  = 3,

    //! プリセットのバンドパスフィルタ (中心周波数 1024Hz) を使用します。
    BIQUAD_FILTER_TYPE_BPF1024 = 4,

    //! プリセットのバンドパスフィルタ (中心周波数 2048Hz) を使用します。
    BIQUAD_FILTER_TYPE_BPF2048 = 5,

    //! ユーザーが登録できるフィルタ係数の種類の最小値です。
    BIQUAD_FILTER_TYPE_USER_MIN = 64,

    BIQUAD_FILTER_TYPE_USER_0 = BIQUAD_FILTER_TYPE_USER_MIN,    //!< ユーザー定義 0 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_1,                                  //!< ユーザー定義 1 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_2,                                  //!< ユーザー定義 2 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_3,                                  //!< ユーザー定義 3 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_4,                                  //!< ユーザー定義 4 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_5,                                  //!< ユーザー定義 5 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_6,                                  //!< ユーザー定義 6 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_7,                                  //!< ユーザー定義 7 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_8,                                  //!< ユーザー定義 8 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_9,                                  //!< ユーザー定義 9 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_10,                                 //!< ユーザー定義 10 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_11,                                 //!< ユーザー定義 11 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_12,                                 //!< ユーザー定義 12 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_13,                                 //!< ユーザー定義 13 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_14,                                 //!< ユーザー定義 14 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_15,                                 //!< ユーザー定義 15 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_16,                                 //!< ユーザー定義 16 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_17,                                 //!< ユーザー定義 17 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_18,                                 //!< ユーザー定義 18 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_19,                                 //!< ユーザー定義 19 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_20,                                 //!< ユーザー定義 20 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_21,                                 //!< ユーザー定義 21 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_22,                                 //!< ユーザー定義 22 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_23,                                 //!< ユーザー定義 23 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_24,                                 //!< ユーザー定義 24 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_25,                                 //!< ユーザー定義 25 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_26,                                 //!< ユーザー定義 26 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_27,                                 //!< ユーザー定義 27 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_28,                                 //!< ユーザー定義 28 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_29,                                 //!< ユーザー定義 29 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_30,                                 //!< ユーザー定義 30 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_31,                                 //!< ユーザー定義 31 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_32,                                 //!< ユーザー定義 32 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_33,                                 //!< ユーザー定義 33 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_34,                                 //!< ユーザー定義 34 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_35,                                 //!< ユーザー定義 35 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_36,                                 //!< ユーザー定義 36 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_37,                                 //!< ユーザー定義 37 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_38,                                 //!< ユーザー定義 38 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_39,                                 //!< ユーザー定義 39 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_40,                                 //!< ユーザー定義 40 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_41,                                 //!< ユーザー定義 41 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_42,                                 //!< ユーザー定義 42 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_43,                                 //!< ユーザー定義 43 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_44,                                 //!< ユーザー定義 44 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_45,                                 //!< ユーザー定義 45 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_46,                                 //!< ユーザー定義 46 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_47,                                 //!< ユーザー定義 47 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_48,                                 //!< ユーザー定義 48 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_49,                                 //!< ユーザー定義 49 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_50,                                 //!< ユーザー定義 50 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_51,                                 //!< ユーザー定義 51 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_52,                                 //!< ユーザー定義 52 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_53,                                 //!< ユーザー定義 53 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_54,                                 //!< ユーザー定義 54 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_55,                                 //!< ユーザー定義 55 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_56,                                 //!< ユーザー定義 56 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_57,                                 //!< ユーザー定義 57 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_58,                                 //!< ユーザー定義 58 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_59,                                 //!< ユーザー定義 59 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_60,                                 //!< ユーザー定義 60 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_61,                                 //!< ユーザー定義 61 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_62,                                 //!< ユーザー定義 62 のフィルタを使用します。
    BIQUAD_FILTER_TYPE_USER_63,                                 //!< ユーザー定義 63 のフィルタを使用します。

    //! ユーザーが登録できるフィルタ係数の種類の最大値です。
    BIQUAD_FILTER_TYPE_USER_MAX = BIQUAD_FILTER_TYPE_USER_63,

    //! フィルタ係数の種類のデータ上の最小値です。
    BIQUAD_FILTER_TYPE_DATA_MIN = BIQUAD_FILTER_TYPE_NONE,

    //! フィルタ係数の種類の最小値です。
    BIQUAD_FILTER_TYPE_MIN = BIQUAD_FILTER_TYPE_INHERIT,

    //! フィルタ係数の種類の最大値です。
    BIQUAD_FILTER_TYPE_MAX = BIQUAD_FILTER_TYPE_USER_MAX
};

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @brief    SRC タイプ (サンプリングレートコンバータの補間方法) の定義です。
//!
//!           サンプリングレート変換時の補間方法を示します。
//!
//! @see      SoundSystem::SetSrcType
//! @see      SoundSystem::GetSrcType
//!
//! @date 2011/10/27 誤字修正 (補完→補間)
//! @date 2010/01/16 文言の調整
//! @date 2010/01/15 初版
//---------------------------------------------------------------------------
enum SrcType
{
    SRC_TYPE_NONE,      //!< 補間しません。
    SRC_TYPE_LINEAR,    //!< 線形補間します。
    SRC_TYPE_4TAP       //!< ４点補間します。
};


/*!
    @briefprivate
    @brief  各チャンネルのゲインを格納する構造体です。
    @date   2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
*/
struct OutputMix
{
    //! メインバスの各チャンネルのゲイン（1.0 が等倍）
    f32 mainBus[CHANNEL_INDEX_NUM];

    //! エフェクトバス の各チャンネルのゲイン（1.0 が等倍）
    f32 auxBus[AUX_BUS_NUM][CHANNEL_INDEX_NUM];
};

//---------------------------------------------------------------------------
//! @brief  リモコン出力の最大個数です。
//!
//!         PAD_MAX_CONTROLLERS つまり 4 が設定されています。
//!
//! @date 2011/11/25 初版
//---------------------------------------------------------------------------
static const int REMOTE_OUT_COUNT = PAD_MAX_CONTROLLERS;

//! @briefprivate
struct RemoteOutputMix
{
    f32 mainBus[REMOTE_OUT_COUNT];
    f32 auxBus[REMOTE_OUT_COUNT];
    bool enable;

    RemoteOutputMix()
    {
        Initialize();
    }
    void Initialize()
    {
        for ( int i = 0; i < REMOTE_OUT_COUNT; i++ )
        {
            mainBus[i] = 0.0f;
            auxBus[i] = 0.0f;
        }
        enable = false;
    }
};

//---------------------------------------------------------------------------
//! @brief  開始時刻と終了時刻を記録する構造体です。
//!
//! @see SoundProfile 構造体
//!
//! @date 2012/02/09 初版
//---------------------------------------------------------------------------
struct TimeRange
{
    //! 開始時刻です。
    nw::ut::Tick begin;

    //! 終了時刻です。
    nw::ut::Tick end;

    //! @brief  開始時刻～終了時刻までの時間を取得します。
    //! @return 開始時刻～終了時刻までの時間を返します。
    //! @date   2012/02/09 初版
    nw::ut::TimeSpan GetSpan() const
    {
        return end - begin;
    }
};

//---------------------------------------------------------------------------
//! @brief  サウンドプロファイル情報を表す構造体です。
//!
//!         内部で AXGetSwapProfile を呼び出し、
//!         取得できた AXPROFILE から算出した値を格納しています。
//!
//!         現状、内部実装の都合で dspCycles で正しい値が取得できません。
//!         ご了承ください。
//!
//!         ■ nwFrameProcess
//!
//!         - プレイヤーやサウンド、トラックなどのパラメータ更新処理が含まれます。
//!         - サウンドの数やサウンドに含まれるトラックの数、チャンネルの数、
//!         ノートオンの数に応じて、処理時間が長くなります。
//!
//!         ■ nwVoiceParamUpdate
//!
//!         - nwFrameProcess で計算されたボイスパラメータの反映処理が含まれます。
//!         - ボイスの数に応じて、処理時間が長くなります。
//!         - AXPROFILE の userCallbackStart/End に相当します。
//!
//!         ■ voiceRendering
//!
//!         - PPC ボイスレンダリング処理が含まれます。
//!         - レンダリングを PPC に指定したボイスの数に応じて、処理時間が長くなります。
//!         - AXPROFILE の axPPCStart/End に相当します。
//!
//!         ■ auxProcess
//!
//!         - AUX コールバック処理 (エフェクト処理) が含まれます。
//!         - @ref SoundSystem::AppendEffect しているエフェクトの数や種類に応じて、
//!         処理時間が長くなります。
//!         - AXPROFILE の auxProcessingStart/End に相当します。
//!
//!         ■ axFrameProcess
//!
//!         - nwVoiceParamUpdate や voiceRendering、auxProcess の他、
//!         AX 割り込み内でのすべての処理が含まれます。
//!         - AXPROFILE の axFrameStart/End に相当します。
//!
//!         各パラメータの対応関係は下記のようなイメージです。
//!
//! @code
//! AX Intr                nw::snd SoundThread
//!   .                            .
//!   .                            .
//!   +-- axFrameProcess           .
//!   |                            .
//!   |+-- voiceRendering          .
//!   ||                           .
//!   ||                           .
//!   |+--                         .
//!   |                            .
//!   |+-- auxProcess              .
//!   ||                           .
//!   ||                           .
//!   |+--                         .
//!   |                            .
//!   |+-- nwVoiceParamUpdate      .
//!   ||                           .
//!   ||                           .
//!   |+--                         .
//!   |                            .
//!   +--                          .
//!   .                            +-- nwFrameProcess
//!   .                            |
//!   .                            |
//!   .                            +--
//!   .                            .
//!   .                            .
//! @endcode
//!
//! @see ProfileReader::Read
//!
//! @date 2012/04/24 dspCycles で正しい値が取得できない旨、追記
//! @date 2012/02/09 初版
//---------------------------------------------------------------------------
struct SoundProfile
{
    //! NW4F サウンドスレッドで 1 フレームの処理をするのにかかった時間時間です。
    TimeRange nwFrameProcess;

    //! nwFrameProcess で計算されたボイスパラメータの反映処理にかかった時間です。
    TimeRange nwVoiceParamUpdate;

    //! PPC ボイスレンダリング処理にかかった時間です。
    TimeRange voiceRendering;

    //! AUX コールバック処理 (エフェクト処理) にかかった時間です。
    TimeRange auxProcess;

    //! @briefprivate
    //! ボイスパラメータ同期処理にかかった時間です。
    TimeRange syncVoiceParam;

    //! @briefprivate
    //! 出力フォーマッティング処理にかかった時間です。
    TimeRange outputFormatProcess;

    //! @briefprivate
    //! メインミックス処理にかかった時間です。
    TimeRange mainMixProcess;

    //! @briefprivate
    //! ファイナルミックス処理にかかった時間です。
    TimeRange finalMixProcess;

    //! AX 割り込みにかかった時間です。
    TimeRange axFrameProcess;

    //! DSP 処理にかかった時間です。
    TimeRange dspFrameProcess;


    //! @briefprivate
    //! 使用した全ボイスの数です
    u32 totalVoiceCount;

    //! @briefprivate
    //! 使用したDSPボイスの数です
    u32 dspVoiceCount;

    //! @briefprivate
    //! 使用したPPCボイスの数です
    u32 ppcVoiceCount;

    //! @briefprivate
    //! 使用したNWボイスの数です
    u32 nwVoiceCount;

    //! 使用したAXボイスの数です (AXPROFILE の axNumVoices に該当します)。
    u32 axNumVoices;

    //! DSP の処理サイクル数です (AXGetDspCycles で取得した値が格納されます)。
    u32 dspCycles;

    //! @briefprivate
    u64 axFrameProcessTick;

    //! @briefprivate
    u64 nwFrameProcessTick;

    //! @briefprivate
    //! @brief NW4F サウンドスレッドのフレーム同士の間隔を取得します。
    //!        通常 3 ミリ秒に近い値になります。
    //! @param prevNwFrameProcessBegin 前回のサウンドスレッドの処理の開始時間です。
    //! @return NW4F サウンドスレッドのフレーム同士の間隔を返します。
    nw::ut::TimeSpan GetNwFrameProcessInterval(const nw::ut::Tick& prevNwFrameProcessBegin) const
    {
        return nwFrameProcess.begin - prevNwFrameProcessBegin;
    }

    //! @briefprivate
    //! @brief NW4F サウンドスレッドのフレーム同士の間隔を取得します。
    //! @brief AXPROFILE の axFrameStart の差分 (今回の axFrameStart - 前回の axFrameStart) に該当します。
    //!        通常 3 ミリ秒に近い値になります。
    //! @param prevAxFrameProcessBegin 前回のAXPROFILE の axFrameStartです。
    //! @return AXPROFILE の axFrameStart の差分を返します。
    nw::ut::TimeSpan GetAxFrameProcessInterval(const nw::ut::Tick& prevAxFrameProcessBegin) const
    {
        return axFrameProcess.begin - prevAxFrameProcessBegin;
    }
};

// (TODO: Voice の公開時に、これも公開する)
/*!
    @briefprivate
    @brief  サンプルデータが格納されているバッファの情報を格納する構造体です。
    @date   2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
*/
struct WaveBuffer
{
    /*!
      @brief    バッファの状態を表す列挙型です。
      @date     2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    */
    enum Status
    {
        STATUS_FREE, //!< 初期状態（一度もAppendWaveBuffer で追加されていない状態）を表します。ボイスの管理外にあります。
        STATUS_WAIT, //!< ボイスのバッファリストにリンクされ、再生待ちの状態を表します。ボイスが管理しています。
        STATUS_PLAY, //!< 現在再生中の状態を表します。ボイスが管理しています。
        STATUS_DONE  //!< 再生完了状態を表します。ボイスの管理外にあります。
    };

    // ユーザ用
    const void * bufferAddress;   //!< サンプルデータ格納先アドレス。必須です。ボイスに登録後は変更しないで下さい。
    u32 sampleLength;             //!< バッファ内のサンプル長。必須です。ボイスに登録後は変更しないで下さい。
    u32 sampleOffset;

    const AdpcmContext * pAdpcmContext; //!< Adpcm コンテクストのアドレス。ボイスに登録後、バッファの再生が終了するまで変更しないで下さい。

    void* userParam;               //!< ユーザーパラメータ。任意です。

    bool loopFlag;                //!< ループフラグ。任意です。
    Status status;                //!< バッファの状態。ライブラリが使用するため、変更しないで下さい。
//    u16 bufferId;                 //!< バッファ管理用 ID。ライブラリが使用するため、変更しないでください。

    struct WaveBuffer * next;     //!< 次のバッファ構造体のアドレス。ライブラリが使用するため、変更しないで下さい。

    /*!
        @brief      コンストラクタです。
     */
    WaveBuffer() : status( STATUS_FREE )
    {
        Initialize();
    }
    ~WaveBuffer()
    {
        NW_ASSERT( status == STATUS_FREE || status == STATUS_DONE );
    }

    void Initialize()
    {
        bufferAddress = NULL;
        sampleLength = 0;
        sampleOffset = 0;
        pAdpcmContext = NULL;
        userParam = NULL;
        loopFlag = false;
        status = STATUS_FREE;
        next = NULL;
    }
#if 0
    void Dump()
    {
        NW_LOG("WaveBuffer(%p)::%s wav(%p) len(%d) ofs(%d) loop(%d) status(%d) next(%p)\n",
                this, __FUNCTION__, bufferAddress, sampleLength, sampleOffset, loopFlag, status, next);
    }
#endif
}; // WaveBuffer

//---------------------------------------------------------------------------
//! @brief  biquad フィルタ係数の構造体です。
//!
//!         各係数については、Cafe SDK リファレンスマニュアルの axvpb.pb.biquad
//!         の同名メンバ変数の説明をご参照ください。
//!
//! @date   2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
struct BiquadFilterCoefficients
{
    s16 b0; //!< x[n] の係数を表します。
    s16 b1; //!< x[n-1] の係数を表します。
    s16 b2; //!< x[n-2] の係数を表します。
    s16 a1; //!< y[n-1] の係数を表します。
    s16 a2; //!< y[n-2] の係数を表します。
};


//---------------------------------------------------------------------------
//! @brief  波形情報を格納した構造体です。
//!
//! @date 2013/10/02 初版
//---------------------------------------------------------------------------
struct FinalMixData
{
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @date 2013/10/02 初版
    //---------------------------------------------------------------------------
    FinalMixData()
    : frontLeft( NULL )
    , frontRight( NULL )
    , rearLeft( NULL )
    , rearRight( NULL )
    , frontCenter( NULL )
    , lfe( NULL )
    , sampleCount( 0 )
    , channelCount( 0 )
    {
    }

    s32* frontLeft;     //!< フロント左チャンネルの波形です。
    s32* frontRight;    //!< フロント右チャンネルの波形です。
    s32* rearLeft;      //!< リア左チャンネルの波形です。
    s32* rearRight;     //!< リア右チャンネルの波形です。
    s32* frontCenter;   //!< フロントセンターチャンネルの波形です。
    s32* lfe;           //!< LFE（低域効果音）チャンネルの波形です。

    u16 sampleCount;    //!< サンプル数です。
    u16 channelCount;   //!< チャンネル数です。
};

//! @briefprivate
enum VoiceRendererType
{
    VOICE_RENDERER_SDK,     //!< AXSetDefaultRenderer で設定したレンダラです。
    VOICE_RENDERER_SDK_DSP, //!< DSP レンダラです。
    VOICE_RENDERER_SDK_PPC, //!< PPC レンダラです。
    VOICE_RENDERER_NW,      //!< NW レンダラです。48KHz でレンダリングされます。
    VOICE_RENDERER_COUNT    //!< レンダラの数です。
};

//! @briefprivate
enum MixMode
{
    MIX_MODE_PAN,          //!< Pan と SurroundPan を指定するミックスです。
    MIX_MODE_MIXPARAMETER, //!< 全てのチャンネルのパラメータを指定するミックスです。
    MIX_MODE_NUM           //!< ミックスの数です。
};

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @brief  サラウンドボリューム(5.1chの各スピーカーの音量)の構造体です。
//!
//! @date   2012/12/10 初版
//---------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#pragma warning(push)
#pragma warning(disable:4201)
#endif
struct MixParameter
{
    union
    {
        struct
        {
            f32 fL;        //!< フロント左チャンネルのパラメータです。
            f32 fR;        //!< フロント右チャンネルのパラメータです。
            f32 rL;        //!< リア左チャンネルのパラメータです。
            f32 rR;        //!< リア右チャンネルのパラメータです。
            f32 fC;        //!< フロントセンターチャンネルのパラメータです。
            f32 lfe;       //!< LFE（低域効果音）チャンネルのパラメータです。
        };

        f32 ch[CHANNEL_INDEX_NUM];
    };

    /* ctor */ MixParameter()
    {
        fL = fR = rL = rR = fC = lfe = 1.0f;
    }

    /* ctor */ MixParameter( f32 _fL, f32 _fR, f32 _rL, f32 _rR, f32 _fC, f32 _lfe )
     : fL ( _fL ), fR ( _fR ), rL ( _rL ), rR ( _rR ), fC ( _fC ), lfe( _lfe )
    {}
};
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#pragma warning(pop)
#endif

/* ========================================================================
        非公開
   ======================================================================== */

namespace internal {

// ------------------------------------------------------------------------
// 波形データ
struct DspAdpcmLoopParam
{
#if defined(NW_PLATFORM_CAFE)
    ut::ResU16 loopPredScale;
    ut::ResU16 loopYn1;
    ut::ResU16 loopYn2;
#else
    u16 loopPredScale;
    u16 loopYn1;
    u16 loopYn2;
#endif
};

struct WaveInfo
{
    SampleFormat sampleFormat;
    bool loopFlag;
    int channelCount;
    int sampleRate;
    u32 loopStartFrame;
    u32 loopEndFrame;
    u32 originalLoopStartFrame;

    struct ChannelParam
    {
        const void* dataAddress;
        DspAdpcmParam adpcmParam;
        DspAdpcmLoopParam adpcmLoopParam;
    } channelParam[ WAVE_CHANNEL_MAX ];

    void Dump()
    {
    #ifdef NW_CONSOLE_ENABLE
        NW_LOG("[WaveInfo] fmt(%d) loop(%d) ch(%d) rate(%d) LS(%d) LE(%d) OLS(%d)\n",
                sampleFormat, loopFlag, channelCount, sampleRate,
                loopStartFrame, loopEndFrame, originalLoopStartFrame );
        for ( int i = 0; i < channelCount; i++ )
        {
            const ChannelParam& p = channelParam[i];
            const DspAdpcmParam& adpcm = p.adpcmParam;
            const DspAdpcmLoopParam& loop = p.adpcmLoopParam;
#if !defined(NW_PLATFORM_ANDROID) && !defined(NW_PLATFORM_IOS)
            NW_LOG("  %d/%d addr(%p)\n       ps(%5d)  yn1(%5d)  yn2(%5d)\n      lps(%5d) lyn1(%5d) lyn2(%5d)\n",
                    i, channelCount-1, p.dataAddress, adpcm.predScale, adpcm.yn1, adpcm.yn2,
                    loop.loopPredScale, loop.loopYn1, loop.loopYn2 );
            NW_LOG("      coef %5d %5d %5d %5d %5d %5d %5d %5d\n",
                    adpcm.coef[0][0], adpcm.coef[0][1], adpcm.coef[1][0], adpcm.coef[1][1],
                    adpcm.coef[2][0], adpcm.coef[2][1], adpcm.coef[3][0], adpcm.coef[3][1] );
            NW_LOG("           %5d %5d %5d %5d %5d %5d %5d %5d\n",
                    adpcm.coef[4][0], adpcm.coef[4][1], adpcm.coef[5][0], adpcm.coef[5][1],
                    adpcm.coef[6][0], adpcm.coef[6][1], adpcm.coef[7][0], adpcm.coef[7][1] );
#endif
        }
    #endif // NW_CONSOLE_ENABLE
    }
};

struct OutputParam
{
    f32 volume;
    u32 mixMode;
    MixParameter mixParameter[ WAVE_CHANNEL_MAX ];
    f32 pan;
    f32 span;
    f32 mainSend;
    f32 fxSend[ AUX_BUS_NUM ];

    void Initialize()
    {
        volume = 1.0f;
        mixMode = 0;
        pan = 0.0f;
        span = 0.0f;
        mainSend = 0.0f;
        for ( int i = 0; i < AUX_BUS_NUM; i++ )
        {
            fxSend[i] = 0.0f;
        }
    };
};

struct RemoteOutputParam
{
    f32 volume;
    f32 mainSend;
    f32 fxSend;

    void Initialize()
    {
        volume = 1.0f;
        mainSend = 0.0f;
        fxSend = 0.0f;
    }
};

enum StreamFileType
{
    STREAM_FILE_BFSTM
};

//---------------------------------------------------------------------------
//! @brief  サウンドライブラリ内部で保持しているプロファイルの最大数です。
//!
//!         @ref ProfileReader::Read の maxCount 引数の最大値としても利用可能です。
//!
//! @see ProfileReader::Read
//!
//! @date 2012/02/01 初版
//---------------------------------------------------------------------------
static const int MAX_PROFILE_COUNT = 32;

enum VoiceState
{
    VOICE_STATE_PLAY,  //!< 再生指令状態を表します。
    VOICE_STATE_STOP,  //!< 停止状態を表します。
    VOICE_STATE_PAUSE  //!< 一時停止状態を表します。再生を再開する際は、SetState により再生状態を設定して下さい。
};

struct VoiceParam
{
    f32                 m_Volume;               // ボリューム
    f32                 m_Pitch;                // ピッチ（SampleRate に対する比）
    OutputMix           m_TvMix;
    OutputMix           m_DrcMix[DRC_OUT_COUNT];
    RemoteOutputMix     m_RemoteMix;

    bool                m_MonoFilterFlag;
    bool                m_BiquadFilterFlag;
    bool                m_RemoteFilterFlag;
    bool                m_DrcFrontBypass;

    BiquadFilterCoefficients m_BiquadFilterCoefficients;
    u16                 m_MonoFilterCutoff;     // LPF のカットオフ周波数
    u8                  m_RemoteFilter;
    u8                  m_InterpolationType;    // 補間タイプ

    void Initialize();
};

struct VoiceInfo
{
    VoiceState voiceState;
    WaveBuffer::Status waveBufferStatus;
    void* waveBufferTag;
    u32 playPosition;
    void* userId;
};

#if defined(NW_PLATFORM_CAFE)
    static const int FILE_PATH_MAX  = FS_MAX_FULLPATH_SIZE;
#elif defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
    static const int FILE_PATH_MAX  = PATH_MAX;
#else
    static const int FILE_PATH_MAX  = 511+128;
#endif

} // namespace nw::snd::internal

} // namespace nw::snd
} // namespace nw

#endif /* NW_SND_GLOBAL_H_ */

