﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_FxMultiChDelay.h
 *
 * @file snd_FxMultiChDelay.h
 */

#ifndef NW_SND_FX_MULTICH_DELAY_H_
#define NW_SND_FX_MULTICH_DELAY_H_

#include <nw/types.h>

#include <nw/snd/snd_FxParam.h>
#include <nw/snd/snd_FxBase.h>
#include <nw/snd/snd_AxfxImpl.h>

#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief  指定されたチャンネル数に対するディレイエフェクトです。
//!
//!         FxMultiChDelay は SDK の AXFXMultiChDelay エフェクトをラッピングしたクラスです。
//!
//! @see FxBase クラス
//!
//! @date 2013/04/05 初版
//---------------------------------------------------------------------------
class FxMultiChDelay : public FxBase
{
    /* ------------------------------------------------------------------------
            constant definition
       ------------------------------------------------------------------------ */
public:
    static const f32 DELAY_MIN;          //!< MultiChDelayParam::delay の最小値 (0.0) です。
    static const f32 DELAY_MAX;          //!< MultiChDelayParam::delay の最大値 (2000.0) です。
    static const f32 FEEDBACK_MIN;       //!< MultiChDelayParam::feedback の最小値 (0.1) です。
    static const f32 FEEDBACK_MAX;       //!< MultiChDelayParam::feedback の最大値 (1.0) です。
    static const f32 LPF_MIN;            //!< MultiChDelayParam::lpf の最小値 (0.1) です。
    static const f32 LPF_MAX;            //!< MultiChDelayParam::lpf の最大値 (1.0) です。
    static const f32 OUT_GAIN_MIN;       //!< MultiChDelayParam::outGain の最小値 (0.1) です。
    static const f32 OUT_GAIN_MAX;       //!< MultiChDelayParam::outGain の最小値 (1.0) です。
    static const f32 CHANNEL_SPREAD_MIN; //!< MultiChDelayParam::channelSpread の最小値 (0.1) です。
    static const f32 CHANNEL_SPREAD_MAX; //!< MultiChDelayParam::channelSpread の最小値 (1.0) です。

    /* ------------------------------------------------------------------------
            typename definition
       ------------------------------------------------------------------------ */
public:
    //---------------------------------------------------------------------------
    //! @brief  チャンネル指定版ディレイエフェクトパラメータの構造体です。
    //!
    //! @see SetParam
    //! @see GetParam
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    struct MultiChDelayParam
    {
        //---------------------------------------------------------------------------
        //! @brief    ディレイタイム（遅延時間）です。
        //!           値の範囲は 0.0～2000.0 (msec 単位) です。
        //!           初期値は 160.0 です。
        //---------------------------------------------------------------------------
        f32 delay;
        //---------------------------------------------------------------------------
        //! @brief    ディレイ音のフィードバックゲインです。
        //!           値の範囲は 0.1～1.0 です。
        //!           初期値は 0.4 です。
        //---------------------------------------------------------------------------
        f32 feedback;
        //---------------------------------------------------------------------------
        //! @brief    出力ゲインです。
        //!           値の範囲は 0.1～1.0 です。
        //!           初期値は 1.0 です。
        //---------------------------------------------------------------------------
        f32 outGain;
        //---------------------------------------------------------------------------
        //! @brief    隣接するチャンネルに配分するサンプルの量です。
        //!           値の範囲は 0.1～1.0 です。
        //!           初期値は 0.4 です。
        //---------------------------------------------------------------------------
        f32 channelSpread;
        //---------------------------------------------------------------------------
        //! @brief    ディレイ音にかけるローパスフィルタの値です。
        //!           値の範囲は 0.1～1.0 です。
        //!           初期値は 1.0 です。
        //---------------------------------------------------------------------------
        f32 lpf;

        //---------------------------------------------------------------------------
        //! @brief    コンストラクタです。
        //! @date 2013/04/05 初版
        //---------------------------------------------------------------------------
        MultiChDelayParam()
        : delay        ( 160.0f ),
          feedback     (   0.4f ),
          outGain      (   1.0f ),
          channelSpread(   0.4f ),
          lpf          (   1.0f )
        {}
    };

    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */
public:
    //! @name コンストラクタ/デストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //!         エフェクトパラメータは、下記のように初期化されます。
    //!
    //!         - delay = 160.0
    //!         - feedback = 0.4
    //!         - outGain = 1.0
    //!         - channelSpread = 0.4
    //!         - lpf = 1.0
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    FxMultiChDelay();

    //---------------------------------------------------------------------------
    //! @brief    デストラクタです。
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    virtual ~FxMultiChDelay() { Finalize(); ReleaseWorkBuffer(); }

    //@}

    //! @name メモリ割り当て
    //@{

    //---------------------------------------------------------------------------
    //! @brief    FxMultiChDelay が必要とするメモリのサイズを取得します。
    //!
    //!           FxMultiChDelay が現在のパラメータ設定でエフェクト処理を行うために必要とする
    //!           メモリのサイズを取得します。
    //!
    //!           この関数を呼び出す前に @ref SetChannelMode を呼び出してチャンネル数を
    //!           設定してください。呼び出さない場合は、初期設定の値が使用されます。
    //!
    //!           エフェクトを使用するためには、取得したメモリサイズのメモリを確保
    //!           した後に @ref AssignWorkBuffer を呼びだし、エフェクトにメモリ領域を
    //!           割り当ててください。
    //!
    //! @return   FxMultiChDelay が必要とするメモリのサイズを返します。
    //!
    //! @see  SetChannelMode
    //! @see  AssignWorkBuffer
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    u32 GetRequiredMemSize();

    //---------------------------------------------------------------------------
    //! @brief    エフェクトで使用するバッファ領域を割り当てます。
    //!
    //!           バッファ領域の割り当てはエフェクトを使用する前に行う必要があります。
    //!
    //!           エフェクトが必要とするバッファサイズはパラメータの値によって変化しますので、
    //!           @ref GetRequiredMemSize を呼び出してバッファサイズを取得してください。
    //!
    //! @param[in] buffer   割り当てるバッファの先頭アドレス。
    //! @param[in] size     割り当てるバッファのサイズ。
    //!
    //! @return   バッファ領域の割り当てに成功した場合は true を、失敗した場合は false を返します。
    //!
    //! @see  GetRequiredMemSize
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    virtual bool AssignWorkBuffer( void* buffer, u32 size );

    //---------------------------------------------------------------------------
    //! @brief  エフェクトで使用していたメモリを解放します。
    //!
    //!         @ref AssignWorkBuffer で割り当てたバッファ領域を解放します。 この関数で
    //!         解放を行う前に、@ref Finalize を呼び出してエフェクト処理を終了させる
    //!         必要があります。
    //!
    //! @see  AssignWorkBuffer
    //! @see  Finalize
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    virtual void ReleaseWorkBuffer();

    //@}

    //! @name エフェクト
    //@{

    //---------------------------------------------------------------------------
    //! @brief    エフェクトの開始処理を行います。
    //!
    //!           @ref SoundSystem クラスを通じてエフェクトを使用する場合、
    //!           @ref SoundSystem クラスからこの関数が呼び出されます。 ユーザがこの関数を
    //!           呼び出す必要はありません。
    //!
    //!           この関数を呼び出す前に、@ref AssignWorkBuffer を呼びだし、エフェクトで
    //!           使用するワークメモリを割り当てる必要があります。
    //!
    //! @return   エフェクトの開始に成功したら true を、失敗したら false を返します。
    //!
    //! @see  SoundSystem クラス
    //! @see  AssignWorkBuffer
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    virtual bool Initialize();

    //---------------------------------------------------------------------------
    //! @brief    エフェクトの終了処理を行います。
    //!
    //!           @ref SoundSystem クラスを通じてエフェクトを使用する場合、
    //!           @ref SoundSystem クラスからこの関数が呼び出されます。 ユーザがこの関数を
    //!           呼び出す必要はありません。
    //!
    //! @see  SoundSystem クラス
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    virtual void Finalize();

    //---------------------------------------------------------------------------
    //! @brief    チャンネル数を設定します。
    //!
    //! @param[in] mode チャンネル数を指定する列挙型です。
    //!
    //! @return   エフェクト処理を開始する前は true を返しますが、
    //!           エフェクト処理中は false を返し、チャンネル数の設定に失敗します。
    //!
    //!           TV 出力に設定するエフェクトには全ての @ref ChannelMode を使用できますが、
    //!           DRC 出力に設定するエフェクトには 2ch, 4ch のみが使用できます。
    //!
    //!           DRC 出力に設定するエフェクトで 6ch を使用した場合、@ref SoundSystem::AppendEffect でエフェクトを追加する際に
    //!           Debug, Develop ビルドではアサートとなり、Release ビルドでは追加に失敗するためご注意ください。
    //!
    //! @see  GetRequiredMemSize
    //! @see  GetChannelMode
    //!
    //! @date 2013/07/31 初版
    //---------------------------------------------------------------------------
    bool SetChannelMode( ChannelMode mode )
    {
        if ( m_IsActive )
        {
            return false;
        }
        else
        {
            m_ChannelMode = mode;
            return true;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    チャンネル数を取得します。
    //!
    //! @return   現在のチャンネル数です。
    //!
    //! @see  SetChannelMode
    //!
    //! @date 2013/07/31 初版
    //---------------------------------------------------------------------------
    const ChannelMode GetChannelMode() const { return m_ChannelMode; }

    //---------------------------------------------------------------------------
    //! @brief    サンプリングレートを設定します。
    //!
    //! @param[in] rate サンプリングレートを指定する列挙型です。
    //!
    //! @return   エフェクト処理を開始する前は true を返しますが、
    //!           エフェクト処理中は false を返し、サンプリングレートの設定に失敗します。
    //!
    //! @see  GetRequiredMemSize
    //! @see  GetSampleRate
    //!
    //! @date 2013/12/04 初版
    //---------------------------------------------------------------------------
    bool SetSampleRate( SampleRate rate )
    {
        if ( m_IsActive )
        {
            return false;
        }
        else
        {
            m_SampleRate = rate;
            return true;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    サンプリングレートを取得します。
    //!
    //! @return   現在のサンプリングレートです。
    //!
    //! @see  SetSampleRate
    //!
    //! @date 2013/12/04 初版
    //---------------------------------------------------------------------------
    const SampleRate GetSampleRate() const { return m_SampleRate; }

    //---------------------------------------------------------------------------
    //! @brief    エフェクトパラメータを設定します。
    //!
    //!           エフェクトパラメータの設定はエフェクト動作中でも行うことができます。
    //!
    //! @param[in] param     エフェクトパラメータ。
    //!
    //! @return   エフェクト動作中は、パラメータ変更に成功したら true を、失敗したら false
    //!           を返します。 エフェクト処理を開始する前は常に true を返します。
    //!
    //! @see  GetParam
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    bool SetParam( const MultiChDelayParam& param );

    //---------------------------------------------------------------------------
    //! @brief    エフェクトのパラメータを取得します。
    //!
    //! @return   現在のエフェクトパラメータの値です。
    //!
    //! @see  SetParam
    //!
    //! @date 2013/04/05 初版
    //---------------------------------------------------------------------------
    const MultiChDelayParam& GetParam() const { return m_Param; }

    // エフェクトコールバック
    virtual void UpdateBuffer(
        int numChannels,
        void* buffer[],
        unsigned long bufferSize,
        SampleFormat format,
        f32 sampleRate,
        OutputMode mode
    );

    //@}

private:
    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @param[in] device チェック対象となるデバイスです。
    //!
    //! @return           チャンネル数が妥当であれば true を、妥当でなければ false を返します。
    //!
    //! @brief            デバイスに対してチャンネル数が妥当かをチェックする仮想関数です。
    //---------------------------------------------------------------------------
    virtual bool IsValidChannelNum( OutputDevice device );

#if defined( NW_PLATFORM_WIN32 ) || defined ( NW_USE_NINTENDO_SDK )
    typedef nw::internal::winext::AXFX_MULTI_CH_DELAY AXFX_MULTI_CH_DELAY;
    typedef nw::internal::winext::AXAUXCBSTRUCT AXAUXCBSTRUCT;
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
    typedef nw::internal::winext::AXFX_MULTI_CH_DELAY AXFX_MULTI_CH_DELAY;
    typedef nw::internal::winext::AXAUXCBSTRUCT AXAUXCBSTRUCT;
#endif // defined( NW_PLATFORM_WIN32 )

    bool m_IsActive;
    internal::AxfxImpl m_Impl;
    MultiChDelayParam  m_Param;
    ChannelMode        m_ChannelMode;
    SampleRate         m_SampleRate;

    AXFX_MULTI_CH_DELAY m_AxfxParam;
    AXAUXCBSTRUCT       m_AxfxInfo;
};

} // namespace nw::snd
} // namespace nw

#endif /* NW_SND_CONFIG_ENABLE_MULTICHEFT */

#endif /* NW_SND_FX_MULTICH_DELAY_H_ */
