﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_FX_BASE_H_
#define NW_SND_FX_BASE_H_

#include <nw/types.h>
#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_Config.h>
#include <nw/ut/ut_LinkList.h>
#include <nw/ut/ut_Preprocessor.h>          // NW_DISALLOW_COPY_AND_ASSIGN

#if defined( NW_PLATFORM_CAFE )
#include <cafe/axfx.h>
#else
#include <winext/cafe/axfx.h>
#endif

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief    エフェクトの抽象クラスです。
//!
//!           FxBase クラスから継承し、エフェクトを実装することができます。
//!
//!           エフェクトの使用方法については、@ref SoundSystem::AppendEffect
//!           などを参照してください。
//!
//!           エフェクトは CPU で処理されます (DSP では処理されません)。
//!
//! @see SoundSystem クラス
//! @see FxReverb クラス
//!
//! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
class FxBase
{
public:
    //---------------------------------------------------------------------------
    //! @brief    エフェクトのチャンネル数の定義です。
    //!
    //! @date   2013/12/11 初版
    //---------------------------------------------------------------------------
    enum ChannelMode
    {
        CHANNEL_MODE_2CH, //!< 2 チャンネルに対しエフェクト処理を行います。
        CHANNEL_MODE_4CH, //!< 4 チャンネルに対しエフェクト処理を行います。
        CHANNEL_MODE_6CH  //!< 6 チャンネルに対しエフェクト処理を行います。
    };

    //---------------------------------------------------------------------------
    //! @brief    エフェクトのサンプリングレートの定義です。
    //!
    //! @date   2013/12/11 初版
    //---------------------------------------------------------------------------
    enum SampleRate
    {
        SAMPLE_RATE_32000    //!< 32 kHz のサンプリングレートです。
#if defined(NW_SND_CONFIG_ENABLE_SOUND2)
        , SAMPLE_RATE_48000  //!< 48 kHz のサンプリングレートです。
#endif
    };

    //! @name コンストラクタ／デストラクタ
    //@{
    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //!
    //! @date 2011/07/04 初版
    //---------------------------------------------------------------------------
    FxBase() {}

    //---------------------------------------------------------------------------
    //! @brief    デストラクタです。
    //!
    //! @date 2011/07/04 初版
    //---------------------------------------------------------------------------
    virtual ~FxBase() {}
    //@}

    //! @name エフェクト
    //@{
    //---------------------------------------------------------------------------
    //! @brief    エフェクトの初期化処理を行います。
    //!
    //!           この関数は、@ref SoundSystem クラスから、
    //!           エフェクトの開始処理が必要となるタイミングで呼び出されます。
    //!
    //!           ユーザーが呼び出す必要はありません。
    //!
    //! @return   初期化に成功すると true を、失敗すると false を返します。
    //!
    //! @date 2011/07/04 初版
    //---------------------------------------------------------------------------
    virtual bool Initialize();

    //---------------------------------------------------------------------------
    //! @brief    エフェクトの終了処理を行います。
    //!
    //!           この関数は、@ref SoundSystem クラスから、
    //!           エフェクトの終了処理が必要となるタイミングで呼び出されます。
    //!
    //!           ユーザーが呼び出す必要はありません。
    //!
    //! @date 2011/07/04 初版
    //---------------------------------------------------------------------------
    virtual void Finalize();

    //---------------------------------------------------------------------------
    //! @brief    エフェクト処理を行います。
    //!
    //!           この関数はエフェクトに送られてくる
    //!           AUX　バスのバッファの更新が必要なタイミングで呼び出されます。
    //!           FxBase を継承したクラスで、この関数にエフェクトの処理を実装しなければなりません。
    //!
    //! @param[in]      numChannels バッファのチャンネル数です。
    //! @param[in,out]  buffer      エフェクト処理を行う波形データのバッファへのポインタ配列です。
    //! @param[in]      bufferSize  1 チャンネルあたりのバッファサイズです。
    //! @param[in]      format      data のサンプルフォーマットです。
    //! @param[in]      sampleRate  data のサンプルレートです。
    //! @param[in]      mode        出力モードです。
    //!
    //! @see SampleFormat
    //! @see OutputMode
    //!
    //! @date 2011/07/04 初版
    //---------------------------------------------------------------------------
    virtual void UpdateBuffer(
        int numChannels,
        void* buffer[],
        unsigned long bufferSize,
        SampleFormat format,
        f32 sampleRate,
        OutputMode mode
    ) = 0;

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief  出力モードが変更されたときに呼ばれる仮想関数です。
    //---------------------------------------------------------------------------
    virtual void OnChangeOutputMode();

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @param[in] device チェック対象となるデバイスです。
    //!
    //! @return           チャンネル数が妥当であれば true を、妥当でなければ false を返します。
    //!
    //! @brief            デバイスに対してチャンネル数が妥当かをチェックする仮想関数です。
    //---------------------------------------------------------------------------
    virtual bool IsValidChannelNum( OutputDevice device );
    //@}

    //! @briefprivate
    ut::LinkListNode m_Link;

protected:
#if defined( NW_PLATFORM_WIN32 )
    typedef nw::internal::winext::AXFX_SAMPLE_RATE AXFX_SAMPLE_RATE;
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
    typedef nw::internal::winext::AXFX_SAMPLE_RATE AXFX_SAMPLE_RATE;
#elif defined( NW_USE_NINTENDO_SDK )
    // TODO: nn_audio
    typedef nw::internal::winext::AXFX_SAMPLE_RATE AXFX_SAMPLE_RATE;
#endif
    //! @brief FxBase の SampleRate 型を AXFX の AXFX_SAMPLE_RATE 型に変換します。
    //!
    //!        今後、変更・修正される可能性があります。
    //!
    //! @param[in] rate FxBase 内でのサンプルレート型です。
    //! @return AXFX の AXFX_SAMPLE_RATE 型を返します
    AXFX_SAMPLE_RATE GetFxSampleRate( SampleRate rate );

private:
    NW_DISALLOW_COPY_AND_ASSIGN( FxBase );
};

} // namespace nw::snd
} // namespace nw

#endif /* NW_SND_FX_BASE_H_ */
