﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_Fx2ReverbI3dl2.h
 *
 * @file snd_Fx2ReverbI3dl2.h
 */


#ifndef NW_SND_FX2_REVERBI3DL2_H_
#define NW_SND_FX2_REVERBI3DL2_H_

#include <nw/types.h>

#include <nw/snd/snd_FxParam.h>
#include <nw/snd/snd_FxBase.h>
#include <nw/snd/snd_AxfxImpl.h>

#if defined(NW_SND_CONFIG_ENABLE_SOUND2)

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief  指定されたチャンネル数に対する I3DL2 版リバーブエフェクトです。
//!
//!         Fx2ReverbI3dl2 は SDK の AXFX2ReverbI3dl2 エフェクトをラッピングしたクラスです。
//!
//! @see FxBase クラス
//!
//! @date 2013/11/08 初版
//---------------------------------------------------------------------------
class Fx2ReverbI3dl2 : public FxBase
{
    /* ------------------------------------------------------------------------
            constant definition
       ------------------------------------------------------------------------ */
public:
    static const s32 ROOM_MIN;              //!< ReverbI3dl2Param::room の最小値 (-10000) です。
    static const s32 ROOM_MAX;              //!< ReverbI3dl2Param::room の最大値 (0) です。
    static const s32 ROOM_HF_MIN;           //!< ReverbI3dl2Param::roomHf の最小値 (-10000) です。
    static const s32 ROOM_HF_MAX;           //!< ReverbI3dl2Param::roomHf の最大値 (0) です。
    static const f32 DECAY_TIME_MIN;        //!< ReverbI3dl2Param::decayTime の最小値 (0.1) です。
    static const f32 DECAY_TIME_MAX;        //!< ReverbI3dl2Param::decayTime の最大値 (20.0) です。
    static const f32 DECAY_HF_RATIO_MIN;    //!< ReverbI3dl2Param::decayHfRatio の最小値 (0.1) です。
    static const f32 DECAY_HF_RATIO_MAX;    //!< ReverbI3dl2Param::decayHfRatio の最大値 (2.0) です。
    static const s32 REFLECTIONS_MIN;       //!< ReverbI3dl2Param::reflections の最小値 (-10000) です。
    static const s32 REFLECTIONS_MAX;       //!< ReverbI3dl2Param::reflections の最大値 (0) です。
    static const f32 REFLECTIONS_DELAY_MIN; //!< ReverbI3dl2Param::reflectionsDelay の最小値 (0.0) です。
    static const f32 REFLECTIONS_DELAY_MAX; //!< ReverbI3dl2Param::reflectionsDelay の最大値 (0.3) です。
    static const s32 REVERB_MIN;            //!< ReverbI3dl2Param::reverb の最小値 (-10000) です。
    static const s32 REVERB_MAX;            //!< ReverbI3dl2Param::reverb の最大値 (20000) です。
    static const f32 REVERB_DELAY_MIN;      //!< ReverbI3dl2Param::reverbDelay の最小値 (0.0) です。
    static const f32 REVERB_DELAY_MAX;      //!< ReverbI3dl2Param::reverbDelay の最大値 (0.1) です。
    static const f32 DIFFUSION_MIN;         //!< ReverbI3dl2Param::diffusion の最小値 (0.0) です。
    static const f32 DIFFUSION_MAX;         //!< ReverbI3dl2Param::diffusion の最大値 (100.0) です。
    static const f32 DENSITY_MIN;           //!< ReverbI3dl2Param::density の最小値 (0.0) です。
    static const f32 DENSITY_MAX;           //!< ReverbI3dl2Param::density の最大値 (100.0) です。
    static const f32 HF_REFERENCE_MIN;      //!< ReverbI3dl2Param::hfReference の最小値 (20.0) です。
    static const f32 HF_REFERENCE_MAX;      //!< ReverbI3dl2Param::hfReference の最大値 (20000.0) です。
    static const f32 OUT_GAIN_MIN;          //!< ReverbI3dl2Param::outGain の最小値 (0.0) です。
    static const f32 OUT_GAIN_MAX;          //!< ReverbI3dl2Param::outGain の最大値 (1.0) です。
    static const f32 DRY_GAIN_MIN;          //!< ReverbI3dl2Param::dryGain の最小値 (0.0) です。
    static const f32 DRY_GAIN_MAX;          //!< ReverbI3dl2Param::dryGain の最大値 (1.0) です。

    /* ------------------------------------------------------------------------
            typename definition
       ------------------------------------------------------------------------ */
public:
    //---------------------------------------------------------------------------
    //! @brief  I3DL2 版リバーブエフェクトパラメータの構造体です。
    //!
    //! @see SetParam
    //! @see GetParam
    //!
    //! @date 2013/11/08 初版
    //---------------------------------------------------------------------------
    struct ReverbI3dl2Param
    {
        //---------------------------------------------------------------------------
        //! @brief    空間エフェクトのための音量レベル指定です。
        //!           値の範囲は -10000～0 (mB 単位)です。
        //!           初期値は -1000 です。
        //---------------------------------------------------------------------------
        s32 room;
        //---------------------------------------------------------------------------
        //! @brief    高域の空間エフェクトのための音量レベル指定です。
        //!           値の範囲は -10000～0 (mB 単位)です。
        //!           初期値は -1000 です。
        //---------------------------------------------------------------------------
        s32 roomHf;
        //---------------------------------------------------------------------------
        //! @brief    後期残響の減衰量が 60dB になるまでの時間です。
        //!           値の範囲は 0.1～20.0 (sec 単位) です。
        //!           初期値は 0.5 です。
        //---------------------------------------------------------------------------
        f32 decayTime;
        //---------------------------------------------------------------------------
        //! @brief    低周波数での decayTime に対する、高周波減衰での decayTime の比率を調整します。
        //!           値を小さくすると低周波成分が主流となり、高周波成分で急速な減衰が発生します。
        //!           大きくすると高周波成分が減衰せずに残るようになります。
        //!           値の範囲は 0.1～2.0 です。
        //!           初期値は 0.3 です。
        //---------------------------------------------------------------------------
        f32 decayHfRatio;
        //---------------------------------------------------------------------------
        //! @brief    初期反射の音量レベルです。
        //!           値の範囲は -10000～0 (mB 単位)です。
        //!           初期値は -1000 です。
        //---------------------------------------------------------------------------
        s32 reflections;
        //---------------------------------------------------------------------------
        //! @brief    初期反射が開始するまでの時間です。
        //!           値の範囲は 0.0～0.3 (sec 単位)です。
        //!           初期値は 0.1 です。
        //---------------------------------------------------------------------------
        f32 reflectionsDelay;
        //---------------------------------------------------------------------------
        //! @brief    後期残響の音量レベルです。
        //!           値の範囲は -10000～20000 (mB 単位)です。
        //!           初期値は 1500 です。
        //---------------------------------------------------------------------------
        s32 reverb;
        //---------------------------------------------------------------------------
        //! @brief    後期残響が開始するまでの時間です。
        //!           値の範囲は 0.0～0.1 (sec 単位)です。
        //!           初期値は 0.05 です。
        //---------------------------------------------------------------------------
        f32 reverbDelay;
        //---------------------------------------------------------------------------
        //! @brief    後期残響中のリバーブ内の音の密度です。
        //!           値の範囲は 0.0～100.0 (% 単位)です。
        //!           初期値は 3.0 です。
        //---------------------------------------------------------------------------
        f32 diffusion;
        //---------------------------------------------------------------------------
        //! @brief    後期残響の１ヘルツあたりの残響量です。
        //!           値の範囲は 0.0～100.0 (% 単位)です。
        //!           初期値は 3.0 です。
        //---------------------------------------------------------------------------
        f32 density;
        //---------------------------------------------------------------------------
        //! @brief    高域とされる周波数の下限を指定します。
        //!           値の範囲は 20.0～20000.0 (Hz 単位) です。
        //!           初期値は 500.0 です。
        //---------------------------------------------------------------------------
        f32 hfReference;
        //---------------------------------------------------------------------------
        //! @brief    エフェクト出力ゲインです。
        //!           値の範囲は 0.0～1.0 です。
        //!           初期値は 0.5 です。
        //---------------------------------------------------------------------------
        f32 outGain;
        //---------------------------------------------------------------------------
        //! @brief    ドライ成分のゲインです。
        //!           値の範囲は 0.0～1.0 です。
        //!           初期値は 0.5 です。
        //---------------------------------------------------------------------------
        f32 dryGain;

        //---------------------------------------------------------------------------
        //! @brief    コンストラクタです。
        //! @date 2013/04/05 初版
        //---------------------------------------------------------------------------
        ReverbI3dl2Param()
        : room             ( -1000     ),
          roomHf           ( -1000     ),
          decayTime        (     0.5f  ),
          decayHfRatio     (     0.3f  ),
          reflections      ( -1000     ),
          reflectionsDelay (     0.1f  ),
          reverb           (  1500     ),
          reverbDelay      (     0.05f ),
          diffusion        (     3.0f  ),
          density          (     3.0f  ),
          hfReference      (   500.0f  ),
          outGain          (     0.5f  ),
          dryGain          (     0.5f  )
        {}
    };

    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */
public:
    //! @name コンストラクタ/デストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //!         エフェクトパラメータは、下記のように初期化されます。
    //!
    //!         - room             = -1000
    //!         - roomHf           = -1000
    //!         - decayTime        =     0.5
    //!         - decayHfRatio     =     0.3
    //!         - reflections      = -1000
    //!         - reflectionsDelay =     0.1
    //!         - reverb           =  1500
    //!         - reverbDelay      =     0.05
    //!         - diffusion        =     3.0
    //!         - density          =     3.0
    //!         - hfReference      =   500.0
    //!         - outGain          =     0.5
    //!         - dryGain          =     0.5
    //!
    //! @date 2013/11/08 初版
    //---------------------------------------------------------------------------
    Fx2ReverbI3dl2();

    //---------------------------------------------------------------------------
    //! @brief    デストラクタです。
    //!
    //! @date 2013/11/08 初版
    //---------------------------------------------------------------------------
    virtual ~Fx2ReverbI3dl2() { Finalize(); ReleaseWorkBuffer(); }

    //@}

    //! @name メモリ割り当て
    //@{

    //---------------------------------------------------------------------------
    //! @brief    Fx2ReverbI3dl2 が必要とするメモリのサイズを取得します。
    //!
    //!           Fx2ReverbI3dl2 が現在のパラメータ設定でエフェクト処理を行うために必要とする
    //!           メモリのサイズを取得します。
    //!
    //!           この関数を呼び出す前に @ref SetParam を呼びだしてエフェクトパラメータを
    //!           設定してください。呼び出さない場合は、初期設定の値が使用されます。
    //!
    //!           エフェクトを使用するためには、取得したメモリサイズのメモリを確保
    //!           した後に @ref AssignWorkBuffer を呼びだし、エフェクトにメモリ領域を
    //!           割り当ててください。
    //!
    //! @return   Fx2ReverbI3dl2 が必要とするメモリのサイズを返します。
    //!
    //! @see  SetParam
    //! @see  AssignWorkBuffer
    //!
    //! @date 2013/11/08 初版
    //---------------------------------------------------------------------------
    u32 GetRequiredMemSize();

    //---------------------------------------------------------------------------
    //! @brief    エフェクトで使用するバッファ領域を割り当てます。
    //!
    //!           バッファ領域の割り当てはエフェクトを使用する前に行う必要があります。
    //!
    //!           エフェクトが必要とするバッファサイズはパラメータの値によって変化しますので、
    //!           @ref GetRequiredMemSize を呼び出してバッファサイズを取得してください。
    //!
    //! @param[in] buffer   割り当てるバッファの先頭アドレス。
    //! @param[in] size     割り当てるバッファのサイズ。
    //!
    //! @return   バッファ領域の割り当てに成功した場合は true を、失敗した場合は false を返します。
    //!
    //! @see  GetRequiredMemSize
    //!
    //! @date 2013/11/08 初版
    //---------------------------------------------------------------------------
    virtual bool AssignWorkBuffer( void* buffer, u32 size );

    //---------------------------------------------------------------------------
    //! @brief  エフェクトで使用していたメモリを解放します。
    //!
    //!         @ref AssignWorkBuffer で割り当てたバッファ領域を解放します。 この関数で
    //!         解放を行う前に、@ref Finalize を呼び出してエフェクト処理を終了させる
    //!         必要があります。
    //!
    //! @see  AssignWorkBuffer
    //! @see  Finalize
    //!
    //! @date 2013/11/08 初版
    //---------------------------------------------------------------------------
    virtual void ReleaseWorkBuffer();

    //@}

    //! @name エフェクト
    //@{

    //---------------------------------------------------------------------------
    //! @brief    エフェクトの開始処理を行います。
    //!
    //!           @ref SoundSystem クラスを通じてエフェクトを使用する場合、
    //!           @ref SoundSystem クラスからこの関数が呼び出されます。 ユーザがこの関数を
    //!           呼び出す必要はありません。
    //!
    //!           この関数を呼び出す前に、@ref AssignWorkBuffer を呼びだし、エフェクトで
    //!           使用するワークメモリを割り当てる必要があります。
    //!
    //! @return   エフェクトの開始に成功したら true を、失敗したら false を返します。
    //!
    //! @see  SoundSystem クラス
    //! @see  AssignWorkBuffer
    //!
    //! @date 2013/11/08 初版
    //---------------------------------------------------------------------------
    virtual bool Initialize();

    //---------------------------------------------------------------------------
    //! @brief    エフェクトの終了処理を行います。
    //!
    //!           @ref SoundSystem クラスを通じてエフェクトを使用する場合、
    //!           @ref SoundSystem クラスからこの関数が呼び出されます。 ユーザがこの関数を
    //!           呼び出す必要はありません。
    //!
    //! @see  SoundSystem クラス
    //!
    //! @date 2013/11/08 初版
    //---------------------------------------------------------------------------
    virtual void Finalize();

    //---------------------------------------------------------------------------
    //! @brief    チャンネル数を設定します。
    //!
    //! @param[in] mode チャンネル数を指定する列挙型です。
    //!
    //! @return   エフェクト処理を開始する前は true を返しますが、
    //!           エフェクト処理中は false を返し、チャンネル数の設定に失敗します。
    //!
    //!           TV 出力に設定するエフェクトには全ての @ref ChannelMode を使用できますが、
    //!           DRC 出力に設定するエフェクトには 2ch, 4ch のみが使用できます。
    //!
    //!           DRC 出力に設定するエフェクトで 6ch を使用した場合、@ref SoundSystem::AppendEffect でエフェクトを追加する際に
    //!           Debug, Develop ビルドではアサートとなり、Release ビルドでは追加に失敗するためご注意ください。
    //!
    //! @see  GetRequiredMemSize
    //! @see  GetChannelMode
    //!
    //! @date 2013/12/11 初版
    //---------------------------------------------------------------------------
    bool SetChannelMode( ChannelMode mode )
    {
        if ( m_IsActive )
        {
            return false;
        }
        else
        {
            m_ChannelMode = mode;
            return true;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    チャンネル数を取得します。
    //!
    //! @return   現在のチャンネル数です。
    //!
    //! @see  SetChannelMode
    //!
    //! @date 2013/12/11 初版
    //---------------------------------------------------------------------------
    const ChannelMode GetChannelMode() const { return m_ChannelMode; }

    //---------------------------------------------------------------------------
    //! @brief    サンプリングレートを設定します。
    //!
    //! @param[in] rate サンプリングレートを指定する列挙型です。
    //!
    //! @return   エフェクト処理を開始する前は true を返しますが、
    //!           エフェクト処理中は false を返し、サンプリングレートの設定に失敗します。
    //!
    //! @see  GetRequiredMemSize
    //! @see  GetSampleRate
    //!
    //! @date 2013/12/11 初版
    //---------------------------------------------------------------------------
    bool SetSampleRate( SampleRate rate )
    {
        if ( m_IsActive )
        {
            return false;
        }
        else
        {
            m_SampleRate = rate;
            return true;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief    サンプリングレートを取得します。
    //!
    //! @return   現在のサンプリングレートです。
    //!
    //! @see  SetSampleRate
    //!
    //! @date 2013/12/11 初版
    //---------------------------------------------------------------------------
    const SampleRate GetSampleRate() const { return m_SampleRate; }

    //---------------------------------------------------------------------------
    //! @brief    エフェクトパラメータを設定します。
    //!
    //!           エフェクトパラメータの設定はエフェクト動作中でも行うことができます。
    //!
    //! @param[in] param     エフェクトパラメータ。
    //!
    //! @return   エフェクト動作中は、パラメータ変更に成功したら true を、失敗したら false
    //!           を返します。 エフェクト処理を開始する前は常に true を返します。
    //!
    //! @see  GetParam
    //!
    //! @date 2013/12/11 引数からチャンネル数とサンプリングレートを削除
    //! @date 2013/11/08 初版
    //---------------------------------------------------------------------------
    bool SetParam( const ReverbI3dl2Param& param );

    //---------------------------------------------------------------------------
    //! @brief    エフェクトのパラメータを取得します。
    //!
    //! @return   現在のエフェクトパラメータの値です。
    //!
    //! @see  SetParam
    //!
    //! @date 2013/11/08 初版
    //---------------------------------------------------------------------------
    const ReverbI3dl2Param& GetParam() const { return m_Param; }

    // エフェクトコールバック
    virtual void UpdateBuffer(
        int numChannels,
        void* buffer[],
        unsigned long bufferSize,
        SampleFormat format,
        f32 sampleRate,
        OutputMode mode
    );

    //@}

private:
    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @param[in] device チェック対象となるデバイスです。
    //!
    //! @return           チャンネル数が妥当であれば true を、妥当でなければ false を返します。
    //!
    //! @brief            デバイスに対してチャンネル数が妥当かをチェックする仮想関数です。
    //---------------------------------------------------------------------------
    virtual bool IsValidChannelNum( OutputDevice device );

#if defined( NW_PLATFORM_WIN32 )
    typedef nw::internal::winext::AXFX2_REVERB_I3DL2 AXFX2_REVERB_I3DL2;
    typedef nw::internal::winext::AXAUXCBSTRUCT      AXAUXCBSTRUCT;
#elif defined( NW_USE_NINTENDO_SDK )
    // TODO: nn_audio
    typedef nw::internal::winext::AXFX2_REVERB_I3DL2 AXFX2_REVERB_I3DL2;
    typedef nw::internal::winext::AXAUXCBSTRUCT      AXAUXCBSTRUCT;
#endif

    bool m_IsActive;
    internal::AxfxImpl m_Impl;
    ReverbI3dl2Param m_Param;
    ChannelMode        m_ChannelMode;
    SampleRate         m_SampleRate;

    AXFX2_REVERB_I3DL2 m_AxfxParam;
    AXAUXCBSTRUCT      m_AxfxInfo;
};

} // namespace nw::snd
} // namespace nw

#endif /* NW_SND_CONFIG_ENABLE_SOUND2 */

#endif /* NW_SND_FX2_REVERBI3DL2_H_ */
