﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_DEVICE_OUT_RECORDER_H_
#define NW_SND_DEVICE_OUT_RECORDER_H_

#include <nw/ut/os/ut_Event.h>
#include <nw/ut/os/ut_MessageQueue.h>
#include <nw/ut/os/ut_Thread.h>
#include <nw/snd/snd_WavOutFileStream.h>

//#define NW_SND_DEBUG_REC_BUFFER

namespace nw {

namespace ut {
class FileStream;
}

namespace snd {

//---------------------------------------------------------------------------
//! @briefprivate
//! @brief  デバイスの出力波形を録音する抽象クラスです。
//!
//!         このクラスは内部でスレッドを生成し、対象デバイスの出力波形を
//!         Wav ファイル形式で指定 FileStream に出力します。
//---------------------------------------------------------------------------
class DeviceOutRecorder : public ut::ThreadHandler
{
    NW_DISALLOW_COPY_AND_ASSIGN( DeviceOutRecorder );

private:
    static const u32 FILE_IO_BUFFER_ALIGNMENT;                              //!< ファイルIOバッファアライメント

    static const u32 MIN_BUFFER_LENGTH       = 256 * 1024;                  //!< 最小バッファサイズ
    static const u32 WRITE_BLOCK_PER_SAMPLES = 128 * 1024 / sizeof(s16);    //!< 一括して書き込むサンプル数
    static const u32 THREAD_STACK_SIZE       = 64 * 1024;                   //!< スレッドのスタックサイズ

public:
    //---------------------------------------------------------------------------
    //! @brief  録音オプションです。
    //---------------------------------------------------------------------------
    struct Options
    {
        Options() :
            channels(0),
            priority(16),
            isLeadSilenceTrimmingEnabled(false),
            maxFrames(0)
        { }

        u32  channels;                      //!< 出力する波形のチャンネル数です。
        u32  priority;                      //!< 録音スレッドのプライオリティです。
        bool isLeadSilenceTrimmingEnabled;  //!< 先頭の無音をトリミングします。
        u32  maxFrames;                     //!< 録音する波形の最大フレーム数です。
    };

    //---------------------------------------------------------------------------
    //! @brief  DeviceOutRecorder の状態です。
    //---------------------------------------------------------------------------
    enum State
    {
        //! @brief  未初期化状態です。
        STATE_NOT_INITIALIZED,

        //! @brief  初期化済みです。
        STATE_INITIALIZED,

        //! @brief  録音中です。
        STATE_RECORDING,

        //! @brief  指定したフレーム数の録音が完了した状態です。
        //! @see    Options
        STATE_RECORDED,

        //! @brief  停止処理中です。
        //!         停止が完了すると、 STATE_INITIALIZED に移行します。
        STATE_STOPPING,
    };

private:
    //---------------------------------------------------------------------------
    //! @brief  スレッドに通知するメッセージです。
    //---------------------------------------------------------------------------
    enum Message
    {
        MESSAGE_PREPARE,        //!< 録音の準備を行います。
        MESSAGE_WRITE_SAMPLES,  //!< 録音サンプルを出力します。
        MESSAGE_EXIT,           //!< スレッドを終了します。
    };

    //---------------------------------------------------------------------------
    //! @brief  録音波形用のバッファクラスです。
    //---------------------------------------------------------------------------
    class RecorderBuffer
    {
    public:
        explicit RecorderBuffer(const char* deviceName=NULL);

    public:
        void Initialize(s16* sampleBuffer, u32 maxSamples);
        void Finalize();

        u32 Push(const s16* sampleBuffer, u32 samples);
        u32 Pop(u32 samples);
        s16* Peek();

        void SetReadBlockSamples(u32 value);
        void Clear();

        inline u32 GetReadableCount() const
        {
            return m_ValidSamples / m_ReadBlockSamples * m_ReadBlockSamples;
        }

        inline u32 GetWritableCount() const;
        inline u32 GetContiguousReadableCount() const;
        inline u32 GetContiguousWritableCount() const;

    private:
        void UpdateMaxSamples();

        inline void Skip(u32 samples);
        inline void Write(const s16* sampleBuffer, u32 samples);

        inline u32 IncrementPosition(u32 position, u32 length);

    private:
        s16* m_SampleBuffer;       //!< バッファ
        u32 m_MaxBufferSamples;    //!< バッファに格納できる最大サンプル数
        u32 m_MaxSamples;          //!< ブロックバッファに格納できる最大サンプル数
        u32 m_ValidSamples;        //!< バッファに格納されている有効サンプル数
        u32 m_ReadPosition;        //!< 読み込みポインタの位置
        u32 m_WritePosition;       //!< 書き込みポインタの位置
        u32 m_ReadBlockSamples;    //!< 読み込みブロックのサンプル数

        const char* m_DeviceName;       //!< デバイス名

#if defined(NW_DEBUG) || defined(NW_DEVELOP)
        bool m_IsWarnedDropSamples;         //!< フレーム落ちの警告フラグ
#endif

#ifdef NW_SND_DEBUG_REC_BUFFER
        static const u32 s_PrintInterval = 200; //!< 1デバッグ出力の感覚（RecorderBuffer::Push() 回数）
        u32              m_PushCount;           //!< RecorderBuffer::Push() 回数
#endif // NW_SND_DEBUG_REC_BUFFER
    };

public:
    //! @briefprivate
    virtual ~DeviceOutRecorder();

protected:
    //! @briefprivate
    //! @param deviceName :private
    explicit DeviceOutRecorder(const char* deviceName=NULL);

public:
    //---------------------------------------------------------------------------
    //! @brief         DeviceOutRecorder を初期化します。
    //!
    //! @param[in]     buf     録音に使用するバッファです。
    //! @param[in]     length  録音に使用するバッファ長です。
    //---------------------------------------------------------------------------
    void Initialize(void* buf, u32 length);

    //---------------------------------------------------------------------------
    //! @brief         DeviceOutRecorder を破棄します。
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief         再生波形の録音を開始します。
    //!
    //! @param[in]     fileStream  波形の出力先ファイルストリームです。
    //!
    //! @return        成功した場合は true、失敗した場合は false を返します。
    //---------------------------------------------------------------------------
    bool Start(ut::FileStream& fileStream)
    {
        Options options;
        return Start(fileStream, options);
    }

    //---------------------------------------------------------------------------
    //! @brief         再生波形の録音を開始します。
    //!                このタイミングで新規に録音用スレッドが生成されます。
    //!
    //! @param[in]     fileStream  波形の出力先ファイルストリームです。
    //! @param[in]     options     録音オプションを指定します。
    //!                             channels                      出力する波形のチャンネル数です。
    //!                                                           0(初期値) を指定すると、出力モードに見て自動選択します。
    //!                                                           サラウンドなら 6ch、それ以外の場合は 2ch で録音します。
    //!                             priority                      録音スレッドのプライオリティです。
    //!                                                           初期値は 16(OS_PRIORITY_APP_DEFAULT)です。
    //!                             isLeadSilenceTrimmingEnabled  true を指定すると、先頭の無音部分をトリミングします。
    //!                             maxFrames                     録音波形の最大フレーム数です。
    //!                                                           0 以外を指定すると、指定フレーム数に到達したら録音を自動停止し、
    //!                                                           状態を STATE_RECORDED に遷移します。
    //!                                                           0(初期値) を指定すると、上限を設定せず、可能な限り録音を継続します。
    //!
    //! @return        成功した場合は true、失敗した場合は false を返します。
    //---------------------------------------------------------------------------
    bool Start(ut::FileStream& fileStream, const Options& options);

    //---------------------------------------------------------------------------
    //! @brief         録音を停止します。
    //!                このタイミングで録音用スレッドが停止されます。
    //!
    //! @param[in]     isBlocking  true を指定すると停止が完了するまで待機します。
    //---------------------------------------------------------------------------
    void Stop(bool isBlocking);

    //---------------------------------------------------------------------------
    //! @brief         初期化の有無を取得します。
    //!
    //! @return        初期化済みの場合は true、未初期化の場合は false を返します。
    //!
    //! @see           GetState
    //---------------------------------------------------------------------------
    bool IsInitialized() const { return m_State != STATE_NOT_INITIALIZED; }

    //---------------------------------------------------------------------------
    //! @brief         DeviceOutRecorder の状態を取得します。
    //!
    //! @return        初期化済みの場合は true、未初期化の場合は false を返します。
    //!
    //! @see           IsInitialized
    //---------------------------------------------------------------------------
    State GetState() const { return m_State; }

    //---------------------------------------------------------------------------
    //! @brief         先頭の無音部分をトリミング中かどうかを調べます。
    //!
    //! @return        トリミング中の場合は true、それ以外の場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsLeadSilenceTrimming() const { return m_IsLeadSilenceTrimming; }

    //---------------------------------------------------------------------------
    //! @brief         録音チャンネル数を取得します。
    //!
    //! @return        録音チャンネル数を返します。
    //!
    //! @see           Start
    //---------------------------------------------------------------------------
    u32 GetRecordingChannels() const { return m_Channels; }

protected:
    OutputMode GetOutputMode() const { return m_OutputMode; }

    virtual u32 GetMaxFrameLength() const = 0;
    virtual u32 GetSamplesPerSec() const = 0;
    virtual u32 GetValidChannels(u32 channels) const = 0;

    virtual void OnStart() { }
    virtual void OnStop() { }
    virtual u32 OnProcessSamples(s16* /* sampleBuffer */, u32 samples) { return samples; }

    void PushSamples(const s16* sampleBuffer, u32 samples);

    s16 ResolveSampleEndian(s16 sample)
    {
#ifdef NW_BIG_ENDIAN
        return ut::ReverseEndian(sample);
#else
        return sample;
#endif
    }

    virtual void ThreadHandlerProc();

private:
    u32 GetReadBlockSamples(u32 channels) const;
    u32 GetLeadSilenceSamples(const s16* sampleBuffer, u32 samples, u32 channels) const;
    u32 GetWritableSamples(u32 samples) const;
    bool IsNoMoreSamples() const;

    bool StartThread(u32 priority);
    void StopThread(bool isBlocking);

    s32  Prepare();
    bool SendMessage(Message message);
    bool PostMessage(Message message);

    s32 OnPrepare();
    void OnExit();
    bool OnWriteSamples();

private:
    volatile State m_State;                     //!< 状態
    u32            m_Channels;                  //!< 録音チャンネル数
    OutputMode     m_OutputMode;                //!< 出力モード
    bool           m_IsLeadSilenceTrimming;     //!< 先頭無音部分のトリミング中
    u32            m_MaxSamples;                //!< 最大サンプル数
    u32            m_WrittenSamples;            //!< 書き込み済みサンプル数

    ut::Thread m_Thread;                            //!< 波形録音スレッド
    u8         m_ThreadStack[THREAD_STACK_SIZE];    //!< 波形録音スレッドスタック
    ut::Event  m_ExitThreadEvent;                   //!< スレッド終了待ちイベント

    ut::MessageQueue m_MessageQueue;            //!< メッセージキュー
    ut::MessageQueue::BufferType m_Message;                 //!< メッセージスタック（格納数は１つ）
    s32              m_MessageResult;           //!< 同期メッセージの結果
    ut::Event        m_MessageDoneEvent;        //!< 同期メッセージの完了待ちイベント

    ut::FileStream*            m_Stream;        //!< 出力先ファイルストリーム
    internal::WavOutFileStream m_WavOutStream;  //!< 波形ファイル出力ストリーム
    RecorderBuffer             m_Buffer;        //!< バッファ
};

} // snd
} // nw

#endif // NW_SND_DEVICE_OUT_RECORDER_H_
