﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_BasicSound.h
 *
 * @file snd_BasicSound.h
 */

#ifndef NW_SND_BASIC_SOUND_H_
#define NW_SND_BASIC_SOUND_H_

#include <nw/types.h>
#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_MoveValue.h>
#include <nw/ut/ut_RuntimeTypeInfo.h>
#include <nw/ut/ut_LinkList.h>

namespace nw {
namespace snd {

class SoundHandle;
class SoundPlayer;
class SoundActor;

// ------------------------------------------------------------------------
// アンビエントパラメータ

//---------------------------------------------------------------------------
//! @brief  出力先別のアンビエントパラメータ構造体です。
//!
//! @see SoundAmbientParam
//! @see AuxBus
//!
//! @date 2011/11/22 初版
//---------------------------------------------------------------------------
struct OutputAmbientParam
{
    //---------------------------------------------------------------------------
    //! @brief  当該出力先に対する音量です。
    //!         0.0 以上の倍率で指定します。他のメイン出力音量と掛け合わされます。
    //---------------------------------------------------------------------------
    f32 volume;

    //---------------------------------------------------------------------------
    //! @brief  当該出力先に対するパン (左右の定位) の相対変化量です。
    //!         0.0 を指定すると変化しません。
    //!         1.0 を指定すると中央に定位していた音が右端に定位するようになり、
    //!         -1.0 を指定すると中央に定位していた音が左端に定位するようになります。
    //---------------------------------------------------------------------------
    f32 pan;

    //---------------------------------------------------------------------------
    //! @brief  当該出力先に対するサラウンドパン（前後の定位）の相対変化量です。
    //!         0.0 を指定すると変化しません。
    //!         1.0 を指定すると最前方に定位していた音が中央に定位するようになり、
    //!         2.0 を指定すると最前方に定位していた音が最後方に定位するようになります。
    //!         前方へ定位を移動させたい場合は負の値を指定します。
    //---------------------------------------------------------------------------
    f32 span;

    //---------------------------------------------------------------------------
    //! @brief  当該出力先に対するエフェクトセンドの相対変化量です。
    //!         0.0 を指定するとセンド量を変更しません。
    //!         1.0 を指定すると AUX バスに送られていなかったサウンドが
    //!         最大のセンド量で送られるようになります。
    //---------------------------------------------------------------------------
    f32 fxSend[AUX_BUS_NUM];

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //!         内部で @ref Initialize を呼び出します。
    //!
    //! @date 2011/11/22 初版
    //---------------------------------------------------------------------------
    OutputAmbientParam() { Initialize(); }

    //---------------------------------------------------------------------------
    //! @brief  メンバー変数を初期化します。
    //!
    //!         下記のように初期化されます。
    //!         - volume = 1.0f;
    //!         - pan = 0.0f;
    //!         - span = 0.0f;
    //!         - fxSend[...] = 0.0f;
    //!
    //! @date 2011/11/22 初版
    //---------------------------------------------------------------------------
    void Initialize()
    {
        volume = 1.0f;
        pan = span = 0.0f;
        for ( int i = 0; i < AUX_BUS_NUM; i++ )
        {
            fxSend[i] = 0.0f;
        }
    };
};

//! @briefprivate
struct SoundParam
{
    f32 volume;
    f32 pitch;
    f32 lpf;
    f32 biquadFilterValue;
    int biquadFilterType;
    int priority;
    int outputLineFlag;
    u32 userData;
    OutputAmbientParam tvParam;
    OutputAmbientParam drcParam[DRC_OUT_COUNT];

    SoundParam() { Initialize(); }

    void Initialize()
    {
        volume            = 1.0f;
        pitch             = 1.0f;
        lpf               = 0.0f;
        biquadFilterValue = 0.0f;
        biquadFilterType  = BIQUAD_FILTER_TYPE_INHERIT;
        priority          = 0;
        userData          = 0;

        outputLineFlag    = -1;

        tvParam.Initialize();
        for ( int i = 0; i < DRC_OUT_COUNT; i++ )
        {
            drcParam[i].Initialize();
        }
    }
};

//---------------------------------------------------------------------------
//! @brief  アンビエントパラメータ構造体です。
//!
//!         3D サウンドを使用している際には、
//!         前回計算された 3D サウンド計算結果を参照するため、そして、
//!         3D サウンドエンジンクラスの計算結果を格納するために用いられます。
//!         詳しくは、@ref Sound3DEngine::UpdateAmbientParam 関数のリファレンスをご参照ください。
//!
//!         biquad フィルタは複数の箇所での設定が重ね合わされず、
//!         以下の優先度に従って設定されます。
//!         優先度が高い箇所でパラメータの設定がされた場合、それより下位の設定は上書きされます。
//!
//!         - サウンドハンドルでの設定
//!         - サウンドプレイヤーでの設定
//!         - 当アンビエントパラメータ構造体での設定
//!         - シーケンスデータでの設定
//!
//! @see Sound3DEngine::UpdateAmbientParam
//! @see SoundHandle::SetFxSend
//! @see OutputAmbientParam
//!
//! @date 2011/11/25 pan, span, fxSend, mainOutVolume, drcOutVolume の廃止、
//!                  tvParam, drcParam の追加
//! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
struct SoundAmbientParam
{
    //---------------------------------------------------------------------------
    //! @brief  サウンドの音量の倍率です。0.0 を指定すると発音されません。
    //---------------------------------------------------------------------------
    f32 volume;

    //---------------------------------------------------------------------------
    //! @brief  サウンドの音程の周波数比率です。
    //!         1.0 を指定すると変化しません。
    //!         2.0 を指定すると再生される周波数が 2 倍になり、1 オクターブ高い音程になります。
    //---------------------------------------------------------------------------
    f32 pitch;

    //---------------------------------------------------------------------------
    //! @brief  サウンドのローパスフィルタのカットオフの相対変化量です。
    //!         0.0 を指定するとカットオフの値を変更しません。
    //!         -1.0 を指定すると、フィルタがかかっていない状態から、
    //!         最もフィルタがかかっている状態（カットオフ周波数が下がる方向）に変更します。
    //---------------------------------------------------------------------------
    f32 lpf;

    //---------------------------------------------------------------------------
    //! @brief  サウンドの biquad フィルタのかかり具合を表す値です。
    //!         値の意味はフィルタの係数の種類によって変化します。
    //---------------------------------------------------------------------------
    f32 biquadFilterValue;

    //---------------------------------------------------------------------------
    //! @brief  サウンドの biquad フィルタの種類です。
    //!         @ref BiquadFilterType の値を使用します。
    //!         プリセットで用意されているフィルタの種類のほか、
    //!         ユーザーが登録したフィルタの種類の値をとります。
    //---------------------------------------------------------------------------
    int biquadFilterType;

    //---------------------------------------------------------------------------
    //! @brief  サウンドのプレイヤープライオリティの相対変化量です。
    //!         もともとのプレイヤープライオリティに加算されます。
    //---------------------------------------------------------------------------
    int priority;

    //---------------------------------------------------------------------------
    //! @brief  ユーザーが自由に利用できるパラメータです。
    //!         サウンド再生時に 0 にリセットされます。
    //---------------------------------------------------------------------------
    u32 userData;

    //---------------------------------------------------------------------------
    //! @brief  サウンドの出力先です。
    //!         -1 を指定すると SoundHandle などの設定を継承します。
    //---------------------------------------------------------------------------
    int outputLineFlag;

    //---------------------------------------------------------------------------
    //! @brief  メイン (TV) 出力に対する音量などのパラメータ群です。
    //---------------------------------------------------------------------------
    OutputAmbientParam tvParam;

    //---------------------------------------------------------------------------
    //! @brief  DRC 出力に対する音量などのパラメータ群です。
    //!         SDK 1.9.x 現在、span や fxSend に値を入れても効果はありません。
    //---------------------------------------------------------------------------
    OutputAmbientParam drcParam[DRC_OUT_COUNT];

    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //!
    //! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    SoundAmbientParam()
    : volume(1.0f),
      pitch(1.0f),
      lpf(0.0f),
      biquadFilterValue(0.0f),
      biquadFilterType(BIQUAD_FILTER_TYPE_INHERIT),
      priority(0),
      userData(0),
      outputLineFlag(-1)
    {
        tvParam.Initialize();
        for ( int i = 0; i < DRC_OUT_COUNT; i++ )
        {
            drcParam[i].Initialize();
        }
    }
};

namespace internal {

struct SoundActorParam
{
    f32 volume;
    f32 pitch;

    f32 tvVolume;
    f32 tvPan;

    f32 lpf;

    f32 drcVolume[DRC_OUT_COUNT];
    f32 drcPan[DRC_OUT_COUNT];

    SoundActorParam()
    {
        Reset();
    }
    void Reset()
    {
        volume = pitch = tvVolume = 1.0f;
        tvPan = 0.0f;
        lpf = 0.0f;

        for ( int i = 0; i < DRC_OUT_COUNT; i++ )
        {
            drcVolume[i] = 1.0f;
            drcPan[i] = 0.0f;
        }
    }
};

/* ========================================================================
        typename declaration
   ======================================================================== */

namespace driver {
class BasicSoundPlayer;
} // nw::snd::internal::driver

class PlayerHeap;
class ExternalSoundPlayer;

/* ========================================================================
        class definition
   ======================================================================== */

class BasicSound
{
    friend class nw::snd::SoundHandle;

public:
    NW_UT_RUNTIME_TYPEINFO_ROOT()

    /* ------------------------------------------------------------------------
            constant declaration
       ------------------------------------------------------------------------ */
public:
    static const int PRIORITY_MIN = 0;
    static const int PRIORITY_MAX = 127;
    static const u32 INVALID_ID = 0xffffffff;

    /* ------------------------------------------------------------------------
            type definition
       ------------------------------------------------------------------------ */
public:
    class AmbientParamUpdateCallback;
    class AmbientArgUpdateCallback;
    class AmbientArgAllocatorCallback;

    struct AmbientInfo
    {
        AmbientParamUpdateCallback* paramUpdateCallback;
        AmbientArgUpdateCallback* argUpdateCallback;
        AmbientArgAllocatorCallback* argAllocatorCallback;
        void* arg;
        unsigned long argSize;
    };

    enum PlayerState
    {
        PLAYER_STATE_INIT,
        PLAYER_STATE_PLAY,
        PLAYER_STATE_STOP
    };

    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */
public:
    BasicSound();
    virtual ~BasicSound() { m_State = STATE_DESTRUCTED; }
    void Update();
    void StartPrepared();
    void Stop( int fadeFrames );
    void Pause( bool flag, int fadeFrames );
    void Mute( bool flag, int fadeFrames );
    void SetAutoStopCounter( int frames );
    void FadeIn( int frames );

    virtual void Initialize();
    virtual void Finalize();

    virtual bool IsPrepared() const = 0;
    bool IsPause() const;
    bool IsMute() const;
    bool IsStarted() const { return m_StartedFlag; }

    //------------------------------------------------------------------
    // サウンド共通パラメータ
    void SetPriority( int priority, int ambientPriority );
    void GetPriority( int* priority, int* ambientPriority ) const;
    void SetInitialVolume( f32 volume );
    f32  GetInitialVolume() const;
    void SetVolume( f32 volume, int frames = 0 );
    f32  GetVolume() const;
    void SetPitch( f32 pitch );
    f32  GetPitch() const;
    void SetLpfFreq( f32 lpfFreq );
    f32  GetLpfFreq() const;
    void SetBiquadFilter( int type, f32 value );
    void GetBiquadFilter( int* type, f32* value ) const;
    void SetOutputLine( u32 lineFlag );
    u32 GetOutputLine() const;
    void ResetOutputLine();
    void SetPlayerPriority( int priority );
    void SetMixMode( MixMode mixMode );
    MixMode GetMixMode();
    void SetPan( f32 pan );
    f32  GetPan() const;
    void SetSurroundPan( f32 pan );
    f32  GetSurroundPan() const;
    void SetMainSend( f32 send );
    f32  GetMainSend() const;
    void SetFxSend( AuxBus bus, f32 send );
    f32  GetFxSend( AuxBus bus ) const;

    void SetPanMode( PanMode mode );
    void SetPanCurve( PanCurve curve );
    void SetFrontBypass( bool isFrontBypass );
    void SetRemoteFilter( u8 filter );
    void SetVoiceRendererType( VoiceRendererType mode );

    void SetOutputVolume     (OutputDevice device, f32 volume);
    void SetOutputPan        (OutputDevice device, f32 pan);
    void SetOutputSurroundPan(OutputDevice device, f32 span);
    void SetOutputMainSend   (OutputDevice device, f32 send);
    void SetOutputFxSend     (OutputDevice device, AuxBus bus, f32 send);
    void SetOutputChannelMixParameter (OutputDevice device, u32 srcChNo, MixParameter param);

    f32 GetOutputVolume     (OutputDevice device) const;
    f32 GetOutputPan        (OutputDevice device) const;
    f32 GetOutputSurroundPan(OutputDevice device) const;
    f32 GetOutputMainSend   (OutputDevice device) const;
    f32 GetOutputFxSend     (OutputDevice device, AuxBus bus) const;
    MixParameter GetOutputChannelMixParameter (OutputDevice device, u32 srcChNo) const;

    // リモコン用パラメータ
    void SetRemoteOutVolume( u32 remoteIndex, f32 volume );
    f32 GetRemoteOutVolume( u32 remoteIndex ) const;
    void SetRemoteMainSend( u32 remoteIndex, f32 send );
    f32  GetRemoteMainSend( u32 remoteIndex ) const;
    void SetRemoteFxSend( u32 remoteIndex, f32 send );
    f32  GetRemoteFxSend( u32 remoteIndex ) const;


    //------------------------------------------------------------------
    // 情報取得
    int GetRemainingFadeFrames() const;
    int GetRemainingPauseFadeFrames() const;
    int GetRemainingMuteFadeFrames() const;
    int GetPlayerPriority() const { return m_Priority; } // for AnimSound

    //------------------------------------------------------------------
    // その他のパラメータ
    void SetId( u32 id );
    u32 GetId() const { return m_Id; }
    u32 GetInstanceId() const { return m_InstanceId; }
    u32 GetPlayFrameCount() const { return m_PlayingCounter; }

    //------------------------------------------------------------------
    int CalcCurrentPlayerPriority() const
    {
        return ut::Clamp(
             static_cast<int>( m_Priority ) + static_cast<int>( m_AmbientParam.priority ),
             PRIORITY_MIN,
             PRIORITY_MAX
        );
    }

    // サウンドプレイヤー
    SoundPlayer* GetSoundPlayer() { return m_pSoundPlayer; }
    const SoundPlayer* GetSoundPlayer() const { return m_pSoundPlayer; }

    void AttachSoundPlayer( SoundPlayer* player );
    void DetachSoundPlayer( SoundPlayer* player );

    void AttachSoundActor( SoundActor* actor );
    void DetachSoundActor( SoundActor* actor );

    void AttachExternalSoundPlayer( ExternalSoundPlayer* extPlayer );
    void DetachExternalSoundPlayer( ExternalSoundPlayer* extPlayer );

    // プレイヤーヒープ
    void AttachPlayerHeap( PlayerHeap* pHeap );
    void DetachPlayerHeap( PlayerHeap* pHeap );
    PlayerHeap* GetPlayerHeap() { return m_pPlayerHeap; }

    // アンビエントパラメータ
    void SetAmbientInfo( const AmbientInfo& info );
    void ClearAmbientArgUpdateCallback() { m_AmbientInfo.argUpdateCallback = NULL; }
    void ClearAmbientParamUpdateCallback() { m_AmbientInfo.paramUpdateCallback = NULL; }
    void ClearAmbientArgAllocatorCallback() { m_AmbientInfo.argAllocatorCallback = NULL; }
    const SoundParam& GetAmbientParam() const { return m_AmbientParam; }
    static int GetAmbientPriority( const AmbientInfo& ambientInfo, u32 soundId  );

    // ハンドル関数
    bool IsAttachedGeneralHandle();
    bool IsAttachedTempGeneralHandle();
    virtual bool IsAttachedTempSpecialHandle() = 0;
    void DetachGeneralHandle();
    void DetachTempGeneralHandle();
    virtual void DetachTempSpecialHandle() = 0;

    void SetUserParamBuffer(void* buffer, size_t size) { m_pUserParam = buffer; m_UserParamSize = size; }
    void* GetUserParam() { return m_pUserParam; }

    void SetSoundStopCallback(const SoundStopCallback callback) { m_SoundStopCallback = callback; }

    //-----------------------------------------------------------------------------
    // PauseState状態遷移
    //
    // state \ event | pause   | unpause   | fade-finish
    // --------------+---------+-----------+--------------
    // normal        | pausing |  -        |  -
    // pausing       | pausing | unpausing | paused
    // paused        |  -      | unpausing |  -
    // unpausing     | pausing | unpausing | normal
    enum PauseState
    {
        PAUSE_STATE_NORMAL,
        PAUSE_STATE_PAUSING,
        PAUSE_STATE_PAUSED,
        PAUSE_STATE_UNPAUSING
    };
    PauseState GetPauseState() const { return static_cast<PauseState>(m_PauseState); }

    enum MuteState
    {
        MUTE_STATE_NORMAL,
        MUTE_STATE_MUTING,
        MUTE_STATE_MUTED,
        MUTE_STATE_UNMUTING
    };
    MuteState GetMuteState() const { return static_cast<MuteState>(m_MuteState); }

protected:
    virtual driver::BasicSoundPlayer* GetBasicSoundPlayerHandle() = 0;

    virtual void OnUpdatePlayerPriority() {}

    virtual void UpdateMoveValue();
    virtual void OnUpdateParam() {}

    bool IsPlayerAvailable() const { return m_PlayerAvailableFlag; }

private:
    enum State
    {
        STATE_CONSTRUCTED,
        STATE_INITIALIZED,
        STATE_FINALIZED,
        STATE_DESTRUCTED
    };

    void UpdateParam();
    void ClearIsFinalizedForCannotAllocatedResourceFlag();
        // Do～ 関数は、サウンドスレッドのインスタンスに対してコマンドを投げる処理
        // ということにする (既存のものは例外アリ)。

    PlayerHeap* m_pPlayerHeap;
    SoundHandle* m_pGeneralHandle;
    SoundHandle* m_pTempGeneralHandle;
    SoundPlayer* m_pSoundPlayer;
    SoundActor* m_pSoundActor;
    ExternalSoundPlayer* m_pExtSoundPlayer;

    AmbientInfo m_AmbientInfo;
    SoundParam m_AmbientParam;
    SoundActorParam m_ActorParam;

    MoveValue<f32, int> m_FadeVolume;
    MoveValue<f32, int> m_PauseFadeVolume;
    MoveValue<f32, int> m_MuteFadeVolume;

    bool m_StartFlag;
    bool m_StartedFlag;
    bool m_AutoStopFlag;
    bool m_FadeOutFlag;

    bool m_PlayerAvailableFlag;
    bool m_UnPauseFlag;
    u8 m_Priority;
    s8 m_BiquadFilterType;

    State m_State;

    u8 m_PlayerState;   // PlayerState (以下、キャッシュ効率のため u8 で保持)
    u8 m_PauseState;    // PauseState
    u8 m_MuteState;     // MuteState
    u8 padding[1];

    s32 m_AutoStopCounter;
    u32 m_UpdateCounter;
    u32 m_PlayingCounter;
    u32 m_Id;
    u32 m_InstanceId;

    f32 m_InitVolume;
    f32 m_Pitch;
    f32 m_LpfFreq;
    f32 m_BiquadFilterValue;
    u32 m_OutputLineFlag;



    // OutputParam の volume が MoveValue になっただけバージョン
    struct CommonParam
    {
        MoveValue<f32, int> volume;
        MixMode mixMode;
        f32 pan;
        f32 span;
        f32 mainSend;
        f32 fxSend[ AUX_BUS_NUM ];
        void Initialize()
        {
            mixMode = MIX_MODE_PAN;
            volume.InitValue( 1.0f );
            pan = span = mainSend = 0.0f;
            for ( int i = 0; i < AUX_BUS_NUM; i++ )
            {
                fxSend[i] = 0.0f;
            }
        }
        void Update() { volume.Update(); }
        f32 GetVolume() const { return volume.GetValue(); }
        void SetVolume( f32 target, int frame ) { volume.SetTarget(target, frame); }
    };
    void ApplyCommonParam( OutputParam& param );
    CommonParam m_CommonParam;
    OutputParam m_OutputParam[OUTPUT_DEVICE_COUNT];     // TV/DRC 向けパラメータ
    RemoteOutputParam m_RemoteParam[REMOTE_OUT_COUNT];  // リモコン出力向けパラメータ

    void* m_pUserParam;
    size_t m_UserParamSize;
    SoundStopCallback m_SoundStopCallback;

    static u32 s_LastInstanceId;

public:
    ut::LinkListNode m_PriorityLink; // for SoundInstanceManager
    ut::LinkListNode m_SoundPlayerPlayLink;
    ut::LinkListNode m_SoundPlayerPriorityLink;
    ut::LinkListNode m_ExtSoundPlayerPlayLink;
};



class BasicSound::AmbientParamUpdateCallback
{
public:
    virtual ~AmbientParamUpdateCallback() {}
    virtual void detail_UpdateAmbientParam(
        const void* arg,
        u32 soundId,
        SoundAmbientParam* param
    ) = 0;
    virtual int detail_GetAmbientPriority(
        const void* arg,
        u32 soundId
    ) = 0;
};

class BasicSound::AmbientArgUpdateCallback
{
public:
    virtual ~AmbientArgUpdateCallback() {}
    virtual void detail_UpdateAmbientArg(
        void* arg,
        const internal::BasicSound* sound
    ) = 0;
};

class BasicSound::AmbientArgAllocatorCallback
{
public:
    virtual ~AmbientArgAllocatorCallback() {}
    virtual void* detail_AllocAmbientArg( size_t argSize ) = 0;
    virtual void detail_FreeAmbientArg(
        void* arg,
        const internal::BasicSound* sound
    ) = 0;
};


} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_BASIC_SOUND_H_ */

