﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_BANK_FILE_H_
#define NW_SND_BANK_FILE_H_

#if defined(NW_PLATFORM_CAFE)
#include <nw/ut/ut_BinaryFileFormat.h>
#else
#include <nw/snd/snd_BinaryFileFormat.h>
#endif
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_Global.h>

namespace nw {
namespace snd {
namespace internal {

/*
    バンクファイル (.bcbnk) の構造

    bcbnk
     |
     +-- FileHeader
     +-- InfoBlock
          |
          +-- ut::BinaryBlockHeader
          +-- InfoBlockBody
               |
               +-- reference to Table of InnerWave
               |    |
               |    +-> Table<InnerWave>
               |         |
               |         +-- s32 count
               |         +-- InnerWave[0]
               |         +-- InnerWave[1]
               |         +-- ...
               |
               +-- reference to Table of [reference to Instrument]
                    |
                    +-> Table<reference to Instrument>
                         |
                         +-- s32 count
                         +-- Reference to Instrument[0] --+ ← このインデックスが
                         +-- Reference to Instrument[1]   |    prg No として扱われる。
                         +-- ...                          |
    +-----------------------------------------------------+
    |
    +-> Instrument
         |
         +-- reference to KeyRegionChunk
              |
             typeId ?
              |
              +-- Direct --> DirectChunk ← キーリージョンの分割数によって
              +-- Range  --> RangeChunk     格納されるデータ構造が異なる
              +-- Index  --> IndexChunk
                              |
                              +-> KeyRegion を取得する
                                   |
    +------------------------------+
    |
    +-- KeyRegion
         |
         +-- reference to VelocityRegionChunk
              |
             typeId ?
              +-- Direct --> DirectChunk
              +-- Range  --> RangeChunk
              +-- Index  --> IndexChunk
                              |
                              +-> VelocityRegion を取得する
*/

struct BankFile
{
    //-----------------------------------------------------------------
    // ファイルヘッダー

    struct InfoBlock;

    struct FileHeader : public Util::SoundFileHeader
    {
        const InfoBlock* GetInfoBlock() const;
    };

    //-----------------------------------------------------------------
    // 情報ブロック

    struct Instrument;

    struct InfoBlockBody
    {
        // データ
        Util::Reference toWaveIdTable;              // "バンク内波形 ID" テーブル
        Util::Reference toInstrumentReferenceTable; // "インストへの参照" テーブル

        // "バンク内波形 ID" テーブルアクセサ

        // アクセサ
        // (テーブル取得)
        const Util::WaveIdTable& GetWaveIdTable() const;
        const Util::ReferenceTable& GetInstrumentReferenceTable() const;

        // (テーブルアイテム数取得)
        NW_INLINE u32 GetWaveIdCount() const
        {
            return GetWaveIdTable().GetCount();
        }
        NW_INLINE s32 GetInstrumentCount() const
        {
            return GetInstrumentReferenceTable().count;
        }

        // (テーブルアイテム取得)
        NW_INLINE const Util::WaveId* GetWaveId( u32 index ) const
        {
            return GetWaveIdTable().GetWaveId( index );
        }
        const Instrument* GetInstrument( int programNo ) const;
    };

    struct InfoBlock
    {
        ut::BinaryBlockHeader   header;
        InfoBlockBody           body;
    };


    struct KeyRegion;
    struct Instrument
    {
        // データ
        Util::Reference toKeyRegionChunk;

        // アクセサ
        const KeyRegion* GetKeyRegion( u32 key ) const;
    };

    struct VelocityRegion;
    struct KeyRegion
    {
        // データ
        Util::Reference toVelocityRegionChunk;

        // アクセサ
        const VelocityRegion* GetVelocityRegion( u32 velocity ) const;
    };

    struct RegionParameter;
    struct VelocityRegion
    {
        // データ
#if defined(NW_PLATFORM_CAFE)
        nw::ut::ResU32  waveIdTableIndex;
#else
        u32  waveIdTableIndex;
#endif
        Util::BitFlag   optionParameter;

        // アクセサ
        u8 GetOriginalKey() const;
        u8 GetVolume() const;
        u8 GetPan() const;
#ifdef NW_PLATFORM_RVL
        u8 GetSurroundPan() const;
#endif /* NW_PLATFORM_RVL */
        f32 GetPitch() const;
        bool IsIgnoreNoteOff() const;
        u8 GetKeyGroup() const;
        u8 GetInterpolationType() const;
        const AdshrCurve& GetAdshrCurve() const;

        const RegionParameter* GetRegionParameter() const;
    };

#if !defined(NW_PLATFORM_CAFE)
    // VelocityRegion::optionParameter に格納されている値
    struct RegionParameter
    {
        // --------------------------------------------
        // VELOCITY_REGION_KEY
        u8 originalKey;
        u8 padding1[3];                     // ignore
        // --------------------------------------------
        // VELOCITY_REGION_VOLUME
        u8 volume;
        u8 padding2[3];                     // ignore
        // --------------------------------------------
        // VELOCITY_REGION_PAN
        u8 pan;
        s8 surroundPan;                     // for RVL
        u8 padding3[2];                     // ignore
        // --------------------------------------------
        // VELOCITY_REGION_PITCH
        f32 pitch;
        // --------------------------------------------
        // VELOCITY_REGION_INSTRUMENT_NOTE_PARAM
        bool isIgnoreNoteOff;
        u8 keyGroup;
        u8 interpolationType;
        u8 padding4[1];                     // ignore
        // --------------------------------------------
        // VELOCITY_REGION_ENVELOPE
        u32 offset;                         // ignore
        Util::Reference refToAdshrCurve;    // ignore
        AdshrCurve adshrCurve;
    };
#else
    // HACK : Cafe(BigEndian) の仮対応です。
    // VelocityRegion::optionParameter に格納されている値
    struct RegionParameter
    {
        // --------------------------------------------
        // VELOCITY_REGION_KEY
        u8 padding1[3];                     // ignore
        u8 originalKey;
        // --------------------------------------------
        // VELOCITY_REGION_VOLUME
        u8 padding2[3];                     // ignore
        u8 volume;
        // --------------------------------------------
        // VELOCITY_REGION_PAN
        u8 padding3[2];                     // ignore
        s8 surroundPan;                     // for RVL
        u8 pan;
        // --------------------------------------------
        // VELOCITY_REGION_PITCH
        ut::ResF32 pitch;
        // --------------------------------------------
        // VELOCITY_REGION_INSTRUMENT_NOTE_PARAM
        u8 padding4[1];                     // ignore
        u8 interpolationType;
        u8 keyGroup;
        bool isIgnoreNoteOff;
        // --------------------------------------------
        // VELOCITY_REGION_ENVELOPE
        u32 offset;                         // ignore
        Util::Reference refToAdshrCurve;    // ignore
        AdshrCurve adshrCurve;
    };
#endif
};

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_BANK_FILE_H_ */

