﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_ANIM_SOUND_IMPL_H_
#define NW_SND_ANIM_SOUND_IMPL_H_

#if defined( NW_PLATFORM_CTR )
    #include <nn/types.h>
#elif defined( NW_PLATFORM_CAFE )
    #include <nw/types.h>
#endif
#include <nw/snd/snd_AnimSoundFile.h>
#include <nw/snd/snd_AnimSoundFileReader.h>

namespace nw {
namespace snd {

class SoundArchive;
class SoundStartable;

namespace internal {

class AnimEventPlayer;

class AnimSoundImpl
{
    NW_DISALLOW_COPY_AND_ASSIGN(AnimSoundImpl);

public:
    //
    // 定義
    //
    enum EventType
    {
        EVENT_TYPE_TRIGGER_START,
        EVENT_TYPE_TRIGGER_STOP,
        EVENT_TYPE_RANGE_START,
        EVENT_TYPE_RANGE_STOP
    };

    enum PlayDirection
    {
        PLAY_DIRECTION_FORWARD,
        PLAY_DIRECTION_BACKWARD
    };

    typedef void (*EventCallback)(
        EventType type,
        s32 frame,
        const char* soundLabel,
        u32 userParam,
        void* arg );

    //
    // 関数
    //
    AnimSoundImpl(
            SoundStartable& starter,
            AnimEventPlayer* eventPlayers,
            int eventPlayerCount );
    ~AnimSoundImpl();

    bool Initialize( const void* bfasdFile );
    void Finalize();
    bool ConvertSoundId( const SoundArchive& arc );
    bool IsAvailable() const { return m_Reader.IsAvailable(); }
    void SetBaseStep( f32 baseStep ) { m_BaseStep = baseStep; }

    void SetEventCallback( EventCallback callback, void* arg )
    {
        m_EventCallback = callback;
        m_EventCallbackArg = arg;
    }

    void ResetFrame( f32 frame, int loopCounter );
    void UpdateFrame( f32 frame, PlayDirection direction = PLAY_DIRECTION_FORWARD );

    void StopAllSound();

    u32 GetFrameSize() const { return m_FrameSize; }
    int GetLoopCount() const { return IsAvailable() ? m_LoopCounter : 0; }
    f32 GetCurrentFrame() const { return IsAvailable() ? m_CurrentFrame : 0.0f; }

    static void Dump( const void* bfasdFile );

private:
    void UpdateForward( f32 frame );
    void UpdateBackward( f32 frame );
    void UpdateOneFrame( s32 current, PlayDirection direction );

    void UpdateTrigger(
            const AnimSoundFile::AnimEvent& event,
            s32 current,
            PlayDirection direction );
    void UpdateRange(
            const AnimSoundFile::AnimEvent& event,
            s32 current,
            PlayDirection direction );

    void UpdateForwardRange( const AnimSoundFile::AnimEvent& event, s32 current );
    void UpdateBackwardRange( const AnimSoundFile::AnimEvent& event, s32 current );

    void StartEvent( const AnimSoundFile::EventInfo& info, bool isStopWhenFinalize );
    void HoldEvent( const AnimSoundFile::EventInfo& info, bool isStopWhenFinalize );
    void StopEvent( const AnimSoundFile::EventInfo& info );

    bool IsPlayableLoopCount( const AnimSoundFile::FrameInfo& info );
    void WritePlaySpeedToSequenceVariable(
            int eventPlayerNo, const AnimSoundFile::EventInfo& info );

    SoundStartable& m_Starter;
    AnimSoundFileReader m_Reader;
    AnimEventPlayer* m_pEventPlayers;

    EventCallback m_EventCallback;
    void* m_EventCallbackArg;

    f32 m_CurrentFrame;
    int m_EventPlayerCount;
    bool m_IsActive;
    bool m_IsInitFrame;
    bool m_IsReset;
    s32 m_LoopCounter;
    f32 m_BaseStep;
    f32 m_CurrentSpeed;
    s32 m_FrameSize;    // 何度も参照するのでキャッシュしておく
};

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_ANIM_SOUND_IMPL_H_ */

