﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_AnimSound.h
 *
 * @file snd_AnimSound.h
 */

#ifndef NW_SND_ANIM_SOUND_H_
#define NW_SND_ANIM_SOUND_H_

#if defined( NW_PLATFORM_CTR )
    #include <nn/types.h>
#elif defined( NW_PLATFORM_CAFE )
    #include <nw/types.h>
#endif
#include <nw/snd/snd_AnimSoundImpl.h>
#include <nw/snd/snd_AnimEventPlayer.h>

namespace nw {
namespace snd {

class SoundStartable;
class SoundArchive;

//---------------------------------------------------------------------------
//! @brief  アニメーションサウンドを再生させるテンプレートクラスです。
//!
//!         アニメーションサウンドとは、アニメーションサウンドデータ (*.bfasd ファイル)
//!         を使うことで、アニメーションの再生に合わせて自動的にサウンドが再生される機能です。
//!
//!         アニメーションサウンドデータには、サウンドのラベル文字列と、
//!         そのサウンドを再生するフレーム位置が定義されたイベントが含まれています。
//!         @ref UpdateFrame を呼び出しアニメーションサウンドのフレーム位置を進めると、
//!         通過したフレーム位置にあるイベントのサウンドが再生されます。
//!
//!         アニメーションサウンドでは、ラベル文字列を使用してサウンドを再生します。
//!         そのため、アニメーションサウンドで再生するサウンドを、
//!         文字列で再生できる状態にしておく必要があります。
//!         FS サウンドアーカイブを使用している場合は、
//!         サウンドのラベル文字列変換テーブルとサウンドデータが、
//!         メモリ上にロードされている必要があります。
//!         ラベル文字列変換テーブルのロードには
//!         @ref FsSoundArchive::LoadLabelStringData を呼び出します。
//!
//!         テンプレート引数 EVENT_PLAYER_NUM は、
//!         同時に扱うことができるイベントの上限数が書きこまれます。
//!         イベントとは、AnimSoundMaker でアニメーションサウンドを編集する画面における
//!         リストの「各行」が該当します。
//!
//!         イベントの上限数以上のイベントを扱おうとした場合、
//!         実行中のイベントで再生しているサウンドのうち、
//!         一番プレイヤー優先度の低いサウンドを再生しているイベントが停止され、
//!         新しいイベントが実行されます。
//!
//!         通常、メモリが足りないなどの問題が無い場合は、
//!         @ref AnimSound クラスをご利用ください。
//!
//! @tparam EVENT_PLAYER_NUM    同時に扱うことのできるるイベントの上限数です。
//!
//! @see AnimSound クラス
//!
//! @date 2011/01/07 初版
//---------------------------------------------------------------------------
template< int EVENT_PLAYER_NUM >
class TAnimSound
{
public:
    //---------------------------------------------------------------------------
    //! @brief  再生方向を表す列挙型です。
    //!
    //!         @ref UpdateFrame でアニメーションサウンドを再生する際の、
    //!         再生方向を表現します。
    //!
    //! @see UpdateFrame
    //!
    //! @date 2011/01/07 初版
    //---------------------------------------------------------------------------
    enum PlayDirection
    {
        //! 順方向への再生を表します。
        PLAY_DIRECTION_FORWARD  = internal::AnimSoundImpl::PLAY_DIRECTION_FORWARD,
        //! 逆方向への再生を表します。
        PLAY_DIRECTION_BACKWARD = internal::AnimSoundImpl::PLAY_DIRECTION_BACKWARD
    };

    //---------------------------------------------------------------------------
    //! @brief  イベントの種類を表す列挙型です。
    //!
    //!         @ref EventCallback の引数で取得することができます。
    //!
    //!         イベントにはトリガイベントとレンジイベントの 2 種類があります。
    //!         トリガイベントは、開始フレームを通過したときにサウンドの再生が始まり、
    //!         終了フレームを通過したときにサウンドの再生が停止します。
    //!         レンジイベントは、
    //!         フレームの現在位置が開始フレームと終了フレームの間にある場合に
    //!         サウンドを再生し続けます。
    //!
    //! @see EventCallback
    //!
    //! @date 2011/01/07 初版
    //---------------------------------------------------------------------------
    enum EventType
    {
        //! トリガイベントの開始を表します。
        EVENT_TYPE_TRIGGER_START = internal::AnimSoundImpl::EVENT_TYPE_TRIGGER_START,
        //! トリガイベントの終了を表します。
        EVENT_TYPE_TRIGGER_STOP  = internal::AnimSoundImpl::EVENT_TYPE_TRIGGER_STOP,
        //! レンジイベントの開始を表します。
        EVENT_TYPE_RANGE_START   = internal::AnimSoundImpl::EVENT_TYPE_RANGE_START,
        //! レンジイベントの終了を表します。
        EVENT_TYPE_RANGE_STOP    = internal::AnimSoundImpl::EVENT_TYPE_RANGE_STOP
    };

    //---------------------------------------------------------------------------
    //! @brief  アニメーションサウンドのイベント発生時に呼び出されるコールバック関数の型です。
    //!
    //!         どのようなイベントが発生したかは type で知ることができます。
    //!
    //! @param[in] type     イベントの種類です。
    //! @param[in] frame    イベントの発生したフレーム位置です。
    //! @param[in] soundLabel   イベントで再生されるサウンドのラベル文字列です。
    //! @param[in] userParam    イベントのユーザーパラメータです。
    //! @param[in] arg      @ref SetEventCallback で設定されたユーザー引数です。
    //!
    //! @see SetEventCallback
    //! @see EventType
    //!
    //! @date 2011/01/07 初版
    //---------------------------------------------------------------------------
    typedef void (*EventCallback)(
        EventType type,
        s32 frame,
        const char* soundLabel,
        u32 userParam,
        void* arg );

    //----------------------------------------
    //! @name コンストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //!         starter で、アニメーションサウンド内のサウンド再生に使用する
    //!         再生インターフェイスを指定します。
    //!         再生インターフェイスには、@ref SoundArchivePlayer または
    //!         @ref Sound3DActor 等を指定することができます。
    //!
    //! @param[in] starter  サウンド再生インターフェイスです。
    //!
    //! @see SoundStartable クラス
    //! @see SoundArchivePlayer クラス
    //! @see Sound3DActor クラス
    //!
    //! @date 2011/01/07 初版
    //---------------------------------------------------------------------------
    explicit TAnimSound( SoundStartable& starter )
    : m_Impl( starter, m_EventPlayers, EVENT_PLAYER_NUM )
    {}
    //@}

    //----------------------------------------
    //! @name 初期化処理、終了処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief  アニメーションサウンドを初期化します。
    //!
    //!         アニメーションサウンドで使用するデータ (*.bfasd ファイル) を
    //!         bfasdFile に渡します。
    //!
    //!         本関数で初期化後、@ref ConvertSoundId 関数を呼ばない場合は、
    //!         サウンドを再生するごとにラベル文字列からサウンド ID への変換が行われます。
    //!         @ref ConvertSoundId 関数を読んでおくと、
    //!         当該 bfasd ファイルに含まれるサウンドのラベル文字列を、
    //!         あらかじめまとめてサウンド ID に変換しておくことができます。
    //!
    //! @param[in] bfasdFile    アニメーションサウンドデータです。
    //!
    //! @return 初期化に成功すると true、失敗すると false を返します。
    //!
    //! @see ConvertSoundId
    //! @see Finalize
    //!
    //! @date 2011/01/07 初版
    //---------------------------------------------------------------------------
    bool Initialize( const void* bfasdFile ) { return m_Impl.Initialize( bfasdFile ); }

    //---------------------------------------------------------------------------
    //! @brief  アニメーションサウンドを破棄します。
    //!
    //!         AnimSoundMaker で「アニメ切替時再生継続」にチェックが入っていると、
    //!         本関数が呼び出されても、該当サウンドは停止しません。
    //!         チェックが入っていないサウンドは、本関数が呼び出されると停止します。
    //!
    //! @see Initialize
    //!
    //! @date 2011/01/07 初版
    //---------------------------------------------------------------------------
    void Finalize() { m_Impl.Finalize(); }

    //---------------------------------------------------------------------------
    //! @brief  アニメーションサウンドデータ内で使用されているサウンド文字列を、
    //!         サウンドID に変換します。
    //!
    //!         @ref Initialize 後に呼び出しておくと、
    //!         アニメーションサウンド再生時に、
    //!         逐次サウンド文字列からサウンドID に変換する処理を省くことができます。
    //!
    //! @param[in] arc  当該アニメーションサウンドで鳴らすサウンドが属している
    //!                 サウンドアーカイブです。
    //!
    //! @return     変換に成功すると true、失敗すると false を返します。
    //!
    //! @see Initialize
    //!
    //! @date 2011/01/07 初版
    //---------------------------------------------------------------------------
    bool ConvertSoundId( const SoundArchive& arc ) { return m_Impl.ConvertSoundId( arc ); }
    //@}

    //----------------------------------------
    //! @name フレーム処理
    //@{
    //---------------------------------------------------------------------------
    //! @brief  アニメーションサウンドのフレームを進めます。
    //!
    //!         アニメーションサウンドの再生を、現在のフレーム位置から frame まで進めます。
    //!         その際、通過したフレーム位置にあるサウンドが再生されます
    //!         (frame のフレーム位置にあるサウンドを含みます)。
    //!         現在のフレーム位置は @ref GetCurrentFrame で取得できます。
    //!
    //!         direction には、フレーム数が増えていく方向の再生の場合
    //!         PLAY_DIRECTION_FORWARD を、
    //!         減っていく方向の場合 PLAY_DIRECTION_BACKWARD を指定します。
    //!         省略時は PLAY_DIRECTION_FORWARD です。
    //!
    //!         再生方向が順方向で、frame が現在位置より小さい場合は、
    //!         アニメーションサウンドデータの終端でループして再生されます。
    //!         アニメーションサウンドデータのフレーム数は
    //!         @ref GetFrameSize で取得できます。
    //!
    //! @param[in] frame        再生を進めるフレーム位置です。
    //! @param[in] direction    再生方向です。
    //!
    //! @see PlayDirection
    //! @see GetCurrentFrame
    //! @see GetFrameSize
    //! @see SetBaseStep
    //!
    //! @date 2011/01/07 初版
    //---------------------------------------------------------------------------
    void UpdateFrame( f32 frame, PlayDirection direction = PLAY_DIRECTION_FORWARD )
    {
        m_Impl.UpdateFrame(
                frame, static_cast<internal::AnimSoundImpl::PlayDirection>( direction ) );
    }
    //---------------------------------------------------------------------------
    //! @brief  アニメーションサウンドの現在のフレーム位置を設定します。
    //!
    //!         この関数による現在位置の更新では、サウンドは再生されません。
    //!         frame のフレーム位置にあるサウンドは、
    //!         次に @ref UpdateFrame が呼び出された際に再生されます。
    //!
    //! @param[in] frame    設定するフレーム位置です。
    //!
    //! @see UpdateFrame
    //!
    //! @date 2011/01/07 初版
    //---------------------------------------------------------------------------
    void ResetFrame( f32 frame ) { m_Impl.ResetFrame( frame, 0 ); }

    //---------------------------------------------------------------------------
    //! @brief  アニメーションサウンドの通常のフレーム更新間隔を設定します。
    //!
    //!         デフォルト値として、あらかじめ 1.0f が設定されています。
    //!
    //!         アニメーションサウンドの再生速度を決めるための、
    //!         通常のフレーム更新間隔を設定します。
    //!         @ref UpdateFrame 関数で、 1.0f ずつ大きな値を設定する場合は、
    //!         1.0f を設定します。
    //!         アプリケーション実行中に FPS を変更した場合などで、
    //!         2.0f ずつ大きな値を設定する場合は、2.0f を設定します。
    //!
    //!         ここで設定する値は、AnimSound インスタンス内で保持される
    //!         「再生速度」の算出に用いられます。
    //!         たとえば、下記のような場合は、
    //!         再生速度は「(4.0f - 2.0f) ÷ 1.0f = 2.0f」と算出されます。
    //!
    //!         - 【通常のフレーム更新間隔 (本関数で設定)】を 1.0f
    //!         - 【前回の UpdateFrame 関数で設定されたフレーム位置】を 2.0f
    //!         - 【今回の UpdateFrame 関数で設定されたフレーム位置】を 4.0f
    //!
    //!         フレーム位置を 4.0f → 2.0f のように逆方向に設定すると、
    //!         再生速度は負値を取ります。
    //!
    //!         ここで算出された「再生速度」を 100 倍し、
    //!         かつ、-32768～32767 の範囲でクランプした値が、
    //!         AnimSoundMaker の「再生速度反映変数」で設定したシーケンス変数に反映されます。
    //!         反映タイミングは、該当サウンドが再生されるときです。
    //!
    //!         また、「再生速度反映変数」がグローバル変数の場合、
    //!         該当サウンドがシーケンスサウンドでなくても値が反映されます。
    //!         「再生速度反映変数」がローカル変数の場合は、
    //!         該当サウンドがシーケンスサウンドでないと値は反映されません。
    //!
    //! @param[in] baseStep     通常のフレーム更新間隔です。
    //! @see UpdateFrame
    //!
    //! @date 2011/01/07 初版
    //---------------------------------------------------------------------------
    void SetBaseStep( f32 baseStep = 1.0f ) { m_Impl.SetBaseStep( baseStep ); }
    //@}

    //----------------------------------------
    //! @name 情報取得
    //@{
    //---------------------------------------------------------------------------
    //! @brief  アニメーションサウンドが使用可能な状態かどうかを取得します。
    //!
    //!         アニメーションサウンドは、
    //!         @ref Initialize が true を返してから、@ref Finalize を呼び出すまで、
    //!         使用可能です。
    //!
    //! @return 使用可能な場合 true、不可能な場合 false を返します。
    //!
    //! @see Initialize
    //! @see Finalize
    //!
    //! @date 2011/01/07 初版
    //---------------------------------------------------------------------------
    bool IsAvailable() const { return m_Impl.IsAvailable(); }

    //---------------------------------------------------------------------------
    //! @brief  アニメーションサウンドデータのフレーム数を取得します。
    //!
    //!         アニメーションサウンドが使用可能でない場合は、 0 を返します。
    //!
    //! @return アニメーションサウンドデータのフレーム数を返します。
    //!
    //! @see IsAvailable
    //!
    //! @date 2011/01/07 初版
    //---------------------------------------------------------------------------
    u32 GetFrameSize() const { return m_Impl.GetFrameSize(); }

    //---------------------------------------------------------------------------
    //! @brief  現在のフレーム位置を取得します。
    //!
    //!         アニメーションサウンドが使用可能でない場合は、 0.0f を返します。
    //!
    //! @return 現在のフレーム位置を返します。
    //!
    //! @see IsAvailable
    //!
    //! @date 2011/01/07 初版
    //---------------------------------------------------------------------------
    f32 GetCurrentFrame() const { return m_Impl.GetCurrentFrame(); }
    //@}

    //---------------------------------------------------------------------------
    //! @brief  アニメーションサウンドで再生中のすべてのイベントの発音を停止します。
    //!
    //! @date 2011/01/07 初版
    //---------------------------------------------------------------------------
    void StopAllSound() { m_Impl.StopAllSound(); }

    //---------------------------------------------------------------------------
    //! @brief  アニメーションサウンドにイベントコールバックを設定します。
    //!
    //!         イベントコールバックは、フレームがイベントを通過し、
    //!         イベントの処理が発生したときに呼び出されます。
    //!
    //! @param[in] callback イベントコールバックです。
    //! @param[in] arg      イベントコールバックに渡されるユーザー引数です。
    //!
    //! @see EventCallback
    //!
    //! @date 2011/01/07 初版
    //---------------------------------------------------------------------------
    void SetEventCallback( EventCallback callback, void* arg )
    {
        m_Impl.SetEventCallback( reinterpret_cast<internal::AnimSoundImpl::EventCallback>( callback ), arg );
    }

    //! @briefprivate
    //! @return :private
    int detail_GetLoopCount() const { return m_Impl.GetLoopCount(); }

    //---------------------------------------------------------------------------
    //! @brief  アニメーションサウンドデータ (*.bfasd ファイル) の内容をダンプします。
    //!
    //!         NW_LOG により下記のような形式でログに出力されます。
    //!
    //! @code
    //! *** Dump begin ***
    //! FrameSize :  30
    //! EventCount:   2
    //! [  0] start(4) end(-1) eventType(TRIG)
    //!       (WSD_FOOTSTEP0) vol(127) pitch(3.000000) userParam(0x00000000)
    //!       notStopSoundWhenAnimFinish?(0) seqVarEnable?(0) seqVar(0)
    //!       playDirection(BOTH)
    //! [  1] start(19) end(-1) eventType(TRIG)
    //!       (WSD_FOOTSTEP1) vol(127) pitch(3.000000) userParam(0x00000000)
    //!       notStopSoundWhenAnimFinish?(0) seqVarEnable?(0) seqVar(0)
    //!       playDirection(BOTH)
    //! *** Dump end *****
    //! @endcode
    //!
    //!         それぞれの意味は下記のとおりです。
    //!
    //!         - FrameSize : アニメーションのフレーム長です。
    //!         - EventCount : アニメーションサウンドデータに含まれるイベントの数です。
    //!         - start/end : イベントのスタートフレーム・エンドフレームです。-1 は
    //!           フレームを指定していないことを表します。
    //!         - eventType : イベントの種類です。
    //!         - (XXXX) : サウンド名です。
    //!         - vol : 音量です。
    //!         - pitch : ピッチです。
    //!         - userParam : ユーザーパラメータです。16 進数で表示します。
    //!         - notStopSoundWhenAnimFinish? : 「アニメ切替時再生継続」するかどうかを表します。
    //!           1 のときに再生継続します。0 のときは再生継続しません。
    //!         - seqVarEnable? : 「再生速度反映変数」が設定されているかどうかを表します。
    //!           1 のときに設定されていることを表します。
    //!         - seqVar : 「再生速度反映変数」で設定しているシーケンス変数を表します。
    //!           0～15 がローカル変数、15～31 がグローバル変数を示します。
    //!
    //! @param[in] bfasdFile    アニメーションサウンドデータです。
    //!
    //! @date 2011/10/12 初版
    //---------------------------------------------------------------------------
    static void Dump( const void* bfasdFile )
    {
        return internal::AnimSoundImpl::Dump(bfasdFile);
    }

private:
    internal::AnimSoundImpl m_Impl;
    internal::AnimEventPlayer m_EventPlayers[ EVENT_PLAYER_NUM ];
};

//---------------------------------------------------------------------------
//! @brief  アニメーションサウンドを再生させるクラスです。
//!
//!         詳しくは @ref TAnimSound クラスの説明をご参照ください。
//!
//!         メモリが足りないなどの問題がない場合は、@ref TAnimSound クラスではなく、
//!         本クラスをご利用ください。
//!
//! @see TAnimSound クラス
//!
//! @date 2011/01/07 初版
//---------------------------------------------------------------------------
typedef TAnimSound<8> AnimSound;

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_ANIM_SOUND_H_ */

