﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_FND_STREAM_CACHE_H_
#define NW_SND_FND_STREAM_CACHE_H_

#include <limits.h>
#include <nw/snd/fnd/basis/sndfnd_Config.h>
#include <nw/snd/fnd/basis/sndfnd_Result.h>
#include <nw/snd/fnd/io/sndfnd_Stream.h>

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
//! @brief  ストリームのキャッシュ機能を提供します。
//---------------------------------------------------------------------------
class StreamCache
{
    NW_DISALLOW_COPY_AND_ASSIGN(StreamCache);

private: // 定数
    static const u32 INVALID_POSITION = UINT_MAX;

private: // 型の定義
    enum CacheState
    {
        CACHE_NONE = 0,
        CACHE_FOR_READ = 1,
        CACHE_FOR_WRITE = 2
    };

public: // コンストラクタ
    StreamCache();

    StreamCache(Stream* stream, void* buffer, u32 length);

    virtual ~StreamCache() { }

public: // メソッド
    void Initialize(Stream* stream, void* buffer, u32 length);

    void Finalize();

    bool IsInitialized() const
    {
        return m_Stream != NULL && m_CacheBuffer != NULL && m_CacheBufferLength > 0;
    }

    bool CanRead() const { return IsInitialized() && m_Stream->CanRead(); }

    bool CanWrite() const { return IsInitialized() && m_Stream->CanWrite(); }

    bool CanSeek() const { return IsInitialized() && m_Stream->CanSeek(); }

    u32 Read(void* buf, u32 length, FndResult* result = NULL);

    u32 Write(const void* buf, u32 length, FndResult* result = NULL);

    FndResult Seek(s32 offset, Stream::SeekOrigin origin);

    u32 GetCurrentPosition() const
    {
        return m_CurrentPosition;
    }

    void ClearCache();

private: // メソッド
    u32 GetReadCacheHitLength(u32 length) const;

    u32 GetWritableCacheLength(u32 length) const;

    //! @brief  対象 Stream の現在位置を StreamCache の現在位置に合わせます。
    FndResult SyncStreamCurrentPosition(u32 position);

    //! @brief  キャッシュの内容を書き込みます。
    FndResult FlushWriteCache();

private: // メンバ変数
    Stream* m_Stream;
    u32     m_CurrentPosition;

    void*   m_CacheBuffer;
    u32     m_CacheBufferLength;
    u32     m_CachePosition;
    u32     m_CachedLength;

    u8      m_CacheState;
    u8      m_Padding1;
    u16     m_Padding2;
};

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_FND_STREAM_CACHE_H_
