﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_FND_RESULT_H_
#define NW_SND_FND_RESULT_H_

#include <nw/snd/fnd/basis/sndfnd_Config.h>

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
//! @brief  sndfnd の処理結果です。
//---------------------------------------------------------------------------
enum FndResultType
{
    //---------------------------------------------------------------------------
    // ビット構成
    //---------------------------------------------------------------------------

    SNDFND_RESULT_ERROR_FLAG_MASK     = 0x80000000,
    SNDFND_RESULT_CATEGORY_MASK       = 0x7f000000,
    SNDFND_RESULT_CODE_MASK           = 0x00ffffff,
    SNDFND_RESULT_CATEGORY_BIT_OFFSET = 24,

    //---------------------------------------------------------------------------
    // カテゴリ
    //---------------------------------------------------------------------------

    SNDFND_RESULT_ERROR_FLAG      = SNDFND_RESULT_ERROR_FLAG_MASK,
    SNDFND_RESULT_CATEGORY_SYSTEM = 0x00 << SNDFND_RESULT_CATEGORY_BIT_OFFSET,
    SNDFND_RESULT_CATEGORY_IO     = 0x01 << SNDFND_RESULT_CATEGORY_BIT_OFFSET,

    //---------------------------------------------------------------------------
    // 成功コード（システム）
    //---------------------------------------------------------------------------

    SNDFND_RESULT_TRUE  = 0x000000 + SNDFND_RESULT_CATEGORY_SYSTEM,   //!< 成功しました。
    SNDFND_RESULT_FALSE = 0x000001 + SNDFND_RESULT_CATEGORY_SYSTEM,   //!< 処理を実行しませんでした。

    //---------------------------------------------------------------------------
    // 失敗コード（システム）
    //---------------------------------------------------------------------------

    SNDFND_RESULT_FAILED            = 0x000000 + SNDFND_RESULT_CATEGORY_SYSTEM + SNDFND_RESULT_ERROR_FLAG,    //!< 失敗しました。
    SNDFND_RESULT_NOT_INITIALIZED   = 0x000001 + SNDFND_RESULT_CATEGORY_SYSTEM + SNDFND_RESULT_ERROR_FLAG,    //!< 初期化されていません。
    SNDFND_RESULT_NOT_SUPPORTED     = 0x000002 + SNDFND_RESULT_CATEGORY_SYSTEM + SNDFND_RESULT_ERROR_FLAG,    //!< サポートされていません。
    SNDFND_RESULT_NOT_OPENED        = 0x000003 + SNDFND_RESULT_CATEGORY_SYSTEM + SNDFND_RESULT_ERROR_FLAG,    //!< 開かれていません。
    SNDFND_RESULT_OUT_OF_MEMORY     = 0x000004 + SNDFND_RESULT_CATEGORY_SYSTEM + SNDFND_RESULT_ERROR_FLAG,    //!< メモリ不足です。
    SNDFND_RESULT_INVALID_ARGUMENT  = 0x000005 + SNDFND_RESULT_CATEGORY_SYSTEM + SNDFND_RESULT_ERROR_FLAG,    //!< 引数が不正です。
    SNDFND_RESULT_INVALID_STATUS    = 0x000006 + SNDFND_RESULT_CATEGORY_SYSTEM + SNDFND_RESULT_ERROR_FLAG,    //!< 状態が不正です。

    //---------------------------------------------------------------------------
    // 失敗コード（入出力）
    //---------------------------------------------------------------------------

    SNDFND_RESULT_IO_ERROR          = 0x000000 + SNDFND_RESULT_CATEGORY_IO + SNDFND_RESULT_ERROR_FLAG,    //!< 入出力でエラーが発生しました。
    SNDFND_RESULT_IO_FILE_NOT_FOUND = 0x000001 + SNDFND_RESULT_CATEGORY_IO + SNDFND_RESULT_ERROR_FLAG,    //!< ファイルが見つかりません。
    SNDFND_RESULT_IO_INVALID_ACCESS = 0x000002 + SNDFND_RESULT_CATEGORY_IO + SNDFND_RESULT_ERROR_FLAG     //!< アクセス権がない等、無効なアクセスが発生しました。
};

//---------------------------------------------------------------------------
//! @brief  sndfnd の処理結果を制御します。
//---------------------------------------------------------------------------
class FndResult
{
public: // コンストラクタ
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //---------------------------------------------------------------------------
    FndResult() : value(SNDFND_RESULT_TRUE) { }

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //! @param[in] value 結果コードです。
    //---------------------------------------------------------------------------
    explicit FndResult(u32 value) : value(value) { }

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //! @param[in] value 処理結果です。
    //---------------------------------------------------------------------------
    explicit FndResult(FndResultType value) : value(value) { }

public: // メソッド
    //---------------------------------------------------------------------------
    //! @brief  結果が成功かどうかを調べます。
    //! @return 成功の場合は true、失敗の場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsSucceeded() const { return !IsFailed(); }

    //---------------------------------------------------------------------------
    //! @brief  結果が成功かつ TRUE かどうかを調べます。
    //! @return 成功かつ TRUE の場合は true、それ以外の場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsTrue() const { return value == SNDFND_RESULT_TRUE; }

    //---------------------------------------------------------------------------
    //! @brief  結果が成功かつ FALSE かどうかを調べます。
    //! @return 成功かつ FALSE の場合は true、それ以外の場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsFalse() const { return value == SNDFND_RESULT_FALSE; }

    //---------------------------------------------------------------------------
    //! @brief  結果が失敗かどうかを調べます。
    //! @return 失敗の場合は true、成功の場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsFailed() const { return (value & SNDFND_RESULT_ERROR_FLAG) != 0; }

    //---------------------------------------------------------------------------
    //! @brief  Result を文字列に変換します。
    //!
    //!         この関数は DEBUG または DEVELOP ビルド時にのみ正しい値を返します。
    //!
    //! @return DEBUG または DEVELOP ビルド時には、変換後の文字列を返します。
    //!
    //!         RELEASE ビルド時には必ず空文字を返します。
    //---------------------------------------------------------------------------
    const char* ToString() const
    {
#if !defined(NW_RELEASE)
        switch(value)
        {
        //---------------------------------------------------------------------------
        // 成功コード（システム）
        case SNDFND_RESULT_TRUE:
            return "SNDFND_RESULT_TRUE";

        case SNDFND_RESULT_FALSE :
            return "SNDFND_RESULT_FALSE";

        //---------------------------------------------------------------------------
        // 失敗コード（システム）
        case SNDFND_RESULT_FAILED:
            return "SNDFND_RESULT_FAILED";

        case SNDFND_RESULT_NOT_INITIALIZED:
            return "SNDFND_RESULT_NOT_INITIALIZED";

        case SNDFND_RESULT_NOT_SUPPORTED:
            return "SNDFND_RESULT_NOT_SUPPORTED";

        case SNDFND_RESULT_NOT_OPENED:
            return "SNDFND_RESULT_NOT_OPENED";

        case SNDFND_RESULT_OUT_OF_MEMORY:
            return "SNDFND_RESULT_OUT_OFMEMORY";

        case SNDFND_RESULT_INVALID_ARGUMENT:
            return "SNDFND_RESULT_INVALID_ARGUMENT";

        case SNDFND_RESULT_INVALID_STATUS:
            return "SNDFND_RESULT_INVALID_STATUS";
        }
#endif

        return "";
    }

public: // オペレータ
    //--------------------------------------------------------------------------
    //! @brief   u32 型へのキャスト演算子です。
    //!
    //! @return  u32 値を返します。
    //---------------------------------------------------------------------------
    operator u32() const
    {
        return value;
    }

    //--------------------------------------------------------------------------
    //! @brief   FndResultType 型へのキャスト演算子です。
    //!
    //! @return  FndResultType で表現した値を返します。
    //---------------------------------------------------------------------------
    operator FndResultType() const
    {
        return static_cast<FndResultType>(value);
    }

private: // メンバ変数
    u32 value;
};


} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_FND_RESULT_H_
