﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_FND_MEMORY_H_
#define NW_SND_FND_MEMORY_H_

#include <nw/ut/ut_Inlines.h>
#include <nw/snd/fnd/basis/sndfnd_Config.h>

#if defined(NW_PLATFORM_CAFE)
#include <cafe/mem.h>
#elif defined(NW_PLATFORM_CTR)
#include <nn/fnd.h>
#elif defined(NW_PLATFORM_WIN32)
#include <winext/cafe/mem.h>
#elif defined(NW_PLATFORM_ANDROID)
#include <winext/cafe/mem.h>
#elif defined(NW_USE_NINTENDO_SDK)
#include <winext/cafe/mem.h>
#endif

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
//! @brief  メモリ特性です。
//---------------------------------------------------------------------------
class MemoryTraits
{
public:
    static const s32 DEFAULT_ALIGNMENT = 4; //!< デフォルトのアライメント
};

//---------------------------------------------------------------------------
//! @brief  メモリアロケータのインターフェイスです。
//---------------------------------------------------------------------------
class IAllocator
{
public:
    virtual ~IAllocator() { }

public:
    virtual void* Alloc(size_t size, s32 alignment = MemoryTraits::DEFAULT_ALIGNMENT) = 0;
    virtual void Free(void* ptr) = 0;
};

//---------------------------------------------------------------------------
//! @brief  前から順にスタック方式でメモリを割り当てるヒープクラスです。
//---------------------------------------------------------------------------
class FrameHeap
{
    NW_DISALLOW_COPY_AND_ASSIGN(FrameHeap);

public: // コンストラクタ
    FrameHeap() :
        m_Buffer(NULL),
        m_BufferLength(0),
        m_UsedLength(0)
    {
    }

    virtual ~FrameHeap()
    {
        Finalize();
    }

public: // メソッド
    void Initialize(void* buffer, size_t length);
    void Finalize();

    bool IsInitialized() const
    {
        return m_Buffer != NULL;
    }

    size_t GetUsedLength() const { return m_UsedLength; }

    size_t GetFreeLength(s32 alignment = MemoryTraits::DEFAULT_ALIGNMENT) const;

    virtual void* Alloc(size_t size, s32 alignment = MemoryTraits::DEFAULT_ALIGNMENT);

    void Clear()
    {
        m_UsedLength = 0;
    }

private: // メンバ変数
    void*  m_Buffer;         //!< バッファです。
    size_t m_BufferLength;   //!< バッファの長さです。
    size_t m_UsedLength;     //!< 使用中のメモリサイズです。
};

//---------------------------------------------------------------------------
//! @brief  固定長ユニット単位でメモリを割り当てるヒープクラスです。
//---------------------------------------------------------------------------
class UnitHeap : public IAllocator
{
    NW_DISALLOW_COPY_AND_ASSIGN(UnitHeap);

public: // コンストラクタ
    UnitHeap() :
        m_Buffer(NULL),
        m_BufferLength(0),
        m_UnitLength(0),
#if !defined(NW_RELEASE)
        m_AllocatedUnitCount(0),
#endif
        m_FirstFreeUnit(NULL)
    {
    }

    virtual ~UnitHeap()
    {
        Finalize();
    }

public: // メソッド
    void Initialize(
        void* buffer,
        size_t length,
        size_t unitLength,
        u32 unitCount,
        u32 unitAlignment = MemoryTraits::DEFAULT_ALIGNMENT);

    void Finalize();

    bool IsInitialized() const
    {
        return m_Buffer != NULL;
    }

    size_t GetBufferLength() const
    {
        return m_BufferLength;
    }

    size_t GetUsedLength() const { return m_AllocatedUnitCount * m_UnitLength; }

    virtual void* Alloc(size_t size = 0, s32 alignment = MemoryTraits::DEFAULT_ALIGNMENT);
    virtual void Free(void* ptr);

private: // メソッド
    void* GetNextFreeUnit(void* unit) const;

private: // メンバ変数
    void*  m_Buffer;               //!< バッファです。
    size_t m_BufferLength;         //!< バッファの長さです。
    size_t m_UnitLength;           //!< ユニットサイズです。
    u32    m_AllocatedUnitCount;   //!< 割り当て済みユニット数です。
    void*  m_FirstFreeUnit;        //!< 最初の空きユニットです。
};

//---------------------------------------------------------------------------
//! @brief  拡張ヒープマネージャを利用したヒープクラスです。
//---------------------------------------------------------------------------
class ExpandedHeap : public IAllocator
{
    NW_DISALLOW_COPY_AND_ASSIGN(ExpandedHeap);

public: // コンストラクタ
    ExpandedHeap() : m_HeapHandle(NULL) { }

    virtual ~ExpandedHeap()
    {
        Finalize();
    }

public: // メソッド
    bool Initialize(void* buffer, size_t length);

    void Finalize();

    bool IsInitialized() const
    {
        return m_HeapHandle != NULL;
    }

    virtual void* Alloc(size_t size, s32 alignment = MemoryTraits::DEFAULT_ALIGNMENT);
    virtual void Free(void* ptr);

public:
#if defined( NW_PLATFORM_WIN32 )
    typedef nw::internal::winext::MEMiHeapHead MEMiHeapHead;
    typedef nw::internal::winext::MEMiExpHeapHead MEMiExpHeapHead;
    typedef nw::internal::winext::MEMiExpHeapMBlockHead MEMiExpHeapMBlockHead;
#elif defined( NW_PLATFORM_ANDROID )
    typedef nw::internal::winext::MEMiHeapHead MEMiHeapHead;
    typedef nw::internal::winext::MEMiExpHeapHead MEMiExpHeapHead;
    typedef nw::internal::winext::MEMiExpHeapMBlockHead MEMiExpHeapMBlockHead;
#elif defined(NW_USE_NINTENDO_SDK)
    // TODO: nn_os
    typedef nw::internal::winext::MEMiHeapHead MEMiHeapHead;
    typedef nw::internal::winext::MEMiExpHeapHead MEMiExpHeapHead;
    typedef nw::internal::winext::MEMiExpHeapMBlockHead MEMiExpHeapMBlockHead;
#endif
     //! ブロックごとの管理領域のサイズ
#if defined( NW_PLATFORM_CTR )
    static const s32 BLOCK_MD_SIZE = sizeof(nn::fnd::detail::NNSiFndExpHeapMBlockHead);
#else
    static const s32 BLOCK_MD_SIZE = sizeof(MEMiExpHeapMBlockHead);
#endif
     //! ヒープ全体の管理領域のサイズ
#if defined( NW_PLATFORM_CTR )
    static const s32 HEAP_MD_SIZE = 4 + 4 + BLOCK_MD_SIZE; // 先頭のRoundUp + 末尾のRoundDown + 拡張ヒープ全体のヘッダサイズ
#else
    static const s32 HEAP_MD_SIZE = 4 + 4 + BLOCK_MD_SIZE + sizeof(MEMiHeapHead) + sizeof(MEMiExpHeapHead); // 先頭のRoundUp + 末尾のRoundDown + 拡張ヒープ全体のヘッダサイズ
#endif

private: // メンバ変数
#if defined( NW_PLATFORM_WIN32 )
    typedef nw::internal::winext::MEMHeapHandle MEMHeapHandle;
#elif defined( NW_PLATFORM_ANDROID )
    typedef nw::internal::winext::MEMHeapHandle MEMHeapHandle;
#elif defined(NW_USE_NINTENDO_SDK)
    // TODO: nn_os
    typedef nw::internal::winext::MEMHeapHandle MEMHeapHandle;
#endif // defined( NW_PLATFORM_WIN32 )

#if defined( NW_PLATFORM_CTR )
    nn::fnd::ExpHeap m_Heap;
    nn::fnd::ExpHeap* m_HeapHandle;
#else
    MEMHeapHandle m_HeapHandle;
#endif
};

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_FND_MEMORY_H_
