﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_EDIT_TYPES_H_
#define NW_SND_EDIT_TYPES_H_

#include <nw/types.h>

namespace nw {
namespace snd {
namespace edit {

//---------------------------------------------------------------------------
//! @brief  【β版】キャッシュ状態を示す列挙体です。
//---------------------------------------------------------------------------
enum CacheState
{
    CACHE_STATE_NONE = 0,   //!< キャッシュされていません。
    CACHE_STATE_CACHING,    //!< キャッシュ中です。
    CACHE_STATE_CACHED      //!< キャッシュ済です。
};

//---------------------------------------------------------------------------
//! @brief  【β版】編集アイテムの種別を示す列挙体です。
//---------------------------------------------------------------------------
enum EditItemType
{
    EDIT_ITEM_TYPE_UNKNOWN = 0,          //!< 不明な種別です。
    EDIT_ITEM_TYPE_STREAM_SOUND = 1,     //!< ストリームサウンドです。
    EDIT_ITEM_TYPE_WAVE_SOUND,           //!< ウェーブサウンドです。
    EDIT_ITEM_TYPE_SEQUENCE_SOUND,       //!< シーケンスサウンドです。
    EDIT_ITEM_TYPE_BANK                  //!< バンクです。
};

//---------------------------------------------------------------------------
//! @brief  【β版】編集アイテム情報を格納する構造体です。
//---------------------------------------------------------------------------
struct EditItemInfo
{
    u32          id;                    //!< 編集アイテムの ID です。
    const char*  name;                  //!< 編集アイテムの名前です。
    EditItemType itemType;              //!< 編集アイテムの種別です。
    bool         isOverrideOriginal;    //!< オリジナルアイテムオーバーライドの有無を示すフラグです。

#if defined(NW_PLATFORM_CTR)
    u8           padding[2];            //!< パディングです。
#endif
};

//! SoundMaker との通信に利用するポート番号の型定義です。
typedef u32 PORT;

//---------------------------------------------------------------------------
//! @briefprivate
//! @brief  編集アイテムの変更を通知するコールバック関数です。
//---------------------------------------------------------------------------
typedef void (*EditItemsChangedCallback)(uptr userParam);

namespace internal {

//---------------------------------------------------------------------------
//! @brief  【β版】32 バイトハッシュ値です。
//---------------------------------------------------------------------------
struct Hash32
{
    static const u32 Size = 32;

    u8 value[Size];

    Hash32() { }

    explicit Hash32(const u8* source)
    {
        const u32* sourceCurrent = reinterpret_cast<const u32*>(source);
        u32* current = reinterpret_cast<u32*>(value);

        // 32 バイトなので、 u32 で 8 回設定します。
        for(u32 count = 0; count < Size/sizeof(u32); ++count)
        {
            *current = *sourceCurrent;

            ++current;
            ++sourceCurrent;
        }
    }

    static const Hash32& Empty()
    {
        static bool isInitialized = false;
        static Hash32 instance;

        if(!isInitialized)
        {
            instance.SetEmpty();
            isInitialized = true;
        }

        return instance;
    }

    void SetEmpty()
    {
        u32* current = reinterpret_cast<u32*>(value);

        // 32 バイトなので、 u32 で 8 回 0 を設定します。
        for(u32 count = 0; count < Size/sizeof(u32); ++count)
        {
            current = 0;
            ++current;
        }
    }

    bool operator ==(const Hash32& rhs) const
    {
        const u32* lhsCurrent = reinterpret_cast<const u32*>(value);
        const u32* rhsCurrent = reinterpret_cast<const u32*>(rhs.value);

        // 32 バイトなので、 u32 で 8 回比較します。
        for(u32 count = 0; count < Size/sizeof(u32); ++count)
        {
            if(*lhsCurrent != *rhsCurrent)
            {
                return false;
            }

            ++lhsCurrent;
            ++rhsCurrent;
        }

        return true;
    }

    bool operator !=(const Hash32& rhs) const { return !(*this == rhs); }

    void ToString(char* buffer, u32 length) const;
};

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_EDIT_TYPES_H_
