﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_EDIT_SOUND_EDIT_H_
#define NW_SND_EDIT_SOUND_EDIT_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

#if defined( NW_PLATFORM_CAFE )
#include <cafe.h>
#endif

#include <nw/sndedit.h>
#include <nw/sndctrl.h>

#include <nw/snd/edit/sndedit_Types.h>
#include <nw/snd/fnd/basis/sndfnd_Memory.h>

#if defined(NW_PLATFORM_CTR)
#if NN_CURRENT_VERSION_NUMBER >= NN_VERSION_NUMBER(4,0,0,0)
#pragma diag_suppress 1301 // padding inserted in struct.
#pragma diag_suppress 2530 // padding added to end of struct.
#endif
#endif

namespace nw {
namespace snd {
namespace edit {

//---------------------------------------------------------------------------
//! @brief  サウンドインゲーム編集機能を提供するクラスです。
//!
//!         このクラスを使用することで、簡単にサウンドインゲーム編集機能を組み込むことができます。@n
//!         細かいカスタマイズを行いたい場合は、（ベータ版扱いとなりますが、）
//!         このクラスではなく、@ref SoundEditSession, @ref SoundArchiveEditor を使用してください。
//!
//---------------------------------------------------------------------------
class SoundEdit
{
public:
    //! サウンドインゲーム編集機能で使用するヒープサイズのデフォルト値です。
    static const s32 DEFAULT_SNDEDIT_HEAP_SIZE = 4 * 1024 * 1024;
    //! サウンドインゲーム編集機能で使用するスレッドのスタックサイズのデフォルト値です。
    static const u32 DEFAULT_THREAD_STACK_SIZE = nw::snd::edit::UpdateCachesHelper::DEFAULT_THREAD_STACK_SIZE;

    //! サウンドインゲーム編集機能で使用するポート番号のデフォルト値です。
    static const PORT DEFAULT_SNDEDIT_PORT0 = internal::HIO_SNDEDIT_SYNC_CHANNEL;
    //! サウンドインゲーム編集機能で使用するポート番号のデフォルト値です。
    static const PORT DEFAULT_SNDEDIT_PORT1 = internal::HIO_SNDEDIT_FUNC_CHANNEL;
    //! サウンドインゲーム編集機能で使用するポート番号のデフォルト値です。
    static const PORT DEFAULT_SNDEDIT_PORT2 = internal::HIO_SNDEDIT_CTRL_CHANNEL;

public:
    //---------------------------------------------------------------------------
    //! @brief  初期化用のパラメータを格納する構造体です。
    //!
    //!         Initialize()に渡すまでに、各パラメータに値を入れておく必要があります。
    //!
    //! @see    Initialize()
    //---------------------------------------------------------------------------
    struct InitializeArg
    {
        //---------------------------------------------------------------------------
        //! @brief  コンストラクタです。
        //---------------------------------------------------------------------------
        InitializeArg()
        : buffer(NULL)
        , bufferSize(0)
        , soundArchive(NULL)
        , soundArchivePlayer(NULL)
        , port0(DEFAULT_SNDEDIT_PORT0)
        , port1(DEFAULT_SNDEDIT_PORT1)
        , port2(DEFAULT_SNDEDIT_PORT2)
#if defined( NW_PLATFORM_CAFE )
        , fsClient(NULL)
        , fsHfioMountPath(NULL)
#endif
        {
        }

        void* buffer;                               //!< インゲーム編集ライブラリが使用するバッファを指定します。
        u32 bufferSize;                             //!< インゲーム編集ライブラリが使用するバッファのサイズを指定します。
        SoundArchive* soundArchive;                 //!< 関連するサウンドアーカイブを指定します。
        SoundArchivePlayer* soundArchivePlayer;     //!< 関連するサウンドアーカイブプレイヤーを指定します。

        PORT port0;                                 //!< 内部で使用するポートです。
        PORT port1;                                 //!< 内部で使用するポートです。
        PORT port2;                                 //!< 内部で使用するポートです。

#if defined( NW_PLATFORM_CAFE )
        FSClient* fsClient;                         //!< FSクライアントを指定します。
        const char* fsHfioMountPath;                //!< HFIOのマウントパスを指定します。
#endif
    };

public:
    //---------------------------------------------------------------------------
    //! @brief  初期化用のオプションパラメータを格納する構造体です。
    //!
    //!         各パラメータには、コンストラクタでデフォルト値が設定されます。@n
    //!         通常の利用において、値を変更する必要はありません。
    //!
    //! @see    Initialize()
    //! @see    GetRequiredMemorySize()
    //---------------------------------------------------------------------------
    struct Option
    {
        //---------------------------------------------------------------------------
        //! @brief  コンストラクタです。
        //---------------------------------------------------------------------------
        Option()
        : sndEditHeapSize(DEFAULT_SNDEDIT_HEAP_SIZE)
        , threadStackSize(DEFAULT_THREAD_STACK_SIZE)
        {
        }

        s32 sndEditHeapSize;            //!< インゲーム編集ライブラリが内部で使用するヒープのサイズを指定します。@n
                                        //!< 指定しないとデフォルト値 DEFAULT_SNDEDIT_HEAP_SIZE が使用されます。@n
                                        //!< 通常の利用において、値を変更する必要はありません。
        u32 threadStackSize;            //!< インゲーム編集ライブラリが内部で使用するスレッドのスタックサイズを指定します。@n
                                        //!< 指定しないとデフォルト値 DEFAULT_THREAD_STACK_SIZE が使用されます。@n
                                        //!< 通常の利用において、値を変更する必要はありません。
    };

public: // コンストラクタ
    //===========================================================================
    //! @name コンストラクタ/デストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //---------------------------------------------------------------------------
    SoundEdit();

    //---------------------------------------------------------------------------
    //! @brief  デストラクタです。
    //---------------------------------------------------------------------------
    ~SoundEdit();

    //@}
    // コンストラクタ/デストラクタ
    //===========================================================================

public: // メソッド
    //===========================================================================
    //! @name 初期化
    //@{

    //---------------------------------------------------------------------------
    //! @brief      SoundEdit の初期化を行います。
    //!
    //!             この関数は、サウンドインゲーム編集機能で利用するメモリを初期化し、
    //!             SoundMaker との接続を開始するための準備を行います。@n
    //!             他の関数を実行する前に、必ずこの関数を呼び出して初期化してください。
    //!
    //! @param[in]  args   InitializeArgを指定します。
    //! @param[in]  param  GetRequiredMemorySize() に渡した Option を指定します。
    //!
    //! @see        InitializeArg
    //! @see        GetRequiredMemorySize()
    //! @see        Finalize()
    //---------------------------------------------------------------------------
    void Initialize(const InitializeArg& args, const Option& param);

    //---------------------------------------------------------------------------
    //! @brief      初期化済みかどうかをチェックします。
    //!
    //! @return     初期化済みであればtrueを返します。
    //!
    //! @see        Initialize()
    //---------------------------------------------------------------------------
    bool IsInitialized() const { return m_IsInitialized; }

    //---------------------------------------------------------------------------
    //! @brief  SoundEdit の終了処理を行います。
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief      サウンドインゲーム編集機能の利用に必要なメモリサイズを取得します。
    //!
    //!             サウンドインゲーム編集機能を利用するには、この関数で取得したサイズ分のバッファを確保し、
    //!             Initialize() の引数 InitializeArg に指定する必要があります。@n
    //!             その際 Initialize() の引数 param には、
    //!             GetRequiredMemorySize() に渡した Option と同じものを指定してください。
    //!
    //! @param[in]  param  Optionを指定します。
    //!
    //! @return     必要なメモリサイズを返します。
    //!
    //! @see        Option
    //! @see        Initialize()
    //---------------------------------------------------------------------------
    u32 GetRequiredMemorySize(const Option& param) const;

    //@}
    // 初期化
    //===========================================================================

    //===========================================================================
    //! @name 開始と終了
    //@{

    //---------------------------------------------------------------------------
    //! @brief  サウンドインゲーム編集機能を開始します。
    //!
    //!         この関数が呼び出されると、Update() のタイミングで
    //!         SoundMaker との通信が行われるようになります。
    //!         また、内部でキャッシュの更新を行うためのスレッドを１つ起動します。
    //!
    //! @see    Stop()
    //! @see    Update()
    //---------------------------------------------------------------------------
    void Start();

    //---------------------------------------------------------------------------
    //! @brief  サウンドインゲーム編集機能が開始されているかチェックします。
    //!
    //! @return サウンドインゲーム編集機能が開始されていれば true を返します。
    //!
    //! @see    Start()
    //---------------------------------------------------------------------------
    bool IsStarted() const { return m_IsStarted; }

    //---------------------------------------------------------------------------
    //! @brief  サウンドインゲーム編集機能を停止します。
    //!
    //!         同時に SoundMakerとの通信や、キャッシュ更新用の内部スレッドを終了します。
    //!
    //! @see    Start()
    //---------------------------------------------------------------------------
    void Stop();

    //---------------------------------------------------------------------------
    //! @brief  サウンドインゲーム編集機能を開始します。
    //!
    //!         この関数の使用は非推奨となります。
    //!         代わりに@ref Start() を使用してください。
    //!
    //! @see    Start()
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION(void Open());

    //---------------------------------------------------------------------------
    //! @brief  サウンドインゲーム編集機能を停止します。
    //!
    //!         この関数の使用は非推奨となります。
    //!         代わりに@ref Stop() を使用してください。
    //!
    //! @see    Stop()
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION(void Close());

    //---------------------------------------------------------------------------
    //! @brief  接続が確立されているかチェックします。
    //!
    //! @return 接続が確立されていれば true を返します。
    //---------------------------------------------------------------------------
    bool IsConnected() const { return m_SoundEditSession.IsConnected() && m_SoundControlSession.IsConnected(); }

    //---------------------------------------------------------------------------
    //! @brief  サウンドインゲーム編集機能の更新処理を行います。
    //!
    //!         SoundMaker への接続要求を投げたり、
    //!         SoundMaker に編集のあったアイテムを問い合わせたりします。
    //!
    //!         セッションが開かれていない場合は、この関数を呼び出しても何も処理しません。
    //!
    //! @see    Start()
    //! @see    Stop()
    //---------------------------------------------------------------------------
    void Update();

    //@}
    // 開始と終了

    //! @briefprivate
    //! @return :private
    nw::snd::edit::SoundArchiveEditor& GetSoundArchiveEditor() { return m_SoundArchiveEditor; }

    //! @briefprivate
    //! @return :private
    nw::snd::edit::SoundEditSession& GetSoundEditSession() { return m_SoundEditSession; }

    //! @briefprivate
    //! @return :private
    nw::snd::ctrl::SoundObjectController& GetSoundObjectController() { return m_SoundObjectController; }

    //! @briefprivate
    //! @return :private
    nw::snd::ctrl::SoundControlSession& GetSoundControlSession() { return m_SoundControlSession; }
    //===========================================================================


private: // メンバ変数
    bool m_IsInitialized;
    bool m_IsStarted;

    nw::snd::internal::fnd::FrameHeap m_Allocator;

    nw::snd::edit::SoundArchiveEditor m_SoundArchiveEditor;
    nw::snd::edit::SoundEditSession m_SoundEditSession;
    nw::snd::ctrl::SoundObjectController m_SoundObjectController;
    nw::snd::ctrl::SoundControlSession m_SoundControlSession;

    nw::snd::edit::UpdateCachesHelper m_UpdateCachesHelper;
    u8* m_pMemoryForUpdateCachesThreadStack;
    u32 m_UpdateCachesThreadStackSize;

    void* m_pMemoryForSoundArchiveEditor;
    void* m_pMemoryForSoundEditSession;
    void* m_pMemoryForSoundControlSession;
};

} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#if defined(NW_PLATFORM_CTR)
#if NN_CURRENT_VERSION_NUMBER >= NN_VERSION_NUMBER(4,0,0,0)
#pragma diag_default 1301 // padding inserted in struct.
#pragma diag_default 2530 // padding added to end of struct.
#endif
#endif

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_EDIT_SOUND_EDIT_H_
