﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_EDIT_CANCELLER_H_
#define NW_SND_EDIT_CANCELLER_H_

#include <nw/snd/snd_Config.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/fnd/basis/sndfnd_Time.h>

#if defined(NW_PLATFORM_CTR)
#if NN_CURRENT_VERSION_NUMBER >= NN_VERSION_NUMBER(4,0,0,0)
#pragma diag_suppress 1301 // padding inserted in struct.
#pragma diag_suppress 2530 // padding added to end of struct.
#endif
#endif

namespace nw {
namespace snd {
namespace edit {

//---------------------------------------------------------------------------
//! @brief  【β版】キャンセルをサポートするインターフェイスです。
//---------------------------------------------------------------------------
class ICanceller
{
public:
    //! @briefprivate
    virtual ~ICanceller() { }

public:
    //---------------------------------------------------------------------------
    //! @brief      キャンセルの有無を取得します。
    //!
    //!             対象の処理が完了するか、この関数が true を返すまで、処理が続行されます。
    //!
    //! @param[in]  param   ユーザーパラメータを指定します。
    //!
    //! @return     true を返すと、対象をキャンセルします。
    //!             @n
    //!             false を返すと、処理を続行します。
    //---------------------------------------------------------------------------
    virtual bool IsCanceled(void* param) = 0;

    //---------------------------------------------------------------------------
    //! @brief      キャンセルオブジェクトの状態をリセットします。
    //!
    //!             主に処理開始前に呼び出され、
    //!             タイムアウト計算用の基準値を設定するため等の目的に利用します。
    //!
    //! @param[in]  param   ユーザーパラメータを指定します。
    //---------------------------------------------------------------------------
    virtual void Reset(void* param) = 0;
};

//---------------------------------------------------------------------------
//! @brief  【β版】キャンセルオブジェクトです。
//---------------------------------------------------------------------------
class Canceller : public ICanceller
{
public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //---------------------------------------------------------------------------
    Canceller() : m_IsCanceled(false) { }

public:
    //---------------------------------------------------------------------------
    //! @brief      キャンセルの有無を取得します。
    //!
    //!             対象の処理が完了するか、この関数が true を返すまで、処理が続行されます。
    //!
    //! @param[in]  param   ユーザーパラメータを指定します。
    //!
    //! @return     Cancel() を呼び出す前は false、呼び出した後は true を返します。
    //---------------------------------------------------------------------------
    virtual bool IsCanceled(void* param) { (void)param; return m_IsCanceled; }

    //---------------------------------------------------------------------------
    //! @brief  キャンセルオブジェクトのリセット実装ですが、ここでは何もしません。
    //!
    //! @param[in]  param   ユーザーパラメータを指定します。
    //---------------------------------------------------------------------------
    virtual void Reset(void* param) { (void)param; m_IsCanceled = false; }

    //---------------------------------------------------------------------------
    //! @brief  キャンセルオブジェクトをキャンセル状態にします。
    //!
    //!         次回の IsCanceled() から true を返すようになります。
    //---------------------------------------------------------------------------
    void Cancel() { m_IsCanceled = true; }

private:
    bool m_IsCanceled;
    u8   m_Padding[3];
};

//---------------------------------------------------------------------------
//! @brief  【β版】タイムアウトでキャンセルするキャンセルオブジェクトです。
//---------------------------------------------------------------------------
class TimeoutCanceller : public ICanceller
{
public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //!         このコンストラクタを使用する場合、別途 Start() を呼び出して、
    //!         タイマを開始する必要があります。
    //---------------------------------------------------------------------------
    TimeoutCanceller() : m_Timeout(0)
    {
    }

    //---------------------------------------------------------------------------
    //! @brief      コンストラクタです。
    //!
    //!             このコンストラクタを使用すると、指定タイムアウト値でタイマを開始します。
    //!             @n
    //!             別途 Start() を呼び出す必要はありません。
    //!
    //! @param[in]  timeoutMsec  タイムアウト値（ミリ秒）を指定します。
    //---------------------------------------------------------------------------
    explicit TimeoutCanceller(u32 timeoutMsec) : m_Timeout(timeoutMsec)
    {
        Start(timeoutMsec);
    }

public:
    //---------------------------------------------------------------------------
    //! @brief      タイマを開始します。
    //!
    //! @param[in]  timeoutMsec  タイムアウト値（ミリ秒）を指定します。
    //---------------------------------------------------------------------------
    void Start(u32 timeoutMsec)
    {
        if(m_StopWatch.IsStarted())
        {
            return;
        }

        m_Timeout = timeoutMsec;
        m_StopWatch.Start();
    }

    //---------------------------------------------------------------------------
    //! @brief      タイマを停止します。
    //---------------------------------------------------------------------------
    void Stop()
    {
        m_StopWatch.Start();
    }

    //---------------------------------------------------------------------------
    //! @brief  タイマをリセットします。
    //---------------------------------------------------------------------------
    void Reset()
    {
        m_StopWatch.Reset();
    }

    //---------------------------------------------------------------------------
    //! @brief      キャンセルの有無を取得します。
    //!
    //!             対象の処理が完了するか、この関数が true を返すまで、処理が続行されます。
    //!
    //! @param[in]  param   ユーザーパラメータを指定します。
    //!
    //! @return     true を返すと、対象をキャンセルします。
    //!             @n
    //!             false を返すと、処理を続行します。
    //---------------------------------------------------------------------------
    virtual bool IsCanceled(void* param)
    {
        (void)param;
        return m_StopWatch.GetElapsedTime().ToMilliSeconds() > m_Timeout;
    }

    //---------------------------------------------------------------------------
    //! @brief      キャンセルオブジェクトの状態をリセットします。
    //!
    //!             主に処理開始前に呼び出され、
    //!             タイムアウト計算用の基準値を設定するため等の目的に利用します。
    //!
    //! @param[in]  param   ユーザーパラメータを指定します。
    //---------------------------------------------------------------------------
    virtual void Reset(void* param)
    {
        (void)param;
        Reset();
    }

private:
    snd::internal::fnd::StopWatch m_StopWatch;
    u32                           m_Timeout;
};

} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#if defined(NW_PLATFORM_CTR)
#if NN_CURRENT_VERSION_NUMBER >= NN_VERSION_NUMBER(4,0,0,0)
#pragma diag_default 1301 // padding inserted in struct.
#pragma diag_default 2530 // padding added to end of struct.
#endif
#endif

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_EDIT_CANCELLER_H_
