﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_EDIT_RES_TYPES_H_
#define NW_SND_EDIT_RES_TYPES_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/fnd/basis/sndfnd_Inlines.h>
#include <nw/snd/fnd/binary/sndfnd_PrimitiveTypes.h>

namespace nw {
namespace snd {
namespace edit {
namespace internal {

//! @brief リソース内での名前を表す構造体です。
struct ResNameData
{
    //! @brief 文字のアライメントサイズです。
    typedef snd::internal::fnd::BinU32 LengthType;
    LengthType len;
    char str[sizeof(LengthType)];   // 実際には長さ len の文字列が入っている。
    // 末尾は sizeof(LengthType) バイトアラインメントされていて、
    // パディングは0である。
};

//! @brief リソース内での名前を表すクラスです。
class ResName : private ResNameData
{
public:
    using ResNameData::LengthType;

public:
    //! @brief 指定した長さの文字列を格納するのに必要なサイズを返します。
    //! @param[in] strLength 文字列の長さです。
    //! @return サイズ(バイト)を返します。
    static u32 GetRequiredSize(u32 strLength)
    {
        // NULL 文字分を含めるため strLength + 1 します。
        return sizeof(LengthType) + ut::RoundUp(strLength + 1, sizeof(LengthType));
    }

    //! @brief 文字列の長さを取得します。
    //! @return 文字列の長さを返します。
    size_t GetLength() const { return static_cast<size_t>(this->len); }

    //! @brief 文字列へのポインタを取得します。
    //! @return 文字列へのポインタを返します。
    const char* GetName() const { return &this->str[0]; }

    //! @brief 長さ付き文字列と比較します。
    //! @param[in] str 文字列です。
    //! @param[in] len 文字列長です。
    //! @return 一致したら true を返します。
    bool Equals(const char* str, size_t len) const
    {
        return (GetLength() == len) && (std::strcmp(GetName(), str) == 0);
    }

    //! @brief ResName と比較します。
    //! @param[in] rhs 比較対象です。
    //! @return 一致したら true を返します。
    bool Equals(const ResName* rhs) const
    {
        return (this == rhs) || Equals(rhs->GetName(), rhs->GetLength());
    }
};

//! @brief リソース内でのデータオフセットを表すクラスです。
class Offset
{
private:
    snd::internal::fnd::BinS32 offset;

public:
    // 共用体に入れるために、コンストラクタを実装しない。

    //! @brief 直接 offset を返す演算子オーバーロードです。
    operator s32() const { return offset; }

    //! @brief 直接 offset に代入する演算子オーバーロードです。
    //! @param[in] ofs 代入するオフセット値です。
    //! @return 自身への参照を返します。
    Offset& operator=(s32 ofs) { offset = ofs; return *this; }

    //! @brief ポインタからオフセットに変換して設定します。
    //! @param[in] ptr ポインタです。
    void  set_ptr(const void* ptr)
    {
        offset = (ptr == NULL) ? 0 :
            static_cast<s32>(reinterpret_cast<intptr_t>(ptr) - reinterpret_cast<intptr_t>(this));
    }

    //! @brief オフセットからポインタに変換して取得します。
    //! @return ポインタを返します。
    void* to_ptr() { return to_ptr<void>(); }

    //! @brief オフセットからポインタに変換して取得します。
    //! @return ポインタを返します。
    const void* to_ptr() const { return to_ptr<void>(); }

    //! @brief オフセットからポインタに変換して取得します。
    //! @return ポインタを返します。
    //! @tparam T ポインタが指すデータ型です。
    template<typename T>
    T* to_ptr() { return (offset == 0) ? NULL : snd::internal::fnd::AddOffsetToPtr<T*>(this, offset); }

    //! @brief オフセットからポインタに変換して取得します。
    //! @return ポインタを返します。
    //! @tparam T ポインタが指すデータ型です。
    template<typename T>
    const T* to_ptr() const { return (offset == 0) ? NULL : snd::internal::fnd::AddOffsetToPtr<const T*>(this, offset); }

    //! @brief オフセットテーブルからオフセットを取得します。
    //! @return オフセットを返します。
    Offset* to_table_ptr() { return static_cast<Offset*>(to_ptr()); }

    //! @brief オフセットテーブルからオフセットを取得します。
    //! @return オフセットを返します。
    const Offset* to_table_ptr() const { return static_cast<const Offset*>(to_ptr()); }
};

//! @brief 文字列への参照を持つクラスです。
class BinString
{
public:
    //! @brief 文字列へのオフセットです。
    Offset offset;

    //! @brief オフセットから文字列へのポインタに変換します。
    //! @return 文字列へのポインタを返します。
    const char* to_ptr() const { return offset.to_ptr<char>(); }

    //! @brief 指定した文字列を参照するように設定します。
    //!
    //! 引数で与えた文字列が破棄された場合、無効なポインタを参照します。
    //!
    //! @param[in] ptr 参照する文字列です。
    void set_ptr(const char* ptr) { offset.set_ptr(ptr); }

    //! @brief リソースでの文字列形式を取得します。
    //! @return リソースでの文字列形式を返します。
    ResName* GetResName()
    {
        return offset == 0 ? NULL : static_cast<ResName*>(
            snd::internal::fnd::AddOffsetToPtr<ResName*>(this, offset - sizeof(ResName::LengthType)));
    }

    //! @brief リソースでの文字列形式を取得します。
    //! @return リソースでの文字列形式を返します。
    const ResName* GetResName() const
    {
        return offset == 0 ? NULL : static_cast<const ResName*>(
            snd::internal::fnd::AddOffsetToPtr<const ResName*>(this, offset - sizeof(ResName::LengthType)));
    }
};

#if defined (NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#pragma warning(push)
#pragma warning(disable:4200)
#endif
//! @brief オフセットテーブルデータを示すクラスです。
struct ResOffsetTableData
{
    snd::internal::fnd::BinU32 itemCount;
    Offset itemOffsets[0];
};
#if defined (NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#pragma warning(pop)
#endif

//! @brief オフセットテーブルを操作するクラスです。
template<typename TItem=void>
class ResOffsetTable
{
public:
    //! @brief アイテム数を取得します。
    //! @return アイテム数を返します。
    u32 GetItemCount() const { return m_Data.itemCount; }

    //! @brief 指定インデックスのアイテムを取得します。
    //! @param[in] index インデックスです。
    //! @return アイテムへのポインタを返します。
    const TItem* GetItem(u32 index) const
    {
        NW_ASSERT(index < m_Data.itemCount);
        return m_Data.itemOffsets[index].template to_ptr<TItem>();
    }

private:
    ResOffsetTableData m_Data;
};

//! @brief バージョンデータを示す共用体です。
union ResVersionData
{
    struct Elements
    {
        u8 major;
        u8 minor;
        u8 micro;
        u8 release;
    };

    ResVersionData(u32 value)
    {
        elements.major   = 0xff & (value >> 24);
        elements.minor   = 0xff & (value >> 16);
        elements.micro   = 0xff & (value >> 8);
        elements.release = 0xff & value;
    }

    u32      value;
    Elements elements;
};

//! @brief バージョンを操作するクラスです。
class ResVersion
{
public:
    //! @brief コンストラクタです。
    //! @param[in] value バージョン番号です。
    explicit ResVersion(u32 value) : m_Data(value) { }

public:
    //! @brief メジャーバージョンを取得します。
    //! @return メジャーバージョンを返します。
    u8 GetMajor() const { return m_Data.elements.major; }

    //! @brief マイナーバージョンを取得します。
    //! @return マイナーバージョンを返します。
    u8 GetMinor() const { return m_Data.elements.minor; }

    //! @brief マイクロバージョンを取得します。
    //! @return マイクロバージョンを返します。
    u8 GetMico() const { return m_Data.elements.micro; }

    //! @brief リリースバージョンを取得します。
    //! @return リリースバージョンを返します。
    u8 GetRelease() const { return m_Data.elements.release; }

private:
    ResVersionData m_Data;
};

//! @brief  32 バイトハッシュ値です。
struct ResHash32
{
    static const u32 Size = 32;
    u8 value[Size];
};

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_EDIT_RES_TYPES_H_
