﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_EDIT_RES_ITEM_INFO_H_
#define NW_SND_EDIT_RES_ITEM_INFO_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

#ifdef NW_PLATFORM_WIN32
#pragma warning(push)
#pragma warning(disable:4200)
#endif

#include <limits.h>
#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_SoundArchive.h>
#include <nw/snd/snd_WaveSoundFileReader.h>
#include <nw/snd/fnd/basis/sndfnd_Memory.h>
#include <nw/snd/edit/res/sndedit_ResTypes.h>

namespace nw {
namespace snd {
namespace edit {
namespace internal {

static const u32 USER_PARAMETER_COUNT     = SoundArchive::USER_PARAM_INDEX_MAX + 1;
static const u32 STRM_TRACK_COUNT         = SoundArchive::STRM_TRACK_NUM;
static const u32 STRM_TRACK_CHANNEL_COUNT = snd::WAVE_CHANNEL_MAX;
static const u32 SEQ_BANK_COUNT           = SoundArchive::SEQ_BANK_MAX;
static const u32 FX_SEND_COUNT            = AUX_BUS_NUM;

//---------------------------------------------------------------------------
//! @brief  リソースデータの種別です。
//---------------------------------------------------------------------------
enum ResDataType
{
    RES_DATA_TYPE_UNKNOWN = 0,

    RES_DATA_TYPE_STREAM_SOUND = 1,
    RES_DATA_TYPE_WAVE_SOUND,
    RES_DATA_TYPE_SEQUENCE_SOUND,
    RES_DATA_TYPE_BANK,
    RED_DATA_TYPE_SOUND_MIN = RES_DATA_TYPE_STREAM_SOUND,
    RED_DATA_TYPE_SOUND_MAX = RES_DATA_TYPE_BANK,

    RES_DATA_TYPE_STREAM_SOUND_FILE = 100,
    RES_DATA_TYPE_WAVE_SOUND_FILE,
    RES_DATA_TYPE_SEQUENCE_FILE,
    RES_DATA_TYPE_BANK_FILE,
    RES_DATA_TYPE_WAVE_ARCHIVE_FILE,
    RES_DATA_TYPE_STREAM_PREFTCH_SOUND_FILE,
    RED_DATA_TYPE_FILE_MIN = RES_DATA_TYPE_STREAM_SOUND_FILE,
    RED_DATA_TYPE_FILE_MAX = RES_DATA_TYPE_STREAM_PREFTCH_SOUND_FILE
};

//---------------------------------------------------------------------------
//! @brief  サウンドの基本パラメータ情報です。
//---------------------------------------------------------------------------
struct SoundBasicParam
{
    snd::internal::fnd::BinU8  volume;                              //!< 音量
    snd::internal::fnd::BinU8  panMode;                             //!< パンモード
    snd::internal::fnd::BinU8  panCurve;                            //!< パンカーブ
    snd::internal::fnd::BinU8  playerPriority;                      //!< プレイヤー優先度
    Offset                     playerNameOffset;                    //!< プレイヤー名
    snd::internal::fnd::BinU8  actorPlayerID;                       //!< アクタープレイヤーID
    snd::internal::fnd::BinU8  remoteFilter;                        //!< リモコンフィルタ
    snd::internal::fnd::BinU8  flags;                               //!< フラグ
    snd::internal::fnd::BinU8  padding[1];                          //!< パディング
    snd::internal::fnd::BinU32 userParams[USER_PARAMETER_COUNT];    //!< ユーザーデータ

    //! @brief  構造体の内容をすべて上書きします。
    //! @param source TBD
    void Set(const SoundBasicParam& source)
    {
        volume = source.volume;
        panMode = source.panMode;
        panCurve = source.panCurve;
        playerPriority = source.playerPriority;
        playerNameOffset = source.playerNameOffset;
        actorPlayerID = source.actorPlayerID;
#if defined(NW_PLATFORM_CTR)
        remoteFilter = 0;
#else
        remoteFilter = source.remoteFilter;
#endif
        flags = source.flags;

        for(u32 i = 0; i < USER_PARAMETER_COUNT; ++i)
        {
            userParams[i] = source.userParams[i];
        }
    }

    //! @brief  パラメータを SoundInfo に適用します。
    //! @param target TBD
    //! @param soundArchive TBD
    void ApplyTo(SoundArchive::SoundInfo& target, const SoundArchive& soundArchive) const
    {
        target.volume = volume;
        target.panMode = static_cast<PanMode>(panMode);
        target.panCurve = static_cast<PanCurve>(panCurve);
        target.playerPriority = playerPriority;

        const ResName* playerName = playerNameOffset.to_ptr<ResName>();

        if(playerName != NULL && playerName->GetLength() > 0)
        {
            SoundArchive::ItemId playerID = soundArchive.GetItemId(playerName->GetName());

            target.playerId = playerID;
            if(playerID == SoundArchive::INVALID_ID)
            {
                NW_WARNING(false, "[sndedit][warning] player '%s' not found.\n", playerName->GetName());
            }
        }

        target.actorPlayerId = actorPlayerID;
#if !defined(NW_PLATFORM_CTR)
        target.remoteFilter = remoteFilter;
#endif
        target.isFrontBypass = ((flags << 0) & 1) == 1;
    }
};

//---------------------------------------------------------------------------
//! @brief  3Dサウンドのパラメータ情報です。
//---------------------------------------------------------------------------
struct Sound3DParam
{
    snd::internal::fnd::BinU32 flags;           //!< [0]Vol [1]Priotity [2]Pan [3]SPan [4]Filter
    snd::internal::fnd::BinF32 decayRatio;      //!< 減衰率
    snd::internal::fnd::BinU8  decayCurve;      //!< 減衰カーブタイプ (DecayCurve が入る)
    snd::internal::fnd::BinU8  dopplerFactor;   //!< ドップラーファクター
    snd::internal::fnd::BinU8  padding[2];      //!< パディング

    //! @brief  構造体の内容をすべて上書きします。
    //! @param source TBD
    void Set(const Sound3DParam& source)
    {
        flags = source.flags;
        decayRatio = source.decayRatio;
        decayCurve = source.decayCurve;
        dopplerFactor = source.dopplerFactor;
    }

    //! @brief  パラメータを Sound3DInfo に適用します。
    //! @param target TBD
    void ApplyTo(SoundArchive::Sound3DInfo& target) const
    {
        target.flags = flags;
        target.decayRatio = decayRatio;
        target.decayCurve = decayCurve;
        target.dopplerFactor = dopplerFactor;
    }
};

//---------------------------------------------------------------------------
//! @brief  センドパラメータ情報です。
//---------------------------------------------------------------------------
struct SendsParam
{
    snd::internal::fnd::BinU8 mainSend;
    snd::internal::fnd::BinU8 fxSend[FX_SEND_COUNT];
#if defined(NW_PLATFORM_CTR)
    snd::internal::fnd::BinU8 padding;
#endif
};

//---------------------------------------------------------------------------
//! @brief  ストリームサウンドトラックのパラメータ情報です。
//---------------------------------------------------------------------------
struct StreamSoundTrackParam
{
    snd::internal::fnd::BinU8 volume;                                       //!< 音量
    snd::internal::fnd::BinU8 pan;                                          //!< パン
    snd::internal::fnd::BinU8 span;                                         //!< サラウンドパン
    snd::internal::fnd::BinU8 channelCount;                                 //!< チャンネル数
    snd::internal::fnd::BinU8 globalChannelIndex[STRM_TRACK_CHANNEL_COUNT]; //!< グローバルチャンネルインデックス
    snd::internal::fnd::BinU8 padding[2];                                   //!< パディング
    snd::internal::fnd::BinU8 lpfFreq;
    snd::internal::fnd::BinU8 biquadType;
    snd::internal::fnd::BinU8 biquadValue;
    snd::internal::fnd::BinU8 flags;
    SendsParam                sends;

    //! @brief  構造体の内容をすべて上書きします。
    //! @param source TBD
    void Set(const StreamSoundTrackParam& source)
    {
        volume = source.volume;
        pan = source.pan;
        span = source.span;
        channelCount = source.channelCount;

        for(u32 i = 0; i < STRM_TRACK_CHANNEL_COUNT; ++i)
        {
            globalChannelIndex[i] = source.globalChannelIndex[i];
        }

        lpfFreq = source.lpfFreq;
        biquadType = source.biquadType;
        biquadValue = source.biquadValue;
        flags = source.flags;

        sends = source.sends;
    }

    //! @brief  パラメータを SoundArchive::StreamTrackInfo に適用します。
    //! @param target TBD
    void ApplyTo(SoundArchive::StreamTrackInfo& target) const
    {
        target.volume = volume;
        target.pan = pan;
        target.span = span;
        target.channelCount = channelCount;

        for(u32 i = 0; i < STRM_TRACK_CHANNEL_COUNT; ++i)
        {
            target.globalChannelIndex[i] = globalChannelIndex[i];
        }

        target.lpfFreq = lpfFreq;
        target.biquadType = biquadType;
        target.biquadValue = biquadValue;
        target.flags = flags;

        target.mainSend = sends.mainSend;
        for(u32 i = 0; i < FX_SEND_COUNT; ++i)
        {
            target.fxSend[i] = sends.fxSend[i];
        }
    }
};

//---------------------------------------------------------------------------
//! @brief  ストリームサウンドのパラメータ情報です。
//---------------------------------------------------------------------------
struct StreamSoundParam
{
    StreamSoundTrackParam trackParams[STRM_TRACK_COUNT];    //!< トラックのパラメータ情報
    snd::internal::fnd::BinU16 trackAllocationFlags;        //!< トラックアロケーションフラグ
    snd::internal::fnd::BinU16 totalChannelCount;           //!< 総チャンネル数
    snd::internal::fnd::BinS32 fileSize;                    //!< bxstm ファイルサイズ
    Offset filePathOffset;                                  //!< bxstm ファイルパス
    snd::internal::fnd::BinF32 pitch;
    SendsParam                 sends;
    snd::internal::fnd::BinU8  streamFileType;
    u8                         padding[3];
    snd::internal::fnd::BinS32 streamPrefetchFileSize;       //!< bxstp ファイルサイズ
    Offset                     streamPrefetchFilePathOffset; //!< bxstp ファイルパス
    ResHash32                  streamPrefetchHashCode;       //!< bxstp ハッシュ値

    //! @brief  構造体の内容をすべて上書きします。
    //! @param source TBD
    void Set(const StreamSoundParam& source)
    {
        for(u32 i = 0; i < STRM_TRACK_COUNT; ++i)
        {
            trackParams[i].Set(source.trackParams[i]);
        }

        trackAllocationFlags = source.trackAllocationFlags;
        totalChannelCount = source.totalChannelCount;
        fileSize = source.fileSize;
        filePathOffset = source.filePathOffset;
        pitch = source.pitch;
        sends = source.sends;
        streamFileType = source.streamFileType;
        streamPrefetchFileSize = source.streamPrefetchFileSize;
        streamPrefetchFilePathOffset = source.streamPrefetchFilePathOffset;
    }

    //! @brief  パラメータを SoundArchive::StreamSoundInfo に適用します。
    //! @param target TBD
    void ApplyTo(SoundArchive::StreamSoundInfo& target) const
    {
        for(u32 i = 0; i < STRM_TRACK_COUNT; ++i)
        {
            trackParams[i].ApplyTo(target.trackInfo[i]);
        }

        target.allocateTrackFlags = trackAllocationFlags;
        target.allocateChannelCount = totalChannelCount;
        target.pitch = pitch;

        target.mainSend = sends.mainSend;
        for(u32 i = 0; i < FX_SEND_COUNT; ++i)
        {
            target.fxSend[i] = sends.fxSend[i];
        }

        target.streamFileType = (SoundArchive::StreamFileType)streamFileType;

        // ファイル関連のパラメータは、ここでは利用しません。
        // SoundArchivePlayer のファイルアクセスをフックする際に利用します。
    }
};

//---------------------------------------------------------------------------
//! @brief  ストリームサウンドのパラメータ情報２です。
//---------------------------------------------------------------------------
struct StreamSoundParam2
{
    snd::internal::fnd::BinBool isLoop;
    snd::internal::fnd::BinU8   padding[3];
    snd::internal::fnd::BinU32  loopStartFrame;
    snd::internal::fnd::BinU32  loopEndFrame;

    //! @brief  構造体の内容をすべて上書きします。
    //! @param source TBD
    void Set(const StreamSoundParam2& source)
    {
        isLoop = source.isLoop;
        loopStartFrame = source.loopStartFrame;
        loopEndFrame = source.loopEndFrame;
    }

    //! @brief  パラメータを SoundArchive::StreamSoundInfo2 に適用します。
    //! @param target TBD
    void ApplyTo(SoundArchive::StreamSoundInfo2& target) const
    {
        target.isLoop = isLoop;
        target.loopStartFrame = loopStartFrame;
        target.loopEndFrame = loopEndFrame;
    }
};

//---------------------------------------------------------------------------
//! @brief  ウェーブサウンドのパラメータ情報です。
//---------------------------------------------------------------------------
struct WaveSoundParam
{
    snd::internal::fnd::BinU8   channelPriority;            //!< チャンネル優先度
    snd::internal::fnd::BinBool isReleasePriorityFix;       //!< リリース時優先度固定
    snd::internal::fnd::BinU8   padding[2];                 //!< パディング
    snd::internal::fnd::BinU32  index;                      //!< 波形インデックス
    snd::internal::fnd::BinS32  fileSize;                   //!< bxwsd ファイルサイズ
    Offset                      filePathOffset;             //!< bxwsd ファイルパス
    ResHash32                   hashCode;                   //!< bxwsd ハッシュ値
    snd::internal::fnd::BinS32  waveArchiveFileSize;        //!< bxwar ファイルサイズ
    Offset                      waveArchiveFilePathOffset;  //!< bxwar ファイルパス
    ResHash32                   waveArchiveHashCode;        //!< bxwar ハッシュ値

    //! @brief  構造体の内容をすべて上書きします。
    //! @param source TBD
    void Set(const WaveSoundParam& source)
    {
        channelPriority = source.channelPriority;
        isReleasePriorityFix = source.isReleasePriorityFix;
        index = source.index;
        fileSize = source.fileSize;
        filePathOffset = source.filePathOffset;
        waveArchiveFileSize = source.waveArchiveFileSize;
        waveArchiveFilePathOffset = source.waveArchiveFilePathOffset;
    }

    //! @brief  パラメータを SoundArchive::WaveSoundInfo に適用します。
    //! @param target TBD
    void ApplyTo(SoundArchive::WaveSoundInfo& target) const
    {
        target.channelPriority = channelPriority;
        target.isReleasePriorityFix = isReleasePriorityFix;
        target.index = index;

        // ファイル関連のパラメータは、ここでは利用しません。
        // SoundArchivePlayer のファイルアクセスをフックする際に利用します。
    }
};

//---------------------------------------------------------------------------
//! @brief  シーケンスサウンドのパラメータ情報です。
//---------------------------------------------------------------------------
struct SequenceSoundParam
{
    snd::internal::fnd::BinU32  startOffset;                        //!< 再生開始位置
    snd::internal::fnd::BinU8   channelPriority;                    //!< チャンネル優先度
    snd::internal::fnd::BinBool isReleasePriorityFix;               //!< リリース時優先度固定
    snd::internal::fnd::BinU8   padding[2];                         //!< パディング
    snd::internal::fnd::BinU32  trackAllocationFlags;               //!< トラックアロケーションフラグ
    snd::internal::fnd::BinS32  fileSize;                           //!< bxseq ファイルサイズ
    Offset                      filePathOffset;                     //!< bxseq ファイルパス
    ResHash32                   hashCode;                           //!< bxseq ハッシュ値
    Offset                      bankNameOffsets[SEQ_BANK_COUNT];    //!< バンク名テーブル

    //! @brief  構造体の内容をすべて上書きします。
    //! @param source TBD
    void Set(const SequenceSoundParam& source)
    {
        startOffset = source.startOffset;
        channelPriority = source.channelPriority;
        isReleasePriorityFix = source.isReleasePriorityFix;
        trackAllocationFlags = source.trackAllocationFlags;
        fileSize = source.fileSize;
        filePathOffset = source.filePathOffset;

        for(u32 i = 0; i < SEQ_BANK_COUNT; ++i)
        {
            bankNameOffsets[i] = source.bankNameOffsets[i];
        }
    }

    //! @brief  パラメータを SequenceSoundInfo に適用します。
    //! @param target TBD
    //! @param soundArchive TBD
    void ApplyTo(SoundArchive::SequenceSoundInfo& target, const SoundArchive& soundArchive) const
    {
        if(startOffset != UINT_MAX)
        {
            target.startOffset = startOffset;
        }

        target.channelPriority = channelPriority;
        target.isReleasePriorityFix = isReleasePriorityFix;
        target.allocateTrackFlags = trackAllocationFlags;

        for(u32 i = 0; i < SEQ_BANK_COUNT; ++i)
        {
            const ResName* name = bankNameOffsets[i].to_ptr<const ResName>();

            if(name == NULL || name->GetLength() == 0)
            {
                continue;
            }

            SoundArchive::ItemId itemID = soundArchive.GetItemId(name->GetName());

            if(itemID == SoundArchive::INVALID_ID)
            {
                continue;
            }

            target.bankIds[i] = itemID;
        }

        // ファイル関連のパラメータは、ここでは利用しません。
        // SoundArchivePlayer のファイルアクセスをフックする際に利用します。
    }
};

//---------------------------------------------------------------------------
//! @brief  バンクのパラメータ情報です。
//---------------------------------------------------------------------------
struct BankParam
{
    snd::internal::fnd::BinS32 fileSize;                //!< bfbnk ファイルサイズ
    Offset     filePathOffset;                          //!< bfbnk ファイルパス
    ResHash32  hashCode;                                //!< bfbnk ハッシュ値
    snd::internal::fnd::BinS32 waveArchiveFileSize;     //!< bfwar ファイルサイズ
    Offset     waveArchiveFilePathOffset;               //!< bfwar ファイルパス
    ResHash32  waveArchiveHashCode;                     //!< bfwar ハッシュ値

    //! @brief  構造体の内容をすべて上書きします。
    //! @param source TBD
    void Set(const BankParam& source)
    {
        fileSize = source.fileSize;
        filePathOffset = source.filePathOffset;
        waveArchiveFileSize = source.waveArchiveFileSize;
        waveArchiveFilePathOffset = source.waveArchiveFilePathOffset;
    }

    //! @brief  パラメータを BankInfo に適用します。
    //! @param target TBD
    //! @param soundArchive SoundArchive TBD
    void ApplyTo(SoundArchive::BankInfo& target, const SoundArchive& soundArchive) const
    {
        (void)target;
        (void)soundArchive;
        // ファイル関連のパラメータは、ここでは利用しません。
    }
};

//---------------------------------------------------------------------------
//! @brief  サウンド情報です。
//---------------------------------------------------------------------------
struct ResSoundInfo
{
    SoundBasicParam soundBasicParam;
    Sound3DParam sound3DParam;

    //! @brief  構造体の内容をすべて上書きします。
    //! @param source TBD
    void Set(const ResSoundInfo& source)
    {
        soundBasicParam.Set(source.soundBasicParam);
        sound3DParam.Set(source.sound3DParam);
    }
};

//---------------------------------------------------------------------------
//! @brief  ストリームサウンド情報です。
//!         HACK : ※この構造体の直後に ResName の実体が続いていることが前提です。
//---------------------------------------------------------------------------
struct ResStreamSoundInfo : public ResSoundInfo
{
    StreamSoundParam streamSoundParam;
    StreamSoundParam2 streamSoundParam2;

    static u32 GetRequiredSize(u32 maxItemName, u32 maxPath)
    {
        // 固定パケットサイズ + プレイヤー名の長さ + bxstm ファイルパスの長さ
        return sizeof(ResStreamSoundInfo) +
            ResName::GetRequiredSize(maxItemName) +
            ResName::GetRequiredSize(maxPath);
    }

    //! @brief  構造体の内容をすべて上書きします。
    //! @param source TBD
    void Set(const ResStreamSoundInfo& source)
    {
        ResSoundInfo::Set(source);
        streamSoundParam.Set(source.streamSoundParam);
        streamSoundParam2.Set(source.streamSoundParam2);
    }
};

//---------------------------------------------------------------------------
//! @brief  ウェーブサウンド情報です。
//!         HACK : ※この構造体の直後に ResName の実体が続いていることが前提です。
//---------------------------------------------------------------------------
struct ResWaveSoundInfo : public ResSoundInfo
{
    WaveSoundParam waveSoundParam;

    static u32 GetRequiredSize(u32 maxItemName, u32 maxPath)
    {
        // 固定パケットサイズ +
        // プレイヤー名の長さ +
        // bxwsd, bxwar ファイルパスの長さ
        return sizeof(ResWaveSoundInfo) +
            ResName::GetRequiredSize(maxItemName) +
            ResName::GetRequiredSize(maxPath) * 2;
    }

    //! @brief  構造体の内容をすべて上書きします。
    //! @param source TBD
    void Set(const ResWaveSoundInfo& source)
    {
        ResSoundInfo::Set(source);
        waveSoundParam.Set(source.waveSoundParam);
    }
};

//---------------------------------------------------------------------------
//! @brief  シーケンスサウンド情報です。
//!         HACK : ※この構造体の直後に ResName の実体が続いていることが前提です。
//---------------------------------------------------------------------------
struct ResSequenceSoundInfo : public ResSoundInfo
{
    SequenceSoundParam sequenceSoundParam;

    static u32 GetRequiredSize(u32 maxItemName, u32 maxPath)
    {
        // 固定パケットサイズ +
        // プレイヤー名、バンク名の長さ +
        // bxseq ファイルパスの長さ
        return sizeof(ResSequenceSoundInfo) +
            ResName::GetRequiredSize(maxItemName) * (1 + SEQ_BANK_COUNT) +
            ResName::GetRequiredSize(maxPath);
    }

    //! @brief  構造体の内容をすべて上書きします。
    //! @param source TBD
    void Set(const ResSequenceSoundInfo& source)
    {
        ResSoundInfo::Set(source);
        sequenceSoundParam.Set(source.sequenceSoundParam);
    }
};

//---------------------------------------------------------------------------
//! @brief  バンク情報です。
//---------------------------------------------------------------------------
struct ResBankInfo
{
    BankParam bankParam;

    static u32 GetRequiredSize(u32 maxPath)
    {
        // 固定パケットサイズ +
        // bxbnk, bxwar ファイルパスの長さ
        return sizeof(ResBankInfo) +
            ResName::GetRequiredSize(maxPath) * 2;
    }

    //! @brief  構造体の内容をすべて上書きします。
    //! @param source TBD
    void Set(const ResBankInfo& source)
    {
        bankParam.Set(source.bankParam);
    }
};

//---------------------------------------------------------------------------
//! @brief  ResSoundInfo のユーティリティクラスです。
//---------------------------------------------------------------------------
class ResItemInfoUtility
{
public:
    //----------------------------------------------------------
    //! @brief  アイテム情報の上限値を取得します。
    //! @param maxItemName TBD
    //! @param maxPath TBD
    //! @return TBD
    static u32 GetMaxItemInfoSize(u32 maxItemName, u32 maxPath)
    {
        u32 result = 0;

        // TODO : アイテム情報の種類が増えたら、ここに追加します。
        // TODO : アイテム情報のサイズが変わったら、ここを編集します。
        result = ut::Max(result, internal::ResStreamSoundInfo::GetRequiredSize(maxItemName, maxPath));
        result = ut::Max(result, internal::ResWaveSoundInfo::GetRequiredSize(maxItemName, maxPath));
        result = ut::Max(result, internal::ResSequenceSoundInfo::GetRequiredSize(maxItemName, maxPath));
        result = ut::Max(result, internal::ResBankInfo::GetRequiredSize(maxPath));

        return ut::RoundUp(result, snd::internal::fnd::MemoryTraits::DEFAULT_ALIGNMENT);
    }

    //----------------------------------------------------------
    //! @brief  サウンド情報データタイプかどうかを調べます。
    //! @param dataType TBD
    //! @return TBD
    static bool IsSoundInfoDataType(u32 dataType)
    {
        return
            RED_DATA_TYPE_SOUND_MIN <= dataType &&
            dataType <= RED_DATA_TYPE_SOUND_MAX &&
            dataType != RES_DATA_TYPE_BANK;
    }

    //----------------------------------------------------------
    //! @brief  ストリームサウンド情報データタイプかどうかを調べます。
    //! @param dataType TBD
    //! @return TBD
    static bool IsStreamSoundInfoDataType(u32 dataType)
    {
        return dataType == RES_DATA_TYPE_STREAM_SOUND;
    }

    //----------------------------------------------------------
    //! @brief  ウェーブサウンド情報データタイプかどうかを調べます。
    //! @param dataType TBD
    //! @return TBD
    static bool IsWaveSoundInfoDataType(u32 dataType)
    {
        return dataType == RES_DATA_TYPE_WAVE_SOUND;
    }

    //----------------------------------------------------------
    //! @brief  シーケンスサウンド情報データタイプかどうかを調べます。
    //! @param dataType TBD
    //! @return TBD
    static bool IsSequenceSoundInfoDataType(u32 dataType)
    {
        return dataType == RES_DATA_TYPE_SEQUENCE_SOUND;
    }

    //----------------------------------------------------------
    //! @brief  バンク情報データタイプかどうかを調べます。
    //! @param dataType TBD
    //! @return TBD
    static bool IsBankInfoDataType(u32 dataType)
    {
        return dataType == RES_DATA_TYPE_BANK;
    }

    //----------------------------------------------------------
    //! @brief  サウンド情報から指定したファイルパスを取得します。
    //! @param soundInfo TBD
    //! @param soundType TBD
    //! @param fileDataType TBD
    //! @return TBD
    static const ResName* GetFilePath(const ResSoundInfo& soundInfo, ResDataType soundType, ResDataType fileDataType)
    {
        switch(soundType)
        {
        case RES_DATA_TYPE_STREAM_SOUND:
            {
                const ResStreamSoundInfo* streamSoundInfo =
                    reinterpret_cast<const ResStreamSoundInfo*>(&soundInfo);

                switch(fileDataType)
                {
                case RES_DATA_TYPE_STREAM_SOUND_FILE:
                    return streamSoundInfo->streamSoundParam.filePathOffset.to_ptr<const ResName>();

                case RES_DATA_TYPE_STREAM_PREFTCH_SOUND_FILE:
                    return streamSoundInfo->streamSoundParam.streamPrefetchFilePathOffset.to_ptr<const ResName>();

                default:
                    break;
                }

                return NULL;
            }

        case RES_DATA_TYPE_WAVE_SOUND:
            {
                const ResWaveSoundInfo* waveSoundInfo =
                    reinterpret_cast<const ResWaveSoundInfo*>(&soundInfo);

                switch(fileDataType)
                {
                case RES_DATA_TYPE_WAVE_SOUND_FILE:
                    return waveSoundInfo->waveSoundParam.filePathOffset.to_ptr<const ResName>();

                case RES_DATA_TYPE_WAVE_ARCHIVE_FILE:
                    return waveSoundInfo->waveSoundParam.waveArchiveFilePathOffset.to_ptr<const ResName>();

                default:
                    break;
                }

                return NULL;
            }

        case RES_DATA_TYPE_SEQUENCE_SOUND:
            {
                const ResSequenceSoundInfo* sequenceSoundInfo =
                    reinterpret_cast<const ResSequenceSoundInfo*>(&soundInfo);

                switch(fileDataType)
                {
                case RES_DATA_TYPE_SEQUENCE_FILE:
                    return sequenceSoundInfo->sequenceSoundParam.filePathOffset.to_ptr<const ResName>();

                default:
                    break;
                }

                return NULL;
            }

        default:
            break;
        }

        return NULL;
    }

    //----------------------------------------------------------
    //! @brief  バク情報から指定したファイルパスを取得します。
    //! @param bankInfo TBD
    //! @param fileDataType TBD
    //! @return TBD
    static const ResName* GetFilePath(const ResBankInfo& bankInfo, ResDataType fileDataType)
    {
        switch(fileDataType)
        {
        case RES_DATA_TYPE_BANK_FILE:
            return bankInfo.bankParam.filePathOffset.to_ptr<const ResName>();

        case RES_DATA_TYPE_WAVE_ARCHIVE_FILE:
            return bankInfo.bankParam.waveArchiveFilePathOffset.to_ptr<const ResName>();

        default:
            break;
        }

        return NULL;
    }
};

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#ifdef NW_PLATFORM_WIN32
#pragma warning(pop)
#endif

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_EDIT_RES_ITEM_INFO_H_
