﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_EDIT_ITEM_CACHE_PACKET_H_
#define NW_SND_EDIT_ITEM_CACHE_PACKET_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/edit/hio/sndedit_HioProtocol.h>
#include <nw/snd/edit/res/sndedit_ResTypes.h>

#if defined(NW_PLATFORM_CTR)
#if NN_CURRENT_VERSION_NUMBER >= NN_VERSION_NUMBER(4,0,0,0)
#pragma diag_suppress 1301 // padding inserted in struct.
#pragma diag_suppress 2530 // padding added to end of struct.
#endif
#endif

namespace nw {
namespace snd {
namespace edit {
namespace internal {

//! @brief  キャッシュ操作の種類を示します。
enum CACHE_ACTION
{
    CACHE_ACTION_UPDATE = 1,      //!< キャッシュを更新します。
    CACHE_ACTION_REMOVE = 2       //!< キャッシュを削除します。
};

//---------------------------------------------------------------------------
//! @brief  アイテムのキャッシュ開始を通知するパケットです。
//---------------------------------------------------------------------------
class BeginItemCachePacket : public HioPacket
{
public:
    class Body : public HioPacketBody
    {
    public:
        Body(const char* itemName, u32 itemNameLength) :
          m_Action(CACHE_ACTION_UPDATE),
          m_IsTargetAll(false),
          m_Padding(0)
        {
            m_ItemName.len = itemNameLength + 1;

            if(itemName != NULL && itemNameLength > 0)
            {
                SetItemName(itemName, itemNameLength);
            }
        }

    public:
        static u32 GetRequiredSize(u32 maxItemName)
        {
            // m_Action, m_IsTargetAll, m_Padding, m_ItemName
            return 4 + ResName::GetRequiredSize(maxItemName);
        }

        CACHE_ACTION GetAction() const { return static_cast<CACHE_ACTION>(m_Action); }
        void SetAction(CACHE_ACTION value) { m_Action = static_cast<u8>(value); }

        bool GetIsTargetAll() const { return m_IsTargetAll; }
        void SetIsTargetAll(bool value) { m_IsTargetAll = value; }

        u32         GetItemNameLength() const { return m_ItemName.len; }
        const char* GetItemName()       const { return m_ItemName.str; }

    private:
        void SetItemName(const char* value, u32 length)
        {
            if(m_ItemName.len == 0)
            {
                return;
            }

            if(value == NULL || *value == '\0')
            {
                m_ItemName.str[0] = '\0';
            }

            ut::strncpy(m_ItemName.str, m_ItemName.len + 1, value, length);
        }

    private:
        u8          m_Action;       //!< キャッシュ操作の種類です。
        bool        m_IsTargetAll;  //!< すべてのアイテムを対象とするフラグです。
        u16         m_Padding;
        ResNameData m_ItemName;     //!< 対象アイテム名です。
   };

public:
    BeginItemCachePacket(const char* itemName, u32 maxItemName) :
      HioPacket(HIO_MESSAGE_BEGIN_ITEM_CACHE, Body::GetRequiredSize(maxItemName)),
          m_Body(itemName, maxItemName)
    {
    }

public:
    static BeginItemCachePacket* CreateInstance(
        void* buffer,
        const char* itemName)
    {
        NW_ASSERT_NOT_NULL(buffer);

        u32 itemNameLength = itemName == NULL ? 0 : std::strlen(itemName);
        return new(buffer) BeginItemCachePacket(itemName, itemNameLength);
    }

    static u32 GetRequiredSize(u32 maxItemName)
    {
        // 固定パケットサイズ + 名前の長さ
        return sizeof(HioPacket) + Body::GetRequiredSize(maxItemName);
    }

    Body&       GetBody() { return m_Body; }
    const Body& GetBody() const { return m_Body; }

private:
    Body m_Body;
};

//---------------------------------------------------------------------------
//! @brief  BeginItemCachePacket の返信パケットです。
//---------------------------------------------------------------------------
class BeginItemCacheReplyPacket : public HioPacket
{
public:
    struct Body : public HioPacketBody
    {
    };

public:
    BeginItemCacheReplyPacket() : HioPacket(HIO_MESSAGE_BEGIN_ITEM_CACHE_REPLY, sizeof(Body))
    {
    }

public:
    static u32 GetRequiredSize()
    {
        return sizeof(BeginItemCacheReplyPacket);
    }

    Body&       GetBody() { return m_Body; }
    const Body& GetBody() const { return m_Body; }

private:
    Body m_Body;
};

//---------------------------------------------------------------------------
//! @brief  アイテムのキャッシュ開始を通知するパケットです。
//---------------------------------------------------------------------------
class EndItemCachePacket : public HioPacket
{
public:
    class Body : public HioPacketBody
    {
    public:
        Body(const char* itemName, u32 itemNameLength) :
          m_Action(CACHE_ACTION_UPDATE),
          m_IsTargetAll(false),
          m_Padding(0)
        {
            m_ItemName.len = itemNameLength + 1;

            if(itemName != NULL && itemNameLength > 0)
            {
                SetItemName(itemName, itemNameLength);
            }
        }

    public:
        static u32 GetRequiredSize(u32 maxItemName)
        {
            // m_Action, m_IsTargetAll, m_Padding, m_ItemName
            return 4 + ResName::GetRequiredSize(maxItemName);
        }

        CACHE_ACTION GetAction() const { return static_cast<CACHE_ACTION>(m_Action); }
        void SetAction(CACHE_ACTION value) { m_Action = static_cast<u8>(value); }

        bool GetIsTargetAll() const { return m_IsTargetAll; }
        void SetIsTargetAll(bool value) { m_IsTargetAll = value; }

        u32   GetItemNameLength() const { return m_ItemName.len; }
        const char* GetItemName()       const { return m_ItemName.str; }

    private:
        void SetItemName(const char* value, u32 length)
        {
            if(m_ItemName.len == 0)
            {
                return;
            }

            if(value == NULL || *value == '\0')
            {
                m_ItemName.str[0] = '\0';
            }

            ut::strncpy(m_ItemName.str, m_ItemName.len + 1, value, length);
        }

    private:
        u8          m_Action;           //!< キャッシュ操作の種類です。
        bool        m_IsTargetAll;      //!< すべてのアイテムを対象とするフラグです。
        u16         m_Padding;
        ResNameData m_ItemName;
    };

public:
    EndItemCachePacket(const char* itemName, u32 maxItemName) :
      HioPacket(HIO_MESSAGE_END_ITEM_CACHE, Body::GetRequiredSize(maxItemName)),
          m_Body(itemName, maxItemName)
      {
      }

public:
    static EndItemCachePacket* CreateInstance(void* buffer, const char* itemName, Result itemCacheResult)
    {
        NW_ASSERT_NOT_NULL(buffer);

        u32 itemNameLength = itemName == NULL ? 0 : std::strlen(itemName);

        EndItemCachePacket* result = new(buffer) EndItemCachePacket(itemName, itemNameLength);

        result->GetHeader().result = HioResult::FromResult(itemCacheResult);

        return result;
    }

    static u32 GetRequiredSize(u32 maxItemName)
    {
        // 固定パケットサイズ + 名前の長さ
        return sizeof(HioPacket) + Body::GetRequiredSize(maxItemName);
    }

    Body&       GetBody() { return m_Body; }
    const Body& GetBody() const { return m_Body; }

private:
    Body m_Body;
};

//---------------------------------------------------------------------------
//! @brief  EndItemCachePacket の返信パケットです。
//---------------------------------------------------------------------------
class EndItemCacheReplyPacket : public HioPacket
{
public:
    struct Body : public HioPacketBody
    {
    };

public:
    EndItemCacheReplyPacket() : HioPacket(HIO_MESSAGE_END_ITEM_CACHE_REPLY, sizeof(Body))
    {
    }

public:
    static u32 GetRequiredSize()
    {
        return sizeof(EndItemCacheReplyPacket);
    }

    Body&       GetBody() { return m_Body; }
    const Body& GetBody() const { return m_Body; }

private:
    Body m_Body;
};


} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#if defined(NW_PLATFORM_CTR)
#if NN_CURRENT_VERSION_NUMBER >= NN_VERSION_NUMBER(4,0,0,0)
#pragma diag_default 1301 // padding inserted in struct.
#pragma diag_default 2530 // padding added to end of struct.
#endif
#endif

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_EDIT_ITEM_CACHE_PACKET_H_
