﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_EDIT_HIO_PROTOCOL_H_
#define NW_SND_EDIT_HIO_PROTOCOL_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

//#define NW_SNDEDIT_HIO_PACKET_DEBUG_DUMP

#include <nw/ut/ut_Inlines.h>
#if defined(NW_PLATFORM_CAFE)
#include <nw/ut/ut_BinaryFileFormat.h>
#elif defined(NW_PLATFORM_CTR)
#include <nw/ut/ut_BinaryFileFormat.h>
#else
#include <nw/snd/snd_BinaryFileFormat.h>
#endif
#include <nw/snd/edit/sndedit_Result.h>
#include <nw/snd/edit/res/sndedit_ResTypes.h>

namespace nw {
namespace snd {
namespace edit {
namespace internal {

static const u32 HIO_PROTOCOL_VERSION = 0x00010000;         //!< SNDEDIT HIO通信プロトコルのバージョン

enum
{
    HIO_MESSAGE_CATEGORY_MASK         = 0x00ff0000,         //!< HIO メッセージカテゴリのマスクです。
    HIO_MESSAGE_REPLY_FLAG_MASK       = 0x00008000,         //!< HIO 返信フラグのマスクです。
    HIO_MESSAGE_ID_MASK               = 0x00007fff,         //!< HIO メッセージ ID のマスクです。

    HIO_MESSAGE_CATEGORY_BIT_OFFSET   = 16,                 //!< HIO メッセージカテゴリ用のビットオフセットです。
    HIO_MESSAGE_REPLY_FLAG_BIT_OFFSET = 15,                 //!< HIO 返信フラグ用のビットオフセットです。
    HIO_MESSAGE_ID_BIT_OFFSET         = 0,                  //!< HIO メッセージ ID 用のビットオフセットです。

    HIO_MESSAGE_REPLY_FLAG = HIO_MESSAGE_REPLY_FLAG_MASK    //!< 返信メッセージであることを示すフラグです。
};

//---------------------------------------------------------------------------
//! @brief  サウンドで利用するメッセージのカテゴリです。
//---------------------------------------------------------------------------
enum HioMessageCategory
{
    HIO_MESSAGE_CATEGORY_SYSTEM  = 0x01 << HIO_MESSAGE_CATEGORY_BIT_OFFSET,     //!< サウンドシステム用のカテゴリです。
    HIO_MESSAGE_CATEGORY_SNDEDIT = 0x02 << HIO_MESSAGE_CATEGORY_BIT_OFFSET,     //!< サウンド編集用のカテゴリです。
    HIO_MESSAGE_CATEGORY_SNDCTRL = 0x03 << HIO_MESSAGE_CATEGORY_BIT_OFFSET      //!< サウンド操作用のカテゴリです。（暫定）
};

//---------------------------------------------------------------------------
//! @brief  サウンドで利用するチャンネル番号です。
//---------------------------------------------------------------------------
enum HioChannelType
{
#if defined(NW_PLATFORM_CTR)
    //HIO_SOUNDPLAYER_PASSIVE_CHANNEL = 0x00000009,               //!< PC->実機 方向に要求するチャンネルです。(仮SoundPlayer用)
    //HIO_SOUNDPLAYER_ACTIVE_CHANNEL  = 0x00000008,               //!< 実機->PC 方向に要求するチャンネルです。(仮SoundPlayer用)
    HIO_SNDEDIT_SYNC_CHANNEL        = 0x0000000A,               //!< SNDEDIT SYNC 用チャンネルです。
    HIO_SNDEDIT_FUNC_CHANNEL        = 0x0000000B,               //!< SNDEDIT 機能チャンネルです。
    HIO_SNDEDIT_CTRL_CHANNEL        = 0x0000000C                //!< SNDCTRL 操作チャンネルです。（暫定）
    //HIO_PING_CHANNEL                = 0x0000000A                //!< 実機の生存確認を行うチャンネルです。
#else
    HIO_BASE_CHANNEL                = 0x8000 + ( 'S' << 8 )+ 'D',
    //HIO_SOUNDPLAYER_PASSIVE_CHANNEL = HIO_BASE_CHANNEL + 0x0,   //!< PC->実機 方向に要求するチャンネルです。(仮SoundPlayer用)
    //HIO_SOUNDPLAYER_ACTIVE_CHANNEL  = HIO_BASE_CHANNEL + 0x1,   //!< 実機->PC 方向に要求するチャンネルです。(仮SoundPlayer用)
    HIO_SNDEDIT_SYNC_CHANNEL        = HIO_BASE_CHANNEL + 0x2,   //!< SNDEDIT SYNC 用チャンネルです。
    HIO_SNDEDIT_FUNC_CHANNEL        = HIO_BASE_CHANNEL + 0x3,   //!< SNDEDIT 機能チャンネルです。
    HIO_SNDEDIT_CTRL_CHANNEL        = HIO_BASE_CHANNEL + 0x4,   //!< SNDCTRL 操作チャンネルです。（暫定）
    //HIO_PING_CHANNEL                = HIO_BASE_CHANNEL + 0xf    //!< 実機の生存確認を行うチャンネルです。
#endif
};

//---------------------------------------------------------------------------
//! @brief  SNDEDIT が受け取れる HIO メッセージの種類です。
//---------------------------------------------------------------------------
typedef u32 HioMessageType;

// HioMessageType の具体値です。
enum
{
    //! @brief  無効なメッセージIDです。
    HIO_MESSAGE_INVALID = 0,

    //---------------------------------------------------------------------------
    // システムカテゴリ
    //---------------------------------------------------------------------------

    //! @brief  実機から定期的に送信される同期メッセージです。
    HIO_MESSAGE_SYNC = 0x00 + HIO_MESSAGE_CATEGORY_SYSTEM,

    //! @brief  HIO_MESSAGE_SYNC の返信メッセージです。
    HIO_MESSAGE_SYNC_REPLY = HIO_MESSAGE_SYNC + HIO_MESSAGE_REPLY_FLAG,

    //! @brief  実機から定期的に送信される無効アイテムの問い合わせ同期メッセージです。
    HIO_MESSAGE_QUERY_ITEMS = 0x01 + HIO_MESSAGE_CATEGORY_SYSTEM,

    //! @brief  HIO_MESSAGE_QUERY_ITEMS の返信メッセージです。
    HIO_MESSAGE_QUERY_ITEMS_REPLY = HIO_MESSAGE_QUERY_ITEMS + HIO_MESSAGE_REPLY_FLAG,

    //---------------------------------------------------------------------------
    // サウンド編集カテゴリ
    //---------------------------------------------------------------------------

    //! @brief  アイテム情報を取得します。
    HIO_MESSAGE_QUERY_ITEM_INFO = 0x00 + HIO_MESSAGE_CATEGORY_SNDEDIT,

    //! @brief  HIO_MESSAGE_QUERY_ITEM_INFO の返信メッセージです。
    HIO_MESSAGE_QUERY_ITEM_INFO_REPLY = HIO_MESSAGE_QUERY_ITEM_INFO + HIO_MESSAGE_REPLY_FLAG,

    //! @brief  アイテムのキャッシュ開始を通知します。
    HIO_MESSAGE_BEGIN_ITEM_CACHE = 0x01 + HIO_MESSAGE_CATEGORY_SNDEDIT,

    //! @brief  HIO_MESSAGE_BEGIN_ITEM_CACHE の返信メッセージです。
    HIO_MESSAGE_BEGIN_ITEM_CACHE_REPLY  = HIO_MESSAGE_BEGIN_ITEM_CACHE + HIO_MESSAGE_REPLY_FLAG,

    //! @brief  アイテムのキャッシュ開始を通知します。
    HIO_MESSAGE_END_ITEM_CACHE = 0x02 + HIO_MESSAGE_CATEGORY_SNDEDIT,

    //! @brief  HIO_MESSAGE_END_ITEM_CACHE の返信メッセージです。
    HIO_MESSAGE_END_ITEM_CACHE_REPLY  = HIO_MESSAGE_END_ITEM_CACHE + HIO_MESSAGE_REPLY_FLAG,

    //---------------------------------------------------------------------------
    // サウンド操作カテゴリ（暫定）
    //---------------------------------------------------------------------------

    //! @brief  サウンドを再生します。（PC -> TARGET)
    HIO_MESSAGE_PLAY_SOUND = 0x00 + HIO_MESSAGE_CATEGORY_SNDCTRL,

    //! @brief  サウンドを停止します。（PC -> TARGET)
    HIO_MESSAGE_STOP_SOUND = 0x01 + HIO_MESSAGE_CATEGORY_SNDCTRL,

    //! @brief  サウンドを一時停止します。（PC -> TARGET)
    HIO_MESSAGE_PAUSE_SOUND = 0x02 + HIO_MESSAGE_CATEGORY_SNDCTRL,

    //! @brief  サウンド情報を更新します。（TARGET -> PC)
    HIO_MESSAGE_UPDATE_SOUND_INFO = 0x80 + HIO_MESSAGE_CATEGORY_SNDCTRL
};

//---------------------------------------------------------------------------
//! @brief  HIO の処理結果です。
//---------------------------------------------------------------------------
enum HioResultType
{
    HIO_RESULT_ERROR_FLAG_MASK  = 0x80000000,
    HIO_RESULT_CATEGORY_MASK    = 0x7f000000,
    HIO_RESULT_CODE_MASK        = 0x00ffffff,

    HIO_RESULT_CATEGORY_BIT_OFFSET = 24,        //!< HIO 結果カテゴリ用のビットオフセットです。

    HIO_RESULT_ERROR_FLAG      = HIO_RESULT_ERROR_FLAG_MASK,
    HIO_RESULT_CATEGORY_NONE   = 0x00 << HIO_RESULT_CATEGORY_BIT_OFFSET,
    HIO_RESULT_CATEGORY_TARGET = 0x01 << HIO_RESULT_CATEGORY_BIT_OFFSET,
    HIO_RESULT_CATEGORY_PC     = 0x02 << HIO_RESULT_CATEGORY_BIT_OFFSET,

    //---------------------------------------------------------------------------
    // 成功コード
    //---------------------------------------------------------------------------

    HIO_RESULT_TRUE  = 0x000000 + HIO_RESULT_CATEGORY_NONE,                            //!< 成功しました。
    HIO_RESULT_FALSE = 0x000001 + HIO_RESULT_CATEGORY_NONE,                            //!< 処理を実行しませんでした。

    //---------------------------------------------------------------------------
    // 失敗コード
    //---------------------------------------------------------------------------

    HIO_RESULT_FAILED = 0x000001 + HIO_RESULT_CATEGORY_NONE + HIO_RESULT_ERROR_FLAG,    //!< 失敗しました。

    //---------------------------------------------------------------------------
    // 失敗コード（実機）
    //---------------------------------------------------------------------------

    HIO_RESULT_TARGET_NOT_INITIALIZED = 0x000001 + HIO_RESULT_CATEGORY_TARGET + HIO_RESULT_ERROR_FLAG,    //!< 初期化されていません。
    HIO_RESULT_TARGET_OUT_OF_MEMORY   = 0x000002 + HIO_RESULT_CATEGORY_TARGET + HIO_RESULT_ERROR_FLAG,    //!< メモリ不足です。
    HIO_RESULT_TARGET_NOT_IMPLEMENTED = 0x000003 + HIO_RESULT_CATEGORY_TARGET + HIO_RESULT_ERROR_FLAG,    //!< 未実装です。
    HIO_RESULT_TARGET_CACHE_OVER_FLOW = 0x000004 + HIO_RESULT_CATEGORY_TARGET + HIO_RESULT_ERROR_FLAG,    //!< キャッシュ可能数を超えました。
    HIO_RESULT_TARGET_NAME_NOT_FOUND  = 0x000005 + HIO_RESULT_CATEGORY_TARGET + HIO_RESULT_ERROR_FLAG,    //!< 名前が見つかりません。

    HIO_RESULT_TARGET_TIMEOUT       = 0x000101 + HIO_RESULT_CATEGORY_TARGET + HIO_RESULT_ERROR_FLAG,    //!< タイムアウトが発生しました。
    HIO_RESULT_TARGET_SEND_ERROR    = 0x000102 + HIO_RESULT_CATEGORY_TARGET + HIO_RESULT_ERROR_FLAG,    //!< 送信エラーです。
    HIO_RESULT_TARGET_RECIEVE_ERROR = 0x000103 + HIO_RESULT_CATEGORY_TARGET + HIO_RESULT_ERROR_FLAG,    //!< 受信エラーです。

    //---------------------------------------------------------------------------
    // 失敗コード（PC）
    //---------------------------------------------------------------------------

    HIO_RESULT_PC_NAME_NOT_FOUND  = 0x000001 + HIO_RESULT_CATEGORY_PC + HIO_RESULT_ERROR_FLAG,    //!< 名前が見つかりません。
    HIO_RESULT_PC_ITEM_PREPAREING = 0x000002 + HIO_RESULT_CATEGORY_PC + HIO_RESULT_ERROR_FLAG     //!< アイテムは準備中です。
};

//---------------------------------------------------------------------------
//! @brief  HIO の処理結果を制御します。
//---------------------------------------------------------------------------
class HioResult
{
public:
    HioResult() : value(HIO_RESULT_TRUE) { }
    explicit HioResult(u32 value) : value(value) { }
    explicit HioResult(HioResultType value) : value(value) { }

    static Result ToResult(HioResultType value)
    {
        switch(value)
        {
        case HIO_RESULT_TRUE:
            return Result(SNDEDIT_RESULT_TRUE);

        case HIO_RESULT_FALSE:
            return Result(SNDEDIT_RESULT_FALSE);

        case  HIO_RESULT_FAILED:
            return Result(SNDEDIT_RESULT_FAILED);

        case  HIO_RESULT_TARGET_NOT_INITIALIZED:
            return Result(SNDEDIT_RESULT_NOT_INITIALIZED);

        case  HIO_RESULT_TARGET_OUT_OF_MEMORY:
            return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);

        case HIO_RESULT_TARGET_CACHE_OVER_FLOW:
            return Result(SNDEDIT_RESULT_CACHE_OVER_FLOW);

        case  HIO_RESULT_TARGET_NAME_NOT_FOUND:
            return Result(SNDEDIT_RESULT_NAME_NOT_FOUND);

        case  HIO_RESULT_TARGET_TIMEOUT:
            return Result(SNDEDIT_RESULT_TIMEOUT);

        case  HIO_RESULT_PC_NAME_NOT_FOUND:
            return Result(SNDEDIT_RESULT_NAME_NOT_FOUND);

        case  HIO_RESULT_PC_ITEM_PREPAREING:
            return Result(SNDEDIT_RESULT_ITEM_PREPAREING);

        default:
            break;
        }

        return HioResult(value).IsSucceeded() ?
            Result(SNDEDIT_RESULT_TRUE) : Result(SNDEDIT_RESULT_FAILED);
    }

    static HioResult FromResult(Result value)
    {
        switch(u32(value))
        {
        case SNDEDIT_RESULT_TRUE:
            return HioResult(HIO_RESULT_TRUE);

        case SNDEDIT_RESULT_FALSE:
            return HioResult(HIO_RESULT_FALSE);

        case  SNDEDIT_RESULT_FAILED:
            return HioResult(HIO_RESULT_FAILED);

        case  SNDEDIT_RESULT_NOT_INITIALIZED:
            return HioResult(HIO_RESULT_TARGET_NOT_INITIALIZED);

        case  SNDEDIT_RESULT_OUT_OF_MEMORY:
            return HioResult(HIO_RESULT_TARGET_OUT_OF_MEMORY);

        case  SNDEDIT_RESULT_TIMEOUT:
            return HioResult(HIO_RESULT_TARGET_TIMEOUT);

        case SNDEDIT_RESULT_CACHE_OVER_FLOW:
            return HioResult(HIO_RESULT_TARGET_CACHE_OVER_FLOW);

        case  SNDEDIT_RESULT_NAME_NOT_FOUND:
            return HioResult(HIO_RESULT_TARGET_NAME_NOT_FOUND);

        case  SNDEDIT_RESULT_ITEM_PREPAREING:
            return HioResult(HIO_RESULT_PC_ITEM_PREPAREING);
        }

        return HioResult(value).IsSucceeded() ?
            HioResult(HIO_RESULT_TRUE) : HioResult(HIO_RESULT_FAILED);
    }

    bool IsSucceeded() const { return !IsFailed(); }
    bool IsTrue() const { return value == HIO_RESULT_TRUE; }
    bool IsFalse() const { return value == HIO_RESULT_FALSE; }
    bool IsFailed() const { return (value & HIO_RESULT_ERROR_FLAG) != 0; }

    //--------------------------------------------------------------------------
    //! @brief   u32 型へのキャスト演算子です。
    //!
    //! @return  u32 値を返します。
    //---------------------------------------------------------------------------
    operator u32() const
    {
        return value;
    }

    //--------------------------------------------------------------------------
    //! @brief   HioResultType 型へのキャスト演算子です。
    //!
    //! @return  HioResultType で表現した値を返します。
    //---------------------------------------------------------------------------
    operator HioResultType() const
    {
        return static_cast<HioResultType>(value);
    }

    //--------------------------------------------------------------------------
    //! @brief   ResultType 型へのキャスト演算子です。
    //!
    //! @return  ResultType で表現した値を返します。
    //---------------------------------------------------------------------------
    operator ResultType() const
    {
        return HioResult::ToResult(static_cast<HioResultType>(value));
    }

private:
    u32 value;
};

//---------------------------------------------------------------------------
//! @brief   HIOメッセージです。
//!
//! @details 32bit の内訳は以下の通りです。
//!          31 - 24 : Reserved
//!          23 - 16 : メッセージカテゴリ
//!          15 -  0 : メッセージID
//---------------------------------------------------------------------------
struct HioMessage
{
    snd::internal::fnd::BinU32 value;

    //---------------------------------------------------------------------------

    HioMessage() : value(0) { }
    explicit HioMessage(HioMessageType value) : value(value) { }

    //---------------------------------------------------------------------------
    //! @brief   HIOメッセージカテゴリを取得します。
    //!
    //! @return  HIOメッセージカテゴリを返します。
    //---------------------------------------------------------------------------
    HioMessageType GetCategory() const
    {
        return (value & HIO_MESSAGE_ID_MASK) >> HIO_MESSAGE_CATEGORY_BIT_OFFSET;
    }

    //---------------------------------------------------------------------------
    //! @brief   HIOメッセージID を取得します。
    //!
    //! @return  HIOメッセージID を返します。
    //---------------------------------------------------------------------------
    HioMessageType GetID() const
    {
        return (value & HIO_MESSAGE_ID_MASK) >> HIO_MESSAGE_ID_BIT_OFFSET;
    }

    //--------------------------------------------------------------------------
    //! @brief   HioMessageType 型へのキャスト演算子です。
    //!
    //! @return  HioMessageType で表現した値を返します。
    //---------------------------------------------------------------------------
    operator HioMessageType() const
    {
        return value;
    }

#if !defined(NW_RELEASE)
    //--------------------------------------------------------------------------
    //! @brief   指定メッセージを文字列に変換します。
    //!
    //! @param[in]  message  メッセージを指定します。
    //!
    //! @return  変換した文字列を返します。
    //---------------------------------------------------------------------------
    static const char* ToString(HioMessageType message)
    {
        switch(message)
        {
        case HIO_MESSAGE_SYNC:
            return "HIO_MESSAGE_SYNC";

        case HIO_MESSAGE_SYNC_REPLY:
            return "HIO_MESSAGE_SYNC_REPLY";

        case HIO_MESSAGE_QUERY_ITEMS:
            return "HIO_MESSAGE_QUERY_ITEMS";

        case HIO_MESSAGE_QUERY_ITEMS_REPLY:
            return "HIO_MESSAGE_QUERY_ITEMS_REPLY";

        case HIO_MESSAGE_QUERY_ITEM_INFO:
            return "HIO_MESSAGE_QUERY_ITEM_INFO";

        case HIO_MESSAGE_QUERY_ITEM_INFO_REPLY:
            return "HIO_MESSAGE_QUERY_ITEM_INFO_REPLY";

        case HIO_MESSAGE_BEGIN_ITEM_CACHE:
            return "HIO_MESSAGE_BEGIN_ITEM_CACHE";

        case HIO_MESSAGE_BEGIN_ITEM_CACHE_REPLY:
            return "HIO_MESSAGE_BEGIN_ITEM_CACHE_REPLY";

        case HIO_MESSAGE_END_ITEM_CACHE:
            return "HIO_MESSAGE_END_ITEM_CACHE";

        case HIO_MESSAGE_END_ITEM_CACHE_REPLY:
            return "HIO_MESSAGE_END_ITEM_CACHE_REPLY";

        case HIO_MESSAGE_PLAY_SOUND:
            return "HIO_MESSAGE_PLAY_SOUND";

        case HIO_MESSAGE_STOP_SOUND:
            return "HIO_MESSAGE_STOP_SOUND";

        case HIO_MESSAGE_PAUSE_SOUND:
            return "HIO_MESSAGE_PAUSE_SOUND";

        case HIO_MESSAGE_UPDATE_SOUND_INFO:
            return "HIO_MESSAGE_UPDATE_SOUND_INFO";
        }

        return "Unknown Message";
    }

    //--------------------------------------------------------------------------
    //! @brief   メッセージを文字列に変換します。
    //!
    //! @return  変換した文字列を返します。
    //---------------------------------------------------------------------------
    const char* ToString() const
    {
        return ToString(value);
    }
#endif
};

//---------------------------------------------------------------------------
//! @brief  HIO パケットヘッダです。
//---------------------------------------------------------------------------
struct HioPacketHeader
{
    static const u32 VALID_SIGNATURE = NW_UT_MAKE_SIGWORD('S', 'E', 'D', 'T');

    u32 signature;
    HioMessage message;
    snd::internal::fnd::BinU32 bodySize;
    snd::internal::fnd::BinU32 result;

    //---------------------------------------------------------------------------

    HioPacketHeader()
    {}

    explicit HioPacketHeader(HioMessage message)
        : signature(VALID_SIGNATURE),
          message(message),
          bodySize(0),
          result(HIO_RESULT_TRUE)
    {
    }

    HioPacketHeader(HioMessage message, u32 bodySize)
        : signature(VALID_SIGNATURE),
          message(message),
          bodySize(bodySize),
          result(HIO_RESULT_TRUE)
    {}

    bool IsValid() const { return signature == VALID_SIGNATURE; }

    HioResult GetResult() const { return HioResult(static_cast<u32>(result)); }
};

//---------------------------------------------------------------------------
//! @brief  HIO パケットボディです。
//---------------------------------------------------------------------------
struct HioPacketBody
{
};

//---------------------------------------------------------------------------
//! @brief  HIO パケットです。
//---------------------------------------------------------------------------
class HioPacket
{
public:
    HioPacket() { }
    explicit HioPacket(HioMessageType message) :
        m_Header(HioMessage(message), sizeof(HioPacketHeader)) { }

    HioPacket(HioMessageType message, u32 bodySize) :
        m_Header(HioMessage(message), bodySize) { }

public:
    static HioPacket& Null()
    {
        static HioPacket nullInstance(HIO_MESSAGE_INVALID, 0);
        return nullInstance;
    }

    HioPacketHeader&       GetHeader()       { return m_Header; }
    const HioPacketHeader& GetHeader() const { return m_Header; }

    HioPacketBody* GetBody()
    {
        return m_Header.bodySize == 0 ?
            NULL : reinterpret_cast<HioPacketBody*>(ut::AddOffsetToPtr(this, sizeof(HioPacketHeader)));
    }

    const HioPacketBody* GetBody() const
    {
        return m_Header.bodySize == 0 ?
            NULL : reinterpret_cast<const HioPacketBody*>(ut::AddOffsetToPtr(this, sizeof(HioPacketHeader)));
    }

    u32 GetSize() const
    {
        return sizeof(HioPacketHeader) + m_Header.bodySize;
    }

#if defined(NW_SNDEDIT_HIO_PACKET_DEBUG_DUMP)
public:
    void Dump()
    {
        NW_LOG("--------------- HioPacketDumpStart ---------------\n");
        DumpHeader();
        NW_LOG("--------------------------------------------------\n");
        DumpBody();
        NW_LOG("---------------- HioPacketDumpEnd ----------------\n");
    }

private:
    void DumpHeader()
    {
        NW_LOG("signature = %x, message = %d\n", m_Header.signature, m_Header.message.value);
        NW_LOG("bodySize = %d, result = %d\n", m_Header.bodySize, m_Header.result);
    }

    void DumpBody()
    {
        const void* body = reinterpret_cast<const void*>(GetBody());
        const u8* startAddress = reinterpret_cast<const u8*>(body);

        for (u32 i = 0; i < m_Header.bodySize; ++i)
        {
            NW_LOG("%02x ", *startAddress);
            startAddress++;
        }
        NW_LOG("\n");

        startAddress = reinterpret_cast<const u8*>(body);
        for (u32 i = 0; i < m_Header.bodySize; ++i)
        {
            NW_LOG("%c ", *startAddress);
            startAddress++;
        }
        NW_LOG("\n");
    }
#endif

private:
    HioPacketHeader m_Header;
};

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_EDIT_HIO_PROTOCOL_H_
