﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_CTRL_SOUND_OBJECT_CONTROLLER_H_
#define NW_SND_CTRL_SOUND_OBJECT_CONTROLLER_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/ut/ut_Preprocessor.h>
#include <nw/snd/snd_SoundHandle.h>
#include <nw/snd/edit/sndedit_Result.h>
#include <nw/snd/edit/sndedit_IErrorProvider.h>
#include <nw/snd/ctrl/sndctrl_SoundControllerImpl.h>

namespace nw {
namespace snd {

class SoundArchivePlayer;

namespace ctrl {

//---------------------------------------------------------------------------
//! @brief    【β版】サウンド関連のオブジェクトを操作するクラスです。
//!
//!           内包するサウンドハンドルを使って外部からサウンド制御するために利用します。
//---------------------------------------------------------------------------
class SoundObjectController : private edit::internal::IErrorProvider
{
    NW_DISALLOW_COPY_AND_ASSIGN(SoundObjectController);

public: // 定数の定義
    static const u32 SOUND_CONTROLLER_COUNT = 4;    //!< SoundObjectController が保持する SoundController の数です。

public: // コンストラクタ
    //===========================================================================
    //! @name コンストラクタ/デストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //---------------------------------------------------------------------------
    SoundObjectController();

    //---------------------------------------------------------------------------
    //! @brief  デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~SoundObjectController() { }

    //@}
    // コンストラクタ/デストラクタ
    //===========================================================================

public: // メソッド
    //===========================================================================
    //! @name 初期化
    //@{

    //---------------------------------------------------------------------------
    //! @brief      SoundObjectController を初期化します。
    //!
    //! @param[in]  soundArchivePlayer  操作対象のサウンドアーカイブプレイヤーを指定します。
    //!
    //! @return     処理結果を返します。
    //!
    //! @see        Finalize()
    //! @see        IsInitialized()
    //---------------------------------------------------------------------------
    edit::Result Initialize(SoundArchivePlayer* soundArchivePlayer);

    //---------------------------------------------------------------------------
    //! @brief       SoundObjectController の終了処理を行います。
    //!
    //! @see         Initialize()
    //---------------------------------------------------------------------------
    void Finalize();

    //@}
    // 初期化
    //===========================================================================

    //===========================================================================
    //! @name 状態の取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief   初期化の有無を取得します。
    //!
    //! @return  初期化済みの場合は true、初期化されていない場合は false を返します。
    //!
    //! @see     Initialize()
    //---------------------------------------------------------------------------
    bool IsInitialized() const { return m_IsInitialized; }

    //@}
    // 状態の取得
    //===========================================================================

    //===========================================================================
    //! @name 操作
    //@{

    //---------------------------------------------------------------------------
    //! @brief   SoundObjectController を更新します。
    //!
    //!          この関数を呼び出すと、サウンドのパラメータや状態を同期します。
    //---------------------------------------------------------------------------
    void Update();

    //---------------------------------------------------------------------------
    //! @brief  サウンドコントローラを取得します。
    //!
    //! @param[in]  index  取得するサウンドコントローラのインデックスを指定します。
    //!                    @n
    //!                    上限値は SOUND_CONTROLLER_COUNT です。
    //!
    //! @return  対象のサウンドコントローラを返します。
    //!
    //! @see     SOUND_CONTROLLER_COUNT
    //---------------------------------------------------------------------------
    SoundController* GetSoundController(u32 index)
    {
        NW_ASSERT(index < SOUND_CONTROLLER_COUNT);
        return &m_SoundController[index];
    }

    //@}
    // 操作
    //===========================================================================

    //===========================================================================
    //! @name エラーハンドリング
    //@{

    //---------------------------------------------------------------------------
    //! @brief         最後に発生したエラー情報を取得します。
    //!
    //! @return        最後に発生したエラー情報を返します。
    //!
    //! @see           SetLastError()
    //---------------------------------------------------------------------------
    virtual edit::Result GetLastError() const { return m_LastError; }

    //---------------------------------------------------------------------------
    //! @brief         最後に発生したエラー情報を設定します。
    //!
    //! @param[in]     value  エラー値を指定します。
    //!
    //! @see           GetLastError()
    //---------------------------------------------------------------------------
    virtual void SetLastError(edit::ResultType value);

    //@}
    // エラーハンドリング
    //===========================================================================

private: // メンバ変数
    edit::Result m_LastError;         //!< 最後に発生したエラー情報です。

    SoundHandle                   m_SoundHandles[SOUND_CONTROLLER_COUNT];
    internal::SoundControllerImpl m_SoundController[SOUND_CONTROLLER_COUNT];

    bool m_IsInitialized;
    u8   m_Padding[3];
};

} // namespace nw::snd::ctrl
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_CTRL_SOUND_OBJECT_CONTROLLER_H_
