﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_CTRL_SOUND_CONTROLLER_H_
#define NW_SND_CTRL_SOUND_CONTROLLER_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/config.h>
#include <nw/snd/snd_SoundStartable.h>
#include <nw/snd/edit/sndedit_Result.h>

namespace nw {
namespace snd {

class SoundHandle;

namespace ctrl {

struct SoundParameters;

//---------------------------------------------------------------------------
//! @brief  【β版】サウンドを制御するクラスです。
//---------------------------------------------------------------------------
class SoundController
{
public: // 型の定義
    //---------------------------------------------------------------------------
    //! @brief  サウンドの状態を示します。
    //---------------------------------------------------------------------------
    enum State
    {
        STATE_DISABLED,     //!< 正しいサウンドIDが設定されていない等、無効な状態です。
        STATE_STOPPED,      //!< 停止中です。
        STATE_PLAYING,      //!< 再生中です。
        STATE_PAUSED        //!< 一時停止中です。
    };

protected: // コンストラクタ
    //! @briefprivate
    SoundController() { }

    //! @briefprivate
    virtual ~SoundController() { }

public: // メソッド
    //===========================================================================
    //! @name 状態の取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief   初期化の有無を取得します。
    //!
    //! @return  初期化済みの場合は true、初期化されていない場合は false を返します。
    //---------------------------------------------------------------------------
    virtual bool IsInitialized() const = 0;

    //---------------------------------------------------------------------------
    //! @brief   サウンドハンドルを取得します。
    //!
    //! @return  サウンドハンドルを返します。
    //---------------------------------------------------------------------------
    virtual const SoundHandle* GetSoundHandle() const = 0;

    //---------------------------------------------------------------------------
    //! @brief   サウンドハンドルを取得します。
    //!
    //! @return  サウンドハンドルを返します。
    //---------------------------------------------------------------------------
    virtual SoundHandle* GetSoundHandle() = 0;

    //---------------------------------------------------------------------------
    //! @brief   対象のサウンドIDを取得します。
    //!
    //! @return  対象のサウンドIDを返します。
    //---------------------------------------------------------------------------
    virtual SoundArchive::ItemId GetSoundID() const = 0;

    //---------------------------------------------------------------------------
    //! @brief   対象のサウンドラベルを取得します。
    //!
    //! @return  対象のサウンドラベルを返します。
    //---------------------------------------------------------------------------
    virtual const char* GetLabel() const = 0;

    //---------------------------------------------------------------------------
    //! @brief   サウンドの状態を取得します。
    //!
    //! @return  サウンドの状態を返します。
    //---------------------------------------------------------------------------
    virtual State GetState() const = 0;

    //@}
    // 状態の取得
    //===========================================================================

    //===========================================================================
    //! @name パラメータ制御
    //@{

    //---------------------------------------------------------------------------
    //! @brief   サウンドのパラメータを取得します。
    //!
    //! @return  サウンドのパラメータを返します。
    //---------------------------------------------------------------------------
    virtual const SoundParameters& GetParameters() const = 0;

    //---------------------------------------------------------------------------
    //! @brief   サウンドのパラメータを取得します。
    //!
    //! @return  サウンドのパラメータを返します。
    //---------------------------------------------------------------------------
    virtual SoundParameters& GetParameters() = 0;

    //---------------------------------------------------------------------------
    //! @brief   サウンドパラメータの変更状態を設定します。
    //!
    //! @param[in] isDirty 変更ありなら true を、無しなら false を指定します。
    //---------------------------------------------------------------------------
    virtual void SetParameterDirty(bool isDirty) = 0;

    //@}
    // パラメータ取得
    //===========================================================================

    //===========================================================================
    //! @name 再生制御
    //@{

    //---------------------------------------------------------------------------
    //! @brief   サウンドを再生します。
    //!
    //! @param   pStartInfo 詳細な再生パラメータです。
    //!
    //! @return  処理結果を返します。
    //---------------------------------------------------------------------------
    virtual edit::Result Play(const SoundStartable::StartInfo * pStartInfo = NULL) = 0;

    //---------------------------------------------------------------------------
    //! @brief   サウンドを停止します。
    //!
    //! @return  処理結果を返します。
    //---------------------------------------------------------------------------
    virtual edit::Result Stop() = 0;

    //---------------------------------------------------------------------------
    //! @brief   サウンドを一時停止します。
    //!
    //! @return  処理結果を返します。
    //---------------------------------------------------------------------------
    virtual edit::Result Pause() = 0;

    //@}
    // 再生制御
    //===========================================================================
};

} // namespace nw::snd::ctrl
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_CTRL_SOUND_CONTROLLER_H_
