﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SCFONT_GLYPH_TREE_MAP_H_
#define NW_SCFONT_GLYPH_TREE_MAP_H_

#include <nw/types.h>
#include <nw/ut/ut_BitFlag.h>
#include <nw/ut/ut_LinkList.h>
#include <nw/ut/ut_Memory.h>
#include <nw/scfont/scfont_FreeList.h>

namespace nw
{
namespace scfont
{

//---------------------------------------------------------------------------
//! @brief テクスチャキャッシュ内のグリフを表すクラスです。
//!
//! @details
//! ライブラリユーザが直接使用するクラスではないため、詳しいDoxyコメントは
//! 付けていません。
//---------------------------------------------------------------------------
class GlyphNode
{
friend class GlyphTreeMap;
public:
    GlyphNode(char16 code, u16 size, u16 face)
     : mLeft(NULL)
     , mRight(NULL)
     , mColor(RED)
     , mLineKind(0)
     , mLineNo(0)
     , mCachePosX(0)
     , mCachePosY(0)
     , mGlyphWidth(0)
     , mGlyphHeight(0)
     , mAdvanceX(0)
     , mLeftOffset(0)
    {
        mKey.detail.code = code;
        mKey.detail.size = size + (face << 11);
    }

    char16 GetCode() const { return mKey.detail.code ;}
    u16 GetSize() const { return mKey.detail.size & 0x7ff;}
    u16 GetFace() const { return mKey.detail.size >> 11;}

    void SetRequested(bool b) { mFlag.ChangeMask(FLAG_MASK_REQUESTED, b); }
    bool IsRequested() const { return mFlag.IsMaskOn(FLAG_MASK_REQUESTED); }

    void SetPlotting(bool b) { mFlag.ChangeMask(FLAG_MASK_PLOTTING, b); }
    bool IsPlotting() const { return mFlag.IsMaskOn(FLAG_MASK_PLOTTING); }

    void SetUsed(bool b) { mFlag.ChangeMask(FLAG_MASK_USED, b); }
    bool IsUsed() const { return mFlag.IsMaskOn(FLAG_MASK_USED); }

    void SetErased(bool b) { mFlag.ChangeMask(FLAG_MASK_ERASED, b); }
    bool IsErased() const { return mFlag.IsMaskOn(FLAG_MASK_ERASED); }

    void SetKeeped(bool b) { mFlag.ChangeMask(FLAG_MASK_KEEPED, b); }
    bool IsKeeped() const { return mFlag.IsMaskOn(FLAG_MASK_KEEPED); }

    void SetSystemReserved(bool b) { mFlag.ChangeMask(FLAG_MASK_SYSTEM_RESERVED, b); }
    bool IsSystemReserved() const { return mFlag.IsMaskOn(FLAG_MASK_SYSTEM_RESERVED); }

    void SetNotInFont(bool b) { mFlag.ChangeMask(FLAG_MASK_NOT_IN_FONT, b); }
    bool IsNotInFont() const { return mFlag.IsMaskOn(FLAG_MASK_NOT_IN_FONT); }

    bool IsPlottingOrNotInFont() const { return mFlag.IsMaskOn(FLAG_MASK_PLOTTING | FLAG_MASK_NOT_IN_FONT); }
    bool IsRequestedOrKeeped() const { return mFlag.IsMaskOn(FLAG_MASK_REQUESTED | FLAG_MASK_KEEPED); }

    u16 GetCachePosX() const { return mCachePosX; }
    u16 GetCachePosY() const { return mCachePosY; }
    u16 GetGlyphWidth() const { return mGlyphWidth; }
    u16 GetGlyphHeight() const { return mGlyphHeight; }
    u16 GetAdvanceX() const { return mAdvanceX; }
    s16 GetLeftOffset() const { return mLeftOffset; }

    void SetCachePosX(u16 posX) { mCachePosX = posX; }
    void SetCachePosY(u16 posY) { mCachePosY = posY; }
    void SetGlyphWidth(u16 width) { mGlyphWidth = width; }
    void SetGlyphHeight(u16 height) { mGlyphHeight = height; }
    void SetAdvanceX(u16 advance) { mAdvanceX = advance; }
    void SetLeftOffset(s16 offset) { mLeftOffset = offset; }

    ut::BitFlag32& GetLockGroup() { return mLockGroup; }
    const ut::BitFlag32& GetLockGroup() const { return mLockGroup; }

    u8 GetLineNo() const { return mLineNo; }
    void SetLineNo(u8 no) { mLineNo = no; }
    u8 GetLineKind() const { return mLineKind; }
    void SetLineKind(u8 kind) { mLineKind = kind; }

    bool IsErasable() const { return (mFlag.IsZero() && mLockGroup.IsZero()); }
    bool IsUsable() const { return ( ! IsPlottingOrNotInFont() && (mFlag.IsMaskOn(FLAG_MASK_REQUESTED | FLAG_MASK_USED | FLAG_MASK_KEEPED | FLAG_MASK_SYSTEM_RESERVED) || ! mLockGroup.IsZero()) ); }

    static u8 CalcLineKind(u16 size);

    ut::LinkListNode m_Link;
    ut::LinkListNode m_LineLink;

protected:
    enum FlagBit {
        FLAG_BIT_REQUESTED,
        FLAG_BIT_PLOTTING,
        FLAG_BIT_USED,
        FLAG_BIT_ERASED,
        FLAG_BIT_KEEPED,
        FLAG_BIT_SYSTEM_RESERVED,
        FLAG_BIT_NOT_IN_FONT
    };

    enum FlagMask {
        FLAG_MASK_REQUESTED = 1 << FLAG_BIT_REQUESTED,
        FLAG_MASK_PLOTTING = 1 << FLAG_BIT_PLOTTING,
        FLAG_MASK_USED = 1 << FLAG_BIT_USED,
        FLAG_MASK_ERASED = 1 << FLAG_BIT_ERASED,
        FLAG_MASK_KEEPED = 1 << FLAG_BIT_KEEPED,
        FLAG_MASK_SYSTEM_RESERVED = 1 << FLAG_BIT_SYSTEM_RESERVED,
        FLAG_MASK_NOT_IN_FONT = 1 << FLAG_BIT_NOT_IN_FONT
    };

    static const u8 RED = 1;
    static const u8 BLACK = 0;

    union KeyType {
        u32 raw;
        struct {
            u16 code;
            u16 size;   // 上位5ビットにfont faceを、下位11ビットにfont sizeを入れる
        } detail;
    };

    NW_STATIC_ASSERT(sizeof(KeyType) == 4);

    GlyphNode* mLeft;
    GlyphNode* mRight;
    KeyType mKey;
    ut::BitFlag8 mFlag;
    u8 mColor;
    u8 mLineKind;
    u8 mLineNo;
    ut::BitFlag32 mLockGroup;
    u16 mCachePosX;
    u16 mCachePosY;
    u16 mGlyphWidth;
    u16 mGlyphHeight;
    u16 mAdvanceX;
    s16 mLeftOffset;

};

typedef ut::LinkList<GlyphNode, offsetof(GlyphNode, m_Link)> GlyphList;
typedef ut::LinkList<GlyphNode, offsetof(GlyphNode, m_LineLink)> GlyphLineList;

//---------------------------------------------------------------------------
//! @brief テクスチャキャッシュ内のグリフを管理するためのTreeMapです。
//!
//! @details
//! 左傾赤黒木という平衡二分木のアルゴリズムを利用してます。
//!
//! ライブラリユーザが直接使用するクラスではないため、詳しいDoxyコメントは
//! 付けていません。
//---------------------------------------------------------------------------
class GlyphTreeMap
{
public:
    GlyphTreeMap();

    void Initialize(nw::ut::IAllocator* allocator, u32 nodeNumMax);

    void Finalize(nw::ut::IAllocator* allocator);

    //! @brief 引数で指定した文字コードとサイズに等しいグリフを検索して返します。
    //!
    //! @param code 検索するグリフの文字コードです。
    //! @param size 検索するグリフのサイズです。
    //! @param face 追加するグリフのフォント字形です。
    //! @return 見つかったグリフ。見つからなかった場合はNULL
    //!
    GlyphNode* Find(char16 code, u16 size, u16 face) const;

    //! @brief 引数で与えた文字コードとサイズのグリフをツリーマップに追加します。
    //!
    //! @param code 追加するグリフの文字コードです。
    //! @param size 追加するグリフのサイズです。
    //! @param face 追加するグリフのフォント字形です。
    //! @return 追加したグリフ。グリフのノード数が最大値を超えている場合は内部でアサートします。製品版ではNULLが返されます。
    //!
    GlyphNode* Insert(char16 code, u16 size, u16 face);

    //! @brief 引数で指定した文字コードとサイズに等しいグリフを検索し、削除します。
    //!
    //! @details
    //! 等しい文字コードとサイズを持っているグリフが存在しない場合、実行時例外で
    //! 停止してしまいますので、
    //! 等しい文字コードとサイズを持っているグリフが含まれていることを確認した上でご使用ください。
    //!
    //! @param code 削除するグリフの文字コードです。
    //! @param size 削除するグリフのサイズです。
    //! @param face 追加するグリフのフォント字形です。
    //!
    void Erase(char16 code, u16 size, u16 face);

    //! @brief ScalableFontのCompleteTextureCacheでグリフのフラグを更新する処理を行います。
    //!
    void UpdateFlagsForCompleteTextureCache()
    {
        if (mRoot)
        {
            UpdateFlagsForCompleteTextureCacheRecursive(mRoot);
        }
    }

    //! @brief ScalableFontのCompleteTextureCacheでグリフのフラグを更新する処理を行います。
    //!
    //! @param[in] groupMask    グループマスクです。
    //!
    void ClearLockGroup(u32 groupMask)
    {
        if (mRoot)
        {
            ClearLockGroupRecursive(mRoot, groupMask);
        }
    }

    //! @brief 内容をダンプします。
    //!
    void Dump();

    //! @brief 内容を初期化し、Initialize直後の状態に戻します。
    //!
    void Reset();

protected:
    GlyphNode* mRoot;
    FreeList mFreeList;
    u32 mNodeNumMax;

    GlyphNode* Find(GlyphNode* node, u32 key_raw) const;
    GlyphNode* Insert(GlyphNode* h, GlyphNode* node);
    GlyphNode* Erase(GlyphNode* h, u32 key_raw);
    static inline GlyphNode* RotateLeft(GlyphNode* h);
    static inline GlyphNode* RotateRight(GlyphNode* h);
    static inline void FlipColors(GlyphNode* h);
    static inline bool IsRed(GlyphNode* h);
    static inline GlyphNode* FixUp(GlyphNode* h);
    static inline GlyphNode* MoveRedLeft(GlyphNode* h);
    static inline GlyphNode* MoveRedRight(GlyphNode* h);
    static inline GlyphNode* EraseMin(GlyphNode* h);
    static inline GlyphNode* GetMin(GlyphNode* h);

    static void UpdateFlagsForCompleteTextureCacheRecursive(GlyphNode* node);
    static void ClearLockGroupRecursive(GlyphNode* node, u32 groupMask);
    static u32 DumpRecursive(GlyphNode* node, u32 level);

};

inline GlyphNode* GlyphTreeMap::RotateLeft(GlyphNode* h)
{
    GlyphNode* x = h->mRight;
    h->mRight = x->mLeft;
    x->mLeft = h;
    x->mColor = h->mColor;
    h->mColor = GlyphNode::RED;
    return x;
}

inline GlyphNode* GlyphTreeMap::RotateRight(GlyphNode* h)
{
    GlyphNode* x = h->mLeft;
    h->mLeft = x->mRight;
    x->mRight = h;
    x->mColor = h->mColor;
    h->mColor = GlyphNode::RED;
    return x;
}

inline void GlyphTreeMap::FlipColors(GlyphNode* h)
{
    h->mColor = ! h->mColor;
    h->mLeft->mColor = ! h->mLeft->mColor;
    h->mRight->mColor = ! h->mRight->mColor;
}

inline bool GlyphTreeMap::IsRed(GlyphNode* h)
{
    if (h) {
        return h->mColor == GlyphNode::RED;
    } else {
        return false;
    }
}

inline GlyphNode* GlyphTreeMap::FixUp(GlyphNode* h)
{
    if (IsRed(h->mRight)) {
        h = RotateLeft(h);
    }
    if(IsRed(h->mLeft) && IsRed(h->mLeft->mLeft)) {
        h = RotateRight(h);
    }
    if(IsRed(h->mLeft) && IsRed(h->mRight)) {
        FlipColors(h);
    }
    return h;
}

inline GlyphNode* GlyphTreeMap::MoveRedLeft(GlyphNode* h)
{
    FlipColors(h);
    if (IsRed(h->mRight->mLeft)) {
        h->mRight = RotateRight(h->mRight);
        h = RotateLeft(h);
        FlipColors(h);
    }
    return h;
}

inline GlyphNode* GlyphTreeMap::MoveRedRight(GlyphNode* h)
{
    FlipColors(h);
    if (IsRed(h->mLeft->mLeft)) {
        h = RotateRight(h);
        FlipColors(h);
    }
    return h;
}

inline GlyphNode* GlyphTreeMap::EraseMin(GlyphNode* h)
{
    if (h->mLeft == NULL) {
        return NULL;
    }
    if ( ! IsRed(h->mLeft) && ! IsRed(h->mLeft->mLeft)) {
        h = MoveRedLeft(h);
    }
    h->mLeft = EraseMin(h->mLeft);

    return FixUp(h);
}

inline GlyphNode* GlyphTreeMap::GetMin(GlyphNode* h)
{
    while (h->mLeft) {
        h = h->mLeft;
    }
    return h;
}

} // namespace nw::scfont
} // namespace nw

#endif // NW_SCFONT_GLYPH_TREE_MAP_H_
