﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MCS_MCS_FILE_STREAM_H_
#define NW_MCS_MCS_FILE_STREAM_H_

#include <nw/mcs/mcs_FileIO.h>
#include <nw/ut/ut_FileStream.h>

namespace nw   {
namespace mcs    {

//---------------------------------------------------------------------------
//! @brief Mcs FileIO を用いたストリームクラスです。
//!
//! @details
//! :category ストリーム入出力
//!
//! McsFileStream は ut::FileStream の派生クラスです。
//! Mcs ライブラリを使用した Mcs サーバ上のファイルへのシーケンシャル
//! アクセス操作を実装します。
//!
//! Mcs FileIO は非同期処理をサポートしないため、
//! 非同期型のメンバー関数は使用できません。
//!
//! @sa ut::FileStream
//---------------------------------------------------------------------------
class McsFileStream : public ut::FileStream
{
#if defined(NW_MCS_ENABLE)

// ------------- public members -------------------
public:
    NW_UT_RUNTIME_TYPEINFO( ut::FileStream )

    //! @brief エラーコードです。
    //!
    //! @sa Read
    //! @sa Write
    enum Result
    {
        //! 通信エラーが発生しました。
        ERR_COMERROR           = -FILEIO_ERROR_COMERROR,
        //! Mcs サーバと接続されていません。
        ERR_NOTCONNECT         = -FILEIO_ERROR_NOTCONNECT,

        //! Mcs サーバはリクエストの実行を継続中です。
        ERR_CONTINUE           = -FILEIO_ERROR_CONTINUE,
        //! Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
        ERR_PROTOCOLERROR      = -FILEIO_ERROR_PROTOCOLERROR,
        //! ファイルが見つかりません。
        ERR_FILENOTFOUND       = -FILEIO_ERROR_FILENOTFOUND,
        //! 指定されたファイル名に含まれるパスが無効です。
        ERR_DIRECTORYNOTFOUND  = -FILEIO_ERROR_DIRECTORYNOTFOUND,
        //! 指定されたファイルは既に存在します。
        ERR_FILEALREADYEXIST   = -FILEIO_ERROR_FILEALREADYEXIST,
        //! 呼び出し元に、必要なアクセス許可がありません。
        ERR_SECURITYERROR      = -FILEIO_ERROR_SECURITYERROR,
        //! 認証されないアクセスです。
        ERR_UNAUTHORIZEDACCESS = -FILEIO_ERROR_UNAUTHORIZEDACCESS,
        //! その他の I/O エラーが発生しました
        ERR_IOERROR            = -FILEIO_ERROR_IOERROR,
        //! Windows 側が認識できないファイルのハンドルを操作しようとしています。
        ERR_INVALIDFILEHANDLE  = -FILEIO_ERROR_INVALIDFILEHANDLE,
        //! パターンに一致するファイルが見つかりません。
        ERR_NOMOREFILES        = -FILEIO_ERROR_NOMOREFILES,
        //! ファイル選択はキャンセルされました。
        ERR_CANCELED           = -FILEIO_ERROR_CANCELED,

        //! 列挙子の最後です。
        ERR_LAST               = ERR_CANCELED
    };

    //! @name コンストラクタ / デストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief      コンストラクタです。
    //!
    //! @details
    //! ファイルをオープンしない状態でオブジェクトが生成されます。
    //---------------------------------------------------------------------------
    /* ctor */              McsFileStream();

    //---------------------------------------------------------------------------
    //! @brief      コンストラクタです。
    //!
    //! @param[in]  fileName    オープンするファイルのファイル名を指定します。
    //! @param[in]  openFlag    ファイルオープン属性を指定します。\n
    //!                         (mcs::FILEIO_FLAG_MODE | mcs::FILEIO_FLAG_OPTION)
    //!
    //! @details
    //! ファイル名とオープン属性を指定してファイルを開きます。
    //---------------------------------------------------------------------------
    /* ctor */              McsFileStream( const char* fileName, u32 openFlag );

    //---------------------------------------------------------------------------
    //! @brief      コンストラクタです。
    //!
    //! @param[in]  openedFile      既にオープン済みのファイル情報を指定します。
    //! @param[in]  openFlag        ファイルを開いた時の属性と同じものを渡します。\n
    //!                             (mcs::FILEIO_FLAG_MODE | mcs::FILEIO_FLAG_OPTION)
    //! @param[in]  closeEnable     ファイルのクローズを有効にするかどうかを指定します。
    //!
    //! @details
    //! 既にオープン済みのファイル情報を元にファイルストリームを生成します。
    //!
    //! 引数 closeEnable に false を指定した場合、ファイルのクローズは無効化され、
    //! Close() を明示的に呼び出してもファイルはクローズされません。
    //! また、デストラクタでもクローズされません。
    //!
    //! この場合、 FileIO_Close() でファイルをクローズしてください。
    //!
    //! @sa ~McsFileStream
    //---------------------------------------------------------------------------
    /* ctor */              McsFileStream( const FileInfo* openedFile, u32 openFlag, bool closeEnable = true );

    //---------------------------------------------------------------------------
    //! @brief      デストラクタです。
    //!
    //! @details
    //! ファイルが開かれていた場合にはクローズします。
    //!
    //! ただし、オープン済みの FileInfo を元にファイルストリームをオープンした場合に、
    //! 引数 closeEnable に false を指定してファイルのクローズを無効化した場合には、
    //! ファイルのクローズは行いません。
    //!
    //! @sa McsFileStream::McsFileStream
    //! @sa Open
    //---------------------------------------------------------------------------
    /* dtor */ virtual      ~McsFileStream();

    //@}

    //! @name ファイルストリームのオープン / クローズ
    //@{

    //---------------------------------------------------------------------------
    //! @brief      ファイルをオープンします。
    //!
    //! @param[in]  fileName    オープンするファイルのファイル名を指定します。
    //! @param[in]  openFlag    ファイルオープン属性を指定します。\n
    //!                         (mcs::FILEIO_FLAG_MODE | mcs::FILEIO_FLAG_OPTION)
    //!
    //! @details
    //! ファイル名とオープン属性を指定してファイルを開きます。
    //!
    //! @sa Close
    //! @sa ~McsFileStream
    //---------------------------------------------------------------------------
    bool                    Open( const char* fileName, u32 openFlag );

    //---------------------------------------------------------------------------
    //! @brief      ファイルをオープンします。
    //!
    //! @param[in]  openedFile  既にオープン済みのファイル情報を指定します。
    //! @param[in]  openFlag    ファイルを開いた時の属性と同じものを渡します。\n
    //!                         (mcs::FILEIO_FLAG_MODE | mcs::FILEIO_FLAG_OPTION)
    //! @param[in]  closeEnable ファイルのクローズを有効にするかどうかを指定します。
    //!
    //! @return     ファイルのオープンに成功した場合にはtrue
    //!             失敗した場合にはfalseが返されます。
    //!
    //! @details
    //! 既にオープン済みのファイル情報を元にファイルストリームを生成します。
    //!
    //! 引数 closeEnable に false を指定した場合、ファイルのクローズは無効化され、
    //! Close() を明示的に呼び出してもファイルはクローズされません。
    //! また、デストラクタでもクローズされません。
    //!
    //! この場合、 FileIO_Close() でファイルをクローズしてください。
    //!
    //! @sa Close
    //! @sa ~McsFileStream
    //---------------------------------------------------------------------------
    bool                    Open( const FileInfo* openedFile, u32 openFlag, bool closeEnable = true );

    //---------------------------------------------------------------------------
    //! @brief      ファイルストリームをクローズします。
    //!
    //! @details
    //! ファイルが開かれていた場合にはクローズします。
    //!
    //! ただし、オープン済みの FileInfo を元にファイルストリームをオープンした場合に、
    //! 引数 closeEnable に false を指定してファイルのクローズを無効化した場合には、
    //! ファイルのクローズは行いません。
    //!
    //! @sa McsFileStream::McsFileStream
    //! @sa Open
    //---------------------------------------------------------------------------
    virtual void            Close( void );

    //@}

    //! @name ファイルストリームの操作
    //@{

    //---------------------------------------------------------------------------
    //! @brief      同期リードします。
    //!
    //! @param[in]  buf     読み込みバッファへのポインタ。\n
    //!                     GetBufferAlign() で取得されるアライメントに合っている必要があります。
    //! @param[in]  length  読み込むバイト数。\n
    //!                     GetSizeAlign() で取得されるアライメントに合っている必要があります。
    //!
    //! @return     正常に読み込めた場合には 読み込んだデータサイズを返します。
    //!             エラーの場合には負の数(McsFileStream::Result)を返します。
    //!
    //! @details
    //! ストリームからバッファへデータを読み出します。
    //! CanRead() の返り値が false になる場合にはアサートで失敗します。
    //!
    //! この関数は読み出しが完了するまでの間、処理をブロックします。
    //!
    //! McsFileStream は非同期処理を行うことができませんので、
    //! 非同期版関数である ReadAsync() を呼び出さないでください。
    //!
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::McsFileStream::Result)
    //!
    //! ''ERR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''ERR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! ''ERR_PROTOCOLERROR''\n
    //! Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''ERR_INVALIDFILEHANDLE''\n
    //! Windows 側が認識できないファイルのハンドルを操作しようとしています。
    //!
    //! ''ERR_IOERROR''\n
    //! その他の I/O エラーが発生しました。
    //!
    //! @sa CanRead
    //! @sa GetBufferAlign
    //! @sa GetSizeAlign
    //! @sa ReadAsync
    //---------------------------------------------------------------------------
    virtual s32             Read( void* buf, u32 length );

    //---------------------------------------------------------------------------
    //! @brief      同期ライトします。
    //!
    //! @param[in]  buf     書き込み元バッファへのポインタ。\n
    //!                     GetBufferAlign() で取得されるアライメントに合っている必要があります。
    //! @param[in]  length  書き込みデータのバイト数。\n
    //!                     GetSizeAlign() で取得されるアライメントに合っている必要があります。
    //!
    //! @return     書き込みに成功すれば、実際に書き込まれたバイト数が返ります。
    //!             エラーの場合には負の数(McsFileStream::Result)を返します。
    //!
    //! @details
    //! ストリームにデータを書き込みます。
    //! CanWrite() の返り値が false になる場合にはアサートで失敗します。
    //!
    //! この関数は書き込みが完了するまでの間、処理をブロックします。
    //!
    //! McsFileStream は非同期処理を行うことができませんので、
    //! 非同期版関数である WriteAsync() を呼び出さないでください。
    //!
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::McsFileStream::Result)
    //!
    //! ''ERR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''ERR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! ''ERR_PROTOCOLERROR''\n
    //! Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''ERR_INVALIDFILEHANDLE''\n
    //! Windows 側が認識できないファイルのハンドルを操作しようとしています。
    //!
    //! ''ERR_IOERROR''\n
    //! その他の I/O エラーが発生しました。
    //!
    //! @sa CanWrite
    //! @sa GetBufferAlign
    //! @sa GetSizeAlign
    //! @sa WriteAsync
    //---------------------------------------------------------------------------
    virtual s32             Write( const void* buf, u32 length );

    //@}

    //! @name ファイルポインタの操作
    //@{

    //---------------------------------------------------------------------------
    //! @brief      ファイルポインタの位置を移動します。
    //!
    //! @param[in]  offset  ファイルポジションを移動するバイト数を指定します。origin で指定された位置が基点となります。
    //! @param[in]  origin  ファイルポジションを移動するための基点を指定します。
    //!
    //! @return     成功した場合には true を返します。
    //!
    //! @details
    //! origin を基点として offset バイト分だけファイルポジションを移動します。
    //!
    //! 移動先のファイルポジションは GetOffsetAlign() で取得できるサイズのアライメントにあっている必要があります。
    //!
    //! 引数 origin には基点として以下の値をとることが可能です。
    //!
    //! ''FILE_STREAM_SEEK_BEGIN''\n
    //! ファイルの先頭を基点とします。
    //!
    //! ''FILE_STREAM_SEEK_CURRENT''\n
    //! 現在のファイルポジションを基点とします。
    //!
    //! ''FILE_STREAM_SEEK_END''\n
    //! ファイルの終端位置を基点とします(offset <= 0で指定します)。
    //!
    //! @sa CanSeek
    //---------------------------------------------------------------------------
    virtual bool            Seek( s32 offset, u32 origin );

    //---------------------------------------------------------------------------
    //! @brief      現在のファイルポインタの位置を取得します。
    //!
    //! @return     現在のファイルポジションを返します。
    //!
    //! @details
    //! この値は Seek() で引数 offset として利用できます。
    //!
    //! @sa Seek
    //---------------------------------------------------------------------------
    virtual u32             Tell()    const { return m_FilePosition.Tell(); }

    //@}

    //! @name ファイルの状態取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief      ファイルサイズを取得します。
    //!
    //! @return     オープン中のファイルのファイルサイズを返します。
    //!             ファイルをオープンしていない場合には 0 が返されます。
    //---------------------------------------------------------------------------
    virtual u32             GetSize() const { return m_FilePosition.GetFileSize(); }

    //---------------------------------------------------------------------------
    //! @brief      ファイルストリームがオープンされているかどうかを取得します。
    //!
    //! @return     このストリームが現在使用可能な状態であれば true を返します。
    //!             ファイルがオープンされていない、既にクローズされた等の理由で使用不能で
    //!             あった場合には false を返します。
    //---------------------------------------------------------------------------
    virtual bool            IsAvailable()    const { return m_Available; }

    //@}

    //! @name ファイルストリームの属性情報を取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief      非同期処理の可否を取得します。
    //!
    //! @return     McsFileStream は非同期処理が不可能であるため、必ず false が返ります。
    //---------------------------------------------------------------------------
    virtual bool            CanAsync( void )  const { return false;     }

    //---------------------------------------------------------------------------
    //! @brief      ファイルポインタをランダムに移動できるかどうかの属性を取得します。
    //!
    //! @return     McsFileStream はファイルポジションの移動を行う
    //!             Seek() の使用が常に可能であるため、必ず true が返ります。
    //!
    //! @sa Seek
    //---------------------------------------------------------------------------
    virtual bool            CanSeek ( void )  const { return true;      }

    //---------------------------------------------------------------------------
    //! @brief      このストリームがリード可能かどうかを取得します。
    //!
    //! @return     このストリームが読み込み可能である場合には true を、
    //!             読み込みができない場合には false を返します。
    //!
    //! @sa Read
    //---------------------------------------------------------------------------
    virtual bool            CanRead ( void )  const { return m_CanRead;  }

    //---------------------------------------------------------------------------
    //! @brief      このストリームがライト可能かどうかを取得します。
    //!
    //! @return     このストリームが書き込み可能である場合には true を、
    //!             書き込みできない場合には false を返します。
    //!
    //! @sa Write
    //---------------------------------------------------------------------------
    virtual bool            CanWrite( void )  const { return m_CanWrite; }

    //---------------------------------------------------------------------------
    //! @brief      このストリームの非同期処理がキャンセル可能かどうかを取得します。
    //!
    //! @return     McsFileStream は非同期処理が不可能であるため、必ず false が返ります。
    //---------------------------------------------------------------------------
    virtual bool            CanCancel( void ) const { return false;     }

    //@}

// ------ private members --------------------------
private:
    //---------------------------------------------------------------------------
    //! @brief      初期化
    //---------------------------------------------------------------------------
    void                    Initialize_( void );

    bool m_Available;

    FileInfo                m_FileInfo;          // The file information structure
    bool                    m_CanRead;           // Can read flag
    bool                    m_CanWrite;          // Can write flag

    FilePosition            m_FilePosition;      // File read position management
    bool                    m_CloseOnDestroyFlg; // flag indicating whether the file is to be closed when the destructor executes
    bool                    m_CloseEnableFlg;    // Flag indicating whether a file close is permitted in the FileStream

#else   // #if defined(NW_MCS_ENABLE)

// ------------- public members -------------------
public:
    NW_UT_RUNTIME_TYPEINFO( ut::FileStream )

    // Constructors/destructors
    /* ctor */              McsFileStream()                                                 {}
    /* ctor*/              McsFileStream( const char* /* fileName*/, u32 /* openFlag */)  {}
    /* ctor*/              McsFileStream( const FileInfo* /* openedFile*/, u32 /* openFlag */ )  {}
    /* ctor*/              McsFileStream( const FileInfo* /* openedFile*/, u32 /* openFlag*/, bool /* closeEnable */ )  {}
    /* dtor */ virtual      ~McsFileStream()                                                {}

    // Open the file
    bool                    Open( const char* /* fileName*/, u32 /* openFlag */ )        { return false; }
    bool                    Open( const FileInfo* /* openedFile*/, u32 /* openFlag */ )    { return false; }

    // Close the file
    virtual void            Close( void )                                   {}

    virtual s32             Read( void* /* buf*/, u32 /* length */)        { return 0; }
    virtual s32             Write( const void* /* buf*/, u32 /* length */) { return 0; }

    virtual bool            Seek(s32 /* offset*/, u32 /* origin */)     { return false; }
    virtual u32             Tell()    const                             { return 0; }
    virtual u32             GetSize() const                             { return 0; }

    virtual bool            CanAsync( void ) const  { return false; }
    virtual bool            CanSeek ( void ) const  { return false; }
    virtual bool            CanRead ( void ) const  { return false; }
    virtual bool            CanWrite( void ) const  { return false; }
    virtual bool            CanCancel( void ) const { return false; }

#endif  // #if defined(NW_MCS_ENABLE)
};



}   /* namespace mcs    */
}   /* namespace nw   */



/* NW_MCS_MCS_FILE_STREAM_H_ */
#endif
