﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MCS_HIOSHAREDBUFFER_H_
#define NW_MCS_HIOSHAREDBUFFER_H_

#include <nw/types.h>
#include <nw/ut/ut_Inlines.h>

//================================================================================
// Definitions of items such as compile options
// #define ENABLE_MCS_REPORT // Debug Print On
//================================================================================

namespace nw     {
namespace mcs    {
namespace internal {

enum
{
    HIO_OK    = 0,
    HIO_ERROR = -1,
};


//---------------------------------------------------------------------------
//! @brief     HIO でやりとりするデータのバッファ
//---------------------------------------------------------------------------
class HioSharedReadBuffer
{
public:
    /* ctor */ HioSharedReadBuffer() {}

    void Initialize(void* buffer, u32 bufferSize)
    {
        m_Buffer = buffer;
        m_BufferSize = bufferSize;
        m_CurrentTop  = 0;
        m_CurrentTail = 0;
    }

    u32 GetReadableSize() const { return m_CurrentTail - m_CurrentTop; }
    u32 GetRemindSize() const { return m_BufferSize - m_CurrentTail; }

    const void* GetReadableData()
    {
        return ut::AddOffsetToPtr(m_Buffer, m_CurrentTop);
    }

    const void* HeadPointer() const
    {
        return ut::AddOffsetToPtr(m_Buffer, m_CurrentTop);
    }

    void* TailPointer()
    {
        return ut::AddOffsetToPtr(m_Buffer, m_CurrentTail);
    }

    void MovePtr(u32 size)
    {
        m_CurrentTop += size;

        NW_ASSERT(m_CurrentTop <= m_CurrentTail);
    }

    void MoveTail(u32 size)
    {
        m_CurrentTail += size;

        NW_ASSERT(m_CurrentTail <= m_BufferSize);
    }

    //-------------------------------------------------------------------------
    //! @brief データをバッファの先頭に詰めます。
    //-------------------------------------------------------------------------
    void Compaction()
    {
        if (m_CurrentTop != m_CurrentTail)
        {
            memmove(
                m_Buffer,
                reinterpret_cast<u8*>(m_Buffer) + m_CurrentTop,
                m_CurrentTail - m_CurrentTop);
        }

        m_CurrentTail -= m_CurrentTop;
        m_CurrentTop = 0;
    }

    void Clear()
    {
        m_CurrentTop = 0;
        m_CurrentTail = 0;
    }

private:
    void*   m_Buffer;
    u32     m_BufferSize;
    u32     m_CurrentTop;
    u32     m_CurrentTail;

    void*   CurrentPointer()
    {
        return ut::AddOffsetToPtr(m_Buffer, m_CurrentTop);
    }
};


class HioSharedWriteBuffer
{
public:
    /* ctor */ HioSharedWriteBuffer() {}

    void Initialize(void* buffer, u32 bufferSize)
    {
        m_Buffer = buffer;
        m_BufferSize = bufferSize;
        m_CurrentTop = 0;
        m_CurrentTail = 0;
    }

    u32 GetWritableSize() const { return m_BufferSize - m_CurrentTail; }

    s32 Write(const void* buffer, u32 size)
    {
        if (size > this->GetWritableSize())
        {
            return HIO_ERROR;
        }

        memcpy(this->CurrentPointer(), buffer, size);

        m_CurrentTail += size;

        return size;
    }

    bool IsEmpty() const
    {
        return (m_CurrentTop == m_CurrentTail);
    }

    void Clear()
    {
        m_CurrentTop = 0;
        m_CurrentTail = 0;
    }

    void MovePtr(u32 size)
    {
        m_CurrentTop += size;

        NW_ASSERT(m_CurrentTop <= m_CurrentTail);
    }

    const void* HeadPointer() const
    {
        return ut::AddOffsetToPtr(m_Buffer, m_CurrentTop);
    }

    u32 GetWrittenSize() const
    {
        return m_CurrentTail - m_CurrentTop;
    }

private:
    void*   m_Buffer;
    u32     m_BufferSize;
    u32     m_CurrentTop;
    u32     m_CurrentTail;

    void*   CurrentPointer()
    {
        return ut::AddOffsetToPtr(m_Buffer, m_CurrentTail);
    }
};



}   // namespace internal
}   // namespace mcs
}   // namespace nw

#endif /* NW_MCS_HIOSHAREDBUFFER_H_ */

