﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MCS_FILEIO_H_
#define NW_MCS_FILEIO_H_

#if defined(__CWCC__)
#pragma once
#endif

#include <nw/mcs/mcs_Base.h>
#include <nw/dev/dev_Fs.h>

#if defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
#include <winext/cafe/fs.h>
#endif

#define NW_MCS_FILEIO_USE_FS

//
// Use Mcs_XXX (mcs_base.cpp) as a submodule.
// The McsFileStream class is a wrapper that uses this module internally.
//
//
namespace nw
{
namespace mcs
{

namespace internal
{

//----------------------------------------------------------------------
// Size of the command buffer for controlling communications:
// The size must be set so that commands will fit into it.
// The size must be unexpectedly large because data such as the file path will be allocated.
const int           FILEIO_COMMAND_SIZE_MAX = 2 * 1024;

// The data chunk size to send at one time.
// The size is configured as appropriate from the internal shared memory size of the HIO communication device.
const int           FILEIO_CHUNK_SIZE_MAX   = 2 * 1024;

//----------------------------------------------------------------------
//
// The work region used by FileIO.
//
struct  FileIOWork
{
#if defined(NW_MCS_ENABLE)

    // Command buffer for controlling communications
    u8          tempBuf[FILEIO_COMMAND_SIZE_MAX];

    // Data buffer
    // Size (* 32) of the buffer that will temporarily accumulate data before reads and writes
    u8          commBuf[FILEIO_CHUNK_SIZE_MAX * 32];

#endif  // #if defined(NW_MCS_ENABLE)
};

}   // namespace internal

//! @name FileIO
//! @{

//! @brief  FileIO が使用するワーク領域のサイズです。
//!
//! @sa FileIO_RegisterBuffer
const int          FILEIO_WORKBUFFER_SIZE   = sizeof(internal::FileIOWork);

//! @brief  ファイル名の最大長です。
//!
//! @details
//!（終端文字を含みます）
const int           FILEIO_PATH_MAX     = 260;



//----------------------------------------------------------------------
//! FileIO 関数の返り値に使用される列挙子です。
//----------------------------------------------------------------------
enum FILEIO_Result
{
    //! 関数は成功しました。
    FILEIO_ERROR_SUCCESS        = 0,
    //! 通信エラーが発生しました。
    FILEIO_ERROR_COMERROR,
    //! Mcs サーバと接続されていません。
    FILEIO_ERROR_NOTCONNECT,

    //! Mcs サーバはリクエストの実行を継続中です。
    FILEIO_ERROR_CONTINUE       = 100,
    //! Mcs ライブラリと Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    FILEIO_ERROR_PROTOCOLERROR,
    //! ファイルが見つかりません。
    FILEIO_ERROR_FILENOTFOUND,
    //! 指定されたファイル名に含まれるパスが無効です。
    FILEIO_ERROR_DIRECTORYNOTFOUND,
    //! 指定されたファイルは既に存在します。
    FILEIO_ERROR_FILEALREADYEXIST,
    //! 呼び出し元に、必要なアクセス許可がありません。
    FILEIO_ERROR_SECURITYERROR,
    //! 認証されないアクセスです。
    FILEIO_ERROR_UNAUTHORIZEDACCESS,
    //! その他のIOエラーが発生しました。
    FILEIO_ERROR_IOERROR,
    //! Windows 側が認識できないファイルのハンドルを操作しようとしています。
    FILEIO_ERROR_INVALIDFILEHANDLE,
    //! パターンに一致するファイルが見つかりません。
    FILEIO_ERROR_NOMOREFILES,
    //! ファイル選択はキャンセルされました。
    FILEIO_ERROR_CANCELED,
    //! FileIO が初期化されていません。
    FILEIO_ERROR_NOT_INITIALIZED,
    //! 不明なエラーです。
    FILEIO_ERROR_UNKNOWN,

    //! 列挙子の最後です。
    FILEIO_ERROR_LAST = FILEIO_ERROR_CANCELED
};

//----------------------------------------------------------------------
//! @brief ファイルを開くモードを指定する列挙子です。
//!
//! @sa FILEIO_FLAG_OPTION
//! @sa FileIO_Open
//! @sa McsFileStream::Open
//----------------------------------------------------------------------
enum FILEIO_FLAG_MODE
{
    //! @brief
    //! 読み込み用に既存のファイルを開きます。
    FILEIO_FLAG_READ        = 1 << 0,

    //! @brief
    //! 書き込み用に新規にファイルを作成してオープンします。\n
    //! 既にファイルが存在する場合は一度消去されます。
    FILEIO_FLAG_WRITE       = 1 << 1,

    //! @brief
    //! 読み書き用にファイルをオープンします。\n
    //! ファイルが存在しない場合は作成されます。
    FILEIO_FLAG_READWRITE   = FILEIO_FLAG_READ | FILEIO_FLAG_WRITE
};

//----------------------------------------------------------------------
//! @brief ファイルを開くオプションを指定する列挙子です。
//!
//! @details
//! mcs::FILEIO_FLAG_MODE に論理和してファイルを開く際のオプションを指定します。
//!
//! @sa FILEIO_FLAG_MODE
//! @sa FileIO_Open
//! @sa McsFileStream::Open
//----------------------------------------------------------------------
enum FILEIO_FLAG_OPTION
{
    //! @brief
    //! openFlag に FILEIO_FLAG_WRITE を指定していて、既にファイルが存在する場合にはエラーとします。\n
    //! 返り値は FILEIO_ERROR_FILEALREADYEXIST になります。
    FILEIO_FLAG_NOOVERWRITE = 1 << 2,

    //! @brief
    //! openFlag に FILEIO_FLAG_READWRITE を指定していて、ファイルが存在しない場合にはエラーとします。\n
    //! 返り値は FILEIO_ERROR_FILENOTFOUND になります。
    FILEIO_FLAG_NOCREATE    = 1 << 3,

    //! @brief
    //! ファイル名文字列の中にパソコンの環境変数文字列を含む場合にこのフラグを指定します。\n
    //! 環境変数は「%%NW4F_ROOT%\\SourceTreeMap.txt」のように「%%」で囲みます。
    FILEIO_FLAG_INCENVVAR   = 1 << 4,

    //! @brief
    //! openFlag に FILEIO_FLAG_WRITE や FILEIO_FLAG_READWRITE を指定してファイルを作成する際に、
    //! ファイルが置かれるディレクトリが存在しない場合にディレクトリを作成します。\n
    //! ただし、FILEIO_FLAG_INCENVVAR が指定されている場合で、指定したファイル名文字列に含まれる
    //! パソコンの環境変数が実際には存在しない場合は、安全のためにディレクトリの作成は行わず、
    //! 関数は失敗します。返り値は FILEIO_ERROR_DIRECTORYNOTFOUND になります。
    FILEIO_FLAG_CREATEDIR   = 1 << 5,

    //! ファイルの共有読み込みを可能にします。
    FILEIO_FLAG_SHAREREAD   = 1 << 16,

    //! ファイルの共有書き込みを可能にします。
    FILEIO_FLAG_SHAREWRITE  = 1 << 17
};

//----------------------------------------------------------------------
//! @brief シークの開始点を指定する列挙子です。
//!
//! @sa FileIO_Seek
//----------------------------------------------------------------------
enum FILEIO_SEEK_MODE
{
    //! ファイルの先頭を開始点とします。
    FILEIO_SEEK_BEGIN,
    //! 現在のファイルポインタの位置を開始点とします。
    FILEIO_SEEK_CURRENT,
    //! ファイルの終端の位置を開始点とします。
    FILEIO_SEEK_END
};

//----------------------------------------------------------------------
//! @brief ファイル属性です。
//!
//! @sa FileIO_FindFirst
//----------------------------------------------------------------------
enum FILEIO_ATTRIBUTE
{
    //! 読み取り専用。
    FILEIO_ATTRIBUTE_READONLY             = 0x00000001,
    //! 隠しファイル。
    FILEIO_ATTRIBUTE_HIDDEN               = 0x00000002,
    //! システムファイル。
    FILEIO_ATTRIBUTE_SYSTEM               = 0x00000004,
    //! ディレクトリ。
    FILEIO_ATTRIBUTE_DIRECTORY            = 0x00000010,
    //! アーカイブ。
    FILEIO_ATTRIBUTE_ARCHIVE              = 0x00000020,
    //! (予約)
    FILEIO_ATTRIBUTE_DEVICE               = 0x00000040,
    //! 属性が無いファイル。
    FILEIO_ATTRIBUTE_NORMAL               = 0x00000080,
    //! (予約)
    FILEIO_ATTRIBUTE_TEMPORARY            = 0x00000100,
    //! (予約)
    FILEIO_ATTRIBUTE_SPARSE_FILE          = 0x00000200,
    //! (予約)
    FILEIO_ATTRIBUTE_REPARSE_POINT        = 0x00000400,
    //! (予約)
    FILEIO_ATTRIBUTE_COMPRESSED           = 0x00000800,
    //! (予約)
    FILEIO_ATTRIBUTE_OFFLINE              = 0x00001000,
    //! (予約)
    FILEIO_ATTRIBUTE_NOT_CONTENT_INDEXED  = 0x00002000,
    //! (予約)
    FILEIO_ATTRIBUTE_ENCRYPTED            = 0x00004000
};

//----------------------------------------------------------------------
//! @brief ファイルダイアログのモードを表す列挙子です。
//!
//! @sa FileIO_ShowFileDialog
//! @sa FILEIO_FILEDIALOG_OPTION
//----------------------------------------------------------------------
enum FILEIO_FILEDIALOG_MODE
{
    //! ファイルを開くためのファイルダイアログを表示します。
    FILEIO_FILEDIALOG_OPEN                  = 0,
    //! ファイルを保存するためのファイルダイアログを表示します。
    FILEIO_FILEDIALOG_SAVE                  = 0x40000000
};

//----------------------------------------------------------------------
//! @brief ファイルダイアログのオプションを表す列挙子です。
//!
//! @details
//! mcs::FILEIO_FILEDIALOG_MODE と論理和することでオプションを指定します。
//!
//! @sa FileIO_ShowFileDialog
//! @sa FILEIO_FILEDIALOG_MODE
//----------------------------------------------------------------------
enum FILEIO_FILEDIALOG_OPTION
{
    //! @brief
    //! ファイルを保存するためのファイルダイアログを表示する際、
    //! このフラグを指定していた場合はユーザが既に存在するファイル名を指定したときに
    //! 警告を表示しません。\n
    //! このフラグを指定しない場合は上書き確認の警告を表示します。\n
    //! FILEIO_FILEDIALOG_SAVE が指定されている場合のみ有効です。
    FILEIO_FILEDIALOG_NOOVERWRITEPROMPT     = 0x00000002,

    //! @brief
    //! ファイルを保存するためのファイルダイアログを表示する際、
    //! このフラグを指定していた場合はユーザが新規にファイルを作成しようとしている場合に
    //! 警告を表示します。\n
    //! FILEIO_FILEDIALOG_SAVE が指定されている場合のみ有効です。
    FILEIO_FILEDIALOG_CREATEPROMPT          = 0x00002000,

    //! @brief
    //! このフラグを指定した場合は、引数 initialPath をファイルへパスとして扱います。\n
    //! このフラグを指定しない場合は、引数 initialPath をディレクトリへのパスとして扱います。
    FILEIO_FILEDIALOG_INCLUDEFILENAME       = 0x80000000
};

/* ========================================================================
    Type Definitions
   ======================================================================== */


//---------------------------------------------------------------------------
//! @brief ファイル検索情報構造体です。
//!
//! @details :category     FileIO
//---------------------------------------------------------------------------
struct FindData
{
    //! ファイルの属性です。\n(mcs::FILEIO_ATTRIBUTE の論理和)
    u32                 attribute;
    //! ファイルのサイズ(バイト数)です。
    u32                 size;
    //! ファイルの 1970年1月1日からの経過時間(秒単位)です。
    u32                 date;
    //! (予約)
    u32                 reserved1;
    //! (予約)
    u32                 reserved2;
    //! ファイル名です。
    char                name[FILEIO_PATH_MAX];
};


//---------------------------------------------------------------------------
//! @brief ファイル情報構造体です。
//!
//! @details :category     FileIO
//---------------------------------------------------------------------------
struct FileInfo
{
public:
#if defined(NW_MCS_ENABLE)
  #if !defined( NW_MCS_FILEIO_USE_FS )
    u32                 handle;    //!< @details :private
  #else
    #if defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
private:
    typedef nw::internal::winext::FSFileHandle FSFileHandle;
public:
    #endif
    FSFileHandle       handle;    //!< @details :private
  #endif
    u32                 fileSize;  //!< @details :private
    u32                 errorCode; //!< @details :private
#endif

    //---------------------------------------------------------------------------
    //! @brief      既存のファイルをオープンあるいは新規のファイルを作成してオープンします。
    //!
    //! @param[in]  fileName    作成または開くファイルを指す、'\\0'で終わる文字列へのポインタを指定します。
    //! @param[in]  openFlag    ファイルの読み書きの種類および作成、オープン方法の指定
    //!                         (mcs::FILEIO_FLAG_MODE | mcs::FILEIO_FLAG_OPTION)。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::FILEIO_Result)を返します。
    //!
    //! @details
    //! openFlag で、ファイルの読み書きの種類、および作成とオープンの方法を指定します。\n
    //! (mcs::FILEIO_FLAG_MODE と mcs::FILEIO_FLAG_OPTION 列挙子の論理和で指定します。)
    //!
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //! Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_FILENOTFOUND''\n
    //! ファイルが見つかりません。openFlag に FILEIO_FLAG_READ が指定されている場合、
    //! および FILEIO_FLAG_READWRITE と FILEIO_FLAG_NOCREATE が指定されている場合に
    //! ファイルが存在しないと、このエラーになります。
    //!
    //! ''FILEIO_ERROR_DIRECTORYNOTFOUND''\n
    //! 指定されたファイル名に含まれるパスが無効です。
    //!
    //! ''FILEIO_ERROR_FILEALREADYEXIST''\n
    //! ファイルが既に存在します。openFlag に FILEIO_FLAG_WRITE と FILEIO_FLAG_NOOVERWRITE が
    //! 指定されていると、このエラーになります。
    //!
    //! ''FILEIO_ERROR_SECURITYERROR''\n
    //! 呼び出し元に、必要なアクセス許可がありません。
    //!
    //! ''FILEIO_ERROR_UNAUTHORIZEDACCESS''\n
    //! ファイルに対するアクセス要求がオペレーティングシステムで許可されていません。
    //! 例えば、openFlag で FILEIO_FLAG_WRITE や FILEIO_FLAG_READWRITE を指定しているのに、
    //! オープンするファイルやファイルが置かれているディレクトリが読み込み専用に設定されている場合は
    //! このエラーになります。
    //!
    //! ''FILEIO_ERROR_IOERROR''\n
    //! その他の I/O エラーが発生しました。
    //!
    //! @sa FileIO_Read
    //! @sa FileIO_Write
    //! @sa FileIO_Close
    //! @sa FileIO_ShowFileDialog
    //---------------------------------------------------------------------------
    NW_INLINE u32 Open(const char* fileName, u32 openFlag);

    //---------------------------------------------------------------------------
    //! @brief      ファイルをクローズします。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::FILEIO_Result)を返します。
    //!
    //! @details
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //! Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_INVALIDFILEHANDLE''\n
    //! Windows 側が認識できないファイルのハンドルを操作しようとしています。
    //! pFileInfo で正しくファイル情報構造体のポインタを渡しているか確認してください。
    //!
    //! ''FILEIO_ERROR_IOERROR''\n
    //! その他の I/O エラーが発生しました。
    //!
    //! @sa FileIO_Open
    //---------------------------------------------------------------------------
    NW_INLINE u32 Close();

    //---------------------------------------------------------------------------
    //! @brief      ファイルから読み込みます。
    //!
    //! @param[in]  buffer      バッファへのポインタ。
    //!                         読み取りが成功した場合、読み込んだ内容が格納されます。
    //! @param[in]  length      読み込むバイト数。
    //! @param[out] pReadBytes  実際に読み込んだバイト数を返す変数へのポインタ。
    //!                         読み取りが成功した場合に、実際に読み込んだバイト数が格納されます。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::FILEIO_Result)を返します。
    //!
    //! @details
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //! Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_INVALIDFILEHANDLE''\n
    //! Windows 側が認識できないファイルのハンドルを操作しようとしています。
    //! pFileInfo で正しくファイル情報構造体のポインタを渡しているか確認してください。
    //!
    //! ''FILEIO_ERROR_IOERROR''\n
    //! その他の I/O エラーが発生しました。
    //!
    //! @sa FileIO_Open
    //! @sa FileIO_Write
    //! @sa FileIO_Close
    //---------------------------------------------------------------------------
    NW_INLINE u32 Read(void* buffer, u32 length, u32* pReadBytes);

    //---------------------------------------------------------------------------
    //! @brief      ファイルへ書き込みます。
    //!
    //! @param[in]  buffer      書き込む内容を保持するバッファへのポインタ。
    //! @param[in]  length      書き込むバイト数。
    //!
    //! @return      関数が成功した場合0を返します。
    //!               失敗した場合エラーコード(mcs::FILEIO_Result)を返します。
    //!
    //! @details
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //! Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_INVALIDFILEHANDLE''\n
    //! Windows 側が認識できないファイルのハンドルを操作しようとしています。
    //! pFileInfo で正しくファイル情報構造体のポインタを渡しているか確認してください。
    //!
    //! ''FILEIO_ERROR_IOERROR''\n
    //! その他の I/O エラーが発生しました。
    //!
    //! @sa FileIO_Open
    //! @sa FileIO_Read
    //! @sa FileIO_Close
    //---------------------------------------------------------------------------
    NW_INLINE u32 Write(const void* buffer, u32 length);

    //---------------------------------------------------------------------------
    //! @brief      指定された位置にファイルポインタを移動します。
    //!
    //! @param[in]  distanceToMove  ファイルポインタを移動するバイト数。
    //! @param[in]  moveMethod      ファイルポインタの移動を開始する位置(mcs::FILEIO_SEEK_MODE)。
    //! @param[out] pNewFilePointer 新しいファイルポインタを受け取る変数へのポインタ、または 0。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::FILEIO_Result)を返します。
    //!
    //! @details
    //! 引数 pNewFilePointer で指定したポインタが指す変数に移動後のファイルポインタの位置がセットされます。
    //! この値が不要な場合は 0 をセットしてください。
    //!
    //! distanceToMove に 0、moveMethod に mcs::FILEIO_SEEK_CURRENT を指定して
    //! 現在のファイルポインタの位置を取得することもできます。
    //!
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //!   通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //!   Mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //!   Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_INVALIDFILEHANDLE''\n
    //!   Windows 側が認識できないファイルのハンドルを操作しようとしています。
    //!   pFileInfo で正しくファイル情報構造体のポインタを渡しているか確認してください。
    //!
    //! ''FILEIO_ERROR_IOERROR''\n
    //!   その他の I/O エラーが発生しました。
    //!
    //! @sa FileIO_Open
    //! @sa FileIO_Read
    //! @sa FileIO_Write
    //---------------------------------------------------------------------------
    NW_INLINE u32 Seek(s32 distanceToMove, u32 moveMethod, u32* pNewFilePointer = 0);

    //---------------------------------------------------------------------------
    //! @brief      ファイルサイズを取得します。
    //!
    //! @return     ファイルサイズを返します。
    //!
    //! @details
    //! Windows 上のファイルのオープンした時点(FileIO_Open()を呼び出した時点)での
    //! ファイルのサイズ(バイト数)を取得します。
    //!
    //! @sa FileIO_Open
    //---------------------------------------------------------------------------
    NW_INLINE u32 GetOpenFileSize() const;

    //---------------------------------------------------------------------------
    //! @brief        FileInfo への処理で最後に発生したエラーコードを取得します。
    //!
    //! @return       最後に発生したエラーを返します。
    //---------------------------------------------------------------------------
    NW_INLINE u32 GetLastErrorCode();

    //---------------------------------------------------------------------------
    //! @brief      指定されたファイル名に一致するファイルをディレクトリ内で
    //!             検索します。
    //!
    //! @param[out] pFindData   見つかったファイルに関する情報を格納する構造体(FindData)へのポインタ。
    //! @param[in]  pattern     検索するファイル名パターンの文字列。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::FILEIO_Result)を返します。
    //!
    //! @details
    //! 見つかった場合、最初に見つかったファイルに関する情報を pFindData で
    //! 指定した FindData 構造体の変数に格納します。
    //!
    //! pFileInfo で指定した FileInfo 構造体の変数に検索のステートが格納されます。
    //!
    //! 残りのファイルを検索するには FileIO_FindNext() を呼び出します。
    //!
    //! 検索を終了するときには FileIO_FindClose() を呼び出してください。
    //!
    //! ファイル名パターンには、「*」または「?」のワイルドカード文字を含めることができます。
    //! また、「%%」で囲むことによって「%%NW4F_ROOT%%\\include\\*.h」のように Windows の環境変数を
    //! 含めることができます。
    //!
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //! Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_DIRECTORYNOTFOUND''\n
    //! 指定されたファイル名に含まれるパスが無効です。
    //!
    //! ''FILEIO_ERROR_SECURITYERROR''\n
    //! 呼び出し元に、必要なアクセス許可がありません。
    //!
    //! ''FILEIO_ERROR_NOMOREFILES''\n
    //! パターンに一致するファイルが見つかりません。
    //!
    //! @sa FileIO_FindNext
    //! @sa FileIO_FindClose
    //---------------------------------------------------------------------------
    NW_INLINE u32 FindFirst( FindData* pFindData, const char* pattern );

    //---------------------------------------------------------------------------
    //! @brief      FileIO_FindFirst()で指定したパターンに一致するファイルの
    //!             続きを検索します。
    //!
    //! @param[out] pFindData   見つかったファイルに関する情報を格納する構造体へのポインタ。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::FILEIO_Result)を返します。
    //!
    //! @details
    //! pFileInfo には FileIO_FindFirst() と同じ FileInfo へのポインタを指定します。
    //!
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //! Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_INVALIDFILEHANDLE''\n
    //! Windows 側が認識できないファイルのハンドルを操作しようとしています。
    //! pFileInfo で正しくファイル情報構造体のポインタを渡しているか確認してください。
    //!
    //! ''FILEIO_ERROR_NOMOREFILES''\n
    //! パターンに一致するファイルが見つかりません。
    //!
    //! @sa FileIO_FindFirst
    //! @sa FileIO_FindClose
    //---------------------------------------------------------------------------
    NW_INLINE u32 FindNext( FindData* pFindData );

    //---------------------------------------------------------------------------
    //! @brief      FileIO_FindFirst()で開始した検索を終了します。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::FILEIO_Result)を返します。
    //!
    //! @details
    //! pFileInfo には FileIO_FindFirst() と同じ FileInfo へのポインタを指定します。
    //!
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //! Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_INVALIDFILEHANDLE''\n
    //! Windows 側が認識できないファイルのハンドルを操作しようとしています。
    //! pFileInfo で正しくファイル情報構造体のポインタを渡しているか確認してください。
    //!
    //! @sa FileIO_FindFirst
    //! @sa FileIO_FindNext
    //---------------------------------------------------------------------------
    NW_INLINE u32 FindClose();
};

//@}

/* ========================================================================
    Function Prototype
   ======================================================================== */

#if defined(NW_MCS_ENABLE)

    //! @name FileIO・初期化
    //@{

    //---------------------------------------------------------------------------
    //! @brief      FileIO ライブラリを初期化します。
    //!
    //! @details
    //! Windows 上のファイルを読み書きする API とディレクトリ内のファイルを列挙する API の
    //! ための初期化を行います。
    //!
    //! この関数を呼び出す前に、 Mcs_Initialize() を呼び出して Mcs ライブラリを
    //! 初期化しておく必要があります。
    //!
    //! この関数は何回よばれても問題ありませんが、McsHID_Initialize 自体は
    //! スレッドセーフではないので注意が必要です。
    //!
    //! @sa Mcs_Initialize
    //! @sa FileIO_RegisterBuffer
    //---------------------------------------------------------------------------
    void            FileIO_Initialize();

    //---------------------------------------------------------------------------
    //! @brief      FileIO APIが初期化済みかどうかを取得します。
    //!
    //! @return     FileIO_Initialize() と FileIO_RegisterBuffer() が完了していれば true、
    //!             そうでなければ false を返します。
    //---------------------------------------------------------------------------
    bool            FileIO_IsInitialized();

    //---------------------------------------------------------------------------
    //! @brief      FileIO ライブラリの終了処理を行います。
    //!
    //! @sa Mcs_Finalize
    //! @sa FileIO_UnRegisterBuffer
    //---------------------------------------------------------------------------
    void            FileIO_Finalize();

    //---------------------------------------------------------------------------
    //! @brief      FileIO ライブラリが使用する作業用メモリを登録します。
    //!
    //! @param[in]  tempBuf     FileIO ライブラリが使用する作業メモリへのポインタ。
    //!
    //! @details
    //! tempBuf には、作業用メモリとして mcs::FILEIO_WORKBUFFER_SIZE のサイズを持つ
    //! メモリへのポインタを渡します。
    //!
    //! @sa FileIO_Initialize
    //! @sa FileIO_UnregisterBuffer
    //---------------------------------------------------------------------------
    void            FileIO_RegisterBuffer(void* tempBuf);

    //---------------------------------------------------------------------------
    //! @brief      FileIO ライブラリが使用する作業メモリを解除します。
    //!
    //! @details
    //! FileIO_RegisterBuffer() で登録した作業用メモリを解除します。
    //!
    //! @sa FileIO_Initialize
    //! @sa FileIO_RegisterBuffer
    //---------------------------------------------------------------------------
    void            FileIO_UnregisterBuffer();

    //---------------------------------------------------------------------------
    //! @brief      データ受信用に登録されたバッファを取得します。
    //!
    //! @return     FileIO_RegisterBuffer で登録されたバッファです。
    //!             登録されたバッファが存在しない場合は NULL を返します。
    //---------------------------------------------------------------------------
    void*           FileIO_GetRegisteredBuffer();

    //---------------------------------------------------------------------------
    //! @brief      FileIO ライブラリが使用するクライアントハンドルを登録します。
    //!
    //! @param[in]  handle     FileIO ライブラリが使用するクライアントハンドル。
    //! @param[in]  cmdBlock   FileIO ライブラリが使用するコマンドブロック。
    //! @param[in]  mountPoint HostFileIO のマウントポイント。
    //!
    //! @sa FileIO_Initialize
    //! @sa FileIO_UnregisterBuffer
    //---------------------------------------------------------------------------
#if defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    typedef ::nw::internal::winext::FSClient FSClient;
    typedef ::nw::internal::winext::FSCmdBlock FSCmdBlock;
#endif // defined( NW_PLATFORM_WIN32 )
    void            FileIO_RegisterClientHandle(FSClient* handle, FSCmdBlock* cmdBlock, const char* mountPoint = "/vol/hfio01");

    //@}

    //! @name FileIO・ファイル読み書き
    //! @{

    //---------------------------------------------------------------------------
    //! @brief      既存のファイルをオープンあるいは新規のファイルを作成してオープンします。
    //!
    //! @param[out] pFileInfo   ファイル情報構造体へのポインタ。
    //! @param[in]  fileName    作成または開くファイルを指す、'\\0'で終わる文字列へのポインタを指定します。
    //! @param[in]  openFlag    ファイルの読み書きの種類および作成、オープン方法の指定
    //!                         (mcs::FILEIO_FLAG_MODE | mcs::FILEIO_FLAG_OPTION)。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::FILEIO_Result)を返します。
    //!
    //! @details
    //! openFlag で、ファイルの読み書きの種類、および作成とオープンの方法を指定します。\n
    //! (mcs::FILEIO_FLAG_MODE と mcs::FILEIO_FLAG_OPTION 列挙子の論理和で指定します。)
    //!
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //! Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_FILENOTFOUND''\n
    //! ファイルが見つかりません。openFlag に FILEIO_FLAG_READ が指定されている場合、
    //! および FILEIO_FLAG_READWRITE と FILEIO_FLAG_NOCREATE が指定されている場合に
    //! ファイルが存在しないと、このエラーになります。
    //!
    //! ''FILEIO_ERROR_DIRECTORYNOTFOUND''\n
    //! 指定されたファイル名に含まれるパスが無効です。
    //!
    //! ''FILEIO_ERROR_FILEALREADYEXIST''\n
    //! ファイルが既に存在します。openFlag に FILEIO_FLAG_WRITE と FILEIO_FLAG_NOOVERWRITE が
    //! 指定されていると、このエラーになります。
    //!
    //! ''FILEIO_ERROR_SECURITYERROR''\n
    //! 呼び出し元に、必要なアクセス許可がありません。
    //!
    //! ''FILEIO_ERROR_UNAUTHORIZEDACCESS''\n
    //! ファイルに対するアクセス要求がオペレーティングシステムで許可されていません。
    //! 例えば、openFlag で FILEIO_FLAG_WRITE や FILEIO_FLAG_READWRITE を指定しているのに、
    //! オープンするファイルやファイルが置かれているディレクトリが読み込み専用に設定されている場合は
    //! このエラーになります。
    //!
    //! ''FILEIO_ERROR_IOERROR''\n
    //! その他の I/O エラーが発生しました。
    //!
    //! @sa FileIO_Read
    //! @sa FileIO_Write
    //! @sa FileIO_Close
    //! @sa FileIO_ShowFileDialog
    //---------------------------------------------------------------------------
    u32             FileIO_Open(
                        FileInfo*           pFileInfo,
                        const char*         fileName,
                        u32                 openFlag);

    //---------------------------------------------------------------------------
    //! @brief      ファイルをクローズします。
    //!
    //! @param[in,out] pFileInfo    ファイル情報構造体(FileInfo)へのポインタ。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::FILEIO_Result)を返します。
    //!
    //! @details
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //! Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_INVALIDFILEHANDLE''\n
    //! Windows 側が認識できないファイルのハンドルを操作しようとしています。
    //! pFileInfo で正しくファイル情報構造体のポインタを渡しているか確認してください。
    //!
    //! ''FILEIO_ERROR_IOERROR''\n
    //! その他の I/O エラーが発生しました。
    //!
    //! @sa FileIO_Open
    //---------------------------------------------------------------------------
    u32             FileIO_Close(FileInfo* pFileInfo);

    //---------------------------------------------------------------------------
    //! @brief      ファイルから読み込みます。
    //!
    //! @param[in,out] pFileInfo  ファイル情報構造体(FileInfo)へのポインタ。
    //! @param[in]  buffer      バッファへのポインタ。
    //!                         読み取りが成功した場合、読み込んだ内容が格納されます。
    //! @param[in]  length      読み込むバイト数。
    //! @param[out] pReadBytes  実際に読み込んだバイト数を返す変数へのポインタ。
    //!                         読み取りが成功した場合に、実際に読み込んだバイト数が格納されます。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::FILEIO_Result)を返します。
    //!
    //! @details
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //! Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_INVALIDFILEHANDLE''\n
    //! Windows 側が認識できないファイルのハンドルを操作しようとしています。
    //! pFileInfo で正しくファイル情報構造体のポインタを渡しているか確認してください。
    //!
    //! ''FILEIO_ERROR_IOERROR''\n
    //! その他の I/O エラーが発生しました。
    //!
    //! @sa FileIO_Open
    //! @sa FileIO_Write
    //! @sa FileIO_Close
    //---------------------------------------------------------------------------
    u32             FileIO_Read(
                        FileInfo*           pFileInfo,
                        void*               buffer,
                        u32                 length,
                        u32*                pReadBytes);

    //---------------------------------------------------------------------------
    //! @brief      ファイルへ書き込みます。
    //!
    //! @param[in,out] pFileInfo    ファイル情報構造体(FileInfo)へのポインタ。
    //! @param[in]  buffer      書き込む内容を保持するバッファへのポインタ。
    //! @param[in]  length      書き込むバイト数。
    //!
    //! @return      関数が成功した場合0を返します。
    //!               失敗した場合エラーコード(mcs::FILEIO_Result)を返します。
    //!
    //! @details
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //! Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_INVALIDFILEHANDLE''\n
    //! Windows 側が認識できないファイルのハンドルを操作しようとしています。
    //! pFileInfo で正しくファイル情報構造体のポインタを渡しているか確認してください。
    //!
    //! ''FILEIO_ERROR_IOERROR''\n
    //! その他の I/O エラーが発生しました。
    //!
    //! @sa FileIO_Open
    //! @sa FileIO_Read
    //! @sa FileIO_Close
    //---------------------------------------------------------------------------
    u32             FileIO_Write(
                        FileInfo*           pFileInfo,
                        const void*         buffer,
                        u32                 length);

    //---------------------------------------------------------------------------
    //! @brief      指定された位置にファイルポインタを移動します。
    //!
    //! @param[in,out] pFileInfo         ファイル情報構造体(FileInfo)へのポインタ。
    //! @param[in]  distanceToMove  ファイルポインタを移動するバイト数。
    //! @param[in]  moveMethod      ファイルポインタの移動を開始する位置(mcs::FILEIO_SEEK_MODE)。
    //! @param[out] pNewFilePointer 新しいファイルポインタを受け取る変数へのポインタ、または 0。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::FILEIO_Result)を返します。
    //!
    //! @details
    //! 引数 pNewFilePointer で指定したポインタが指す変数に移動後のファイルポインタの位置がセットされます。
    //! この値が不要な場合は 0 をセットしてください。
    //!
    //! distanceToMove に 0、moveMethod に mcs::FILEIO_SEEK_CURRENT を指定して
    //! 現在のファイルポインタの位置を取得することもできます。
    //!
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //!   通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //!   Mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //!   Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_INVALIDFILEHANDLE''\n
    //!   Windows 側が認識できないファイルのハンドルを操作しようとしています。
    //!   pFileInfo で正しくファイル情報構造体のポインタを渡しているか確認してください。
    //!
    //! ''FILEIO_ERROR_IOERROR''\n
    //!   その他の I/O エラーが発生しました。
    //!
    //! @sa FileIO_Open
    //! @sa FileIO_Read
    //! @sa FileIO_Write
    //---------------------------------------------------------------------------
    u32             FileIO_Seek(
                        FileInfo*           pFileInfo,
                        s32                 distanceToMove,
                        u32                 moveMethod,
                        u32*                pNewFilePointer     = 0
                    );

    //---------------------------------------------------------------------------
    //! @brief      ファイルサイズを取得します。
    //!
    //! @param[in]  pFileInfo   ファイル情報構造体(FileInfo)へのポインタ。
    //!
    //! @return     ファイルサイズを返します。
    //!
    //! @details
    //! Windows 上のファイルのオープンした時点(FileIO_Open()を呼び出した時点)での
    //! ファイルのサイズ(バイト数)を取得します。
    //!
    //! @sa FileIO_Open
    //---------------------------------------------------------------------------
    inline u32      FileIO_GetOpenFileSize( const FileInfo* pFileInfo ) { return pFileInfo->fileSize; }

    //@}

    //! @name FileIO・ファイル列挙
    //@{

    //---------------------------------------------------------------------------
    //! @brief      指定されたファイル名に一致するファイルをディレクトリ内で
    //!             検索します。
    //!
    //! @param[out] pFileInfo   ファイル情報構造体(FileInfo)へのポインタ。
    //! @param[out] pFindData   見つかったファイルに関する情報を格納する構造体(FindData)へのポインタ。
    //! @param[in]  pattern     検索するファイル名パターンの文字列。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::FILEIO_Result)を返します。
    //!
    //! @details
    //! 見つかった場合、最初に見つかったファイルに関する情報を pFindData で
    //! 指定した FindData 構造体の変数に格納します。
    //!
    //! pFileInfo で指定した FileInfo 構造体の変数に検索のステートが格納されます。
    //!
    //! 残りのファイルを検索するには FileIO_FindNext() を呼び出します。
    //!
    //! 検索を終了するときには FileIO_FindClose() を呼び出してください。
    //!
    //! ファイル名パターンには、「*」または「?」のワイルドカード文字を含めることができます。
    //! また、「%%」で囲むことによって「%%NW4F_ROOT%%\\include\\*.h」のように Windows の環境変数を
    //! 含めることができます。
    //!
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //! Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_DIRECTORYNOTFOUND''\n
    //! 指定されたファイル名に含まれるパスが無効です。
    //!
    //! ''FILEIO_ERROR_SECURITYERROR''\n
    //! 呼び出し元に、必要なアクセス許可がありません。
    //!
    //! ''FILEIO_ERROR_NOMOREFILES''\n
    //! パターンに一致するファイルが見つかりません。
    //!
    //! @sa FileIO_FindNext
    //! @sa FileIO_FindClose
    //---------------------------------------------------------------------------
    u32             FileIO_FindFirst(
                        FileInfo*           pFileInfo,
                        FindData*           pFindData,
                        const char*         pattern);

    //---------------------------------------------------------------------------
    //! @brief        FileInfo への処理で最後に発生したエラーコードを取得します。
    //!
    //! @param[out]   pFileInfo ファイル情報構造体 (FileInfo) へのポインタです。
    //!
    //! @return       最後に発生したエラーを返します。
    //---------------------------------------------------------------------------
    NW_INLINE u32
    FileIO_GetLastErrorCode(FileInfo* pFileInfo)
    {
        u32 code = pFileInfo->errorCode;
        pFileInfo->errorCode = FILEIO_ERROR_SUCCESS;
        return code;
    }

    //---------------------------------------------------------------------------
    //! @brief      FileIO_FindFirst()で指定したパターンに一致するファイルの
    //!             続きを検索します。
    //!
    //! @param[in,out]  pFileInfo   ファイル情報構造体(FileInfo)へのポインタ。
    //! @param[out] pFindData   見つかったファイルに関する情報を格納する構造体へのポインタ。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::FILEIO_Result)を返します。
    //!
    //! @details
    //! pFileInfo には FileIO_FindFirst() と同じ FileInfo へのポインタを指定します。
    //!
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //! Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_INVALIDFILEHANDLE''\n
    //! Windows 側が認識できないファイルのハンドルを操作しようとしています。
    //! pFileInfo で正しくファイル情報構造体のポインタを渡しているか確認してください。
    //!
    //! ''FILEIO_ERROR_NOMOREFILES''\n
    //! パターンに一致するファイルが見つかりません。
    //!
    //! @sa FileIO_FindFirst
    //! @sa FileIO_FindClose
    //---------------------------------------------------------------------------
    u32             FileIO_FindNext(
                        FileInfo*           pFileInfo,
                        FindData*           pFindData);

    //---------------------------------------------------------------------------
    //! @brief      FileIO_FindFirst()で開始した検索を終了します。
    //!
    //! @param[in,out]  pFileInfo   ファイル情報構造体(FileInfo)へのポインタ。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::FILEIO_Result)を返します。
    //!
    //! @details
    //! pFileInfo には FileIO_FindFirst() と同じ FileInfo へのポインタを指定します。
    //!
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //! Mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //! Mcs ライブラリ - Mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_INVALIDFILEHANDLE''\n
    //! Windows 側が認識できないファイルのハンドルを操作しようとしています。
    //! pFileInfo で正しくファイル情報構造体のポインタを渡しているか確認してください。
    //!
    //! @sa FileIO_FindFirst
    //! @sa FileIO_FindNext
    //---------------------------------------------------------------------------
    u32             FileIO_FindClose(FileInfo* pFileInfo);

    //@}

    //! @name FileIO・ファイルダイアログ
    //@{

    //---------------------------------------------------------------------------
    //! @brief      ファイルダイアログを表示します。
    //!
    //! @param[out] fileNameBuf ファイル名を保持するバッファ。
    //! @param[in]  fileNameBufSize fileNameBuf のサイズ。
    //! @param[in]  flag        ダイアログのモードを指定します\n
    //!                         (mcs::FILEIO_FILEDIALOG_MODE | mcs::FILEIO_FILEDIALOG_OPTION)。
    //! @param[in]  initialPath 初期ディレクトリおよびファイル名。
    //!                         特に指定しない場合は、空文字列""かNULLを指定します。
    //! @param[in]  filter      フィルタ文字列。
    //!                         特に指定しない場合は、空文字列""かNULLを指定します。
    //! @param[in]  defaultExt  規定の拡張子。
    //!                         特に指定しない場合は、空文字列""かNULLを指定します。
    //! @param[in]  title       ファイルダイアログのタイトル文字列。
    //!                         特に指定しない場合は、空文字列""かNULLを指定します。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::FILEIO_Result)を返します。
    //!
    //! @details
    //! Windows 上にファイルダイアログを表示します。
    //!
    //! ユーザがファイル名を確定した場合は FILEIO_ERROR_SUCCESS が返り、
    //! fileNameBuf で指定したバッファにファイル名が格納されます。
    //!
    //! ユーザがキャンセルした場合は、FILEIO_ERROR_CANCELED が返ります。
    //!
    //! 引数 flag でファイルダイアログの動作とオプションを指定します。\n
    //! (mcs::FILEIO_FILEDIALOG_MODE と mcs::FILEIO_FILEDIALOG_OPTION の論理和)
    //!
    //! 引数 initialPath には初期ディレクトリおよびファイル名を指定します。\n
    //! 引数 flag に FILEIO_FILEDIALOG_INCLUDEFILENAME が指定されていない場合は、
    //! 引数の内容はディレクトリへのパスとして扱います。\n
    //! 引数 flag に FILEIO_FILEDIALOG_INCLUDEFILENAME が指定されている場合は、
    //! 引数の内容はファイルへのパスとして扱い、初期ディレクトリとファイル名が設定されます。
    //! ただし、ファイル名のみが含まれる場合は初期ディレクトリを設定しません。\n
    //!「%%」で囲むことによって「%%NW4F_ROOT%\\include」のように
    //! Windows の環境変数を含めることができます。\n
    //! 特に指定しない場合は、空文字列""かNULLを指定します。
    //!
    //! 引数 filter には表示するファイルを選択する文字列を指定します。\n
    //! フィルタの説明とフィルタパターンの組み合わせで指定し、縦棒'|'で区切ります。\n
    //! 例えば、テキストファイル(*.txt)と全てのファイル(*.*)の 2 つを指定する場合は
    //! "テキストファイル (*.txt)|*.txt|全てのファイル(*.*)|*.*" のように指定します。\n
    //! 特に指定しない場合は、空文字列 "" か NULL を指定します。
    //!
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //! mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //! mcs ライブラリ - mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_IOERROR''\n
    //! その他の I/O エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_CANCELED''\n
    //! キャンセルされました。
    //!
    //! @sa FileIO_Open
    //---------------------------------------------------------------------------
    u32             FileIO_ShowFileDialog(
                        char*       fileNameBuf,
                        u32         fileNameBufSize,
                        u32         flag,
                        const char* initialPath         = 0,
                        const char* filter              = 0,
                        const char* defaultExt          = 0,
                        const char* title               = 0);

    //---------------------------------------------------------------------------
    //! @brief        環境変数を展開したパスを取得します。
    //!
    //! @details      DOS 形式の環境変数混じりのパス文字列を McsServer を通して展開します。
    //!
    //!               FileIO_Open 関数でも openFlag に FILEIO_FLAG_INCENVVAR を指定する事で
    //!               本関数と同様に環境変数を解決してファイルのオープンが可能です。
    //!
    //! @param[in]    fileName      元となるパス名です。
    //! @param[out]   resultBuf     展開した環境変数の格納先バッファです。
    //! @param[in]    resultBufSize 環境変数の展開先バッファのバッファサイズです。
    //!
    //! @return       エラーコードを返します。
    //! この関数が返すエラーの詳細は以下の通りです：\n
    //! (mcs::FILEIO_Result)
    //!
    //! ''FILEIO_ERROR_COMERROR''\n
    //! 通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_NOTCONNECT''\n
    //! mcs サーバと接続されていません。
    //!
    //! ''FILEIO_ERROR_PROTOCOLERROR''\n
    //! mcs ライブラリ - mcs サーバ間のバージョン違いによる通信エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_IOERROR''\n
    //! その他の I/O エラーが発生しました。
    //!
    //! ''FILEIO_ERROR_FILENOTFOUND''\n
    //! パス文字列の展開に失敗しました。
    //!
    //! @sa FileIO_Open
    //---------------------------------------------------------------------------
    u32             FileIO_ExpandPath(
                        const char* fileName,
                        char*       resultBuf,
                        u32         resultBufSize );

    //@}

#else   // #if defined(NW_MCS_ENABLE)

    inline void     FileIO_Initialize()                                 {}

    inline void     FileIO_Finalize()                                   {}

    inline void     FileIO_RegisterBuffer(void* /* tempBuf */)          {}

    inline void     FileIO_UnregisterBuffer()                           {}

    inline void*    FileIO_GetRegisteredBuffer() { return NULL; }

#if defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    typedef ::nw::internal::winext::FSClient FSClient;
    typedef ::nw::internal::winext::FSCmdBlock FSCmdBlock;
#endif // defined( NW_PLATFORM_WIN32 )
    inline void     FileIO_RegisterClientHandle(FSClient /* client */, FSCmdBlock* /* cmdBlock */, const char* /* mountPoint */ = 0) {}

    inline u32      FileIO_Open(FileInfo* /* pFileInfo*/, const char* /* fileName*/, u32 /* openFlag */)                { return FILEIO_ERROR_COMERROR; }

    inline u32      FileIO_Close(FileInfo* /* pFileInfo */)                                                             { return FILEIO_ERROR_COMERROR; }

    inline u32      FileIO_Read(FileInfo* /* pFileInfo*/, void* /* buffer*/, u32 /* length*/, u32* /* pReadBytes */)    { return FILEIO_ERROR_COMERROR; }

    inline u32      FileIO_Write(FileInfo* /* pFileInfo*/, const void* /* buffer*/, u32 /* length */)                   { return FILEIO_ERROR_COMERROR; }

    inline u32      FileIO_Seek(FileInfo* /* pFileInfo*/, s32 /* distanceToMove*/, u32 /* moveMethod*/, u32* /* pNewFilePointer */ = 0)  { return FILEIO_ERROR_COMERROR; }

    inline u32      FileIO_GetOpenFileSize( const FileInfo* /* pFileInfo */)   { return 0; }

    inline u32      FileIO_GetLastErrorCode(FileInfo* /* pFileInfo */)  { return FILEIO_ERROR_COMERROR; }

    inline u32      FileIO_FindFirst(FileInfo* /* pFileInfo*/, FindData* /* pFindData*/, const char* /* pattern */)     { return FILEIO_ERROR_COMERROR; }

    inline u32      FileIO_FindNext(FileInfo* /* pFileInfo*/, FindData* /* pFindData */)                                { return FILEIO_ERROR_COMERROR; }

    inline u32      FileIO_FindClose(FileInfo* /* pFileInfo */)                                                         { return FILEIO_ERROR_COMERROR; }

    inline u32      FileIO_ShowFileDialog(char* /* fileNameBuf */, u32 /* fileNameBufSize*/, u32 /* flag*/, const char* /* initialPath*/ = 0, const char* /* filter*/ = 0, const char* /* defaultExt*/ = 0, const char* /* title */ = 0)
                    { return FILEIO_ERROR_COMERROR; }

    inline u32      FileIO_ExpandPath(const char* /* fileName */, char* /* resultBuf */, u32 /* resultBufSize */ )                        { return FILEIO_ERROR_COMERROR; }


#endif  // #if defined(NW_MCS_ENABLE)


    //---------------------------------------------------------------------------
    NW_INLINE u32
    FileInfo::Open(const char* fileName, u32 openFlag)
    {
        return FileIO_Open( this, fileName, openFlag );
    }

    //---------------------------------------------------------------------------
    NW_INLINE u32
    FileInfo::Close()
    {
        return FileIO_Close( this );
    }

    //---------------------------------------------------------------------------
    NW_INLINE u32
    FileInfo::Read(void* buffer, u32 length, u32* pReadBytes)
    {
        return FileIO_Read( this, buffer, length, pReadBytes );
    }

    //---------------------------------------------------------------------------
    NW_INLINE u32
    FileInfo::Write(const void* buffer, u32 length)
    {
        return FileIO_Write( this, buffer, length );
    }

    //---------------------------------------------------------------------------
    NW_INLINE u32
    FileInfo::Seek(s32 distanceToMove, u32 moveMethod, u32* pNewFilePointer /* = 0 */)
    {
        return FileIO_Seek( this, distanceToMove, moveMethod, pNewFilePointer );
    }

    //---------------------------------------------------------------------------
    NW_INLINE u32
    FileInfo::GetOpenFileSize() const
    {
        return FileIO_GetOpenFileSize( this );
    }

    //---------------------------------------------------------------------------
    NW_INLINE u32
    FileInfo::GetLastErrorCode()
    {
        return FileIO_GetLastErrorCode( this );
    }

    //---------------------------------------------------------------------------
    NW_INLINE u32
    FileInfo::FindFirst( FindData* pFindData, const char* pattern )
    {
        return FileIO_FindFirst( this, pFindData, pattern );
    }

    //---------------------------------------------------------------------------
    NW_INLINE u32
    FileInfo::FindNext( FindData* pFindData )
    {
        return FileIO_FindNext( this, pFindData );
    }

    //---------------------------------------------------------------------------
    NW_INLINE u32
    FileInfo::FindClose()
    {
        return FileIO_FindClose( this );
    }


#if defined(NW_MCS_ENABLE)

namespace internal
{

// MCS channel number for FileIO
const ChannelType   FILEIO_CHANNEL      = 0xFF7F;

//
// MCS-FileIO command types
//
enum
{
    FILEIO_COMMAND_FILEOPEN     = 0,
    FILEIO_COMMAND_FILECLOSE,
    FILEIO_COMMAND_FILEREAD,
    FILEIO_COMMAND_FILEWRITE,
    FILEIO_COMMAND_FILESEEK,

    FILEIO_COMMAND_FINDFIRST    = 16,
    FILEIO_COMMAND_FINDNEXT,
    FILEIO_COMMAND_FINDCLOSE,

    FILEIO_COMMAND_SHOWFILEDLG  = 32,
    FILEIO_COMMAND_EXPANDPATH
};

typedef void*                   VoidPtr;
typedef FileInfo*               FileInfoPtr;
typedef FindData*               FindDataPtr;

/* ------------------------------------------------------------------------
    Command to send to file I/O server
   ------------------------------------------------------------------------ */

struct FileIOChunkHeader
{
    u32                     chunkID;
    u32                     chunkSize;
};

/*
    General-purpose command
*/
struct FileIOCommand : FileIOChunkHeader
{
    FileInfoPtr             pFileInfo;
    u32                     handle;
};

/*
    File open command
*/
struct FileIOOpenCommand : FileIOChunkHeader
{
    FileInfoPtr             pFileInfo;
    u32                     flag;
    u32                     mode;

    char16                  fileName[FILEIO_PATH_MAX];
};

/*
    File read command
*/
struct FileIOReadCommand : FileIOChunkHeader
{
    FileInfoPtr             pFileInfo;
    u32                     handle;
    VoidPtr                 pBuffer;
    u32                     size;
};

/*
  File write command
*/
struct FileIOWriteCommand : FileIOChunkHeader
{
    FileInfoPtr             pFileInfo;
    u32                     handle;
    u32                     size;
};

/*
  File seek command
*/
struct FileIOSeekCommand : FileIOChunkHeader
{
    FileInfoPtr             pFileInfo;
    u32                     handle;
    s32                     distanceToMove;
    u32                     moveMethod;
};

/*
    FindFirstFile command
*/
struct FileIOFindFirstCommand : FileIOChunkHeader
{
    FileInfoPtr             pFileInfo;
    FindDataPtr             pFindData;
    u32                     mode;
    char16                  fileName[FILEIO_PATH_MAX];
};

/*
    FindNextFile command
*/
struct FileIOFindNextCommand : FileIOChunkHeader
{
    FileInfoPtr             pFileInfo;
    u32                     handle;
    FindDataPtr             pFindData;
};

/*
    File open command
*/
struct FileIOShowFileDlgCommand : FileIOChunkHeader
{
    u32                     flag;
    u32                     mode;
    u32                     reserved;

    char16                  initialPath[FILEIO_PATH_MAX];
    char16                  filter[256];
    char16                  defaultExt[16];
    char16                  title[256];
};


/*
    Path expand command
*/
struct FileIOExpandPathCommand : FileIOChunkHeader
{
    char16                  fileName[FILEIO_PATH_MAX];
};


/* ------------------------------------------------------------------------
    Result from Windows
   ------------------------------------------------------------------------ */
//------------------------------------------------------------------------
// Structure that describes the file I/O response
// The following are command responses from the PC to the console.
//
//------------------------------------------------------------------------
struct FileIOResponse
{
    u32                     errorCode;
    FileInfoPtr             pFileInfo;
};

/*
    FileOpen result
*/
struct FileIOOpenResponse
{
    u32                     errorCode;
    FileInfoPtr             pFileInfo;
    u32                     handle;
    u32                     fileSize;
};

struct FileIOReadResponse
{
    u32                     errorCode;
    FileInfoPtr             pFileInfo;
    VoidPtr                 pBuffer;
    u32                     size;
    u32                     totalSize;
};

struct FileIOSeekResponse
{
    u32                     errorCode;
    FileInfoPtr             pFileInfo;
    u32                     filePointer;
};

struct FileIOFindResponse
{
    u32                     errorCode;
    FileInfoPtr             pFileInfo;
    FindDataPtr             pFindData;
    u32                     handle;
    u32                     fileSize;
    u32                     fileAttribute;
    u32                     fileDate;
    u32                     reserved1;
    u32                     reserved2;
    char16                  fileName[FILEIO_PATH_MAX];
};

struct FileIOShowFileDlgResponse
{
    u32                     errorCode;
    u32                     size;
};

struct FileIOExpandPathResponse
{
    u32                     errorCode;
    u32                     size;
};

}   // namespace internal

#endif  // #if defined(NW_MCS_ENABLE)

}   // namespace mcs
}   // namespace nw

/* NW_MCS_FILEIO_H_ */
#endif
