﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MCS_HIOCOMMDEVICE_H_
#define NW_MCS_HIOCOMMDEVICE_H_

#if defined(__CWCC__)
#pragma once
#endif

#include <types.h>
#include <cafe/hio.h>
#include <nw/ut/ut_LinkList.h>
#include <nw/mcs/mcs_Base.h>
#include <nw/mcs/mcs_Common.h>
#include <nw/mcs/mcs_HioSharedBuffer.h>

namespace nw
{
namespace mcs
{
namespace internal
{

class IHIOCommDeviceEnumerate
{
public:
    virtual ~IHIOCommDeviceEnumerate() {}
    virtual bool            Find(int devType) = 0;
};

#if defined(NW_MCS_ENABLE)

class HIOCommDevice : public CommDevice
{
public:
    //
    // Buffer size for sending and receiving data
    //
#ifndef RVL_PCGX
    static const u32        READ_TRANSBUF_SIZE  = MCS_INTERNAL_BUFFER_SIZE;
    static const u32        WRITE_TRANSBUF_SIZE = MCS_INTERNAL_BUFFER_SIZE;
    static const u32        INVALID_HANDLE_VALUE = 0;
#else
//PC_COMMENT Changed the buffer size
    static const u32        READ_TRANSBUF_SIZE = 0x10000;
    static const u32        WRITE_TRANSBUF_SIZE = 0x8000;
#endif

    static const u32        TEMP_BUF_SIZE_MIN = READ_TRANSBUF_SIZE + WRITE_TRANSBUF_SIZE;
    static const u32        COPY_ALIGNMENT = 32;

    //---------------------------------------------------------------------------
    //! @brief       通信デバイスの列挙を行います。
    //!              通信デバイスを見つけると、pEnumerateのFindメソッドを呼び出します。
    //!
    //! @param[out]  pEnumerate   通信デバイスを見つけたときに呼び出される
    //!                      Findメソッドを実装するオブジェクト。
    //!
    //! @return      列挙に成功すれば true を、列挙に失敗すれば false を返します。
    //---------------------------------------------------------------------------
    static bool             EnumerateDevice(IHIOCommDeviceEnumerate* pEnumerate);


    //---------------------------------------------------------------------------
    //! @brief       HIOCommDeviceのコンストラクタです。
    //!
    //! @param[in]   tempBuf       HIOCommDeviceが使用する作業メモリへのポインタ。
    //! @param[in]   tempBufSize   tempBufが指すメモリのサイズ。
    //---------------------------------------------------------------------------
    /* ctor */              HIOCommDevice(
                                void*           tempBuf,
                                u32             tempBufSize);

    //---------------------------------------------------------------------------
    //! @brief       HIOCommDeviceのデストラクタです。
    //---------------------------------------------------------------------------
    /* dtor */ virtual      ~HIOCommDevice();

    //---------------------------------------------------------------------------
    //! @brief       データ受信用のバッファを登録します。
    //!
    //!              受信用バッファが受信データで一杯になり、新規に受信したデータを格納する
    //!              だけの空き容量が無い場合は、その受信データは捨てられます。
    //!              従って、通信で使用するデータ量に合わせてバッファサイズを十分な大きさに
    //!              設定する必要があります。
    //!
    //! @param[in]   channel    データ送受信の通信相手を一意に識別するための値。
    //! @param[in]   buf        登録する受信用バッファ。
    //! @param[in]   bufSize    登録する受信用バッファのサイズ。
    //---------------------------------------------------------------------------
    virtual void            RegisterBuffer(
                                ChannelType channel,
                                void*       buf,
                                u32         bufSize);

    //---------------------------------------------------------------------------
    //! @brief       RegisterBufferメソッドで登録した受信用バッファの
    //!              登録を解除します。
    //!
    //! @param[in]   channel   データ送受信の通信相手を一意に識別するための値。
    //---------------------------------------------------------------------------
    virtual void*           UnregisterBuffer(ChannelType channel);

    //---------------------------------------------------------------------------
    //! @brief      データ受信用に登録されたバッファを取得します。
    //!
    //! @return     Mcs_RegisterBuffer で登録されたバッファです。
    //!             登録されたバッファが存在しない場合は NULL を返します。
    //---------------------------------------------------------------------------
    virtual void*            GetRegisteredBuffer(ChannelType channel);

    //---------------------------------------------------------------------------
    //! @brief       通信デバイスをオープンします。接続が完了するまでブロックします。
    //!
    //! @return      関数が成功した場合0を返します。
    //!              失敗した場合エラーコード(0以外の値)を返します。
    //---------------------------------------------------------------------------
    virtual u32             Open();

    //---------------------------------------------------------------------------
    //! @brief       通信デバイスをクローズします。
    //---------------------------------------------------------------------------
    virtual void            Close();

    //---------------------------------------------------------------------------
    //! @brief       メインループ内でこの関数を呼び出してください。
    //!
    //! @return      関数が成功した場合0を返します。
    //!               失敗した場合エラーコード(0以外の値)を返します。
    //---------------------------------------------------------------------------
    virtual u32             Polling();

    //---------------------------------------------------------------------------
    //! @brief       Readメソッドで、ブロックすることなく読み込めるデータのサイズを
    //!              取得します。
    //!
    //! @param[in]   channel   データ送受信の通信相手を一意に識別するための値。
    //!
    //! @return      読み込み可能なデータのサイズを返します。
    //---------------------------------------------------------------------------
    virtual u32             GetReadableBytes(ChannelType channel);

    //---------------------------------------------------------------------------
    //! @brief       データを読み取ります。読み取ったデータは受信バッファから
    //!              削除しません。
    //!
    //!              sizeで指定したサイズが、その時点で受信バッファに格納されている
    //!              データサイズより大きい場合は、受信バッファに格納されている
    //!              サイズだけ読み込みます。
    //!
    //! @param[in]   channel   データ送受信の通信相手を一意に識別するための値。
    //! @param[in]   data      読み込むデータを格納するバッファへのポインタ。
    //! @param[in]   size      読み込むデータを格納するバッファのサイズ。
    //!
    //! @return      読み取ったバイト数を返します。
    //---------------------------------------------------------------------------
    virtual u32             Peek(
                                ChannelType channel,
                                void*       data,
                                u32         size);

    //---------------------------------------------------------------------------
    //! @brief       データの受信を行います。
    //!
    //!              sizeで指定したサイズ全てを受信するまで制御を返しません。
    //!              従って、その時点で受信バッファに格納されているデータサイズ
    //!              より大きいサイズを読み込む場合、関数呼び出しはブロックされます。
    //!
    //! @param[in]   channel   データ受信の識別用の値。ユーザー任意に決められます。
    //! @param[in]   data      読み込むデータを格納するバッファへのポインタ。
    //! @param[in]   size      読み込むデータを格納するバッファのサイズ。
    //!
    //! @return      関数が成功した場合0を返します。
    //!               失敗した場合エラーコード(0以外の値)を返します。
    //---------------------------------------------------------------------------
    virtual u32             Read(
                                ChannelType channel,
                                void*       data,
                                u32         size);

    //---------------------------------------------------------------------------
    //! @brief       一度に送信可能なサイズを取得します。
    //!
    //! @param[in]   channel          現時点では使用しません。
    //! @param[out]  pWritableBytes   送信可能なバイト数を格納する変数へのポインタ。
    //!
    //! @return      関数が成功した場合0を返します。
    //!               失敗した場合エラーコード(0以外の値)を返します。
    //---------------------------------------------------------------------------
    virtual u32             GetWritableBytes(
                                ChannelType channel,
                                u32*        pWritableBytes);

    //---------------------------------------------------------------------------
    //! @brief       データの送信を行います。
    //!
    //!              sizeで指定したサイズ全てを送信するまで制御を返しません。
    //!              GetWritableBytesメソッドが返すサイズよりも大きいサイズの
    //!              データを送信する場合、関数呼び出しはブロックされます。
    //!
    //! @param[in]   channel   データ送受信の通信相手を一意に識別するための値。
    //! @param[in]   data      送信するデータを格納するバッファへのポインタ。
    //! @param[in]   size      送信するデータのサイズ。
    //!
    //! @return      関数が成功した場合0を返します。
    //!               失敗した場合エラーコード(0以外の値)を返します。
    //---------------------------------------------------------------------------
    virtual u32             Write(
                                ChannelType channel,
                                const void* data,
                                u32         size);

    //---------------------------------------------------------------------------
    //! @brief       mcsサーバとの接続状態を示す値を取得します。
    //!
    //! @return      mcsサーバと接続していたらtrue、接続していない場合はfalseを
    //!               返します。
    //---------------------------------------------------------------------------
    virtual bool            IsServerConnect();

    //---------------------------------------------------------------------------
    //! @brief       mcsサーバの時間を取得します。
    //!
    //! @return      2000年 1 月 1 日 を基準とした msec です。
    //---------------------------------------------------------------------------
    virtual s64             GetServerTime();

private:

    void                    OnConnect();
    void                    OnDisConnect();
    void                    OnRead(HIOStatus status);
    void                    OnWrite(HIOStatus status);
    void                    StartWritingBuffer();
    void                    StartReadingBuffer();
    void                    StopReadingBuffer();
    void                    SetServerTime(s64 time);
    void                    CheckInitBuffer();
    void                    CheckAlive();

    //---------------------------------------------------------------------------
    //! @brief       エラーが発生しているかどうかの状態を取得します。
    //!
    //! @return      エラーが発生していたらtrue、そうでない場合はfalseを
    //!               返します。
    //---------------------------------------------------------------------------
    bool                    IsError();

    //---------------------------------------------------------------------------
    //! @brief       mcsサーバとの通信状態の同期を取ります。
    //!
    //! @return      関数が成功した場合0を返します。
    //!               失敗した場合エラーコード(0以外の値)を返します。
    //---------------------------------------------------------------------------
    u32                     Negotiate();

    //---------------------------------------------------------------------------
    //! @brief       mcsサーバと通信を行うための共有メモリの情報を初期化します。
    //!
    //! @return      なし。
    //---------------------------------------------------------------------------
    void                    InitHioRingBuffer();

    //---------------------------------------------------------------------------
    //! @brief       チャンネル毎に登録されている受信バッファの内容をクリアします。
    //!
    //! @return      なし。
    //---------------------------------------------------------------------------
    void                    InitRegisteredBuffer();

    //---------------------------------------------------------------------------
    //! @brief       チャンネルを指定して、ChannelInfoへのポインタを取得します。
    //!
    //! @param[in]   channel   データ送受信の通信相手を一意に識別するための値。
    //!
    //! @return      指定したチャンネルに対するChannelInfoが存在すれば、
    //!               その変数へのポインタを返します。
    //!               見つからないときは0を返します。
    //---------------------------------------------------------------------------
    ChannelInfo*            GetChannelInfo(ChannelType channel);

    //---------------------------------------------------------------------------
    //! @brief       送信のために一時的に待ちます。
    //!
    //! @return      関数が成功した場合0を返します。
    //!               失敗した場合エラーコード(0以外の値)を返します。
    //---------------------------------------------------------------------------
    u32                     WaitSendData();

    //---------------------------------------------------------------------------
    //! @brief       受信のために一時的に待ちます。
    //!
    //! @return      関数が成功した場合0を返します。
    //!               失敗した場合エラーコード(0以外の値)を返します。
    //---------------------------------------------------------------------------
    u32                     WaitRecvData();

    static void             ConnectionCallback(HIOStatus status, void* context);
    static void             ReadCallback(HIOStatus status, void* context);
    static void             WriteCallback(HIOStatus status, void* context);
    static int              McsThreadProc(int argc, void *ptrArg);

    const char*             m_ChannelName;
    HIOHandle               m_HioHandle;
    void*                   m_TempBuf;
    u32                     m_TempBufSize;
    bool                    m_IsReadInitCodeMail;
    u32                     m_NegoPhase;
    ut::Mutex               m_Lock;
    s64                     m_ServerTime; // 2000/01/01 からの msec.
    OSTime                  m_ClientTime; // ServerTime を取得した時点の Client の OSTime
    bool                    m_IsOpen;
    bool                    m_IsWriting;
    bool                    m_IsReading;
    bool                    m_EnableReadChain;
    bool                    m_InitBufferPending;
    bool                    m_CheckAlivePending;

    HioSharedReadBuffer     m_SharedReadBuffer;
    HioSharedWriteBuffer    m_SharedWriteBuffer;

    ut::LinkList<ChannelInfo, offsetof(ChannelInfo,link)>  m_ChannelInfoList;       // Channel information list
};

#endif  // #if defined(NW_MCS_ENABLE)

}   // namespace internal
}   // namespace mcs
}   // namespace nw

/* NW_MCS_HIOCOMMDEVICE_H_ */
#endif
