﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MATH_VECTOR2_H_
#define NW_MATH_VECTOR2_H_

#include <cstring>
#include <nw/math/math_Config.h>
#include <nw/math/math_Arithmetic.h>

namespace nw {
namespace math {

struct VEC2;
struct MTX23;

namespace internal { namespace standard {

    VEC2* VEC2Maximize(VEC2* pOut, const VEC2* p1, const VEC2* p2);
    VEC2* VEC2Minimize(VEC2* pOut, const VEC2* p1, const VEC2* p2);

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

    VEC2* VEC2Maximize(VEC2* pOut, const VEC2* p1, const VEC2* p2);
    VEC2* VEC2Minimize(VEC2* pOut, const VEC2* p1, const VEC2* p2);

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

NW_MATH_INLINE bool VEC2IsZero(const VEC2* p);

NW_MATH_INLINE VEC2* VEC2Lerp(VEC2* pOut, const VEC2* p1, const VEC2* p2, f32 t);
NW_MATH_INLINE f32 VEC2Dot(const VEC2* p1, const VEC2* p2);
NW_MATH_INLINE VEC2* VEC2Maximize(VEC2* pOut, const VEC2* p1, const VEC2* p2);
NW_MATH_INLINE VEC2* VEC2Minimize(VEC2* pOut, const VEC2* p1, const VEC2* p2);
NW_MATH_INLINE VEC2* VEC2Normalize(VEC2* pOut, const VEC2* p);
NW_MATH_INLINE VEC2* VEC2FastNormalize(VEC2* pOut, const VEC2* p);
NW_MATH_INLINE VEC2* VEC2SafeNormalize(VEC2* pOut, const VEC2* p, const VEC2& alt);
NW_MATH_INLINE VEC2* VEC2FastSafeNormalize(VEC2* pOut, const VEC2* p, const VEC2& alt);
NW_MATH_INLINE f32 VEC2DistSq(const VEC2* p1, const VEC2* p2);
NW_MATH_INLINE VEC2* VEC2Transform(VEC2* pOut, const MTX23* pM, const VEC2* pV);

//========================================================================
//        クラスの定義
//========================================================================
//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        2次のベクトルの構造体です
//---------------------------------------------------------------------------
struct VEC2_
{
    f32 x;
    f32 y;
};

//---------------------------------------------------------------------------
//!   @brief        2次のベクトルクラスです。
//---------------------------------------------------------------------------
struct VEC2 : public VEC2_
{
public:
    static const int DIMENSION = 2; //!< 次元数です。

    //!< @brief ゼロベクトルです。
    static const VEC2& Zero()
    {
        static const VEC2 zero(0.0f, 0.0f);

        return zero;
    }

    typedef VEC2 self_type; //!< 自分の型です
    typedef f32  value_type; //!< 要素の型です。
public:

    //----------------------------------------
    //! @name 作成
    //@{

    //! @brief コンストラクタです。
    VEC2() {}
    //! @brief コンストラクタです。
    //!
    //! @param[in] p メンバ変数の初期化に用いる浮動小数点配列の先頭です。
    explicit VEC2(const f32* p) { x = p[0]; y = p[1]; }
    //! @brief コンストラクタです。
    //!
    //! @param[in] v メンバ変数の初期化に用いる VEC2 です。
    explicit VEC2(const VEC2_& v) { x = v.x; y = v.y; }
    //! @brief コンストラクタです。
    //!
    //! @param[in] fx メンバ変数 x の初期化に用いる浮動小数点値です。
    //! @param[in] fy メンバ変数 y の初期化に用いる浮動小数点値です。
    VEC2(f32 fx, f32 fy) { x = fx; y = fy; }

    //@}

    //----------------------------------------
    //! @name 変換
    //@{
    //! @brief f32 型へのキャストです。
    operator f32*() { return &x; }

    //! @brief f32 型へのキャストです。
    operator const f32*() const { return &x; }

    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    f32* ToF32() { return &x; }
    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    const f32* ToF32() const { return &x; }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast()
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelPointer<ToPtr>::Type) <= sizeof(VEC2));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast() const
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelPointer<ToPtr>::Type) <= sizeof(VEC2));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    //@}

    //----------------------------------------
    //! @name 演算
    //@{
    self_type& operator += (const self_type& rhs) { x += rhs.x; y += rhs.y; return *this; }
    self_type& operator -= (const self_type& rhs) { x -= rhs.x; y -= rhs.y; return *this; }
    self_type& operator *= (f32 f) { x *= f; y *= f; return *this; }
    self_type& operator /= (f32 f) { f32 r = 1.f / f; x *= r; y *= r; return *this; }

    self_type operator + () const { return *this; }
    self_type operator - () const { return self_type(-x, -y); }

    self_type operator + (const self_type& rhs) const { return self_type(x + rhs.x, y + rhs.y); }
    self_type operator - (const self_type& rhs) const { return self_type(x - rhs.x, y - rhs.y); }
    self_type operator * (f32 f) const { return self_type(f * x, f * y); }
    self_type operator / (f32 f) const { f32 r = 1.f / f; return self_type(r * x, r * y); }

    //! @brief 2つのベクトル間の線形補間を計算し設定します。
    //!
    //! @param[in] lhs 線形補間の始点となるベクトルです。
    //! @param[in] rhs 線形補間の終点となるベクトルです。
    //! @param[in] t   線形補間のパラメータ。0.0 であれば lhs が 1.0 であれば rhs が結果となります。
    //! @return 設定結果として、オブジェクト自身への参照を返します。
    //!
    self_type& Lerp(const VEC2& lhs, const VEC2& rhs, f32 t)
    {
        return *VEC2Lerp(this, &lhs, &rhs, t);
    }

    //! @brief 指定したベクトルとの内積を計算します。
    //!
    //! @param[in] vec 内積を行うベクトルです。
    //! @return 計算結果です。
    //!
    f32 Dot(const VEC2& vec) const
    {
        return VEC2Dot(this, &vec);
    }

    //! @brief ベクトルの長さの2乗を計算します。
    //!
    //! @return 計算結果です。
    f32 LengthSquare() const { return x * x + y * y; }

    //! @brief ベクトルの長さを計算します。
    //!
    //! @return 計算結果です。
    f32 Length() const { return FSqrt(this->x * this->x + this->y * this->y); }

    //! @brief ベクトルを正規化します。
    //!
    //! @return 処理結果として、オブジェクト自身への参照を返します。
    self_type& Normalize()
    {
        return *VEC2Normalize(this, this);
    }

    //! @brief ベクトルを正規化して設定します。
    //!
    //! @param[in]    src   正規化する対象のベクトルです。
    //! @return 設定結果として、オブジェクト自身への参照を返します。
    self_type& SetNormalize(const VEC2& src)
    {
        return *VEC2Normalize(this, &src);
    }

    //! @brief ベクトルを正規化します。
    //!        正規化に失敗した場合は指定されたベクトルを設定します。
    //!
    //! @param[in]    alt   正規化に失敗した場合に設定するベクトルです。
    //! @return 処理結果として、正規化に成功した場合はオブジェクト自身への参照を、失敗した場合は指定されたベクトルを返します。
    self_type& SafeNormalize(const VEC2& alt)
    {
        return *VEC2SafeNormalize(this, this, alt);
    }

    //! @brief ベクトルを正規化して設定します。
    //!        正規化に失敗した場合は指定されたベクトルを設定します。
    //!
    //! @param[in]    src   正規化する対象のベクトルです。
    //! @param[in]    alt   正規化に失敗した場合に設定するベクトルです。
    //! @return 設定結果として、正規化に成功した場合はオブジェクト自身への参照を、失敗した場合は指定されたベクトルを返します。
    self_type& SetSafeNormalize(const VEC2& src, const VEC2& alt)
    {
        return *VEC2SafeNormalize(this, &src, alt);
    }

    //! @brief 指定したベクトルとの距離の二乗を計算します。
    //!
    //! @param[in] vec 距離を計算するベクトルです。
    //! @return 計算結果です。
    f32 DistanceSquare(const VEC2& vec)
    {
        return VEC2DistSq(this, &vec);
    }

    //! @brief 2つのベクトルのそれぞれの成分の最大値から構成されるベクトルを設定します。
    //!
    //! @param[in] lhs 最大値を計算する左辺ベクトルです。
    //! @param[in] rhs 最大値を計算する右辺ベクトルです。
    //! @return 設定結果として、オブジェクト自身への参照を返します。
    self_type& SetMaximize(const VEC2& lhs, const VEC2& rhs)
    {
        return *VEC2Maximize(this, &lhs, &rhs);
    }

    //! @brief 2つのベクトルのそれぞれの成分の最小値から構成されるベクトルを設定します。
    //!
    //! @param[in] lhs 最小値を計算する左辺ベクトルです。
    //! @param[in] rhs 最小値を計算する右辺ベクトルです。
    //! @return 設定結果として、オブジェクト自身への参照を返します。
    self_type& SetMinimize(const VEC2& lhs, const VEC2& rhs)
    {
        return *VEC2Minimize(this, &lhs, &rhs);
    }

    //@}

    //----------------------------------------
    //! @name 設定
    //@{

    //! @brief 値を個別に設定します。
    //!
    //! @param[in] fx メンバ変数 x に設定される浮動小数点値です。
    //! @param[in] fy メンバ変数 y に設定される浮動小数点値です。
    void Set(f32 fx, f32 fy) { x = fx; y = fy; }

    //@}

    //----------------------------------------
    //! @name 比較
    //@{

    //! @brief 同値であれば true を返します。
    //!
    //! @param[in] rhs 比較対象となる VEC2 です。
    //! @return 比較結果です。
    bool operator == (const self_type& rhs) const { return x == rhs.x && y == rhs.y; }

    //! @brief 同値でなければ true を返します。
    //!
    //! @param[in] rhs 比較対象となる VEC2 です。
    //! @return 比較結果です。
    bool operator != (const self_type& rhs) const { return x != rhs.x || y != rhs.y; }

    //! @brief ゼロベクトルかどうかを判定します。
    //!
    //! @return ゼロベクトルであれば true を返します。
    bool IsZero() const { return VEC2IsZero(this); }
    //@}

    //! @brief ベクトルの内容をデバッグ出力に書き出します。
    //!
    //! @param[in] bNewline  true なら出力の最後に改行を出力します。
    //! @param[in] name      ベクトルの名前を指定します。NULL指定可能です。

    void Report(bool bNewline = true, const char* name = NULL) const;
};

typedef struct VEC2 Vector2;  //!< @briefprivate

//---------------------------------------------------------------------------
//    VEC2用の関数
//---------------------------------------------------------------------------

namespace internal { namespace standard {

NW_INLINE VEC2*
VEC2Add(VEC2* pOut, const VEC2* p1, const VEC2* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    pOut->x = p1->x + p2->x; pOut->y = p1->y + p2->y;
    return pOut;
}

NW_INLINE VEC2*
VEC2Sub(VEC2* pOut, const VEC2* p1, const VEC2* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    pOut->x = p1->x - p2->x; pOut->y = p1->y - p2->y;
    return pOut;
}

NW_INLINE VEC2*
VEC2Scale(VEC2* pOut, const VEC2* p, f32 scale)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p );

    pOut->x = p->x * scale; pOut->y = p->y * scale;
    return pOut;
}

NW_INLINE VEC2*
VEC2Lerp(VEC2* pOut, const VEC2* p1, const VEC2* p2, f32 t)
{
    // (1-t)*p1 + t*p2
    pOut->x = p1->x + t * (p2->x - p1->x);
    pOut->y = p1->y + t * (p2->y - p1->y);
    return pOut;
}

NW_INLINE f32
VEC2Dot(const VEC2* p1, const VEC2* p2)
{
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    return p1->x * p2->x + p1->y * p2->y;
}

NW_INLINE f32
VEC2LenSq(const VEC2* p)
{
    NW_ASSERT_NOT_NULL( p );
    return p->x * p->x + p->y * p->y;
}

NW_INLINE f32
VEC2DistSq(const VEC2* p1, const VEC2* p2)
{
    VEC2 tmp;
    return standard::VEC2LenSq(VEC2Sub(&tmp, p1, p2));
}

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

NW_INLINE VEC2*
VEC2Add(VEC2* pOut, const VEC2* p1, const VEC2* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    tof32x2(pOut->x) = __PS_ADD(tof32x2(p1->x), tof32x2(p2->x));

    return pOut;
}

NW_INLINE VEC2*
VEC2Sub(VEC2* pOut, const VEC2* p1, const VEC2* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    tof32x2(pOut->x) = __PS_SUB(tof32x2(p1->x), tof32x2(p2->x));

    return pOut;
}

NW_INLINE VEC2*
VEC2Scale(VEC2* pOut, const VEC2* p, f32 scale)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p );

    tof32x2(pOut->x) = __PS_MULS0F(tof32x2(p->x), scale);

    return pOut;
}

NW_INLINE VEC2*
VEC2Lerp(VEC2* pOut, const VEC2* p1, const VEC2* p2, f32 t)
{
    // (1-t)*p1 + t*p2

    f32x2 tt = __PS_FDUP(t);
    f32x2 f0 = tof32x2(p1->x);
    f0 = __PS_NMSUB(f0, tt, f0);
    tof32x2(pOut->x) = __PS_MADD(tof32x2(p2->x), tt, f0);

    return pOut;
}

NW_INLINE f32
VEC2Dot(const VEC2* p1, const VEC2* p2)
{
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    f32x2 f0 = __PS_MUL(tof32x2(p1->x), tof32x2(p2->x));
    f0 = __PS_SUM0(f0, f0, f0);

    return f0[0];
}

NW_INLINE f32
VEC2LenSq(const VEC2* p)
{
    NW_ASSERT_NOT_NULL( p );

    f32x2 f0 = tof32x2(p->x);
    f0 =__PS_MUL(f0, f0);
    f0 = __PS_SUM0(f0, f0, f0);

    return f0[0];
}

NW_INLINE f32
VEC2DistSq(const VEC2* p1, const VEC2* p2)
{
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    f32x2 f0 = __PS_SUB(tof32x2(p1->x), tof32x2(p2->x));
    f0 =__PS_MUL(f0, f0);
    f0 = __PS_SUM0(f0, f0, f0);

    return f0[0];
}

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

//----------------------------------------
//! @name    ベクトル
//@{

//---------------------------------------------------------------------------
//! @brief        ベクトルの和を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じベクトルを指していても構いません。
//! @param[in]    p1    左辺値へのポインタ
//! @param[in]    p2    右辺値へのポインタ
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline VEC2*
VEC2Add(VEC2* pOut, const VEC2* p1, const VEC2* p2)
{
    return NW_MATH_IMPL_NS::VEC2Add(pOut, p1, p2);
}

//---------------------------------------------------------------------------
//! @brief        ベクトルの差を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じベクトルを指していても構いません。
//! @param[in]    p1    左辺値へのポインタ
//! @param[in]    p2    右辺値へのポインタ
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline VEC2*
VEC2Sub(VEC2* pOut, const VEC2* p1, const VEC2* p2)
{
    return NW_MATH_IMPL_NS::VEC2Sub(pOut, p1, p2);
}


//---------------------------------------------------------------------------
//! @brief        ベクトルのスカラー倍を計算します。
//!
//! @param[out]   pOut   計算結果を受け取るバッファへのポインタ。p と同じベクトルを指していても構いません。
//! @param[in]    p      左辺値へのポインタ
//! @param[in]    scale  掛ける数
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline VEC2*
VEC2Scale(VEC2* pOut, const VEC2* p, f32 scale)
{
    return NW_MATH_IMPL_NS::VEC2Scale(pOut, p, scale);
}


//---------------------------------------------------------------------------
//! @brief        ベクトル間の線形補間を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じベクトルを指していても構いません。
//! @param[in]    p1    線形補間の始点となるベクトルへのポインタ
//! @param[in]    p2    線形補間の終点となるベクトルへのポインタ
//! @param[in]    t     線形補間のパラメータ。0.0 であれば p1 が 1.0 であれば p2 が結果となります。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline VEC2*
VEC2Lerp(VEC2* pOut, const VEC2* p1, const VEC2* p2, f32 t)
{
    return NW_MATH_IMPL_NS::VEC2Lerp(pOut, p1, p2, t);
}


//---------------------------------------------------------------------------
//! @brief        ベクトルの内積を計算します。
//!
//! @param[in]    p1    左辺値へのポインタ
//! @param[in]    p2    右辺値へのポインタ
//!
//! @return       p1 と p2 の内積を返します。
//---------------------------------------------------------------------------
inline f32
VEC2Dot(const VEC2* p1, const VEC2* p2)
{
    return NW_MATH_IMPL_NS::VEC2Dot(p1, p2);
}

//---------------------------------------------------------------------------
//! @brief        ベクトルの長さの2乗を計算します。
//!
//! @param[in]    p  対象のベクトルへのポインタ。
//!
//! @return       p の長さの2乗を返します。
//---------------------------------------------------------------------------
inline f32
VEC2LenSq(const VEC2* p)
{
    return internal::standard::VEC2LenSq(p);
}

//---------------------------------------------------------------------------
//! @brief        ベクトルの長さを計算します。
//!
//! @param[in]    p  対象のベクトルへのポインタ。
//!
//! @return       p の長さを返します。
//---------------------------------------------------------------------------
inline f32
VEC2Len(const VEC2* p) { return FSqrt(p->x * p->x + p->y * p->y); }

//---------------------------------------------------------------------------
//! @brief        2つのベクトル間の距離の2乗を計算します。
//!
//! @param[in]    p1    左辺値へのポインタ
//! @param[in]    p2    右辺値へのポインタ
//!
//! @return       p1 と p2 の距離の2乗を返します。
//---------------------------------------------------------------------------
inline f32
VEC2DistSq(const VEC2* p1, const VEC2* p2)
{
    return NW_MATH_IMPL_NS::VEC2DistSq(p1, p2);
}

//---------------------------------------------------------------------------
//! @briefprivate
//! @details VEC2 と浮動小数点数の積を計算します。
//! @param[in] f 積に用いる浮動小数点数です。
//! @param[in] rhs 積に用いる VEC2 です。
//! @return 計算結果の VEC2 を返します。
//---------------------------------------------------------------------------
inline VEC2
operator * (f32 f, const VEC2& rhs) { return VEC2(f * rhs.x, f * rhs.y); }

//@}

}  // namespace math
}  // namespace nw

#if defined(NW_MATH_AS_INLINE)
#include <nw/math/inline/math_Vector2.ipp>
#endif

namespace nw {
namespace math {

//-- const 引数を参照にしたオーバーロード
inline bool VEC2IsZero(const VEC2& v){ return VEC2IsZero( &v ); }
inline VEC2* VEC2Add(VEC2* pOut, const VEC2& p1, const VEC2& p2) { return VEC2Add(pOut, &p1, &p2); }
inline VEC2* VEC2Sub(VEC2* pOut, const VEC2& v1, const VEC2& v2) { return VEC2Sub(pOut, &v1, &v2); }
inline VEC2* VEC2Scale(VEC2* pOut, const VEC2& v, f32 scale) { return VEC2Scale(pOut, &v, scale); }
inline VEC2* VEC2Lerp(VEC2* pOut, const VEC2& v1, const VEC2& v2, f32 t) { return VEC2Lerp(pOut, &v1, &v2, t); }
inline f32 VEC2Dot(const VEC2& v1, const VEC2& v2) { return VEC2Dot(&v1, &v2); }
inline f32 VEC2LenSq(const VEC2& v) { return VEC2LenSq( &v ); }
inline f32 VEC2Len(const VEC2& v) { return VEC2Len( &v ); }
inline f32 VEC2DistSq(const VEC2& v1, const VEC2& v2) { return VEC2DistSq( &v1, &v2 ); }

inline VEC2* VEC2Maximize(VEC2* pOut, const VEC2& v1, const VEC2& v2) { return VEC2Maximize( pOut, &v1, &v2 ); }
inline VEC2* VEC2Minimize(VEC2* pOut, const VEC2& v1, const VEC2& v2) { return VEC2Minimize( pOut, &v1, &v2 ); }
inline VEC2* VEC2Normalize(VEC2* pOut, const VEC2& v) { return VEC2Normalize( pOut, &v ); }
inline VEC2* VEC2FastNormalize(VEC2* pOut, const VEC2& v) { return VEC2FastNormalize( pOut, &v ); }
inline VEC2* VEC2SafeNormalize(VEC2* pOut, const VEC2& v, const VEC2& alt) { return VEC2SafeNormalize( pOut, &v, alt ); }
inline VEC2* VEC2FastSafeNormalize(VEC2* pOut, const VEC2& v, const VEC2& alt) { return VEC2FastSafeNormalize( pOut, &v, alt ); }
inline VEC2* VEC2Transform(VEC2* pOut, const MTX23& m, const VEC2& v) { return VEC2Transform( pOut, &m, &v ); }

}  // namespace math
}  // namespace nw


#endif
