﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MATH_MATRIX44_H_
#define NW_MATH_MATRIX44_H_

// 不具合修正前の、NW4F 1.4.0 以前の行列計算を使用する。
//#define NW_MATH_USE_OLDMTX

// MEMO: SDK で MTX44* を define している箇所があり、インクルードの順番によって
//       ビルドが出来なくなることがあるため、回避策として明示的に SDK のヘッダを
//       ここでインクルードしています。
#if defined(NW_PLATFORM_CAFE)
    #include <cafe/mtx/mtx44.h>
#endif

#include <cstring>
#include <nw/math/math_Config.h>

namespace nw {
namespace math {

struct MTX44;

namespace internal { namespace standard {

    VEC4* VEC3Transform(VEC4* pOut, const MTX44* pM, const VEC3* pV);
    MTX44* MTX44Add(MTX44* pOut, const MTX44* p1, const MTX44* p2);
    MTX44* MTX44Sub(MTX44* pOut, const MTX44* p1, const MTX44* p2);
    MTX44* MTX44Mult(MTX44* pOut, const MTX44* p, f32 f);
    MTX44* MTX44Mult(MTX44* pOut, const MTX44* __restrict p1, const MTX44* __restrict p2);
    MTX44* MTX44MultScale(MTX44* pOut, const MTX44* pM, const VEC3* pS);
    MTX44* MTX44MultScale(MTX44* pOut, const VEC3* pS, const MTX44* pM);

}} // namespace internal::standard

namespace internal { namespace intrinsics {

    VEC4* VEC3Transform(VEC4* pOut, const MTX44* pM, const VEC3* pV);
    MTX44* MTX44Add(MTX44* pOut, const MTX44* p1, const MTX44* p2);
    MTX44* MTX44Sub(MTX44* pOut, const MTX44* p1, const MTX44* p2);
    MTX44* MTX44Mult(MTX44* pOut, const MTX44* p, f32 f);
    MTX44* MTX44Mult(MTX44* pOut, const MTX44* __restrict p1, const MTX44* __restrict p2);
    MTX44* MTX44MultScale(MTX44* pOut, const MTX44* pM, const VEC3* pS);
    MTX44* MTX44MultScale(MTX44* pOut, const VEC3* pS, const MTX44* pM);

}} // namespace internal::intrinsics

//---------------------------------------------------------------------------
//    MTX44用の関数
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX44* MTX44Zero(MTX44* pOut);
NW_MATH_INLINE bool   MTX44IsIdentity(const MTX44* p);
NW_MATH_INLINE MTX44* MTX44Identity(MTX44* pOut);
NW_MATH_INLINE MTX44* MTX44Copy(MTX44* pOut, const MTX44* p);

NW_MATH_INLINE MTX44* MTX44Add(MTX44* pOut, const MTX44* p1, const MTX44* p2);
NW_MATH_INLINE MTX44* MTX44Sub(MTX44* pOut, const MTX44* p1, const MTX44* p2);
NW_MATH_INLINE MTX44* MTX44Mult(MTX44* pOut, const MTX44* p, f32 f);
NW_MATH_INLINE MTX44* MTX44Mult(MTX44* pOut, const MTX44* p1, const MTX44* p2);

NW_MATH_INLINE MTX44* MTX44FrustumNew(MTX44* pOut, f32 l, f32 r, f32 b, f32 t, f32 n, f32 f);
NW_MATH_INLINE MTX44* MTX44OrthoNew(MTX44* pOut, f32 l, f32 r, f32 b, f32 t, f32 n, f32 f);
NW_MATH_INLINE MTX44* MTX44PerspectiveRadNew(MTX44* pOut, f32 fovyRad, f32 aspect, f32 n, f32 f);

NW_MATH_INLINE MTX44* MTX44FrustumOld(MTX44* pOut, f32 l, f32 r, f32 b, f32 t, f32 n, f32 f);
NW_MATH_INLINE MTX44* MTX44OrthoOld(MTX44* pOut, f32 l, f32 r, f32 b, f32 t, f32 n, f32 f);
NW_MATH_INLINE MTX44* MTX44PerspectiveRadOld(MTX44* pOut, f32 fovyRad, f32 aspect, f32 n, f32 f);

NW_MATH_INLINE MTX44* MTX44Frustum(MTX44* pOut, f32 l, f32 r, f32 b, f32 t, f32 n, f32 f)
{
#ifdef NW_MATH_USE_OLDMTX
    return MTX44FrustumOld(pOut, l, r, b, t, n, f);
#else
    return MTX44FrustumNew(pOut, l, r, b, t, n, f);
#endif
}
NW_MATH_INLINE MTX44* MTX44Ortho(MTX44* pOut, f32 l, f32 r, f32 b, f32 t, f32 n, f32 f)
{
#ifdef NW_MATH_USE_OLDMTX
    return MTX44OrthoOld(pOut, l, r, b, t, n, f);
#else
    return MTX44OrthoNew(pOut, l, r, b, t, n, f);
#endif
}
NW_MATH_INLINE MTX44* MTX44PerspectiveRad(MTX44* pOut, f32 fovyRad, f32 aspect, f32 n, f32 f)
{
#ifdef NW_MATH_USE_OLDMTX
    return MTX44PerspectiveRadOld(pOut, fovyRad, aspect, n, f);
#else
    return MTX44PerspectiveRadNew(pOut, fovyRad, aspect, n, f);
#endif
}

NW_MATH_INLINE MTX44* MTX44Transpose(MTX44* pOut, const MTX44 *pSrc);
NW_MATH_INLINE MTX44* MTX44MultArray(MTX44* pOut, const MTX44* p1, const MTX44* pSrc, s32 count);
NW_MATH_INLINE u32    MTX44Inverse(MTX44* pOut, const MTX44* p);

NW_MATH_INLINE MTX44* MTX44RotXYZIdx(MTX44* pOut, u32 idxX, u32 idxY, u32 idxZ);
NW_MATH_INLINE MTX44* MTX44RotAxisIdx(MTX44* pOut, const VEC3* pAxis, u32 idx);

NW_MATH_INLINE MTX44* MTX44Scale(MTX44* pOut, const VEC3* pS);
NW_MATH_INLINE MTX44* MTX44MultScale(MTX44* pOut, const MTX44* pM, const VEC3* pS);
NW_MATH_INLINE MTX44* MTX44MultScale(MTX44* pOut, const VEC3* pS, const MTX44* pM);

NW_MATH_INLINE MTX44* MTX44Translate(MTX44* pOut, const VEC3* pT);
NW_MATH_INLINE MTX44* MTX44MultTranslate(MTX44* pOut, const MTX44* pM, const VEC3* pT);
NW_MATH_INLINE MTX44* MTX44MultTranslate(MTX44* pOut, const VEC3* pT, const MTX44* pM);

//---------------------------------------------------------------------------
//! @brief        スクリーンの回転方向です。
//---------------------------------------------------------------------------
enum PivotDirection
{
    PIVOT_NONE,             //!< 回転処理をおこないません。
    PIVOT_UPSIDE_TO_TOP,    //!< カメラ上方向が画面上方向を向くように回転します。
    PIVOT_UPSIDE_TO_RIGHT,  //!< カメラ上方向が画面右方向を向くように回転します。
    PIVOT_UPSIDE_TO_BOTTOM, //!< カメラ上方向が画面下方向を向くように回転します。
    PIVOT_UPSIDE_TO_LEFT,   //!< カメラ上方向が画面左方向を向くように回転します。
    PIVOT_NUM
};

NW_MATH_INLINE MTX44* MTX44FrustumPivot(MTX44* pOut, f32 l, f32 r, f32 b, f32 t, f32 n, f32 f, PivotDirection pivot = PIVOT_NONE);
NW_MATH_INLINE MTX44* MTX44OrthoPivot(MTX44* pOut, f32 l, f32 r, f32 b, f32 t, f32 n, f32 f, PivotDirection pivot = PIVOT_NONE);
NW_MATH_INLINE MTX44* MTX44PerspectivePivotRad(MTX44* pOut, f32 fovyRad, f32 aspect, f32 n, f32 f, PivotDirection pivot = PIVOT_NONE);

//----------------------------------------
//! @name    行列
//@{

//---------------------------------------------------------------------------
//! @brief        射影行列を視野角とアスペクト比から作成します。
//!
//! @param[out]   pOut    射影行列を格納する行列へのポインタ。
//! @param[in]    fovyDeg 縦方向の視野角（Degree）
//! @param[in]    aspect  視野のアスペクト比(幅/高さ)
//! @param[in]    n       ニアクリッピング面までの距離。
//! @param[in]    f       ファークリッピング面までの距離。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline MTX44*
MTX44PerspectiveDeg(MTX44* pOut, f32 fovyDeg, f32 aspect, f32 n, f32 f)
{
    return MTX44PerspectiveRad(pOut, NW_MATH_DEG_TO_RAD(fovyDeg), aspect, n, f);
}

//---------------------------------------------------------------------------
//! @brief        射影行列を視野角とアスペクト比から作成します。
//!
//! @param[out]   pOut    射影行列を格納する行列へのポインタ。
//! @param[in]    fovyRad 縦方向の視野角（Rad）
//! @param[in]    aspect  視野のアスペクト比(幅/高さ)
//! @param[in]    n       ニアクリッピング面までの距離。
//! @param[in]    f       ファークリッピング面までの距離。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
// MTX44PerspectiveDeg 関数の別名です。
// 互換性の為に接尾語の無い Perspective 関数を degree 版としています。
inline MTX44*
MTX44Perspective(MTX44* pOut, f32 fovyRad, f32 aspect, f32 n, f32 f)
{
    return MTX44PerspectiveRad(pOut, fovyRad, aspect, n, f);
}

//---------------------------------------------------------------------------
//! @brief        射影行列を視野角とアスペクト比から作成します。
//!
//! @param[out]   pOut    射影行列を格納する行列へのポインタ。
//! @param[in]    fovyDeg 縦方向の視野角（Degree）
//! @param[in]    aspect  視野のアスペクト比(幅/高さ)
//! @param[in]    n       ニアクリッピング面までの距離。
//! @param[in]    f       ファークリッピング面までの距離。
//! @param[in]    pivot   スクリーンの回転方向。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline MTX44*
MTX44PerspectivePivotDeg(MTX44* pOut, f32 fovyDeg, f32 aspect, f32 n, f32 f, PivotDirection pivot = PIVOT_NONE )
{
    return MTX44PerspectivePivotRad(pOut, NW_MATH_DEG_TO_RAD(fovyDeg), aspect, n, f, pivot);
}


//---------------------------------------------------------------------------
//! @brief        回転行列を作成します。
//!
//! @param[out]   pOut   計算結果を受け取るバッファへのポインタ。
//! @param[in]    fRadX  ラジアン単位での X 軸周りの角度
//! @param[in]    fRadY  ラジアン単位での Y 軸周りの角度
//! @param[in]    fRadZ  ラジアン単位での Z 軸周りの角度
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline MTX44*
MTX44RotXYZRad(MTX44* pOut, f32 fRadX, f32 fRadY, f32 fRadZ)
{
    return MTX44RotXYZIdx(pOut, NW_MATH_RAD_TO_IDX(fRadX), NW_MATH_RAD_TO_IDX(fRadY), NW_MATH_RAD_TO_IDX(fRadZ));
}

//---------------------------------------------------------------------------
//! @brief        回転行列を作成します。
//!
//! @param[out]   pOut   計算結果を受け取るバッファへのポインタ。
//! @param[in]    fDegX  1 円周を 360.0 とする単位での X 軸周りの角度
//! @param[in]    fDegY  1 円周を 360.0 とする単位での Y 軸周りの角度
//! @param[in]    fDegZ  1 円周を 360.0 とする単位での Z 軸周りの角度
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline MTX44*
MTX44RotXYZDeg(MTX44* pOut, f32 fDegX, f32 fDegY, f32 fDegZ)
{
    return MTX44RotXYZIdx(pOut, NW_MATH_DEG_TO_IDX(fDegX), NW_MATH_DEG_TO_IDX(fDegY), NW_MATH_DEG_TO_IDX(fDegZ));
}

//---------------------------------------------------------------------------
//! @brief        指定する軸の周りを回転させる回転行列を作成します。
//!
//! @param[out]   pOut   計算結果を受け取るバッファへのポインタ。
//! @param[in]    pAxis  回転軸を指定するベクトルへのポインタ。
//! @param[in]    fRad   ラジアン単位での回転量
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline MTX44*
MTX44RotAxisRad(MTX44* pOut, const VEC3* pAxis, f32 fRad)
{
    return MTX44RotAxisIdx(pOut, pAxis, NW_MATH_RAD_TO_IDX(fRad));
}

//---------------------------------------------------------------------------
//! @brief        指定する軸の周りを回転させる回転行列を作成します。
//!
//! @param[out]   pOut   計算結果を受け取るバッファへのポインタ。
//! @param[in]    pAxis  回転軸を指定するベクトルへのポインタ。
//! @param[in]    fDeg   1 円周を 360.0 とする単位での回転量
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline MTX44*
MTX44RotAxisDeg(MTX44* pOut, const VEC3* pAxis, f32 fDeg)
{
    return MTX44RotAxisIdx(pOut, pAxis, NW_MATH_DEG_TO_IDX(fDeg));
}

//@}

//========================================================================
//        クラスの定義
//========================================================================
struct MTX34;

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        4行4列の行列の構造体です
//---------------------------------------------------------------------------
#if defined(_WIN32)
#pragma warning(push)
#pragma warning(disable:4201)
#endif
struct MTX44_
{
    struct BaseData
    {
        f32 _00, _01, _02, _03;
        f32 _10, _11, _12, _13;
        f32 _20, _21, _22, _23;
        f32 _30, _31, _32, _33;
    };

    union
    {
        struct
        {
            f32 _00, _01, _02, _03;
            f32 _10, _11, _12, _13;
            f32 _20, _21, _22, _23;
            f32 _30, _31, _32, _33;
        };
        BaseData f;
        f32 m[4][4];
        f32 a[16];
        VEC4_ v[4];
    };
};
#if defined(_WIN32)
#pragma warning(pop)
#endif
//---------------------------------------------------------------------------
//! @brief        ４行４列の行列クラスです。
//---------------------------------------------------------------------------
struct MTX44 : public MTX44_
{
public:
    static const int ROW_COUNT = 4; //!< 行数です。
    static const int COLUMN_COUNT = 4; //!< 列数です。

    //! @brief 単位行列です。
    //!
    //! @return 単位行列を返します。
    static const MTX44& Identity()
    {
        static const MTX44 identity(
            1.0f, 0.0f, 0.0f, 0.0f,
            0.0f, 1.0f, 0.0f, 0.0f,
            0.0f, 0.0f, 1.0f, 0.0f,
            0.0f, 0.0f, 0.0f, 1.0f);

        return identity;
    }
    typedef MTX44 self_type; //!< 自分の型です。
    typedef f32   value_type; //!< 要素の型です。

public:
    //----------------------------------------
    //! @name 作成
    //@{

    //! @brief コンストラクタです。
    MTX44() {}

    //! @brief コンストラクタです。
    //!
    //! @param[in] p メンバ変数の初期化に用いる浮動小数点配列の先頭です。
    explicit MTX44(const f32* p) { (void)MTX44Copy(this, (MTX44*)p); }

    //! @brief コンストラクタです。
    //!
    //! @param[in] rhs メンバ変数の初期化に用いる MTX44 です。メンバ変数 _30, _31, _32 には 0.0f、_33 には 1.0f が入ります。
    explicit MTX44(const MTX34& rhs)
    {
        (void)MTX34Copy((MTX34*)this, (MTX34*)&rhs);
        f._30 = f._31 = f._32 = 0.f; f._33 = 1.f;
    }

    //! @brief コピーコンストラクタです。
    //!
    //! @param[in] rhs メンバ変数の初期化に用いる MTX44 です。
    MTX44(const MTX44& rhs) { (void)MTX44Copy(this, &rhs); }

    //! @brief コンストラクタです。
    //!
    //! @param[in] x00 メンバ変数 _00 の初期化に用いる浮動小数点値です。
    //! @param[in] x01 メンバ変数 _01 の初期化に用いる浮動小数点値です。
    //! @param[in] x02 メンバ変数 _02 の初期化に用いる浮動小数点値です。
    //! @param[in] x03 メンバ変数 _03 の初期化に用いる浮動小数点値です。
    //! @param[in] x10 メンバ変数 _10 の初期化に用いる浮動小数点値です。
    //! @param[in] x11 メンバ変数 _11 の初期化に用いる浮動小数点値です。
    //! @param[in] x12 メンバ変数 _12 の初期化に用いる浮動小数点値です。
    //! @param[in] x13 メンバ変数 _13 の初期化に用いる浮動小数点値です。
    //! @param[in] x20 メンバ変数 _20 の初期化に用いる浮動小数点値です。
    //! @param[in] x21 メンバ変数 _21 の初期化に用いる浮動小数点値です。
    //! @param[in] x22 メンバ変数 _22 の初期化に用いる浮動小数点値です。
    //! @param[in] x23 メンバ変数 _23 の初期化に用いる浮動小数点値です。
    //! @param[in] x30 メンバ変数 _30 の初期化に用いる浮動小数点値です。
    //! @param[in] x31 メンバ変数 _31 の初期化に用いる浮動小数点値です。
    //! @param[in] x32 メンバ変数 _32 の初期化に用いる浮動小数点値です。
    //! @param[in] x33 メンバ変数 _33 の初期化に用いる浮動小数点値です。
    MTX44(f32 x00, f32 x01, f32 x02, f32 x03,
          f32 x10, f32 x11, f32 x12, f32 x13,
          f32 x20, f32 x21, f32 x22, f32 x23,
          f32 x30, f32 x31, f32 x32, f32 x33)
    {
        f._00 = x00; f._01 = x01; f._02 = x02; f._03 = x03;
        f._10 = x10; f._11 = x11; f._12 = x12; f._13 = x13;
        f._20 = x20; f._21 = x21; f._22 = x22; f._23 = x23;
        f._30 = x30; f._31 = x31; f._32 = x32; f._33 = x33;
    }

    //@}

    //----------------------------------------
    //! @name 変換
    //@{

    //! @brief f32 型へのキャストです。
    operator f32*() { return this->a; }

    //! @brief f32 型へのキャストです。
    operator const f32*() const { return this->a; }

    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    f32* ToF32() { return this->a; }
    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    const f32* ToF32() const { return this->a; }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast()
    {
        NW_STATIC_ASSERT(sizeof(typename ut::PeelPointer<ToPtr>::Type) <= sizeof(MTX44));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast() const
    {
        NW_STATIC_ASSERT(sizeof(typename ut::PeelPointer<ToPtr>::Type) <= sizeof(MTX44));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    template <typename ToRef>
    ToRef CastToRef()
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelReference<ToRef>::Type) <= sizeof(MTX44));
        NW_STATIC_ASSERT(ut::IsReference<ToRef>::value);

        return *reinterpret_cast<typename ut::PeelReference<ToRef>::Type*>( this );
    }

    template <typename ToRef>
    ToRef CastToRef() const
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelReference<ToRef>::Type) <= sizeof(MTX44));
        NW_STATIC_ASSERT(ut::IsReference<ToRef>::value);

        return *reinterpret_cast<const typename ut::PeelReference<ToRef>::Type*>( this );
    }

    //! @brief メモリフォーマットが同じ型からのキャストです。
    //!
    //! @param[in] fromPtr キャスト元となるポインタです。
    //! @return キャスト結果です。
    template <typename FromPtr>
    static MTX44* CastFrom(FromPtr* fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX44));
        return reinterpret_cast<MTX44*>( fromPtr );
    }

    template <typename FromPtr>
    static const MTX44* CastFrom(const FromPtr* fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX44));
        return reinterpret_cast<const MTX44*>( fromPtr );
    }

    template <typename FromPtr>
    static MTX44& CastFrom(FromPtr& fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX44));
        return *reinterpret_cast<MTX44*>( &fromPtr );
    }

    template <typename FromPtr>
    static const MTX44& CastFrom(const FromPtr& fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX44));
        return *reinterpret_cast<const MTX44*>( &fromPtr );
    }

    //! @brief VEC4 型として行を取得します。
    //!
    //! @param[in] index 取得する行のインデックスです。
    //! @return 行の取得結果です。
    VEC4& GetRow(int index)
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, ROW_COUNT);
        return *reinterpret_cast<VEC4*>(&this->v[index]);
    }

    //! @brief VEC4 型として行を取得します。
    //!
    //! @param[in] index 取得する行のインデックスです。
    //! @return 行の取得結果です。
    const VEC4& GetRow(int index) const
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, ROW_COUNT);
        return *reinterpret_cast<const VEC4*>(&this->v[index]);
    }

    //! @brief VEC4 型として列を取得します。
    //!
    //! @param[in] index 取得する列のインデックスです。
    //! @return 列の取得結果です。
    VEC4 GetColumn(int index) const
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, COLUMN_COUNT);
        VEC4 column;
        column.x = this->m[0][index];
        column.y = this->m[1][index];
        column.z = this->m[2][index];
        column.w = this->m[3][index];
        return column;
    }

    //! @brief VEC4 型で列を設定します。
    //!
    //! @param[in] index 設定する列のインデックスです。
    //! @param[in] column 設定する VEC4 です。
    void SetColumn(int index, const VEC4& column)
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, COLUMN_COUNT);
        this->m[0][index] = column.x;
        this->m[1][index] = column.y;
        this->m[2][index] = column.z;
        this->m[3][index] = column.w;
    }
    //@}

    //----------------------------------------
    //! @name 演算
    //@{

    //! @brief 行列を右からかけた値を代入します。
    //!
    //! @param[in] rhs 右からかける行列です。
    //! @return 行列をかけた結果です。
    self_type& operator *= (const self_type& rhs) { return *MTX44Mult(this, this, &rhs); }

    self_type& operator += (const self_type& rhs) { return *MTX44Add(this, this, &rhs); }
    self_type& operator -= (const self_type& rhs) { return *MTX44Sub(this, this, &rhs); }

    self_type& operator *= (f32 value) { return *MTX44Mult(this, this, value); }
    self_type& operator /= (f32 value) { return operator*=(1.f / value); }

    self_type operator + () const { return *this; }
    self_type operator - () const
    {
        return MTX44(-f._00, -f._01, -f._02, -f._03,
                     -f._10, -f._11, -f._12, -f._13,
                     -f._20, -f._21, -f._22, -f._23,
                     -f._30, -f._31, -f._32, -f._33);
    }

    self_type operator + (const self_type& rhs) const { MTX44 tmp; return *MTX44Add(&tmp, this, &rhs); }
    self_type operator - (const self_type& rhs) const { MTX44 tmp; return *MTX44Sub(&tmp, this, &rhs); }

    self_type operator * (f32 value) const { MTX44 tmp; return *MTX44Mult(&tmp, this, value); }
    self_type operator / (f32 value) const { return *this * (1.f / value); }

    //! 転置を行います。
    //!
    //! @return 転置した結果です。
    self_type& Transpose() { return *MTX44Transpose(this, this); }
    //@}

    //----------------------------------------
    //! @name 設定
    //@{

    //! @brief     単位行列に設定します。
    //!
    //! @return    設定された行列を返します。
    self_type& SetIdentity() { return *MTX44Identity(this); }

    //! @brief     逆行列を設定します。
    //!
    //! @param[in] mtx 逆行列の元となる行列です。
    //!
    //! @return    設定された行列を返します。
    self_type& SetInverse( const MTX44& mtx )
    {
        MTX44Inverse(this, &mtx); return *this;
    }

    //! @brief     逆行列を設定します。
    //!
    //! @return    設定された行列を返します。
    self_type& Inverse()
    {
        MTX44Inverse(this, this); return *this;
    }

    //! @brief     行列の積を設定します。
    //!
    //! @param[in] lhs 左辺の行列です。
    //! @param[in] rhs 右辺の行列です。
    //!
    //! @return 設定された行列を返します。
    self_type& SetMult( const MTX44& lhs, const MTX44& rhs )
    {
        return *MTX44Mult( this, &lhs, &rhs );
    }

    //! @brief     スケール変換行列に設定します。
    //!
    //! @param[in] scale スケール値です。
    //!
    //! @return    設定された行列を返します。
    //!
    self_type& SetScale(const VEC3& scale) { return *MTX44Scale(this, &scale); }

    //! @brief     平行移動行列に設定します。
    //!
    //! @param[in] translate 平行移動値です。
    //!
    //! @return    設定された行列を返します。
    //!
    self_type& SetTranslate(const VEC3& translate)
    {
        return *MTX44Translate(this, &translate);
    }

    //! @brief     XYZ 順の回転行列に設定します。
    //!
    //! @param[in] rotate 回転値です。単位はラジアンです。
    //!
    //! @return    設定された行列を返します。
    //!
    self_type& SetRotateXyz(const VEC3& rotate)
    {
        return *MTX44RotXYZRad(this, rotate.x, rotate.y, rotate.z);
    }

    //! @brief     指定した軸周りの回転行列に設定します。
    //!
    //! @param[in] axis 基準となる軸のベクトルです。
    //! @param[in] theta 回転させる角度です。
    //!
    //! @return    設定された行列を返します。
    //!
    self_type& SetRotate(const VEC3& axis, f32 theta)
    {
        return *MTX44RotAxisRad(this, &axis, theta);
    }

    //! @brief        射影行列をニアクリッピング面での視錐台を元に設定します。
    //!               引数の順序が OpenGL 準拠になっているので注意してください。
    //!
    //! @param[in]    l     ニアクリッピング面での視錐台左辺の X 座標です。
    //! @param[in]    r     ニアクリッピング面での視錐台右辺の X 座標です。
    //! @param[in]    b     ニアクリッピング面での視錐台下辺の Y 座標です。
    //! @param[in]    t     ニアクリッピング面での視錐台上辺の Y 座標です。
    //! @param[in]    screenNear     ニアクリッピング面までの距離です。
    //! @param[in]    screenFar     ファークリッピング面までの距離です。
    //! @param[in]    pivot   画面の回転方向です。
    //!
    //! @return       設定された行列を返します。
    //!
    self_type& SetFrustum(f32 l, f32 r, f32 b, f32 t, f32 screenNear, f32 screenFar, PivotDirection pivot = PIVOT_NONE)
    {
        return *MTX44FrustumPivot(this, l, r, b, t, screenNear, screenFar, pivot);
    }

    //! @brief        正射影行列を設定します。
    //!               引数の順序が OpenGL 準拠になっているので注意してください。
    //!
    //! @param[in]    l     ニアクリッピング面での視錐台左辺の X 座標です。
    //! @param[in]    r     ニアクリッピング面での視錐台右辺の X 座標です。
    //! @param[in]    b     ニアクリッピング面での視錐台下辺の Y 座標です。
    //! @param[in]    t     ニアクリッピング面での視錐台上辺の Y 座標です。
    //! @param[in]    screenNear     ニアクリッピング面までの距離です。
    //! @param[in]    screenFar     ファークリッピング面までの距離です。
    //! @param[in]    pivot   画面の回転方向です。
    //!
    //! @return       設定された行列を返します。
    //!
    self_type& SetOrtho(f32 l, f32 r, f32 b, f32 t, f32 screenNear, f32 screenFar, PivotDirection pivot = PIVOT_NONE)
    {
        return *MTX44OrthoPivot(this, l, r, b, t, screenNear, screenFar, pivot);
    }

    //! @brief        射影行列を視野角とアスペクト比から設定します。
    //!
    //! @param[in]    fovyRad 縦方向の視野角（Radian）です。
    //! @param[in]    aspect  視野のアスペクト比(幅/高さ)です。
    //! @param[in]    screenNear       ニアクリッピング面までの距離です。
    //! @param[in]    screenFar       ファークリッピング面までの距離です。
    //! @param[in]    pivot   画面の回転方向です。
    //!
    //! @return       設定された行列を返します。
    //!
    self_type& SetPerspective(f32 fovyRad, f32 aspect, f32 screenNear, f32 screenFar, PivotDirection pivot = PIVOT_NONE)
    {
        return *MTX44PerspectivePivotRad(this, fovyRad, aspect, screenNear, screenFar, pivot);
    }

    //@}

    //----------------------------------------
    //! @name 比較
    //@{

    //! @brief 同値であれば true を返します。
    //!
    //! @param[in] rhs 比較対象となる MTX44 です。
    //! @return 比較結果です。
    bool operator == (const self_type& rhs) const { return ::std::memcmp(this, &rhs, sizeof(MTX44)) == 0; }

    //! @brief 同値でなければ true を返します。
    //!
    //! @param[in] rhs 比較対象となる MTX44 です。
    //! @return 比較結果です。
    bool operator != (const self_type& rhs) const { return ::std::memcmp(this, &rhs, sizeof(MTX44)) != 0; }

    //! @brief 単位行列かどうかを判定します。
    //!
    //! @return 単位行列であれば true を返します。
    bool IsIdentity() const { return MTX44IsIdentity(this); }
    //@}

    //! @brief 行列の内容をデバッグ出力に書き出します。
    //!
    //! @param[in] bNewline  true なら出力の最後に改行を出力します。
    //! @param[in] name      行列の名前を指定します。NULL指定可能です。
    void Report(bool bNewline = true, const char* name = NULL) const;
};

typedef struct MTX44 Matrix44;  //!< @briefprivate

}  // namespace math
}  // namespace nw

namespace nw {
namespace math {

//-- const 引数を参照にしたオーバーロード
inline MTX44* MTX44Copy(MTX44* pOut, const MTX44& m) { return MTX44Copy( pOut, &m ); }
inline bool   MTX44IsIdentity(const MTX44& m) { return MTX44IsIdentity( &m ); }

inline MTX44* MTX44Add(MTX44* pOut, const MTX44& m1, const MTX44& m2) { return MTX44Add( pOut, &m1, &m2 ); }
inline MTX44* MTX44Sub(MTX44* pOut, const MTX44& m1, const MTX44& m2) { return MTX44Sub( pOut, &m1, &m2 ); }
inline MTX44* MTX44Mult(MTX44* pOut, const MTX44& m, f32 f) { return MTX44Mult( pOut, &m, f ); }
inline MTX44* MTX44Mult(MTX44* pOut, const MTX44& m1, const MTX44& m2) { return MTX44Mult( pOut, &m1, &m2 ); }

inline MTX44* MTX44Transpose(MTX44* pOut, const MTX44& m) { return MTX44Transpose( pOut, &m ); }
inline MTX44* MTX44MultArray(MTX44* pOut, const MTX44& m1, const MTX44* pSrc, s32 count) { return MTX44MultArray( pOut, &m1, pSrc, count ); }
inline u32    MTX44Inverse(MTX44* pOut, const MTX44& m) { return MTX44Inverse( pOut, &m ); }

inline MTX44* MTX44RotAxisIdx(MTX44* pOut, const VEC3& vAxis, u32 idx) { return MTX44RotAxisIdx( pOut, &vAxis, idx ); }
inline MTX44* MTX44RotAxisRad(MTX44* pOut, const VEC3& vAxis, f32 fRad) { return MTX44RotAxisRad( pOut, &vAxis, fRad ); }
inline MTX44* MTX44RotAxisDeg(MTX44* pOut, const VEC3& vAxis, f32 fDeg) { return MTX44RotAxisDeg( pOut, &vAxis, fDeg ); }

inline MTX44* MTX44Scale(MTX44* pOut, const VEC3& S) { return MTX44Scale(pOut, &S); }
inline MTX44* MTX44MultScale(MTX44* pOut, const MTX44& M, const VEC3& S) { return MTX44MultScale(pOut, &M, &S); }
inline MTX44* MTX44MultScale(MTX44* pOut, const VEC3& S, const MTX44& M) { return MTX44MultScale(pOut, &S, &M); }

inline MTX44* MTX44Translate(MTX44* pOut, const VEC3& T) { return MTX44Translate(pOut, &T); }
inline MTX44* MTX44MultTranslate(MTX44* pOut, const MTX44& M, const VEC3& T) { return MTX44MultTranslate(pOut, &M, &T); }
inline MTX44* MTX44MultTranslate(MTX44* pOut, const VEC3& T, const MTX44& M) { return MTX44MultTranslate(pOut, &T, &M); }

}  // namespace math
}  // namespace nw

#if defined(NW_MATH_AS_INLINE)
#include <nw/math/inline/math_Matrix44.ipp>
#endif

#if defined(NW_PLATFORM_CTR)
#pragma pop
#endif

#endif
