﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MATH_MATRIX43_H_
#define NW_MATH_MATRIX43_H_

#include <cstring>
#include <nw/math/math_Config.h>

namespace nw {
namespace math {

struct MTX43;

namespace internal { namespace standard {

    MTX43* MTX43Add(MTX43* pOut, const MTX43* p1, const MTX43* p2);
    MTX43* MTX43Sub(MTX43* pOut, const MTX43* p1, const MTX43* p2);
    MTX43* MTX43Mult(MTX43* pOut, const MTX43* p1, const MTX43* p2);
    MTX43* MTX43Mult(MTX43* pOut, const MTX43* p, f32 f);

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

    MTX43* MTX43Add(MTX43* pOut, const MTX43* p1, const MTX43* p2);
    MTX43* MTX43Sub(MTX43* pOut, const MTX43* p1, const MTX43* p2);
    MTX43* MTX43Mult(MTX43* pOut, const MTX43* p1, const MTX43* p2);
    MTX43* MTX43Mult(MTX43* pOut, const MTX43* p, f32 f);

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

//---------------------------------------------------------------------------
//    MTX43用の関数
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX43* MTX43Zero(MTX43* pOut);
NW_MATH_INLINE MTX43* MTX43Identity(MTX43* pOut);
NW_MATH_INLINE bool   MTX43IsIdentity(const MTX43* p);
NW_MATH_INLINE MTX43* MTX43Copy(MTX43* pOut, const MTX43* p);
NW_MATH_INLINE MTX43* MTX43Add(MTX43* pOut, const MTX43* p1, const MTX43* p2);
NW_MATH_INLINE MTX43* MTX43Sub(MTX43* pOut, const MTX43* p1, const MTX43* p2);
NW_MATH_INLINE MTX43* MTX43Mult(MTX43* pOut, const MTX43* p1, const MTX43* p2);
NW_MATH_INLINE MTX43* MTX43Mult(MTX43* pOut, const MTX43* p, f32 f);

//========================================================================
//        クラスの定義
//========================================================================

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        4行3列の行列の構造体です
//---------------------------------------------------------------------------
#if defined(_WIN32)
#pragma warning(push)
#pragma warning(disable:4201)
#endif
struct MTX43_
{
    struct BaseData
    {
        f32 _00, _01, _02;
        f32 _10, _11, _12;
        f32 _20, _21, _22;
        f32 _30, _31, _32;
    };

    union
    {
        struct
        {
            f32 _00, _01, _02;
            f32 _10, _11, _12;
            f32 _20, _21, _22;
            f32 _30, _31, _32;
        };
        BaseData f;
        f32 m[4][3];
        f32 a[12];
        VEC3_ v[4];
    };
};
#if defined(_WIN32)
#pragma warning(pop)
#endif
//---------------------------------------------------------------------------
//! @brief        ４行3列の行列クラスです。
//---------------------------------------------------------------------------
struct MTX43 : public MTX43_
{
public:
    static const int ROW_COUNT    = 4; //!< 行数です。
    static const int COLUMN_COUNT = 3; //!< 列数です。

    //! @brief 単位行列です。
    //!
    //! @return 単位行列を返します。
    static const MTX43& Identity()
    {
        static const MTX43 identity(
            1.0f, 0.0f, 0.0f,
            0.0f, 1.0f, 0.0f,
            0.0f, 0.0f, 1.0f,
            0.0f, 0.0f, 0.0f);

        return identity;
    }
    typedef MTX43 self_type; //!< 自分の型です。
    typedef f32   value_type; //!< 要素の型です。

public:
    //----------------------------------------
    //! @name 作成
    //@{

    //! @brief コンストラクタです。
    MTX43() {}

    //! @brief コンストラクタです。
    //!
    //! @param[in] p メンバ変数の初期化に用いる浮動小数点配列の先頭です。
    explicit MTX43(const f32* p) { (void)MTX43Copy(this, (MTX43*)p); }

    //! @brief コンストラクタです。
    //!
    //! @param[in] x00 メンバ変数 _00 の初期化に用いる浮動小数点値です。
    //! @param[in] x01 メンバ変数 _01 の初期化に用いる浮動小数点値です。
    //! @param[in] x02 メンバ変数 _02 の初期化に用いる浮動小数点値です。
    //! @param[in] x10 メンバ変数 _10 の初期化に用いる浮動小数点値です。
    //! @param[in] x11 メンバ変数 _11 の初期化に用いる浮動小数点値です。
    //! @param[in] x12 メンバ変数 _12 の初期化に用いる浮動小数点値です。
    //! @param[in] x20 メンバ変数 _20 の初期化に用いる浮動小数点値です。
    //! @param[in] x21 メンバ変数 _21 の初期化に用いる浮動小数点値です。
    //! @param[in] x22 メンバ変数 _22 の初期化に用いる浮動小数点値です。
    //! @param[in] x30 メンバ変数 _30 の初期化に用いる浮動小数点値です。
    //! @param[in] x31 メンバ変数 _31 の初期化に用いる浮動小数点値です。
    //! @param[in] x32 メンバ変数 _32 の初期化に用いる浮動小数点値です。
    MTX43(f32 x00, f32 x01, f32 x02,
          f32 x10, f32 x11, f32 x12,
          f32 x20, f32 x21, f32 x22,
          f32 x30, f32 x31, f32 x32)
    {
        f._00 = x00; f._01 = x01; f._02 = x02;
        f._10 = x10; f._11 = x11; f._12 = x12;
        f._20 = x20; f._21 = x21; f._22 = x22;
        f._30 = x30; f._31 = x31; f._32 = x32;
    }
    //@}

    //----------------------------------------
    //! @name 変換
    //@{

    //! @brief f32 型へのキャストです。
    operator f32*() { return this->a; }
    //! @brief f32 型へのキャストです。
    operator const f32*() const { return this->a; }

    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    f32* ToF32() { return this->a; }
    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    const f32* ToF32() const { return this->a; }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast()
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelPointer<ToPtr>::Type) <= sizeof(MTX43));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast() const
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelPointer<ToPtr>::Type) <= sizeof(MTX43));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    template <typename ToRef>
    ToRef CastToRef()
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelReference<ToRef>::Type) <= sizeof(MTX43));
        NW_STATIC_ASSERT(ut::IsReference<ToRef>::value);

        return *reinterpret_cast<typename ut::PeelReference<ToRef>::Type*>( this );
    }

    template <typename ToRef>
    ToRef CastToRef() const
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelReference<ToRef>::Type) <= sizeof(MTX43));
        NW_STATIC_ASSERT(ut::IsReference<ToRef>::value);

        return *reinterpret_cast<const typename ut::PeelReference<ToRef>::Type*>( this );
    }

    //! @brief メモリフォーマットが同じ型からのキャストです。
    //!
    //! @param[in] fromPtr キャスト元となるポインタです。
    //! @return キャスト結果です。
    template <typename FromPtr>
    static MTX43* CastFrom(FromPtr* fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX43));
        return reinterpret_cast<MTX43*>( fromPtr );
    }

    template <typename FromPtr>
    static const MTX43* CastFrom(const FromPtr* fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX43));
        return reinterpret_cast<const MTX43*>( fromPtr );
    }

    template <typename FromPtr>
    static MTX43& CastFrom(FromPtr& fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX43));
        return *reinterpret_cast<MTX43*>( &fromPtr );
    }

    template <typename FromPtr>
    static const MTX43& CastFrom(const FromPtr& fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX43));
        return *reinterpret_cast<const MTX43*>( &fromPtr );
    }

    //! @brief VEC3 型として行を取得します。
    //!
    //! @param[in] index 取得する行のインデックスです。
    //! @return 行の取得結果です。
    VEC3& GetRow(int index)
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, ROW_COUNT);
        return *reinterpret_cast<VEC3*>(&this->v[index]);
    }

    //! @brief VEC3 型として行を取得します。
    //!
    //! @param[in] index 取得する行のインデックスです。
    //! @return 行の取得結果です。
    const VEC3& GetRow(int index) const
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, ROW_COUNT);
        return *reinterpret_cast<const VEC3*>(&this->v[index]);
    }

    //! @brief VEC4 型として列を取得します。
    //!
    //! @param[in] index 取得する列のインデックスです。
    //! @return 列の取得結果です。
    VEC4 GetColumn(int index) const
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, COLUMN_COUNT);
        VEC4 column;
        column.x = this->m[0][index];
        column.y = this->m[1][index];
        column.z = this->m[2][index];
        column.w = this->m[3][index];
        return column;
    }

    //! @brief VEC4 型で列を設定します。
    //!
    //! @param[in] index 設定する列のインデックスです。
    //! @param[in] column 設定する VEC3 です。
    void SetColumn(int index, const VEC4& column)
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, COLUMN_COUNT);
        this->m[0][index] = column.x;
        this->m[1][index] = column.y;
        this->m[2][index] = column.z;
        this->m[3][index] = column.w;
    }

    //@}

    //----------------------------------------
    //! @name 演算
    //@{

    self_type& operator += (const self_type& rhs) { return *MTX43Add(this, this, &rhs); }
    self_type& operator -= (const self_type& rhs) { return *MTX43Sub(this, this, &rhs); }

    self_type& operator *= (const self_type& rhs) { return *MTX43Mult(this, this, &rhs); }
    self_type& operator *= (f32 value) { return *MTX43Mult(this, this, value); }
    self_type& operator /= (f32 value) { return operator*=(1.f / value); }

    self_type operator + () const { return *this; }
    self_type operator - () const
        {
            return MTX43(-f._00, -f._01, -f._02,
                         -f._10, -f._11, -f._12,
                         -f._20, -f._21, -f._22,
                         -f._30, -f._31, -f._32);
        }

    self_type operator + (const self_type& rhs) const { MTX43 tmp; return *MTX43Add(&tmp, this, &rhs); }
    self_type operator - (const self_type& rhs) const { MTX43 tmp; return *MTX43Add(&tmp, this, &rhs); }

    self_type operator * (f32 value) const { MTX43 tmp; return *MTX43Mult(&tmp, this, value); }
    self_type operator / (f32 value) const { return *this * (1.f / value); }

    //@}

    //----------------------------------------
    //! @name 設定
    //@{

    //! @brief 単位行列に設定します。
    //!
    //! @return    設定された行列を返します。
    self_type& SetIdentity() { return *MTX43Identity(this); }

    //! @brief 行列の積を設定します。
    //!
    //!
    //! @param[in] lhs 左辺の行列です。
    //! @param[in] rhs 右辺の行列です。
    //!
    //! @return 設定された行列を返します。
    self_type& SetMult( const MTX43& lhs, const MTX43& rhs )
    {
        return *MTX43Mult( this, &lhs, &rhs );
    }

    //@}

    //----------------------------------------
    //! @name 比較
    //@{

    //! @brief 同値であれば true を返します。
    //!
    //! @param[in] rhs 比較対象となる MTX43 です。
    //! @return 比較結果です。
    bool operator == (const self_type& rhs) const { return ::std::memcmp(this, &rhs, sizeof(MTX43)) == 0; }

    //! @brief 同値でなければ true を返します。
    //!
    //! @param[in] rhs 比較対象となる MTX43 です。
    //! @return 比較結果です。
    bool operator != (const self_type& rhs) const { return ::std::memcmp(this, &rhs, sizeof(MTX43)) != 0; }

    //! @brief 単位行列かどうかを判定します。
    //!
    //! @return 単位行列であれば true を返します。
    bool IsIdentity() const { return MTX43IsIdentity(this); }

    //@}

    //! @brief 行列の内容をデバッグ出力に書き出します。
    //!
    //! @param[in] bNewline  true なら出力の最後に改行を出力します。
    //! @param[in] name      行列の名前を指定します。NULL指定可能です。
    void Report(bool bNewline = true, const char* name = NULL) const;
};

typedef struct MTX43 Matrix43;  //!< @briefprivate

//-- const 引数を参照にしたオーバーロード
inline bool   MTX43IsIdentity(const MTX43& m) { return MTX43IsIdentity( &m ); }
inline MTX43* MTX43Copy(MTX43* pOut, const MTX43& m) { return MTX43Copy( pOut, &m ); }
inline MTX43* MTX43Add(MTX43* pOut, const MTX43& m1, const MTX43& m2) { return MTX43Add( pOut, &m1, &m2 ); }
inline MTX43* MTX43Sub(MTX43* pOut, const MTX43& m1, const MTX43& m2) { return MTX43Sub( pOut, &m1, &m2 ); }
inline MTX43* MTX43Mult(MTX43* pOut, const MTX43& m1, const MTX43& m2) { return MTX43Mult( pOut, &m1, &m2 ); }
inline MTX43* MTX43Mult(MTX43* pOut, const MTX43& m, f32 f) { return MTX43Mult( pOut, &m, f); }

}  // namespace math
}  // namespace nw


#if defined(NW_MATH_AS_INLINE)
#include <nw/math/inline/math_Matrix43.ipp>
#endif

#endif
