﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MATH_MATRIX34_H_
#define NW_MATH_MATRIX34_H_

#include <cstring>
#include <nw/math/math_Config.h>

namespace nw {
namespace math {

struct MTX34;

namespace internal { namespace standard {

    VEC3* VEC3Transform(VEC3* pOut, const MTX34* pM, const VEC3* pV);
    MTX34* MTX34Add(MTX34* pOut, const MTX34* p1, const MTX34* p2);
    MTX34* MTX34Sub(MTX34* pOut, const MTX34* p1, const MTX34* p2);
    MTX34* MTX34MultScale(MTX34* pOut, const MTX34* pM, const VEC3* pS);
    MTX34* MTX34MultScale(MTX34* pOut, const VEC3* pS, const MTX34* pM);
    MTX34* MTX34Mult(MTX34* pOut, const MTX34* p, f32 f);
    MTX34* MTX34MAdd(MTX34* pOut, f32 t, const MTX34* p1, const MTX34* p2);
    MTX34* MTX34Mult(MTX34* pOut, const MTX34* p1, const MTX34* p2);
    u32 MTX34Inverse(MTX34* pOut, const MTX34* p);
    u32 MTX34InvTranspose(MTX34* pOut, const MTX34* p);

}} // namespace internal::standard

namespace internal { namespace intrinsics {

    VEC3* VEC3Transform(VEC3* pOut, const MTX34* pM, const VEC3* pV);
    MTX34* MTX34Add(MTX34* pOut, const MTX34* p1, const MTX34* p2);
    MTX34* MTX34Sub(MTX34* pOut, const MTX34* p1, const MTX34* p2);
    MTX34* MTX34MultScale(MTX34* pOut, const MTX34* pM, const VEC3* pS);
    MTX34* MTX34MultScale(MTX34* pOut, const VEC3* pS, const MTX34* pM);
    MTX34* MTX34Mult(MTX34* pOut, const MTX34* p, f32 f);
    MTX34* MTX34MAdd(MTX34* pOut, f32 t, const MTX34* p1, const MTX34* p2);
    MTX34* MTX34Mult(MTX34* pOut, const MTX34* p1, const MTX34* p2);
    u32 MTX34Inverse(MTX34* pOut, const MTX34* p);
    u32 MTX34InvTranspose(MTX34* pOut, const MTX34* p);

}} // namespace internal::intrinsics

//---------------------------------------------------------------------------
//    MTX34用の関数
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX34* MTX34Zero(MTX34* pOut);
NW_MATH_INLINE bool MTX34IsIdentity(const MTX34* p);
NW_MATH_INLINE MTX34* MTX34Identity(MTX34* pOut);
NW_MATH_INLINE MTX34* MTX34Copy(MTX34* pOut, const MTX34* p);

NW_MATH_INLINE MTX34* MTX34Add(MTX34* pOut, const MTX34* p1, const MTX34* p2);
NW_MATH_INLINE MTX34* MTX34Sub(MTX34* pOut, const MTX34* p1, const MTX34* p2);
NW_MATH_INLINE MTX34* MTX34Mult(MTX34* pOut, const MTX34* p, f32 f);
NW_MATH_INLINE MTX34* MTX34Mult(MTX34* pOut, const MTX34* p1, const MTX34* p2);
NW_MATH_INLINE MTX34* MTX34MAdd(MTX34* pOut, f32 t, const MTX34* p1, const MTX34* p2);

NW_MATH_INLINE MTX33* MTX34ToMTX33(MTX33* pOut, const MTX34* pM);
NW_MATH_INLINE MTX34* MTX33ToMTX34(MTX34* pOut, const MTX33* pM);

NW_MATH_INLINE MTX34* MTX34MultArray(MTX34* pOut, const MTX34* p1, const MTX34* pSrc, s32 count);
NW_MATH_INLINE u32    MTX34Inverse(MTX34* pOut, const MTX34* p);
NW_MATH_INLINE MTX34* MTX34Transpose(MTX34* pOut, const MTX34* p);
NW_MATH_INLINE u32    MTX34InvTranspose(MTX34* pOut, const MTX34* p);

NW_MATH_INLINE MTX34* MTX34LookAt(MTX34* pOut, const VEC3* pCamPos, const VEC3* pCamUp, const VEC3* pTarget);
NW_MATH_INLINE MTX34* MTX34LookAt(MTX34* pOut, const VEC3* pCamPos, f32 twist, const VEC3* pTarget);
NW_MATH_INLINE MTX34* MTX34CameraRotate(MTX34* pOut, const VEC3* pCamPos, const VEC3* pCamRotate);

NW_MATH_INLINE MTX34* MTX34TextureProjectionFrustum(MTX34* pOut, f32 l, f32 r, f32 b, f32 t, f32 n, f32 scaleS, f32 scaleT, f32 translateS, f32 translateT);
NW_MATH_INLINE MTX34* MTX34TextureProjectionPerspective(MTX34* pOut, f32 fovy, f32 aspect, f32 scaleS, f32 scaleT, f32 translateS, f32 translateT);
NW_MATH_INLINE MTX34* MTX34TextureProjectionOrtho(MTX34* pOut, f32 l, f32 r, f32 b, f32 t, f32 scaleS, f32 scaleT, f32 translateS, f32 translateT);

NW_MATH_INLINE MTX34* MTX34RotXYZIdx(MTX34* pOut, u32 idxX, u32 idxY, u32 idxZ);
NW_MATH_INLINE MTX34* MTX34RotXYZTranslateIdx(MTX34* pOut, u32 idxX, u32 idxY, u32 idxZ, const VEC3* pT);
NW_MATH_INLINE MTX34* MTX34RotAxisIdx(MTX34* pOut, const VEC3* pAxis, u32 idx);

NW_MATH_INLINE MTX34* MTX34Scale(MTX34* pOut, const VEC3* pS);
NW_MATH_INLINE MTX34* MTX34MultScale(MTX34* pOut, const MTX34* pM, const VEC3* pS);
NW_MATH_INLINE MTX34* MTX34MultScale(MTX34* pOut, const VEC3* pS, const MTX34* pM);

NW_MATH_INLINE MTX34* MTX34Translate(MTX34* pOut, const VEC3* pT);
NW_MATH_INLINE MTX34* MTX34MultTranslate(MTX34* pOut, const MTX34* pM, const VEC3* pT);
NW_MATH_INLINE MTX34* MTX34MultTranslate(MTX34* pOut, const VEC3* pT, const MTX34* pM);

NW_MATH_INLINE MTX34* MTX34MakeSRT(MTX34* pOut, const VEC3* pS, const VEC3* pR, const VEC3* pT);
NW_MATH_INLINE MTX34* MTX34MakeST(MTX34* pOut, const VEC3* pS, const VEC3* pT);
NW_MATH_INLINE MTX34* MTX34MakeS(MTX34* pOut, const VEC3* pS);

NW_MATH_INLINE MTX34* QUATToMTX34(MTX34* pOut, const QUAT* p);

NW_FORCE_INLINE MTX34* MTX34Copy(MTX34* pOut, const MTX34* p);
NW_FORCE_INLINE MTX34* MTX34Mult(MTX34* pOut, const MTX34* p1, const MTX34* p2);


// そのうち実装する
//MTX34* MTX34Inv(MTX34* pOut, f32* pDet, const MTX34* p);
//MTX34* MTX34InvTranspose(MTX34* pOut, f32* pDet, const MTX34* p);
//f32 MTX34Det(const MTX34* p);

//----------------------------------------
//! @name    行列
//@{

//---------------------------------------------------------------------------
//! @brief        回転行列を作成します。
//!
//! @param[out]   pOut   計算結果を受け取るバッファへのポインタ。
//! @param[in]    fRadX  ラジアン単位での X 軸周りの角度
//! @param[in]    fRadY  ラジアン単位での Y 軸周りの角度
//! @param[in]    fRadZ  ラジアン単位での Z 軸周りの角度
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline MTX34*
MTX34RotXYZRad(MTX34* pOut, f32 fRadX, f32 fRadY, f32 fRadZ)
{
    return MTX34RotXYZIdx(pOut, NW_MATH_RAD_TO_IDX(fRadX), NW_MATH_RAD_TO_IDX(fRadY), NW_MATH_RAD_TO_IDX(fRadZ));
}

//---------------------------------------------------------------------------
//! @brief        回転行列を作成します。
//!
//! @param[out]   pOut   計算結果を受け取るバッファへのポインタ。
//! @param[in]    fDegX  1 円周を 360.0 とする単位での X 軸周りの角度
//! @param[in]    fDegY  1 円周を 360.0 とする単位での Y 軸周りの角度
//! @param[in]    fDegZ  1 円周を 360.0 とする単位での Z 軸周りの角度
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline MTX34*
MTX34RotXYZDeg(MTX34* pOut, f32 fDegX, f32 fDegY, f32 fDegZ)
{
    return MTX34RotXYZIdx(pOut, NW_MATH_DEG_TO_IDX(fDegX), NW_MATH_DEG_TO_IDX(fDegY), NW_MATH_DEG_TO_IDX(fDegZ));
}

//---------------------------------------------------------------------------
//! @brief        指定する軸の周りを回転させる回転行列を作成します。
//!
//! @param[out]   pOut   計算結果を受け取るバッファへのポインタ。
//! @param[in]    pAxis  回転軸を指定するベクトルへのポインタ。
//! @param[in]    fRad   ラジアン単位での回転量
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline MTX34*
MTX34RotAxisRad(MTX34* pOut, const VEC3* pAxis, f32 fRad)
{
    return MTX34RotAxisIdx(pOut, pAxis, NW_MATH_RAD_TO_IDX(fRad));
}

//---------------------------------------------------------------------------
//! @brief        指定する軸の周りを回転させる回転行列を作成します。
//!
//! @param[out]   pOut   計算結果を受け取るバッファへのポインタ。
//! @param[in]    pAxis  回転軸を指定するベクトルへのポインタ。
//! @param[in]    fDeg   1 円周を 360.0 とする単位での回転量
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline MTX34*
MTX34RotAxisDeg(MTX34* pOut, const VEC3* pAxis, f32 fDeg)
{
    return MTX34RotAxisIdx(pOut, pAxis, NW_MATH_DEG_TO_IDX(fDeg));
}

//@}

//========================================================================
//        クラスの定義
//========================================================================

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        3行4列の行列の構造体です
//---------------------------------------------------------------------------
#if defined(_WIN32)
#pragma warning(push)
#pragma warning(disable:4201)
#endif
struct MTX34_
{
    struct BaseData
    {
        f32 _00, _01, _02, _03;
        f32 _10, _11, _12, _13;
        f32 _20, _21, _22, _23;
    };
    union
    {
        struct
        {
            f32 _00, _01, _02, _03;
            f32 _10, _11, _12, _13;
            f32 _20, _21, _22, _23;
        };
        BaseData f;
        f32 m[3][4];
        f32 a[12];
        VEC4_ v[3];
    };
};
#if defined(_WIN32)
#pragma warning(pop)
#endif
//---------------------------------------------------------------------------
//! @brief        3行４列の行列クラスです。
//---------------------------------------------------------------------------
struct MTX34 : public MTX34_
{
public:
    static const int ROW_COUNT = 3; //!< 行数です。
    static const int COLUMN_COUNT = 4; //!< 列数です。

    //! @brief 単位行列です。
    //!
    //! @return 単位行列を返します。
    static const MTX34& Identity()
    {
        static const MTX34 identity(
            1.0f, 0.0f, 0.0f, 0.0f,
            0.0f, 1.0f, 0.0f, 0.0f,
            0.0f, 0.0f, 1.0f, 0.0f);

        return identity;
    }

    typedef MTX34 self_type; //!< 自分の型です。
    typedef f32   value_type; //!< 要素の型です。
public:
    //----------------------------------------
    //! @name 作成
    //@{

    //! @brief コンストラクタです。
    MTX34() {}

    //! @brief コンストラクタです。
    //!
    //! @param[in] p メンバ変数の初期化に用いる浮動小数点配列の先頭です。
    explicit MTX34(const f32* p) { (void)MTX34Copy(this, (MTX34*)p); }

    //! @brief コピーコンストラクタです。
    //!
    //! @param[in] rhs メンバ変数の初期化に用いる MTX34 です。
    MTX34(const MTX34& rhs) { (void)MTX34Copy(this, &rhs); }

    //! @brief コンストラクタです。
    //!
    //! @param[in] rhs メンバ変数の初期化に用いる MTX33 です。
    explicit MTX34(const MTX33& rhs) { MTX33ToMTX34(this, &rhs); }

    //! @brief コンストラクタです。
    //!
    //! @param[in] x00 メンバ変数 _00 の初期化に用いる浮動小数点値です。
    //! @param[in] x01 メンバ変数 _01 の初期化に用いる浮動小数点値です。
    //! @param[in] x02 メンバ変数 _02 の初期化に用いる浮動小数点値です。
    //! @param[in] x03 メンバ変数 _03 の初期化に用いる浮動小数点値です。
    //! @param[in] x10 メンバ変数 _10 の初期化に用いる浮動小数点値です。
    //! @param[in] x11 メンバ変数 _11 の初期化に用いる浮動小数点値です。
    //! @param[in] x12 メンバ変数 _12 の初期化に用いる浮動小数点値です。
    //! @param[in] x13 メンバ変数 _13 の初期化に用いる浮動小数点値です。
    //! @param[in] x20 メンバ変数 _20 の初期化に用いる浮動小数点値です。
    //! @param[in] x21 メンバ変数 _21 の初期化に用いる浮動小数点値です。
    //! @param[in] x22 メンバ変数 _22 の初期化に用いる浮動小数点値です。
    //! @param[in] x23 メンバ変数 _23 の初期化に用いる浮動小数点値です。
    MTX34(f32 x00, f32 x01, f32 x02, f32 x03,
          f32 x10, f32 x11, f32 x12, f32 x13,
          f32 x20, f32 x21, f32 x22, f32 x23)
    {
        f._00 = x00; f._01 = x01; f._02 = x02; f._03 = x03;
        f._10 = x10; f._11 = x11; f._12 = x12; f._13 = x13;
        f._20 = x20; f._21 = x21; f._22 = x22; f._23 = x23;
    }

    //@}

    //----------------------------------------
    //! @name 変換
    //@{

    //! @brief f32 型へのキャストです。
    operator f32*() { return this->a; }

    //! @brief f32 型へのキャストです。
    operator const f32*() const { return this->a; }

    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    f32* ToF32() { return this->a; }
    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    const f32* ToF32() const { return this->a; }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast()
    {
        NW_STATIC_ASSERT(sizeof(typename ut::PeelPointer<ToPtr>::Type) <= sizeof(MTX34));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast() const
    {
        NW_STATIC_ASSERT(sizeof(typename ut::PeelPointer<ToPtr>::Type) <= sizeof(MTX34));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    template <typename ToRef>
    ToRef CastToRef()
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelReference<ToRef>::Type) <= sizeof(MTX34));
        NW_STATIC_ASSERT(ut::IsReference<ToRef>::value);

        return *reinterpret_cast<typename ut::PeelReference<ToRef>::Type*>( this );
    }

    template <typename ToRef>
    ToRef CastToRef() const
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelReference<ToRef>::Type) <= sizeof(MTX34));
        NW_STATIC_ASSERT(ut::IsReference<ToRef>::value);

        return *reinterpret_cast<const typename ut::PeelReference<ToRef>::Type*>( this );
    }

    //! @brief メモリフォーマットが同じ型からのキャストです。
    //!
    //! @param[in] fromPtr キャスト元となるポインタです。
    //! @return キャスト結果です。
    template <typename FromPtr>
    static MTX34* CastFrom(FromPtr* fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX34));
        return reinterpret_cast<MTX34*>( fromPtr );
    }

    template <typename FromPtr>
    static const MTX34* CastFrom(const FromPtr* fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX34));
        return reinterpret_cast<const MTX34*>( fromPtr );
    }

    template <typename FromPtr>
    static MTX34& CastFrom(FromPtr& fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX34));
        return *reinterpret_cast<MTX34*>( &fromPtr );
    }

    template <typename FromPtr>
    static const MTX34& CastFrom(const FromPtr& fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX34));
        return *reinterpret_cast<const MTX34*>( &fromPtr );
    }

    //! @brief VEC4 型として行を取得します。
    //!
    //! @param[in] index 取得する行のインデックスです。
    //! @return 行の取得結果です。
    VEC4& GetRow(int index)
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, ROW_COUNT);
        return *reinterpret_cast<VEC4*>(&this->v[index]);
    }

    //! @brief VEC4 型として行を取得します。
    //!
    //! @param[in] index 取得する行のインデックスです。
    //! @return 行の取得結果です。
    const VEC4& GetRow(int index) const
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, ROW_COUNT);
        return *reinterpret_cast<const VEC4*>(&this->v[index]);
    }

    //! @brief VEC3 型として列を取得します。
    //!
    //! @param[in] index 取得する列のインデックスです。
    //! @return 列の取得結果です。
    VEC3 GetColumn(int index) const
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, COLUMN_COUNT);
        VEC3 column;
        column.x = this->m[0][index];
        column.y = this->m[1][index];
        column.z = this->m[2][index];
        return column;
    }

    //! @brief VEC3 型で列を設定します。
    //!
    //! @param[in] index 設定する列のインデックスです。
    //! @param[in] column 設定する VEC3 です。
    void SetColumn(int index, const VEC3& column)
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, COLUMN_COUNT);
        this->m[0][index] = column.x;
        this->m[1][index] = column.y;
        this->m[2][index] = column.z;
    }
    //@}

    //----------------------------------------
    //! @name 演算
    //@{
    self_type& operator += (const self_type& rhs) { return *MTX34Add(this, this, &rhs); }
    self_type& operator -= (const self_type& rhs) { return *MTX34Sub(this, this, &rhs); }

    //! @brief 浮動小数を右からかけた値を代入します。
    //!
    //! @param[in] f 右からかける浮動小数です。
    //! @return 浮動小数をかけた結果です。
    self_type& operator *= (f32 value) { return *MTX34Mult(this, this, value); }
    self_type& operator /= (f32 value) { return operator*=(1.f / value); }

    self_type operator + () const { return *this; }
    self_type operator - () const
    {
        return MTX34(-f._00, -f._01, -f._02, -f._03,
                     -f._10, -f._11, -f._12, -f._13,
                     -f._20, -f._21, -f._22, -f._23);
    }

    self_type operator + (const self_type& rhs) const { MTX34 tmp; return *MTX34Add(&tmp, this, &rhs); }
    self_type operator - (const self_type& rhs) const { MTX34 tmp; return *MTX34Sub(&tmp, this, &rhs); }

    self_type operator * (f32 value) const { MTX34 tmp; return *MTX34Mult(&tmp, this, value); }
    self_type operator / (f32 value) const { return *this * (1.f / value); }

    //! @brief 転置を行います。
    //!
    //! @return 転置した結果です。
    self_type& Transpose() { return *MTX34Transpose(this, this); }

    //@}

    //----------------------------------------
    //! @name 設定
    //@{

    //! @brief     逆行列を設定します。
    //!
    //! @param[in] mtx 逆行列の元となる行列です。
    //!
    //! @return    設定結果として、オブジェクト自身の参照を返します。
    self_type& SetInverse( const MTX34& mtx )
    {
        MTX34Inverse(this, &mtx); return *this;
    }

    //! @brief     逆行列を設定します。
    //!
    //! @return    設定結果として、オブジェクト自身の参照を返します。
    self_type& Inverse()
    {
        MTX34Inverse(this, this); return *this;
    }

    //! @brief     行列の積を設定します。
    //!
    //! @param[in] lhs 左辺の行列です。
    //! @param[in] rhs 右辺の行列です。
    //!
    //! @return    設定結果として、オブジェクト自身の参照を返します。
    self_type& SetMult( const MTX34& lhs, const MTX34& rhs )
    {
        return *MTX34Mult( this, &lhs, &rhs );
    }


    //! @brief     単位行列に設定します。
    //!
    //! @return    設定結果として、オブジェクト自身の参照を返します。
    self_type& SetIdentity() { return *MTX34Identity(this); }

    //! @brief     スケール変換行列に設定します。
    //!
    //! @param[in] scale スケール値です。
    //!
    //! @return    設定結果として、オブジェクト自身の参照を返します。
    //!
    self_type& SetScale(const VEC3& scale) { return *MTX34Scale(this, &scale); }

    //! @brief     平行移動行列に設定します。
    //!
    //! @param[in] translate 平行移動値です。
    //!
    //! @return    設定結果として、オブジェクト自身への参照を返します。
    //!
    self_type& SetTranslate(const VEC3& translate)
    {
        return *MTX34Translate(this, &translate);
    }

    //! @brief     XYZ 順の回転行列に設定します。
    //!
    //! @param[in] rotate 回転値です。単位はラジアンです。
    //!
    //! @return    設定結果として、オブジェクト自身への参照を返します。
    //!
    self_type& SetRotateXyz(const VEC3& rotate)
    {
        return *MTX34RotXYZRad(this, rotate.x, rotate.y, rotate.z);
    }

    //! @brief     指定した軸周りの回転行列に設定します。
    //!
    //! @param[in] axis 基準となる軸のベクトルです。
    //! @param[in] theta 回転させる角度です。
    //!
    //! @return    設定結果として、オブジェクト自身への参照を返します。
    //!
    self_type& SetRotate(const VEC3& axis, f32 theta)
    {
        return *MTX34RotAxisRad(this, &axis, theta);
    }

    //! @brief     クォータニオンから回転行列を設定します。
    //!
    //! @param[in] quaternion 元となるクォータニオンです。
    //!
    //! @return    設定結果として、オブジェクト自身への参照を返します。
    //!
    self_type& SetRotate(const QUAT& quaternion)
    {
        return *QUATToMTX34(this, &quaternion);
    }

    //---------------------------------------------------------------------------
    //! @brief        カメラ行列を設定します。
    //!
    //! @param[in]    camPos  カメラの位置を指定するベクトルへのポインタ。
    //! @param[in]    camUp   カメラの上方向を指定するベクトルへのポインタ。
    //! @param[in]    target  カメラの注視点を指定するベクトルへのポインタ。
    //!
    //! @return       設定された行列を返します。
    //---------------------------------------------------------------------------
    self_type& SetLookAt(const VEC3& camPos, const VEC3& camUp, const VEC3& target)
    {
        return *MTX34LookAt(this, &camPos, &camUp, &target);
    }

    //---------------------------------------------------------------------------
    //! @brief        カメラ行列を設定します。
    //!
    //! @param[in]    camPos  カメラの位置を指定するベクトルへのポインタ。
    //! @param[in]    twist   カメラの視線方向に対する回転角度。
    //! @param[in]    target  カメラの注視点を指定するベクトルへのポインタ。
    //!
    //! @return       設定された行列を返します。
    //---------------------------------------------------------------------------
    self_type& SetLookAt(const VEC3& camPos, f32 twist, const VEC3& target)
    {
        return *MTX34LookAt(this, &camPos, twist, &target);
    }

    //---------------------------------------------------------------------------
    //! @brief        カメラ行列を設定します。
    //!
    //! @param[in]    camPos     カメラの位置を指定するベクトルへのポインタ。
    //! @param[in]    camRotate  カメラの回転量を指定するベクトルへのポインタ。
    //!
    //! @return       設定された行列を返します。
    //---------------------------------------------------------------------------
    self_type& SetCameraRotate(const VEC3& camPos, const VEC3& camRotate)
    {
        return *MTX34CameraRotate(this, &camPos, &camRotate);
    }

    //---------------------------------------------------------------------------
    //! @brief        テクスチャマトリクス用射影行列を視錐体を元に作成します。
    //!
    //! @param[in]    l           ニアクリップの左辺です。
    //! @param[in]    r           ニアクリップの右辺です。
    //! @param[in]    b           ニアクリップの下辺です。
    //! @param[in]    t           ニアクリップの上辺です。
    //! @param[in]    n           カメラからニアクリップまでの距離です。
    //! @param[in]    scaleS      S軸方向のスケール値です。
    //! @param[in]    scaleT      T軸方向のスケール値です。
    //! @param[in]    translateS  S軸方向の移動値です。
    //! @param[in]    translateT  T軸方向の移動値です。
    //!
    //! @return       設定された行列を返します。
    //---------------------------------------------------------------------------
    self_type& SetTextureProjectionFrustum(f32 l, f32 r, f32 b, f32 t, f32 n, f32 scaleS, f32 scaleT, f32 translateS, f32 translateT)
    {
        return *MTX34TextureProjectionFrustum(this, l, r, b, t, n, scaleS, scaleT, translateS, translateT);
    }

    //---------------------------------------------------------------------------
    //! @brief        テクスチャマトリクス用射影行列を作成します。
    //!
    //! @param[in]    fovy        垂直視野角です。
    //! @param[in]    aspect      クリップ面のアスペクト比です。
    //! @param[in]    scaleS      S軸方向のスケール値です。
    //! @param[in]    scaleT      T軸方向のスケール値です。
    //! @param[in]    translateS  S軸方向の移動値です。
    //! @param[in]    translateT  T軸方向の移動値です。
    //!
    //! @return       設定された行列を返します。
    //---------------------------------------------------------------------------
    self_type& SetTextureProjectionPerspective(f32 fovy, f32 aspect, f32 scaleS, f32 scaleT, f32 translateS, f32 translateT)
    {
        return *MTX34TextureProjectionPerspective(this, fovy, aspect, scaleS, scaleT, translateS, translateT);
    }

    //---------------------------------------------------------------------------
    //! @brief        平行射影行列を作成します。
    //!
    //! @param[in]    l           ニアクリップの左辺です。
    //! @param[in]    r           ニアクリップの右辺です。
    //! @param[in]    b           ニアクリップの下辺です。
    //! @param[in]    t           ニアクリップの上辺です。
    //! @param[in]    scaleS      S軸方向のスケール値です。
    //! @param[in]    scaleT      T軸方向のスケール値です。
    //! @param[in]    translateS  S軸方向の移動値です。
    //! @param[in]    translateT  T軸方向の移動値です。
    //!
    //! @return       設定された行列を返します。
    //---------------------------------------------------------------------------
    self_type& SetTextureProjectionOrtho(f32 l, f32 r, f32 b, f32 t, f32 scaleS, f32 scaleT, f32 translateS, f32 translateT)
    {
        return *MTX34TextureProjectionOrtho(this, l, r, b, t, scaleS, scaleT, translateS, translateT);
    }

    //---------------------------------------------------------------------------
    //! @brief        SRT 行列を作成します。
    //!
    //! @param[in]    s           スケール成分です。
    //! @param[in]    r           回転成分です。
    //! @param[in]    t           平行移動成分です。
    //!
    //! @return       設定された行列を返します。
    //---------------------------------------------------------------------------
    self_type& SetSRTMtx(const VEC3& s, const VEC3& r, const VEC3& t)
    {
        return *MTX34MakeSRT(this, &s, &r, &t);
    }

    //---------------------------------------------------------------------------
    //! @brief        ST 行列を作成します。
    //!
    //! @param[in]    s           スケール成分です。
    //! @param[in]    t           平行移動成分です。
    //!
    //! @return       設定された行列を返します。
    //---------------------------------------------------------------------------
    self_type& SetSTMtx(const VEC3& s, const VEC3& t)
    {
        return *MTX34MakeST(this, &s, &t);
    }

    //---------------------------------------------------------------------------
    //! @brief        S 行列を作成します。
    //!
    //! @param[in]    s           スケール成分です。
    //!
    //! @return       設定された行列を返します。
    //---------------------------------------------------------------------------
    self_type& SetSMtx(const VEC3& s)
    {
        return *MTX34MakeS(this, &s);
    }

    //@}

    //----------------------------------------
    //! @name 比較
    //@{

    //! @brief 同値であれば true を返します。
    //!
    //! @param[in] rhs 比較対象となる MTX34 です。
    //! @return 比較結果です。
    bool operator == (const self_type& rhs) const { return ::std::memcmp(this, &rhs, sizeof(MTX34)) == 0; }

    //! @brief 同値でなければ true を返します。
    //!
    //! @param[in] rhs 比較対象となる MTX34 です。
    //! @return 比較結果です。
    bool operator != (const self_type& rhs) const { return ::std::memcmp(this, &rhs, sizeof(MTX34)) != 0; }

    //! @brief 単位行列かどうかを判定します。
    //!
    //! @return 単位行列であれば true を返します。
    bool IsIdentity() const { return MTX34IsIdentity(this); }
    //@}

    //! @brief 行列の内容をデバッグ出力に書き出します。
    //!
    //! @param[in] bNewline  true なら出力の最後に改行を出力します。
    //! @param[in] name      行列の名前を指定します。NULL指定可能です。
    void Report(bool bNewline = true, const char* name = NULL) const;
};

typedef struct MTX34 Matrix34; //!< @briefprivate

//---------------------------------------------------------------------------
//! @briefprivate
//! @details MTX34 と浮動小数点数の積を計算します。
//! @param[in] f 積に用いる浮動小数点数です。
//! @param[in] rhs 積に用いる MTX34 です。
//! @return 計算結果の MTX34 を返します。
//---------------------------------------------------------------------------
inline MTX34 operator * (f32 f, const MTX34& rhs)
{
    MTX34 tmp;
    (void)MTX34Mult(&tmp, &rhs, f);
    return tmp;
}

//----------------------------------------
//! @name    行列
//@{

//---------------------------------------------------------------------------
//! @brief        行列をコピーします。
//!
//! @param[out]   pOut  コピー先の行列へのポインタ。
//! @param[in]    p     コピー元の行列へのポインタ
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_FORCE_INLINE MTX34*
MTX34Copy(MTX34* pOut, const MTX34* p)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p );

    if ( p != pOut )
    {
        *pOut = *p;
    }

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        行列の積を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。
//!               p1, p2 と同じ行列を指していても構いません。
//! @param[in]    p1    左辺値へのポインタ。
//! @param[in]    p2    右辺値へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_INLINE MTX34*
MTX34Mult(MTX34* pOut, const MTX34* p1, const MTX34* p2)
{
    return NW_MATH_IMPL_NS::MTX34Mult(pOut, p1, p2);
}

//@}

}  // namespace math
}  // namespace nw

namespace nw {
namespace math {

//-- const 引数を参照にしたオーバーロード

inline bool MTX34IsIdentity(const MTX34& m) { return MTX34IsIdentity( &m ); }

inline MTX34* MTX34Add(MTX34* pOut, const MTX34& m1, const MTX34& m2) { return MTX34Add( pOut, &m1, &m2 ); }
inline MTX34* MTX34Sub(MTX34* pOut, const MTX34& m1, const MTX34& m2) { return MTX34Sub( pOut, &m1, &m2 ); }
inline MTX34* MTX34Mult(MTX34* pOut, const MTX34& m, f32 f) { return MTX34Mult( pOut, &m, f ); }
inline MTX34* MTX34Mult(MTX34* pOut, const MTX34& m1, const MTX34& m2) { return MTX34Mult( pOut, &m1, &m2 ); }
inline MTX34* MTX34MAdd(MTX34* pOut, f32 t, const MTX34& m1, const MTX34& m2) { return MTX34MAdd( pOut, t, &m1, &m2 ); }

inline MTX34* MTX34Copy(MTX34* pOut, const MTX34& m) { return MTX34Copy( pOut, &m ); }
inline MTX34* MTX34MultArray(MTX34* pOut, const MTX34& m1, const MTX34* pSrc, s32 count) { return MTX34MultArray( pOut, m1, pSrc, count ); }
inline u32    MTX34Inverse(MTX34* pOut, const MTX34& m) { return MTX34Inverse( pOut, &m ); }
inline MTX34* MTX34Transpose(MTX34* pOut, const MTX34& m) { return MTX34Transpose( pOut, &m ); }
inline u32    MTX34InvTranspose(MTX34* pOut, const MTX34& m) { return MTX34InvTranspose( pOut, &m ); }

inline MTX34* MTX34LookAt(MTX34* pOut, const VEC3& vCamPos, const VEC3& vCamUp, const VEC3& vTarget) { return MTX34LookAt( pOut, &vCamPos, &vCamUp, &vTarget ); }
inline MTX34* MTX34LookAt(MTX34* pOut, const VEC3& vCamPos, f32 twist, const VEC3& vTarget) { return MTX34LookAt( pOut, &vCamPos, twist, &vTarget ); }
inline MTX34* MTX34CameraRotate(MTX34* pOut, const VEC3& vCamPos, const VEC3& vCamRotate) { return MTX34CameraRotate( pOut, &vCamPos, &vCamRotate ); }

inline MTX34* MTX34RotXYZTranslateIdx(MTX34* pOut, u32 idxX, u32 idxY, u32 idxZ, const VEC3& vT) { return MTX34RotXYZTranslateIdx( pOut, idxX, idxY, idxZ, &vT ); }
inline MTX34* MTX34RotAxisIdx(MTX34* pOut, const VEC3& vAxis, u32 idx) { return MTX34RotAxisIdx( pOut, &vAxis, idx ); }
inline MTX34* MTX34RotAxisRad(MTX34* pOut, const VEC3& vAxis, f32 fRad) { return MTX34RotAxisRad( pOut, &vAxis, fRad ); }
inline MTX34* MTX34RotAxisDeg(MTX34* pOut, const VEC3& vAxis, f32 fDeg) { return MTX34RotAxisDeg( pOut, &vAxis, fDeg ); }

inline MTX34* MTX34Scale(MTX34* pOut, const VEC3& vS) { return MTX34Scale( pOut, &vS ); }
inline MTX34* MTX34MultScale(MTX34* pOut, const MTX34& m, const VEC3& vS) { return MTX34MultScale( pOut, &m, &vS ); }
inline MTX34* MTX34MultScale(MTX34* pOut, const VEC3& vS, const MTX34& m) { return MTX34MultScale( pOut, &vS, &m ); }

inline MTX34* MTX34Translate(MTX34* pOut, const VEC3& vT) { return MTX34Translate( pOut, &vT ); }
inline MTX34* MTX34MultTranslate(MTX34* pOut, const MTX34& m, const VEC3& vT) { return MTX34MultTranslate( pOut, &m, &vT ); }
inline MTX34* MTX34MultTranslate(MTX34* pOut, const VEC3& vT, const MTX34& m) { return MTX34MultTranslate( pOut, &vT, &m ); }

inline MTX34* MTX34MakeSRT(MTX34* pOut, const VEC3& s, const VEC3& r, const VEC3& t) { return MTX34MakeSRT( pOut, &s, &r, &t ); }
inline MTX34* MTX34MakeST(MTX34* pOut, const VEC3& s, const VEC3& t) { return MTX34MakeST( pOut, &s, &t ); }
inline MTX34* MTX34MakeS(MTX34* pOut, const VEC3& s) { return MTX34MakeS( pOut, &s ); }

inline MTX34* QUATToMTX34(MTX34* pOut, const QUAT& q) { return QUATToMTX34( pOut, &q ); }

}  // namespace math
}  // namespace nw

#if defined(NW_MATH_AS_INLINE)
#include <nw/math/inline/math_Matrix34.ipp>
#endif // #if defined(NW_MATH_AS_INLINE)

#endif
