﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MATH_ARITHMETIC_H_
#define NW_MATH_ARITHMETIC_H_

#include <nw/assert.h>
#include <nw/config.h>
#include <nw/math/math_Config.h>
#include <nw/math/math_Constant.h>
#include <nw/ut/ut_Inlines.h>

#include <cmath>

namespace nw { namespace math {


//========================================================================
//        浮動少数に関する数学関数
//========================================================================

//---------------------------------------------------------------------------
//! @brief        浮動小数点数の指数部を取得します。
//!
//! @param[in]    f   指数部を取得する数。
//!
//! @return       f の指数部を返します。ただし負の値はオフセット表現ではなく 2 の補数表現です。
//---------------------------------------------------------------------------
inline s32
FGetExpPart(f32 f)
{
    s32 s = static_cast<s32>((ut::F32AsU32(f) >> 23) & 0xFF);
    return s - 127;
}


//---------------------------------------------------------------------------
//! @brief        浮動小数点数の仮数部を求めます。
//!
//! @param[in]    f   仮数部を取得する数。
//!
//! @return       f の仮数部を返します。返り値は f と同じ符号を持っています。
//---------------------------------------------------------------------------
inline f32
FGetMantPart(f32 f)
{
    // 指数部を 127 にする
    u32 u = (ut::F32AsU32(f) & 0x807FFFFF) | 0x3F800000;
    return ut::U32AsF32(u);
}


//---------------------------------------------------------------------------
//! @brief        数の正負に応じて値を選択します。
//!
//! @param[in]    cond    返り値を選択する条件。
//! @param[in]    ifPos   cond が0以上の場合の返り値。
//! @param[in]    ifNeg   cond が負の場合の返り値。
//!
//! @return       cond が 0 以上であれば ifPos を、そうでなければ ifNeg を返します。
//---------------------------------------------------------------------------
inline f32
FSelect(f32 cond, f32 ifPos, f32 ifNeg)
{
    f32 ret;

    ret = (cond >= 0) ? ifPos: ifNeg;


    return ret;
}


//---------------------------------------------------------------------------
//! @brief        絶対値を求めます
//!
//! @param[in]    x   絶対値を求める値
//!
//! @return       x の絶対値を返します
//---------------------------------------------------------------------------
inline f32
FAbs(f32 x)
{
    f32 ret;

    ret = ::std::fabsf(x);


    return ret;
}


//---------------------------------------------------------------------------
//! @brief        絶対値が等しく、符号が負である数を求めます
//!
//! @param[in]    x   元の数。
//!
//! @return       x と絶対値が等しく、符号が負である数を返します
//---------------------------------------------------------------------------
inline f32
FNAbs(f32 x)
{
    f32 ret;

    ret = - FAbs(x);


    return ret;
}


//---------------------------------------------------------------------------
//! @brief        一方の数の符号を他方の数にコピーします
//!
//! @param[in]    abs     絶対値の元となる数。
//! @param[in]    sign    符号を採用する数。
//!
//! @return       abs の絶対値と sign の符号を持った数を返します。
//---------------------------------------------------------------------------
inline f32
FCopySign(f32 abs, f32 sign)
{
    f32 pos = FAbs(abs);
    f32 neg = FNAbs(abs);

    return FSelect(sign, pos, neg);
}

namespace internal
{
    f32 FExp(f32 x);
    f32 FLog(f32 x);
}


//---------------------------------------------------------------------------
//! @brief        e^x を求めます
//!
//! @param[in]    x   指数の値
//!
//! @return       e^x を返します
//---------------------------------------------------------------------------
inline f32
FExp(f32 x)
{
    return ::std::expf(x);

}


//---------------------------------------------------------------------------
//! @brief        自然対数を求めます
//!
//! @param[in]    x   自然対数を求める値
//!
//! @return       x の自然対数を返します
//---------------------------------------------------------------------------
inline f32
FLog(f32 x)
{
    NW_MATH_WARNING(x > 0, "FLog: Input is out of the domain.");

    return ::std::logf(x);

}


//---------------------------------------------------------------------------
//! @brief        常用対数を求めます
//!
//! @param[in]    x   常用対数を求める値
//!
//! @return       x の常用対数を返します
//---------------------------------------------------------------------------
inline f32
FLog10(f32 x)
{
    NW_MATH_WARNING(x > 0, "FLog10: Input is out of the domain.");

    return ::std::log10f(x);
}


//---------------------------------------------------------------------------
//! @brief        剰余を計算します
//!
//! @param[in]    x   割られる数
//! @param[in]    y   割る数
//!
//! @return       x / y の剰余を返します
//---------------------------------------------------------------------------
inline f32
FMod(f32 x, f32 y)
{
    return ::std::fmodf(x, y);
}


//---------------------------------------------------------------------------
//! @brief        浮動小数点数を整数部と小数部に分けます。整数部、小数部共に x と同じ符号を持ちます。
//!
//! @param[in]    x   元の浮動小数点数。
//! @param[out]   y   整数部を格納するバッファへのポインタ。
//!
//! @return       小数部を返します
//---------------------------------------------------------------------------
inline f32
FModf(f32 x, f32* y)
{
    return ::std::modff(x, y);
}


//---------------------------------------------------------------------------
//! @brief        x を下回らない最小の整数値を求めます
//!
//! @param[in]    x   対象の値
//!
//! @return       x を下回らない最小の整数値を返します
//---------------------------------------------------------------------------
inline f32
FCeil(f32 x)
{
    return ::std::ceilf(x);
}


//---------------------------------------------------------------------------
//! @brief        x を超えない最大の整数値を求めます
//!
//! @param[in]    x   対象の値
//!
//! @return       x を超えない最大の整数値を返します
//---------------------------------------------------------------------------
inline f32
FFloor(f32 x)
{
    return ::std::floorf(x);
}


//---------------------------------------------------------------------------
//! @brief        f32型からs16型に変換します
//!
//! @param[in]    x   s16型に変換する値
//!
//! @return       x と同じ値を持ったs16型の値を返します
//---------------------------------------------------------------------------
inline s16
F32ToS16(f32 x)
{
    return s16(x);

}


//---------------------------------------------------------------------------
//! @brief        f32型からu16型に変換します
//!
//! @param[in]    x   u16型に変換する値
//!
//! @return       x と同じ値を持ったu16型の値を返します
//---------------------------------------------------------------------------
inline u16
F32ToU16(f32 x)
{
    return u16(x);

}


//---------------------------------------------------------------------------
//! @brief        u16型からf32型に変換します
//!
//! @param[in]    x   f32型に変換する値
//!
//! @return       x と同じ値を持ったf32型の値を返します
//---------------------------------------------------------------------------
inline f32
U16ToF32(u16 x)
{
    return f32(x);

}


//---------------------------------------------------------------------------
//! @brief        s16型からf32型に変換します
//!
//! @param[in]    x   f32型に変換する値
//!
//! @return       x と同じ値を持ったf32型の値を返します
//---------------------------------------------------------------------------
inline f32
S16ToF32(s16 x)
{
    return f32(x);

}



//---------------------------------------------------------------------------
//! @brief        逆数を高速低精度に求めます
//!
//! @param[in]    x   逆数を求める値
//!
//! @return       x の逆数を返します
//---------------------------------------------------------------------------
inline f32
FInv(f32 x)
{
    f32 ix;

    ix = 1.0f / x;


    return ix;
}


//---------------------------------------------------------------------------
//! @brief        平方根の逆数を求めます。
//!
//! @param[in]    x 平方根の逆数を求める値。
//!
//! @return       x の平方根の逆数を返します。
//---------------------------------------------------------------------------
f32 FrSqrt(f32 x);

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details      平方根の逆数を求めます。
//!               FrSqrt より高速ですが、求められる計算結果の精度に差があります。
//!
//! @param[in]    x 平方根の逆数を求める値。
//!
//! @return       x の平方根の逆数を返します。
//---------------------------------------------------------------------------
f32 FrFastSqrt(f32 x);

//---------------------------------------------------------------------------
//! @brief        平方根を求めます
//!
//! @param[in]    x   平方根を求める値
//!
//! @return       x の平方根を返します
//---------------------------------------------------------------------------
inline f32
FSqrt(f32 x)
{
    NW_MATH_WARNING(x >= 0, "FSqrt: Input is out of the domain.");

    return ::std::sqrtf(x);

}


//---------------------------------------------------------------------------
//! @brief        立方根を求めます
//!
//! @param[in]    x   立方根を求める値
//!
//! @return       x の立方根を返します
//---------------------------------------------------------------------------
inline f32
FCbrt(f32 x)
{
#if defined(NW_CW3)
    return ::std::cbrtf(x);
#else
    return ::std::pow(x, 1/3.f);
#endif
}


//---------------------------------------------------------------------------
//! @brief        float 値の逆数を求めます
//!
//! @param[in]    x     元の float 値です。
//!
//! @return       逆数を返します。
//---------------------------------------------------------------------------
inline f32
Reciprocal(f32 x)
{
    //  #if defined(NW_PLATFORM_WIN32)
#if 1
    return 1.0f / x;
#else
    f32 ix = __FRES(x);
    // Newton-Raphson refinement
    // ( ix = est. of 1/x ) -> ( ix = 2*ix - x*ix*ix )
    return 2.0f * ix - x * ix * ix;
#endif
}


//---------------------------------------------------------------------------
//! @brief        エルミート補間を行います
//!
//! @param[in]    v0  点1での値。
//! @param[in]    t0  点1での傾き。
//! @param[in]    v1  点2での値。
//! @param[in]    t1  点2での傾き。
//! @param[in]    s   補間対象位置。(点1:0.0～1.0:点2)
//!
//! @return       補間結果の値。
//---------------------------------------------------------------------------
inline f32
Hermite(f32 v0, f32 t0, f32 v1, f32 t1, f32 s)
{
    f32 SS   = s * s;
    f32 SS_S = s * s - s;
    f32 b1 = SS_S * s - SS_S;
    f32 b2 = SS_S * s;
    f32 a2 = SS - 2.f * b2;

    return v0 - a2 * v0 + a2 * v1 + b1 * t0 + b2 * t1;
}

//---------------------------------------------------------------------------
//! @brief        エルミート補間計算です。
//!
//! @param[in]    v0  点1での値。
//! @param[in]    t0  点1での傾き。
//! @param[in]    v1  点2での値。
//! @param[in]    t1  点2での傾き。
//! @param[in]    p   点1から補間対象位置の距離。
//! @param[in]    d   点1と点2の距離。
//!
//! @return       エルミート補間の結果です。
//---------------------------------------------------------------------------
NW_FORCE_INLINE f32
Hermite(f32 v0, f32 t0, f32 v1, f32 t1, f32 p, f32 d)
{
    f32 inv_d = 1 / d;
    f32 s = p * inv_d;
    f32 s_1 = s - 1;
    return v0 + (v0 - v1) * (2 * s - 3) * s * s + s * s_1 * (s_1 * t0 + s * t1);
}

//---------------------------------------------------------------------------
//! @brief        ベジェ補間を行います。
//!
//! @param[in]    p1  点1での値
//! @param[in]    p2  点1での制御値
//! @param[in]    p3  点2での制御値
//! @param[in]    p4  点2での値
//! @param[in]    s   補間対象位置。(点1:0.0～1.0:点2)
//!
//! @return       補間結果の値。
//---------------------------------------------------------------------------
f32 Bezier(f32 p1, f32 p2, f32 p3, f32 p4, f32 s);


//---------------------------------------------------------------------------
//! @brief        Catmull-Rom 補間 を行います。
//!
//! @param[in]    p0  点1での制御値
//! @param[in]    p1  点1での値
//! @param[in]    p2  点2での値
//! @param[in]    p3  点2での制御値
//! @param[in]    s   補間対象位置。(点1:0.0～1.0:点2)
//!
//! @return       補間結果の値。
//---------------------------------------------------------------------------
f32 CatmullRom(f32 p0, f32 p1, f32 p2, f32 p3, f32 s);

//@}

//========================================================================
//        整数関連
//========================================================================

//----------------------------------------
//! @name    整数に関する数学関数
//@{

// 非インライン関数

//---------------------------------------------------------------------------
//! @brief        ビット列の中で1となっているビットの数を数えます。
//!
//! @param[in]    x  走査するビット列。
//!
//! @return       ビット列中で1となっているビットの数を返します。
//---------------------------------------------------------------------------
u32 CntBit1(u32 x);

//---------------------------------------------------------------------------
//! @brief        ビット列の中で1となっているビットの数を数えます。
//!
//! @param[in]    first  ビット列の先頭ワードへのポインタ。
//! @param[in]    last   ビット列の最終ワードの次のワードへのポインタ。
//!
//! @return       ビット列中で1となっているビットの数を返します。
//---------------------------------------------------------------------------
u32 CntBit1(const u32* first, const u32* last);

//---------------------------------------------------------------------------
//! @brief        ビット列間の距離を求めます。
//!
//!               ビット列間の距離はビット列同士をビットごとに比較した場合の
//!               等しいビットの数です。
//!
//! @param[in]    first1  ビット列 1 の先頭ワードへのポインタ。
//! @param[in]    last1   ビット列 1 の最終ワードの次のワードへのポインタ。
//! @param[in]    first2  ビット列 2 の先頭ワードへのポインタ。
//!
//! @return       ビット列 1 とビット列 2 の距離を返します。
//---------------------------------------------------------------------------
u32 DistBit(const u32* first1, const u32* last1, const u32* first2);

//---------------------------------------------------------------------------
//! @brief        ビット列の順序を反転します。
//!
//! @param[in]    x  反転するビット列
//!
//! @return       x のビット順序を逆順にしたビット列を返します。
//---------------------------------------------------------------------------
u32 RevBit(u32 x);

//---------------------------------------------------------------------------
//! @brief        整数の非負整数乗を求めます。
//!
//! @param[in]    x  底となる値
//! @param[in]    n  指数の値。非負でなければなりません。
//!
//! @return       x^n を返します
//---------------------------------------------------------------------------
int IExp(int x, u32 n);

//---------------------------------------------------------------------------
//! @brief        整数の常用対数を計算し、結果を整数で返します。
//!
//! @param[in]    x  常用対数を求める値。
//!
//! @return       x の常用対数を整数で返します。
//---------------------------------------------------------------------------
u32 ILog10(u32 x);

namespace internal
{
namespace standard {
u32 CntLz_(u32 x);
} // standard
}

#if defined(NW_MATH_USE_INTRINSICS)

namespace internal { namespace intrinsics {

inline u32 CntLz_(u32 x)
{
    return __cntlzw(x);
}

}} // namespace internal::intrinsics

#endif // NW_MATH_USE_INTRINSICS

//---------------------------------------------------------------------------
//! @brief        MSBからの連続する 0 のビットを数えます。
//!
//! @param[in]    x  対象のビット列
//!
//! @return       MSBからの連続する 0 のビット数を返します。
//---------------------------------------------------------------------------
inline u32
CntLz(u32 x)
{
    return NW_MATH_IMPL_NS::CntLz_(x);
}


//---------------------------------------------------------------------------
//! @brief        整数値が2の整数乗かどうかを判定します。
//!
//! @param[in]    x  判定対象の整数値
//!
//! @return       x が 2 の累乗であれば true を、そうでなければ false を返します。
//---------------------------------------------------------------------------
//          ただし x が 0 の場合は true を返します。
inline bool IsPwr2(int x) { return 0 == (x & (x - 1)); }

//---------------------------------------------------------------------------
//! @brief        1 になっているビットで一番右(LSB側)のビットを取得します。
//!
//! @param[in]    x  対象とするビット列
//!
//! @return       x で一番右の 1 となっているビット位置のみが 1 となっているビット列を返します。
//---------------------------------------------------------------------------
inline int Rightmost1(int x) { return x & (-x); }

//---------------------------------------------------------------------------
//! @brief        0 になっているビットで一番右(LSB側)のビットを取得します。
//!
//! @param[in]    x  対象とするビット列
//!
//! @return       x で一番右の 0 となっているビット位置のみが 1 となっているビット列を返します。
//---------------------------------------------------------------------------
inline int Rightmost0(int x) { return ~x & (x + 1); }

//---------------------------------------------------------------------------
//! @brief        ビット列同士の距離を計算します。
//!
//! @param[in]    x  ビット列 1
//! @param[in]    y  ビット列 2
//!
//! @return       ビット列 1 とビット列 2 の距離を返します。
//---------------------------------------------------------------------------
inline u32 DistBit(u32 x, u32 y) { return CntBit1(x ^ y); }

//---------------------------------------------------------------------------
//! @brief        LSBからの連続する 0 のビットを数えます。
//!
//! @param[in]    x  対象のビット列
//!
//! @return       LSBからの連続する 0 のビット数を返します。
//---------------------------------------------------------------------------
inline u32 CntTz(u32 x) { return 32 - CntLz(~x & (x - 1)); }

//---------------------------------------------------------------------------
//! @brief        整数の 2 を底とする対数を計算し、結果を整数で返します。
//!
//! @param[in]    x  真数
//!
//! @return       整数の 2 を底とする対数を整数で返します。
//---------------------------------------------------------------------------
inline u32 ILog2(u32 x) { return 31 - CntLz(x); }

//---------------------------------------------------------------------------
//! @brief        与えられた整数以下の最大の 2 の累乗を計算します。
//!
//! @param[in]    x  対象の整数
//!
//! @return       x 以下の最大の2の累乗を返します。
//---------------------------------------------------------------------------
inline u32 FloorPwr2(u32 x) { return 0x80000000 >> CntLz(x); }

//---------------------------------------------------------------------------
//! @brief        与えられた整数以上の最小の 2 の累乗を計算します。
//!
//! @param[in]    x  対象の整数
//!
//! @return       x 以上の最小の 2 の累乗を返します。
//---------------------------------------------------------------------------
inline u32 CeilPwr2(u32 x) { return 0x80000000 >> (CntLz(x - 1) - 1); }

//@}

}}  // nw::math

/* NW_MATH_ARITHMETIC_H_ */
#endif
