﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo/HAL Laboratory, Inc. All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>

namespace nw {
namespace math {

//---------------------------------------------------------------------------
//        VEC3
//---------------------------------------------------------------------------

namespace internal { namespace standard {

NW_MATH_INLINE VEC3*
VEC3Maximize(VEC3* pOut, const VEC3* p1, const VEC3* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    pOut->x = (p1->x > p2->x) ? p1->x : p2->x;
    pOut->y = (p1->y > p2->y) ? p1->y : p2->y;
    pOut->z = (p1->z > p2->z) ? p1->z : p2->z;

    return pOut;
}

NW_MATH_INLINE VEC3*
VEC3Minimize(VEC3* pOut, const VEC3* p1, const VEC3* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    pOut->x = (p1->x < p2->x) ? p1->x : p2->x;
    pOut->y = (p1->y < p2->y) ? p1->y : p2->y;
    pOut->z = (p1->z < p2->z) ? p1->z : p2->z;

    return pOut;
}

NW_MATH_INLINE f32
VEC3SquareDist(const VEC3* p1, const VEC3* p2)
{
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    VEC3 diff;

    diff.x = p1->x - p2->x;
    diff.y = p1->y - p2->y;
    diff.z = p1->z - p2->z;

    return (diff.x * diff.x) + (diff.y * diff.y) + (diff.z * diff.z);
}

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

NW_MATH_INLINE VEC3*
VEC3Maximize(VEC3* pOut, const VEC3* p1, const VEC3* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    f32x2 f0, f1, f2;

    f0 = tof32x2(p1->x);
    f1 = tof32x2(p2->x);
    f2 = __PS_SUB(f0, f1);
    f2 = __PS_SEL(f2, f0, f1);

    tof32x2(pOut->x) = f2;

    pOut->z = (p1->z > p2->z) ? p1->z : p2->z;

    return pOut;
}

NW_MATH_INLINE VEC3*
VEC3Minimize(VEC3* pOut, const VEC3* p1, const VEC3* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    f32x2 f0, f1, f2;

    f0 = tof32x2(p1->x);
    f1 = tof32x2(p2->x);
    f2 = __PS_SUB(f1, f0);
    f2 = __PS_SEL(f2, f0, f1);

    tof32x2(pOut->x) = f2;

    f32 z1 = p1->z;
    f32 z2 = p2->z;

    pOut->z = (z1 < z2) ? z1 : z2;

    return pOut;
}

NW_MATH_INLINE f32
VEC3SquareDist(const VEC3* p1, const VEC3* p2)
{
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    f32x2 f0;
    f0 = __PS_SUB(tof32x2(p1->x), tof32x2(p2->x));
    f0 = __PS_MUL(f0, f0);
    f0 = __PS_SUM0(f0, f0, f0);

    f32 f1 = p1->z - p2->z;

    return f0[0] + f1 * f1;
}

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

//----------------------------------------
//! @name    ベクトル
//@{

//---------------------------------------------------------------------------
//! @brief        ベクトルがゼロベクトルかどうか判定します。
//!
//! @param[in]    p    判定対象のベクトルへのポインタ。
//!
//! @return       p がゼロベクトルであれば true そうでなければ false を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE bool
VEC3IsZero(const VEC3* p)
{
    return p->x == 0.f && p->y == 0.f && p->z == 0.f;
}

//---------------------------------------------------------------------------
//! @brief        2つのベクトルのそれぞれの成分の最大値から構成されるベクトルを作成します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じベクトルを指していても構いません。
//! @param[in]    p1    対象のベクトル1へのポインタ。
//! @param[in]    p2    対象のベクトル2へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC3*
VEC3Maximize(VEC3* pOut, const VEC3* p1, const VEC3* p2)
{
    return NW_MATH_IMPL_NS::VEC3Maximize(pOut, p1, p2);
}



//---------------------------------------------------------------------------
//! @brief        2つのベクトルのそれぞれの成分の最小値から構成されるベクトルを作成します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じベクトルを指していても構いません。
//! @param[in]    p1    対象のベクトル1へのポインタ。
//! @param[in]    p2    対象のベクトル2へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC3*
VEC3Minimize(VEC3* pOut, const VEC3* p1, const VEC3* p2)
{
    return NW_MATH_IMPL_NS::VEC3Minimize(pOut, p1, p2);
}


//---------------------------------------------------------------------------
//! @brief        ベクトルの外積を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。
//!                     p1, p2 と同じベクトルを指していても構いません。
//! @param[in]    p1    左辺値へのポインタ
//! @param[in]    p2    右辺値へのポインタ
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC3*
VEC3Cross(VEC3* pOut, const VEC3* p1, const VEC3* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    VEC3 tmpVec;

    tmpVec.x = ( p1->y * p2->z ) - ( p1->z * p2->y );
    tmpVec.y = ( p1->z * p2->x ) - ( p1->x * p2->z );
    tmpVec.z = ( p1->x * p2->y ) - ( p1->y * p2->x );

    pOut->x = tmpVec.x;
    pOut->y = tmpVec.y;
    pOut->z = tmpVec.z;

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        ベクトルを正規化します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p と同じベクトルを指していても構いません。
//! @param[in]    p     対象のベクトルへのポインタ
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC3*
VEC3Normalize(VEC3* pOut, const VEC3* p)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p);

    f32 mag = (p->x * p->x) + (p->y * p->y) + (p->z * p->z);

    NW_TASSERTMSG(mag != 0, "MATHNormalize3():  zero magnitude vector");

    mag = 1.0f / ::std::sqrtf(mag);

    pOut->x = p->x * mag;
    pOut->y = p->y * mag;
    pOut->z = p->z * mag;

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        ベクトルを正規化します。
//!               VEC3Normalize より高速ですが、求められる計算結果の精度に差があります。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p と同じベクトルを指していても構いません。
//! @param[in]    p     対象のベクトルへのポインタ
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC3*
VEC3FastNormalize(VEC3* pOut, const VEC3* p)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p);

    f32 mag = (p->x * p->x) + (p->y * p->y) + (p->z * p->z);

    NW_TASSERTMSG(mag != 0, "MATHNormalize3():  zero magnitude vector");

    mag = FrFastSqrt(mag);

    pOut->x = p->x * mag;
    pOut->y = p->y * mag;
    pOut->z = p->z * mag;

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        ベクトルを正規化します。
//!               正規化に失敗した場合は指定されたベクトルを設定します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p と同じベクトルを指していても構いません。
//! @param[in]    p     対象のベクトルへのポインタ
//! @param[in]    alt   正規化に失敗した場合に設定するベクトル
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC3*
VEC3SafeNormalize(VEC3* pOut, const VEC3* p, const VEC3& alt)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p);

    f32 mag = (p->x * p->x) + (p->y * p->y) + (p->z * p->z);

    if (mag == 0 || mag == F_INF || isnan(mag))
    {
        *pOut = alt;

        return pOut;
    }

    mag = 1.0f / ::std::sqrtf(mag);

    pOut->x = p->x * mag;
    pOut->y = p->y * mag;
    pOut->z = p->z * mag;

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        ベクトルを正規化します。
//!               正規化に失敗した場合は指定されたベクトルを設定します。
//!               VEC3SafeNormalize より高速ですが、求められる計算結果の精度に差があります。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p と同じベクトルを指していても構いません。
//! @param[in]    p     対象のベクトルへのポインタ
//! @param[in]    alt   正規化に失敗した場合に設定するベクトル
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC3*
VEC3FastSafeNormalize(VEC3* pOut, const VEC3* p, const VEC3& alt)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p);

    f32 mag = (p->x * p->x) + (p->y * p->y) + (p->z * p->z);

    if (mag == 0 || mag == F_INF || isnan(mag))
    {
        *pOut = alt;

        return pOut;
    }

    mag = FrFastSqrt(mag);

    pOut->x = p->x * mag;
    pOut->y = p->y * mag;
    pOut->z = p->z * mag;

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        2つのベクトル間の距離の2乗を計算します。
//!
//! @param[in]    p1    左辺値へのポインタ
//! @param[in]    p2    右辺値へのポインタ
//!
//! @return       p1 と p2 の距離の2乗を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE f32
VEC3SquareDist(const VEC3* p1, const VEC3* p2)
{
    return NW_MATH_IMPL_NS::VEC3SquareDist(p1, p2);
}

//@}

}  // namespace math
}  // namespace nw
