﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo/HAL Laboratory, Inc. All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/math/math_Matrix43.h>

namespace nw { namespace math {

namespace internal { namespace standard {

NW_MATH_INLINE MTX43*
MTX43Add(MTX43* pOut, const MTX43* p1, const MTX43* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    pOut->f._00 = p1->f._00 + p2->f._00;
    pOut->f._01 = p1->f._01 + p2->f._01;
    pOut->f._02 = p1->f._02 + p2->f._02;

    pOut->f._10 = p1->f._10 + p2->f._10;
    pOut->f._11 = p1->f._11 + p2->f._11;
    pOut->f._12 = p1->f._12 + p2->f._12;

    pOut->f._20 = p1->f._20 + p2->f._20;
    pOut->f._21 = p1->f._21 + p2->f._21;
    pOut->f._22 = p1->f._22 + p2->f._22;

    pOut->f._30 = p1->f._30 + p2->f._30;
    pOut->f._31 = p1->f._31 + p2->f._31;
    pOut->f._32 = p1->f._32 + p2->f._32;

    return pOut;
}

NW_MATH_INLINE MTX43*
MTX43Sub(MTX43* pOut, const MTX43* p1, const MTX43* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    pOut->f._00 = p1->f._00 - p2->f._00;
    pOut->f._01 = p1->f._01 - p2->f._01;
    pOut->f._02 = p1->f._02 - p2->f._02;

    pOut->f._10 = p1->f._10 - p2->f._10;
    pOut->f._11 = p1->f._11 - p2->f._11;
    pOut->f._12 = p1->f._12 - p2->f._12;

    pOut->f._20 = p1->f._20 - p2->f._20;
    pOut->f._21 = p1->f._21 - p2->f._21;
    pOut->f._22 = p1->f._22 - p2->f._22;

    pOut->f._30 = p1->f._30 - p2->f._30;
    pOut->f._31 = p1->f._31 - p2->f._31;
    pOut->f._32 = p1->f._32 - p2->f._32;

    return pOut;
}

NW_MATH_INLINE MTX43*
MTX43Mult(MTX43* pOut, const MTX43* __restrict p1, const MTX43* __restrict p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    MTX43 mTmp;

    MTX43* __restrict pDst = ( pOut == p1 || pOut == p2 ) ? &mTmp : pOut;

    pDst->f._00 = p1->f._00 * p2->f._00 + p1->f._01 * p2->f._10 + p1->f._02 * p2->f._20;
    pDst->f._01 = p1->f._00 * p2->f._01 + p1->f._01 * p2->f._11 + p1->f._02 * p2->f._21;
    pDst->f._02 = p1->f._00 * p2->f._02 + p1->f._01 * p2->f._12 + p1->f._02 * p2->f._22;

    pDst->f._10 = p1->f._10 * p2->f._00 + p1->f._11 * p2->f._10 + p1->f._12 * p2->f._20;
    pDst->f._11 = p1->f._10 * p2->f._01 + p1->f._11 * p2->f._11 + p1->f._12 * p2->f._21;
    pDst->f._12 = p1->f._10 * p2->f._02 + p1->f._11 * p2->f._12 + p1->f._12 * p2->f._22;

    pDst->f._20 = p1->f._20 * p2->f._00 + p1->f._21 * p2->f._10 + p1->f._22 * p2->f._20;
    pDst->f._21 = p1->f._20 * p2->f._01 + p1->f._21 * p2->f._11 + p1->f._22 * p2->f._21;
    pDst->f._22 = p1->f._20 * p2->f._02 + p1->f._21 * p2->f._12 + p1->f._22 * p2->f._22;

    pDst->f._30 = p1->f._30 * p2->f._00 + p1->f._31 * p2->f._10 + p1->f._32 * p2->f._20 + p2->f._30;
    pDst->f._31 = p1->f._30 * p2->f._01 + p1->f._31 * p2->f._11 + p1->f._32 * p2->f._21 + p2->f._31;
    pDst->f._32 = p1->f._30 * p2->f._02 + p1->f._31 * p2->f._12 + p1->f._32 * p2->f._22 + p2->f._32;

    if ( pDst != pOut )
    {
        MTX43Copy( pOut, pDst );
    }

    return pOut;
}

NW_MATH_INLINE MTX43*
MTX43Mult(MTX43* pOut, const MTX43* p, f32 f)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p );

    pOut->f._00 = p->f._00 * f;
    pOut->f._01 = p->f._01 * f;
    pOut->f._02 = p->f._02 * f;

    pOut->f._10 = p->f._10 * f;
    pOut->f._11 = p->f._11 * f;
    pOut->f._12 = p->f._12 * f;

    pOut->f._20 = p->f._20 * f;
    pOut->f._21 = p->f._21 * f;
    pOut->f._22 = p->f._22 * f;

    pOut->f._30 = p->f._30 * f;
    pOut->f._31 = p->f._31 * f;
    pOut->f._32 = p->f._32 * f;

    return pOut;
}

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

NW_MATH_INLINE MTX43*
MTX43Add(MTX43* pOut, const MTX43* p1, const MTX43* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    tof32x2(pOut->a[0]) = __PS_ADD(tof32x2(p1->a[0]), tof32x2(p2->a[0]));
    tof32x2(pOut->a[2]) = __PS_ADD(tof32x2(p1->a[2]), tof32x2(p2->a[2]));
    tof32x2(pOut->a[4]) = __PS_ADD(tof32x2(p1->a[4]), tof32x2(p2->a[4]));
    tof32x2(pOut->a[6]) = __PS_ADD(tof32x2(p1->a[6]), tof32x2(p2->a[6]));
    tof32x2(pOut->a[8]) = __PS_ADD(tof32x2(p1->a[8]), tof32x2(p2->a[8]));
    tof32x2(pOut->a[10]) = __PS_ADD(tof32x2(p1->a[10]), tof32x2(p2->a[10]));

    return pOut;
}

NW_MATH_INLINE MTX43*
MTX43Sub(MTX43* pOut, const MTX43* p1, const MTX43* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    tof32x2(pOut->a[0]) = __PS_SUB(tof32x2(p1->a[0]), tof32x2(p2->a[0]));
    tof32x2(pOut->a[2]) = __PS_SUB(tof32x2(p1->a[2]), tof32x2(p2->a[2]));
    tof32x2(pOut->a[4]) = __PS_SUB(tof32x2(p1->a[4]), tof32x2(p2->a[4]));
    tof32x2(pOut->a[6]) = __PS_SUB(tof32x2(p1->a[6]), tof32x2(p2->a[6]));
    tof32x2(pOut->a[8]) = __PS_SUB(tof32x2(p1->a[8]), tof32x2(p2->a[8]));
    tof32x2(pOut->a[10]) = __PS_SUB(tof32x2(p1->a[10]), tof32x2(p2->a[10]));

    return pOut;
}

NW_MATH_INLINE MTX43*
MTX43Mult(MTX43* pOut, const MTX43* __restrict p1, const MTX43* __restrict p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    // d = a . b;
    //
    // { f0 f4 } = { b00 b01  b02 }
    // { f1 f5 }   { b10 b11  b12 }
    // { f2 f6 }   { b20 b21  b22 }
    // { f3 f7 }   { b30 b31  b32 }

    const f32x2 f0 = __PSQ_LX(p2, offsetof(MTX43, f._00), 0, 0);
    const f32x2 f1 = __PSQ_LX(p2, offsetof(MTX43, f._10), 0, 0);
    const f32x2 f2 = __PSQ_LX(p2, offsetof(MTX43, f._20), 0, 0);
    const f32x2 f3 = __PSQ_LX(p2, offsetof(MTX43, f._30), 0, 0);
    const f32x2 f4 = __PS_FDUP(p2->f._02);
    const f32x2 f5 = __PS_FDUP(p2->f._12);
    const f32x2 f6 = __PS_FDUP(p2->f._22);
    const f32x2 f7 = __PS_FDUP(p2->f._32);

    f32x2 f8, f9, f10;

    // { d00 d01 d02 } = { f9 f10 } := { a00 a01 a02 } . { f0 f4 }
    //                                                   { f1 f5 }
    //                                                   { f2 f6 }

    f8 = __PSQ_LX(p1, offsetof(MTX43, f._00), 0, 0); // { a00 a01 }
    f9 = __PS_MULS0(f0, f8);
    f10 = __PS_MULS0(f4, f8);
    f9 = __PS_MADDS1(f1, f8, f9);
    f10 = __PS_MADDS1(f5, f8, f10);
    f8 = __PSQ_LX(p1, offsetof(MTX43, f._02), 0, 0); // { a02 a10 }
    f9 = __PS_MADDS0(f2, f8, f9);
    f10 = __PS_MADDS0(f6, f8, f10);
    __PSQ_STX(pOut, offsetof(MTX43, m[0][0]), f9, 0, 0);
    __PSQ_STX(pOut, offsetof(MTX43, m[0][2]), f10, 1, 0);

    //                                                   { f0 f4 }
    // { d10 d11 d12 } = { f9 f10 } := { a10 a11 a12 } . { f1 f5 }
    //                                                   { f2 f6 }

    f9 = __PS_MULS1(f0, f8);
    f10 = __PS_MULS1(f4, f8);
    f8 = __PSQ_LX(p1, offsetof(MTX43, f._11), 0, 0); // { a11 a12 }
    f9 = __PS_MADDS0(f1, f8, f9);
    f10 = __PS_MADDS0(f5, f8, f10);
    f9 = __PS_MADDS1(f2, f8, f9);
    f10 = __PS_MADDS1(f6, f8, f10);
    __PSQ_STX(pOut, offsetof(MTX43, m[1][0]), f9, 0, 0);
    __PSQ_STX(pOut, offsetof(MTX43, m[1][2]), f10, 1, 0);

    //                                                   { f0 f4 }
    //                                                   { f1 f5 }
    // { d20 d21 d22 } = { f9 f10 } := { a20 a21 a22 } . { f2 f6 }

    f8 = __PSQ_LX(p1, offsetof(MTX43, f._20), 0, 0); // { a20 a21 }
    f9 = __PS_MULS0(f0, f8);
    f10 = __PS_MULS0(f4, f8);
    f9 = __PS_MADDS1(f1, f8, f9);
    f10 = __PS_MADDS1(f5, f8, f10);
    f8 = __PSQ_LX(p1, offsetof(MTX43, f._22), 0, 0); // { a22 a30 }
    f9 = __PS_MADDS0(f2, f8, f9);
    f10 = __PS_MADDS0(f6, f8, f10);
    __PSQ_STX(pOut, offsetof(MTX43, m[2][0]), f9, 0, 0);
    __PSQ_STX(pOut, offsetof(MTX43, m[2][2]), f10, 1, 0);

    //                                                         { f0 f4 }
    //                                                         { f1 f5 }
    //                                                         { f2 f6 }
    // { d30 d31 d32 d33 } = { f9 f10 } := { a30 a31 a32 1 } . { f3 f7 }

    f9 = __PS_MULS1(f0, f8);
    f10 = __PS_MULS1(f4, f8);
    f8 = __PSQ_LX(p1, offsetof(MTX43, f._31), 0, 0); // { a31 a32 }
    f9 = __PS_MADDS0(f1, f8, f9);
    f10 = __PS_MADDS0(f5, f8, f10);
    f9 = __PS_MADDS1(f2, f8, f9);
    f10 = __PS_MADDS1(f6, f8, f10);
    f9 = __PS_ADD(f3, f9);
    f10 = __PS_ADD(f7, f10);
    __PSQ_STX(pOut, offsetof(MTX43, m[3][0]), f9, 0, 0);
    __PSQ_STX(pOut, offsetof(MTX43, m[3][2]), f10, 1, 0);

    return pOut;
}

NW_MATH_INLINE MTX43*
MTX43Mult(MTX43* pOut, const MTX43* p, f32 f)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p );

    tof32x2(pOut->a[0]) = __PS_MULS0F(tof32x2(p->a[0]), f);
    tof32x2(pOut->a[2]) = __PS_MULS0F(tof32x2(p->a[2]), f);
    tof32x2(pOut->a[4]) = __PS_MULS0F(tof32x2(p->a[4]), f);
    tof32x2(pOut->a[6]) = __PS_MULS0F(tof32x2(p->a[6]), f);
    tof32x2(pOut->a[8]) = __PS_MULS0F(tof32x2(p->a[8]), f);
    tof32x2(pOut->a[10]) = __PS_MULS0F(tof32x2(p->a[10]), f);

    return pOut;
}

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

//----------------------------------------
//! @name    行列
//@{

//---------------------------------------------------------------------------
//! @brief        行列が単位行列かどうか判定します。
//!
//! @param[in]    p  判定対象の行列へのポインタ。
//!
//! @return       p が単位行列であれば true そうでなければ false を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE bool
MTX43IsIdentity(const MTX43* p)
{
    return p->f._00 == 1.f && p->f._01 == 0.f && p->f._02 == 0.f &&
           p->f._10 == 0.f && p->f._11 == 1.f && p->f._12 == 0.f &&
           p->f._20 == 0.f && p->f._21 == 0.f && p->f._22 == 1.f &&
           p->f._30 == 0.f && p->f._31 == 0.f && p->f._32 == 0.f;
}

//---------------------------------------------------------------------------
//! @brief        零行列を作成します。
//!
//! @param[out]   pOut  零行列を格納するバッファへのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX43*
MTX43Zero(MTX43* pOut)
{
    NW_ASSERT_NOT_NULL( pOut );

    pOut->f._00 = pOut->f._01 = pOut->f._02 =
    pOut->f._10 = pOut->f._11 = pOut->f._12 =
    pOut->f._20 = pOut->f._21 = pOut->f._22 =
    pOut->f._30 = pOut->f._31 = pOut->f._32 = 0.f;

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        単位行列を作成します。
//!
//! @param[out]   pOut  単位行列を格納するバッファへのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX43*
MTX43Identity(MTX43* pOut)
{
    NW_ASSERT_NOT_NULL( pOut );

    MTX43Copy(pOut, MTX43::Identity());

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        行列をコピーします。
//!
//! @param[out]   pOut  コピー先の行列へのポインタ。
//! @param[in]    p     コピー元の行列へのポインタ
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX43*
MTX43Copy(MTX43* pOut, const MTX43* p)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p );

    if (pOut != p)
    {
        *pOut = *p;
    }

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        行列の和を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じ行列を指していても構いません。
//! @param[in]    p1    左辺値へのポインタ。
//! @param[in]    p2    右辺値へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX43*
MTX43Add(MTX43* pOut, const MTX43* p1, const MTX43* p2)
{
    return NW_MATH_IMPL_NS::MTX43Add(pOut, p1, p2);
}

//---------------------------------------------------------------------------
//! @brief        行列の差を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じ行列を指していても構いません。
//! @param[in]    p1    左辺値へのポインタ。
//! @param[in]    p2    右辺値へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX43*
MTX43Sub(MTX43* pOut, const MTX43* p1, const MTX43* p2)
{
    return NW_MATH_IMPL_NS::MTX43Sub(pOut, p1, p2);
}

//---------------------------------------------------------------------------
//! @brief        行列の積を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じ行列を指していても構いません。
//! @param[in]    p1    左辺値へのポインタ。
//! @param[in]    p2    右辺値へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX43*
MTX43Mult(MTX43* pOut, const MTX43* __restrict p1, const MTX43* __restrict p2)
{
    return NW_MATH_IMPL_NS::MTX43Mult(pOut, p1, p2);
}

//---------------------------------------------------------------------------
//! @brief        行列にスカラを乗算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p と同じ行列を指していても構いません。
//! @param[in]    p   元となる行列へのポインタ。
//! @param[in]    f   乗算するスカラ値。。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX43*
MTX43Mult(MTX43* pOut, const MTX43* p, f32 f)
{
    return NW_MATH_IMPL_NS::MTX43Mult(pOut, p, f);
}

//@}

}  // namespace math
}  // namespace nw
