﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo/HAL Laboratory, Inc. All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/math/math_Vector3.h>

namespace nw {
namespace math {

namespace internal { namespace standard {

NW_MATH_INLINE VEC3*
VEC3Transform(VEC3* pOut, const MTX33* pM, const VEC3* pV)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(pM);
    NW_ASSERT_NOT_NULL(pV);

    VEC3 vTmp;
    VEC3* pDst = (pOut == pV) ? &vTmp : pOut;
    pDst->x = pM->f._00 * pV->x + pM->f._01 * pV->y + pM->f._02 * pV->z;
    pDst->y = pM->f._10 * pV->x + pM->f._11 * pV->y + pM->f._12 * pV->z;
    pDst->z = pM->f._20 * pV->x + pM->f._21 * pV->y + pM->f._22 * pV->z;

    if (pDst == &vTmp)
    {
        pOut->x = pDst->x;
        pOut->y = pDst->y;
        pOut->z = pDst->z;
    }

    return pOut;
}

NW_MATH_INLINE MTX33*
MTX33MAdd(MTX33* pOut, f32 t, const MTX33* p1, const MTX33* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    pOut->f._00 = t * p1->f._00 + p2->f._00;
    pOut->f._01 = t * p1->f._01 + p2->f._01;
    pOut->f._02 = t * p1->f._02 + p2->f._02;

    pOut->f._10 = t * p1->f._10 + p2->f._10;
    pOut->f._11 = t * p1->f._11 + p2->f._11;
    pOut->f._12 = t * p1->f._12 + p2->f._12;

    pOut->f._20 = t * p1->f._20 + p2->f._20;
    pOut->f._21 = t * p1->f._21 + p2->f._21;
    pOut->f._22 = t * p1->f._22 + p2->f._22;

    return pOut;
}

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

NW_MATH_INLINE VEC3*
VEC3Transform(VEC3* pOut, const MTX33* pM, const VEC3* pV)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(pM);
    NW_ASSERT_NOT_NULL(pV);

    f32x2 fp0, fp1, fp2, fp3, fp4;

    // { fp0 fp1 } = { x y z 1 }
    fp0 = __PSQ_LX(pV, offsetof(VEC3, x), 0, 0);
    fp1 = __PSQ_LX(pV, offsetof(VEC3, z), 1, 0);

    // fp4 = [ m00 m01 m02 ] . T[ x y z 1 ]
    fp2 = __PSQ_LX(pM, offsetof(MTX33, m[0][0]), 0, 0);
    fp3 = __PSQ_LX(pM, offsetof(MTX33, m[0][2]), 1, 0);
    fp4 = __PS_MUL(fp2, fp0);
    fp4 = __PS_SUM0(fp4, fp4, fp4);
    fp4 = __PS_MADD(fp3, fp1, fp4);

    pOut->x = fp4[0];

    // fp4 = [ m10 m11 m12 ] . T[ x y z 1 ]
    fp2 = __PSQ_LX(pM, offsetof(MTX33, m[1][0]), 0, 0);
    fp3 = __PSQ_LX(pM, offsetof(MTX33, m[1][2]), 1, 0);
    fp4 = __PS_MUL(fp2, fp0);
    fp4 = __PS_SUM0(fp4, fp4, fp4);
    fp4 = __PS_MADD(fp3, fp1, fp4);

    pOut->y = fp4[0];

    // fp4 = [ m20 m21 m22 ] . T[ x y z 1 ]
    fp2 = __PSQ_LX(pM, offsetof(MTX33, m[2][0]), 0, 0);
    fp3 = __PSQ_LX(pM, offsetof(MTX33, m[2][2]), 1, 0);
    fp4 = __PS_MUL(fp2, fp0);
    fp4 = __PS_SUM0(fp4, fp4, fp4);
    fp4 = __PS_MADD(fp3, fp1, fp4);

    pOut->z = fp4[0];

    return pOut;
}

NW_MATH_INLINE MTX33*
MTX33MAdd(MTX33* pOut, f32 t, const MTX33* p1, const MTX33* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    tof32x2(pOut->a[0]) = __PS_MADDS0F(tof32x2(p1->a[0]), t, tof32x2(p2->a[0]));
    tof32x2(pOut->a[2]) = __PS_MADDS0F(tof32x2(p1->a[2]), t, tof32x2(p2->a[2]));
    tof32x2(pOut->a[4]) = __PS_MADDS0F(tof32x2(p1->a[4]), t, tof32x2(p2->a[4]));
    tof32x2(pOut->a[6]) = __PS_MADDS0F(tof32x2(p1->a[6]), t, tof32x2(p2->a[6]));
    pOut->a[8] = t * p1->a[8] + p2->a[8];

    return pOut;
}

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS


//----------------------------------------
//! @name    行列
//@{

//---------------------------------------------------------------------------
//! @brief        行列が単位行列かどうか判定します。
//!
//! @param[in]    p  判定対象の行列へのポインタ。
//!
//! @return       p が単位行列であれば true そうでなければ false を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE bool
MTX33IsIdentity(const MTX33* p)
{
    return p->f._00 == 1.f && p->f._01 == 0.f && p->f._02 == 0.f &&
           p->f._10 == 0.f && p->f._11 == 1.f && p->f._12 == 0.f &&
           p->f._20 == 0.f && p->f._21 == 0.f && p->f._22 == 1.f;
}

//@}

//----------------------------------------
//! @name    ユーティリティ
//@{

//---------------------------------------------------------------------------
//! @brief        ベクトルを行列で変換します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。pV と同じベクトルを指していても構いません。
//! @param[in]    pM    変換行列へのポインタ。
//! @param[in]    pV    元となるベクトルへのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC3*
VEC3Transform(VEC3* pOut, const MTX33* pM, const VEC3* pV)
{
    return NW_MATH_IMPL_NS::VEC3Transform(pOut, pM, pV);
}

//---------------------------------------------------------------------------
//! @brief        回転行列とみなしてRPYを計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。
//! @param[in]    pM    入力の行列へのポインタです。
//!
//! @return       RPY をラジアン単位で返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC3*
VEC3CalcRPY(VEC3* pOut, const MTX33* pM)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(pM);

    f32 tmp = math::FAbs(pM->_20);

    if (1.0f - tmp < math::F_ULP)
    {
        pOut->x = 0.f;
        pOut->y = -math::F_PI / 2.0f * (pM->_20 / tmp);
        pOut->z = math::Atan2Rad(-pM->_01, -pM->_20 * pM->_02);
    }
    else
    {
        pOut->x = math::Atan2Rad(pM->_21, pM->_22);
        pOut->y = math::AsinRad(-pM->_20);
        pOut->z = math::Atan2Rad(pM->_10, pM->_00);
    }

    return pOut;
}

//@}

//---------------------------------------------------------------------------
//        MTX33
//---------------------------------------------------------------------------

//----------------------------------------
//! @name    行列
//@{

//---------------------------------------------------------------------------
//! @brief        3x3行列をコピーします。
//!
//! @param[out]   pOut  コピー先の行列のポインタ
//! @param[in]    p     コピー元の行列のポインタ
//!
//! @return       pOutを返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX33*
MTX33Copy(MTX33* pOut, const MTX33* p)
{
    if (pOut != p)
    {
        *pOut = *p;
    }

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        零行列を作成します。
//!
//! @param[out]   pOut  零行列を格納するバッファへのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX33*
MTX33Zero(MTX33* pOut)
{
    pOut->f._00 = pOut->f._01 = pOut->f._02 =
    pOut->f._10 = pOut->f._11 = pOut->f._12 =
    pOut->f._20 = pOut->f._21 = pOut->f._22 = 0.f;

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        単位行列を作成します。
//!
//! @param[out]   pOut  単位行列を格納するバッファへのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX33*
MTX33Identity(MTX33* pOut)
{
    NW_ASSERT_NOT_NULL( pOut );

    MTX33Copy(pOut, MTX33::Identity());

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        行列を実数倍して、別の行列を足します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じ行列を指していても構いません。
//! @param[in]    t     掛ける数。
//! @param[in]    p1    元の行列へのポインタ。
//! @param[in]    p2    足す行列へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX33*
MTX33MAdd(MTX33* pOut, f32 t, const MTX33* p1, const MTX33* p2)
{
    return NW_MATH_IMPL_NS::MTX33MAdd(pOut, t, p1, p2);
}

//@}

}  // namespace math
}  // namespace nw
