﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_WINDOW_H_
#define NW_LYT_WINDOW_H_

#include <nw/lyt/lyt_WindowFoundation.h>
#include <nw/ut/ut_RuntimeTypeInfo.h>

#include <nw/lyt/lyt_Pane.h>
#include <nw/lyt/lyt_Material.h>

namespace nw
{
namespace lyt
{

namespace res
{

struct Window;

}   // namespace res

class DrawInfo;

//---------------------------------------------------------------------------
//! @brief ウィンドウの表示を行うペインです。
//!
//---------------------------------------------------------------------------
class Window : public Pane
{
    typedef Pane Base;

    //! @brief 内部用機能のため使用禁止です。
    struct Frame
    {
        Frame()
        :   textureFlip(0),
            pMaterial(0)
        {}

        ~Frame();

        TextureFlip GetTextureFlip() const
        {
            return (TextureFlip) this->textureFlip;
        }

        u8 textureFlip;
        Material* pMaterial;
    };

public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO(Window::Base);

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    //! @details
    //! コンテンツ領域とフレーム(4辺共通)でそれぞれ指定したテクスチャの
    //! 枚数分のメモリを確保して、オブジェクトを生成します。
    //!
    //! @param[in] contentTexNum    コンテンツ領域で使用するテクスチャの最大数です。
    //! @param[in] frameTexNum      フレームで使用するテクスチャの最大数です。
    //!
    Window(
        u8  contentTexNum,
        u8  frameTexNum);

    //! @brief コンストラクタです。
    //!
    //! @details
    //! コンテンツ領域とフレーム(4辺)でそれぞれ指定したテクスチャの
    //! 枚数分のメモリを確保して、オブジェクトを生成します。
    //!
    //! @param[in] contentTexNum    コンテンツ領域で使用するテクスチャの最大数です。
    //! @param[in] frameLTTexNum    左上辺フレームで使用するテクスチャの最大数です。
    //! @param[in] frameRTTexNum    右上辺フレームで使用するテクスチャの最大数です。
    //! @param[in] frameRBTexNum    右下辺フレームで使用するテクスチャの最大数です。
    //! @param[in] frameLBTexNum    左下辺フレームで使用するテクスチャの最大数です。
    //!
    Window(
        u8  contentTexNum,
        u8  frameLTTexNum,
        u8  frameRTTexNum,
        u8  frameRBTexNum,
        u8  frameLBTexNum);

    //! @brief コンストラクタです。
    //!
    //! @details
    //! コンテンツ領域とフレーム(4隅と4辺)でそれぞれ指定したテクスチャの
    //! 枚数分のメモリを確保して、オブジェクトを生成します。
    //!
    //! @param[in] contentTexNum    コンテンツ領域で使用するテクスチャの最大数です。
    //! @param[in] cornerLTTexNum   左上隅フレームで使用するテクスチャの最大数です。
    //! @param[in] cornerRTTexNum   右上隅フレームで使用するテクスチャの最大数です。
    //! @param[in] cornerRBTexNum   右下隅フレームで使用するテクスチャの最大数です。
    //! @param[in] cornerLBTexNum   左下隅フレームで使用するテクスチャの最大数です。
    //! @param[in] frameLTexNum     左辺フレームで使用するテクスチャの最大数です。
    //! @param[in] frameTTexNum     上辺フレームで使用するテクスチャの最大数です。
    //! @param[in] frameRTexNum     右辺フレームで使用するテクスチャの最大数です。
    //! @param[in] frameBTexNum     下辺フレームで使用するテクスチャの最大数です。
    //!
    Window(
        u8  contentTexNum,
        u8  cornerLTTexNum,
        u8  cornerRTTexNum,
        u8  cornerRBTexNum,
        u8  cornerLBTexNum,
        u8  frameLTexNum,
        u8  frameTTexNum,
        u8  frameRTexNum,
        u8  frameBTexNum);

    //! @brief コンストラクタです。
    //!
    //! @details
    //! リソースからオブジェクトを生成します。
    //!
    //! @param[in] pBaseBlock       このペインが所属するレイアウトが保持しているリソースへのポインタです。
    //! @param[in] pOverrideBlock   上書きを行ったリソースへのポインタです。上書きが行われていないときはNULLになります。
    //! @param[in] buildArgSet      構築時の引数への参照です。
    //!
    Window(
        const res::Window* pBaseBlock,
        const res::Window* pOverrideBlock,
        const BuildArgSet& buildArgSet);

    //! @brief コピーコンストラクタです。
    //!
    //! @details
    //! 引数で指定したペインをコピーしたペインを作成します。
    //!
    //! ペインの元々のレイアウトデータの状態ではなく、そのときの状態でコピーします
    //! のでご注意ください。つまり、アニメーションを再生した後はそのままの状態で
    //! コピーします。
    //!
    //! ただし、このメソッドではペインの親子関係はコピーしませんので、ご注意ください。
    //! つまり、ペインは親を持たず、子のリストが空の状態で作られます。
    //! ペインツリーに登録するためには、明示的にペインにAppendChildしてください。
    //!
    //! また、アニメーションについても、何もバインドされていない状態になります。
    //!
    //! @param[in] window   コピー元のペインです。
    //!
    explicit Window(const Window& window);

    //! @brief デストラクタです。
    //!
    virtual ~Window();

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief 頂点カラーを取得します。
    //!
    //! @param[in] idx  インデックスです。 lyt::VertexColor を指定します。
    //!
    //! @return ペインの四隅の頂点カラーを返します。
    //!
    //! @sa SetVtxColor
    //! @sa GetVtxColorElement
    //! @sa lyt::VertexColor
    //!
    virtual const ut::Color8 GetVtxColor(u32 idx) const;

    //! @brief 頂点カラーを設定します。
    //!
    //! @param[in] idx      インデックスです。 lyt::VertexColor を指定します。
    //! @param[in] value    頂点カラーです。
    //!
    //! @sa GetVtxColor
    //! @sa SetVtxColorElement
    //! @sa lyt::VertexColor
    //!
    virtual void SetVtxColor(u32 idx, ut::Color8 value);

    //! @brief テキストの表示色（頂点カラー）の各カラー成分を取得します。
    //!
    //! @details
    //! このメソッドは、R,G,B など色成分を個別に取得します。色成分をまとめて取得する場合は GetVtxColor を利用してください。
    //!
    //! @param[in] idx  インデックスです。 lyt::AnimTargetPaneColor を指定します。
    //!
    //! @return テキストの表示色の各成分を返します。
    //!
    //! @sa SetVtxColorElement
    //! @sa GetVtxColor
    //! @sa lyt::AnimTargetPaneColor
    //!
    virtual u8 GetVtxColorElement(u32 idx) const;

    //! @brief テキストの表示色（頂点カラー）の各カラー成分を設定します。
    //!
    //! @details
    //! このメソッドは、R,G,B など色成分を個別に設定します。色成分をまとめて設定する場合は SetVtxColor を利用してください。
    //!
    //! @param[in] idx      インデックスです。 lyt::AnimTargetPaneColor を指定します。
    //! @param[in] value    カラーの成分値です。
    //!
    //! @sa GetVtxColorElement
    //! @sa SetVtxColor
    //! @sa lyt::AnimTargetPaneColor
    //!
    virtual void SetVtxColorElement(u32 idx, u8 value);

    using Base::GetMaterial;

    //! @brief ペインが持つマテリアルの数を取得します。
    //!
    //! @details
    //! Window ペインはコンテンツ領域＋フレーム数のマテリアルを持ちます。
    //!
    //! @return フレーム数 + 1 を返します。
    //!
    //! @sa GetMaterial
    //!
    virtual u8 GetMaterialNum() const;

    //! @brief ペインのマテリアルを取得します。
    //!
    //! @details
    //! idx に指定する値は GetMaterialNum() の返り値未満でなければなりません。
    //!
    //! @param[in] idx  インデックスです。
    //!
    //! @return マテリアルへのポインタを返します。
    //!
    //! @sa GetMaterialNum
    //! @sa GetContentMaterial
    //! @sa SetContentMaterial
    //! @sa GetFrameMaterial
    //! @sa SetFrameMaterial
    //!
    virtual Material* GetMaterial(u32 idx) const;

    //! @brief コンテント領域のマテリアルを取得します。
    //!
    //! @return マテリアルへのポインタを返します。
    //!
    //! @sa GetMaterial
    //! @sa SetContentMaterial
    //! @sa GetFrameMaterial
    //! @sa SetFrameMaterial
    //!
    Material* GetContentMaterial() const;

    //! @brief コンテント領域のマテリアルを設定します。
    //!
    //! @details
    //! 現在設定されているマテリアルが Window オブジェクトの生成時に
    //! 同時に生成されたものだった場合には、そのマテリアルは破棄されます。
    //!
    //! @param[in] pMaterial    マテリアルへのポインタです。
    //!
    //! @sa GetMaterial
    //! @sa GetContentMaterial
    //! @sa GetFrameMaterial
    //! @sa SetFrameMaterial
    //!
    void SetContentMaterial(Material* pMaterial);

    //! @brief ウィンドウフレーム領域のマテリアルを取得します。
    //!
    //! @param[in] frameIdx フレームのインデックスです。
    //!
    //! @return マテリアルへのポインタを返します。
    //!
    //! @sa GetMaterial
    //! @sa GetContentMaterial
    //! @sa SetContentMaterial
    //! @sa SetFrameMaterial
    //!
    Material* GetFrameMaterial(WindowFrame frameIdx) const;

    //! @brief ウィンドウフレーム領域のマテリアルを設定します。
    //!
    //! @details
    //! 現在設定されているマテリアルが Window オブジェクトの生成時に
    //! 同時に生成されたものだった場合には、そのマテリアルは破棄されます。
    //!
    //! @param[in] frameIdx     フレームのインデックスです。
    //! @param[in] pMaterial    マテリアルへのポインタです。
    //!
    //! @sa GetMaterial
    //! @sa GetContentMaterial
    //! @sa SetContentMaterial
    //! @sa GetFrameMaterial
    //!
    void SetFrameMaterial(WindowFrame frameIdx, Material* pMaterial);

    //! @brief コンテント領域で使うテクスチャ座標を保持するためのメモリ領域を確保します。
    //!
    //! @details
    //! 保持可能なテクスチャ座標の数の初期値はコンテント領域のテクスチャ数と同じです。
    //!
    //! @param[in] num  テクスチャ座標の保持数です。
    //!
    //! @sa Window::Window
    //!
    void ReserveTexCoord(u8 num);

    //! @brief コンテント領域で使うテクスチャ座標の保持数を取得します。
    //!
    //! @return 保持しているテクスチャ座標の数を返します。
    //!
    //! @sa SetTexCoordNum
    //!
    u8 GetTexCoordNum() const;

    //! @brief コンテント領域で使うテクスチャ座標の保持数を設定します。
    //!
    //! @details
    //! num に指定できる値は保持可能なテクスチャ座標数以下でなければなりません。
    //!
    //! @param[in] num  テクスチャ座標の保持数です。
    //!
    //! @sa ReserveTexCoord
    //! @sa GetTexCoord
    //!
    void SetTexCoordNum(u8 num);

    //! @brief コンテント領域で使うテクスチャ座標を取得します。
    //!
    //! @param[in] idx      インデックスです。
    //! @param[out] coords  テクスチャ座標を格納する領域へのポインタです。
    //!
    void GetTexCoord(u32 idx, TexCoordQuad coords) const;

    //! @brief コンテント領域で使うテクスチャ座標を設定します。
    //!
    //! @param[in] idx      インデックスです。
    //! @param[in] coords   テクスチャ座標です。
    //!
    void SetTexCoord(u32 idx, const TexCoordQuad coords);

    //! @brief ウィンドウフレーム数を取得します。
    //!
    //! @details
    //! ウィンドウフレームの数はオブジェクト生成時に決定されます。
    //!
    //! @return オブジェクトの生成方法によって 1, 2, 4, 8 のいずれかを返します。
    //!
    //! @sa Window::Window
    //!
    u8 GetFrameNum() const
    {
        return m_FrameNum;
    }

    //! @brief ウィンドウフレームのサイズを取得します。
    //!
    //! @details
    //! HORIZONTAL_MODE の場合、左右のフレームはそれぞれ WINDOWFRAME_LT, WINDOWFRAME_RT
    //! になります。
    //!
    //! @param[out] size    フレームのサイズです。
    //!
    //! @sa Window::Window
    //!
    void CalcFrameSize(WindowFrameSize* size)
    {
        NW_ASSERTMSG(size, "size must not be NULL for Window[%s]", GetName());

        size->l = m_WindowSize.frameSize.l;
        size->r = m_WindowSize.frameSize.r;
        size->t = m_WindowSize.frameSize.t;
        size->b = m_WindowSize.frameSize.b;
    }

    //! @brief ウィンドウフレームのモードを取得します。
    //!
    //! @return ウィンドウフレームのモード
    //!
    WindowFrameMode GetWindowFrameMode() const
    {
        return static_cast<WindowFrameMode>(m_FrameMode);
    }

    //@}

    //----------------------------------------
    //! @name 検索
    //@{

    //! @brief 名前でマテリアルを検索します。
    //!
    //! @details
    //! findName のマテリアル名を持つマテリアルを検索します。
    //!
    //! bRecursive に true を渡した場合は子ペインと、子ペインが子を
    //! 持つ場合はさらにその子ペインからも検索します。
    //!
    //! @param[in] findName     検索するマテリアル名です。
    //! @param[in] bRecursive   子ペインも検索する場合は true を指定します。
    //!
    //! @return マテリアルが見つかった場合はそのマテリアルへのポインタを、
    //! 見つからなかった場合は NULL を返します。
    //!
    virtual Material* FindMaterialByName(
        const char* findName,
        bool bRecursive = true);

    //@}

protected:
    virtual void DrawSelf(DrawInfo& drawInfo);

private:
    void InitTexNum(
        u8 contentTexNum,
        u8 frameTexNums[],
        u8 frameNum);

    void InitContent(u8 texNum);

    void InitFrame(u8 frameNum);

    void DrawAroundFrameWindow(DrawInfo& drawInfo);
    void DrawHorizontalFrameWindow(DrawInfo& drawInfo);
    void DrawHorizontalFrameNocontentWindow(DrawInfo& drawInfo);

    void DrawContent(
        DrawInfo& draInfo,
        const math::VEC2& basePt,
        const WindowFrameSize& frameSize,
        u8 alpha);

    void DrawFrame(
        DrawInfo& draInfo,
        const math::VEC2& basePt,
        const Frame& frame,
        const WindowFrameSize& frameSize,
        u8 alpha);

    void DrawHorizontalFrame(
        DrawInfo& draInfo,
        const math::VEC2& basePt,
        const Frame& frame,
        const WindowFrameSize& frameSize,
        u8 alpha);

    void DrawHorizontalNocontentFrame(
        DrawInfo& draInfo,
        const math::VEC2& basePt,
        const Frame& frame,
        const WindowFrameSize& frameSize,
        u8 alpha);

    void DrawHorizontalFrame2(
        DrawInfo& draInfo,
        const math::VEC2& basePt,
        const Frame* frames,
        const WindowFrameSize& frameSize,
        u8 alpha);

    void DrawHorizontalNocontentFrame2(
        DrawInfo& draInfo,
        const math::VEC2& basePt,
        const Frame* frames,
        const WindowFrameSize& frameSize,
        u8 alpha);

    void DrawFrame4(
        DrawInfo& draInfo,
        const math::VEC2& basePt,
        const Frame* frames,
        const WindowFrameSize& frameSize,
        u8 alpha);

    void DrawFrame8(
        DrawInfo& draInfo,
        const math::VEC2& basePt,
        const Frame* frames,
        const WindowFrameSize& frameSize,
        u8 alpha);

    // コピー演算子を禁止します。
    const Window& operator=( const Window& );

private:
    enum WindowFlags
    {
        OVERALL_VERTEX_COLOR_ENABLED = (1 << 0),
        ONEMATERIAL_FOR_ALLFRAME_ENABLED = (1 << 1),
        NOT_DRAW_CONTENT_ENABLED = (1 << 2),

        CONTENT_INFLATION_ENABLED = (1 << 3)
    };
    WindowSize m_WindowSize;
    WindowContent m_Content;
    u8 m_FrameMode;
    u8 m_FrameNum;
    u8 m_WindowFlags;
    Frame* m_Frames;
    Material* m_pMaterial;
};

//----------------------------------------
NW_INLINE void
Window::ReserveTexCoord(u8 num)
{
    m_Content.texCoordAry.Reserve(num);
}

//----------------------------------------
NW_INLINE u8
Window::GetTexCoordNum() const
{
    return m_Content.texCoordAry.GetSize();
}

//----------------------------------------
NW_INLINE void
Window::SetTexCoordNum(u8 num)
{
    m_Content.texCoordAry.SetSize(num);
}

//----------------------------------------
NW_INLINE void
Window::GetTexCoord(
    u32 idx,
    TexCoordQuad coords
) const
{
    m_Content.texCoordAry.GetCoord(idx, coords);
}

//----------------------------------------
NW_INLINE void
Window::SetTexCoord(
    u32 idx,
    const TexCoordQuad coords
)
{
    m_Content.texCoordAry.SetCoord(idx, coords);
}

//----------------------------------------
NW_INLINE Material*
Window::GetFrameMaterial(WindowFrame frameIdx) const
{
    NW_ASSERTMSG(frameIdx < WINDOWFRAME_MAX, "out of bounds: frameIdx[%d] < WINDOWFRAME_MAX for Window[%s]", frameIdx, GetName());

    if (frameIdx >= m_FrameNum)
    {
        return NULL;
    }

    return m_Frames[frameIdx].pMaterial;
}

//----------------------------------------
NW_INLINE Material*
Window::GetContentMaterial() const
{
    return m_pMaterial;
}


} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_WINDOW_H_

