﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_UTIL_H_
#define NW_LYT_UTIL_H_

#include <nw/lyt/lyt_Types.h>
#include <nw/ut/ut_Color.h>

namespace nw
{
namespace math
{

struct VEC2;
struct MTX23;

} // namespace nw::math

namespace lyt
{

class Pane;
class Layout;
class DrawInfo;
class Group;
class AnimTransform;
class TexMap;

namespace res
{
struct HermiteKey;
}

//----------------------------------------
//! @name アニメーション
//@{

//---------------------------------------------------------------------------
//! @brief グループ内の個々のペインにアニメーションを関連付けます。
//!
//! @param[in] pGroup           グループへのポインタです。
//! @param[in,out] pAnimTrans   アニメーションへのポインタです。
//! @param[in] bEnable          アニメーションを無効状態で関連付ける場合は false を指定します。
//!
//! @details
//! bEnable に false を渡して呼び出した場合は、アニメーションを無効状態で
//! 関連付けます。
//! 有効にする場合は、 AnimTransform::SetEnable() を使用してください。
//!
//! @sa nw::lyt::UnbindAnimation
//! @sa nw::lyt::Pane::BindAnimation
//! @sa nw::lyt::AnimTransform::SetEnable
//!
//---------------------------------------------------------------------------
void                BindAnimation(
                        Group*          pGroup,
                        AnimTransform*  pAnimTrans,
                        bool            bEnable = true);

//---------------------------------------------------------------------------
//! @brief グループ内の個々のペインからアニメーションの関連付けを解除します。
//!
//! @param[in] pGroup           グループへのポインタです。
//! @param[in,out] pAnimTrans   アニメーションへのポインタです。
//!
//! @sa nw::lyt::BindAnimation
//! @sa nw::lyt::Pane::UnbindAnimation
//!
//---------------------------------------------------------------------------
void                UnbindAnimation(
                        Group*          pGroup,
                        AnimTransform*  pAnimTrans);

//@}

//----------------------------------------
//! @name ヒットチェック
//@{

//---------------------------------------------------------------------------
//! @brief ペインが指定した点を含むかどうかを判定します。
//!
//! @param[in] pPane    ペインへのポインタです。
//! @param[in] pos      チェックする点の位置です。
//!
//! @return ペインが pos で指定した点を含む場合は true を返します。
//!
//! @details
//! ペインの で指定した点を含むかどうかを判別します。
//!
//! Pane::CalculateMtx() によって計算された行列の値を使用するため、
//! この関数を呼び出す前に、 Pane::CalculateMtx() が呼び出されている
//! 必要があります。
//!
//! この関数は、ペインの回転の x, y の値が 0 であることを前提にしています。
//! 0 以外の値の場合は正しく判定できません。
//!
//! @sa nw::lyt::FindHitPane
//!
//---------------------------------------------------------------------------
bool                IsContain(
                        Pane*               pPane,
                        const math::VEC2&   pos);

//---------------------------------------------------------------------------
//! @brief 指定した点を含むペインを検索します。
//!
//! @param[in] pPane    ペインへのポインタです。
//! @param[in] pos      チェックする点の位置です。
//!
//! @return 指定した点を含む境界ペインが見つかった場合は境界ペインへの
//! ポインタを、見つからなかった場合は NULL を返します。
//!
//! @details
//! pos で指定した点を含む境界ペインを検索します。検索順は、描画順と逆の
//! 順序になります。非表示のペインとその子階層は検索対象になりません。
//!
//! pPane で指定したペインをルートとして検索します。
//!
//---------------------------------------------------------------------------
Pane*               FindHitPane(
                        Pane*               pPane,
                        const math::VEC2&   pos);

//---------------------------------------------------------------------------
//! @brief 指定した点を含むペインを検索します。
//!
//! @param[in] pLayout  レイアウトへのポインタです。
//! @param[in] pos      チェックする点の位置です。
//!
//! @return 指定した点を含む境界ペインが見つかった場合は境界ペインへの
//! ポインタを、見つからなかった場合は NULL を返します。
//!
//! @details
//! pos で指定した点を含む境界ペインを検索します。検索順は、描画順と逆の
//! 順序になります。非表示のペインとその子階層は検索対象になりません。
//!
//! レイアウト全てのペインを検索対象とします。
//!
//! Pane::CalculateMtx() によって計算された行列の値を使用するため、
//! この関数を呼び出す前に、 Pane::CalculateMtx() が呼び出されている
//! 必要があります。
//!
//! この関数は、ペインの回転の x, y の値が 0 であることを前提にしています。」
//! 0 以外の値の場合は正しく判定できません。
//!
//---------------------------------------------------------------------------
Pane*               FindHitPane(
                        Layout*             pLayout,
                        const math::VEC2&   pos);

//@}

//---------------------------------------------------------------------------
//! @brief ペインの子供、あるいは兄弟のペインのポインタを返します。
//!
//! @param[in] pPane    ペインへのポインタです。
//!
//! @return ペインの子供、あるいは兄弟のペインのポインタを返します。
//! 該当するペインが無い場合は NULL を返します。
//!
//! @details
//! 引数 pPane で指定されたペインの子供、あるいは兄弟のペインを返します。
//! 最初にルートペインを引数にして呼び出し、以後 NULL が返るまで返り値を引数に
//! 設定して呼び出すことで、全てのペインを列挙することが出来ます。
//!
//---------------------------------------------------------------------------
Pane*               GetNextPane(Pane* pPane);

//---------------------------------------------------------------------------
//! @brief ペインを、その子ペインも一緒にコピーします。
//!
//! @param[in] pPane    コピー元のペインへのポインタです。
//!
//! @return コピーしたペインを返します。
//!
//! @details
//! このメソッドは、与えられたペインの子ペインに再帰的にコピーコンストラクタを
//! 呼び出して、子ペインも一緒にコピーします。
//!
//! ペインを単体でコピーしたい場合は、各ペインのコピーコンストラクタを使用してください。
//!
//! このメソッドはレイアウトの標準のペインクラスのみサポートしますので、アプリケーション
//! で独自に各ペインを継承して拡張している場合は、このメソッドを再実装する必要があります。
//!
//! 返り値のペインの親は何も指定されていない状態になっていますので、
//! ペインツリーに登録するためには、明示的にペインにAppendChildしてください。
//!
//! また、アニメーションについては、何もバインドされていない状態になります。
//!
//---------------------------------------------------------------------------
Pane*               ClonePaneTree(const Pane* pPane);

//---------------------------------------------------------------------------
//! @brief ペインを、ペインツリー階層ごと(部品レイアウトも)コピーします。
//!
//! @details
//! このメソッドは、与えられたペインの子ペインに再帰的にコピーコンストラクタを
//! 呼び出して、子ペインも一緒にコピーします。
//!
//! ClonePaneTreeの方は、ペインツリー階層の中に部品レイアウトがあってもレイアウトの
//! コピーは行いませんが、こちらは部品レイアウトもコピーします。
//!
//! このメソッドはレイアウトの標準のペインクラスのみサポートしますので、アプリケーション
//! で独自に各ペインを継承して拡張している場合は、このメソッドを再実装する必要があります。
//!
//! 返り値のペインの親は何も指定されていない状態になっていますので、
//! ペインツリーに登録するためには、明示的にペインにAppendChildしてください。
//!
//! また、アニメーションについては、何もバインドされていない状態になります。
//!
//! @param[in] pPane        コピー元のペインへのポインタです。
//! @param[in] pPartsLayout コピーするペインが部品ペインのとき、セットする部品レイアウトを指定します。
//!
//! @return コピーしたペインを返します。
//!
//---------------------------------------------------------------------------
Pane*               ClonePaneTreeWithPartsLayout(const Pane* pPane, Layout* pPartsLayout);

//---------------------------------------------------------------------------
//! @brief テクスチャをデバイスにロードします。
//!
//! @param[out] textureInfo ロードしたテクスチャの情報を格納します。
//! @param[in] pImgRes リソースです。
//! @param[in] size リソースのサイズです。
//!
//! @return ロードが成功した場合は true が返ります。
//!
//---------------------------------------------------------------------------
bool LoadTexture(TextureInfo* textureInfo, const void* pImgRes, u32 size);

//---------------------------------------------------------------------------
//! @brief アーカイブシェーダをデバイスにロードします。
//!
//! @param[out] pShaderInfo ロードしたアーカイブシェーダの情報を格納します。
//! @param[in] pShaderRes リソースです。
//! @param[in] size リソースのサイズです。
//! @param[in] name シェーダアーカイブの名前です。
//!
//! @return ロードが成功した場合は true が返ります。
//!
//---------------------------------------------------------------------------
bool LoadArchiveShader(ArchiveShaderInfo* pShaderInfo, const void* pShaderRes, u32 size, const char *name);

//---------------------------------------------------------------------------
//! @brief アーカイブシェーダを解放します。
//!
//! @param[in] pShaderInfo ロードしたアーカイブシェーダの情報を格納します。
//!
//---------------------------------------------------------------------------
void FreeArchiveShader(ArchiveShaderInfo* pShaderInfo);

//---------------------------------------------------------------------------
//! @brief GX2のテクスチャフォーマットを対応するレイアウトのテクスチャフォーマットに変換します。
//!
//! @param[out] pTexFormat  対応するテクスチャフォーマット
//! @param[in] gx2Format    変換元のフォーマット。GX2SurfaceFormatを指定してください。GX2をインクルードしていない場合にコンパイルエラーにならないように、型をintにしています。
//! @param[in] is1chUsaAlpha    1チャンネルのフォーマットのとき、アルファに割り当てるか。falseに割り当てる場合はルミナンスに割り当てることになります。
//!
//! @return 対応するフォーマットがある場合はtrue、ない場合はfalseが返されます。
//!
//---------------------------------------------------------------------------
bool GX2SurfaceFormatToLytFormat(TexFormat* pTexFormat, int gx2Format, bool is1chUsaAlpha);

//---------------------------------------------------------------------------
//! @brief 指定したペインツリー以下のヌルペインと境界ペインを枠線で描画します。
//!
//! @param[in] drawInfo         描画情報です。
//! @param[in] pPane            コピー元のペインへのポインタです。
//! @param[in] nullColor        ヌルペインの枠線の色です。
//! @param[in] boundingColor    境界ペインの枠線の色です。
//!
//---------------------------------------------------------------------------
void DrawNullAndBoundingPane(DrawInfo& drawInfo, const Pane* pPane, ut::Color8 nullColor, ut::Color8 boundingColor);

//---------------------------------------------------------------------------
//! @brief エルミート形式カーブの特定フレームでの値を取得します。
//!
//! @param[in] frame    値を取得するフレームです。
//! @param[in] keyArray エルミート形式キーの配列です。
//! @param[in] keySize  キー数です。
//!
//! @return 値です。
//!
//---------------------------------------------------------------------------
f32 GetHermiteCurveValue(f32 frame, const res::HermiteKey*  keyArray, u32 keySize);

} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_UTIL_H_
