﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_LYT_TEXTBOX_H_
#define NW_LYT_TEXTBOX_H_

#include <nw/ut/ut_Color.h>
#include <nw/ut/ut_RuntimeTypeInfo.h>
#include <nw/font/font_DispStringBuffer.h>
#include <nw/font/font_WideTextWriter.h>
#include <nw/lyt/lyt_TextSearcher.h>

#include <nw/lyt/lyt_Pane.h>

namespace nw
{
namespace font
{

class Font;

template <typename CharType>
class TagProcessorBase;

template <typename CharType>
class TextWriterBase;

} // namespace nw::font

namespace lyt
{

namespace res
{

struct TextBox;
struct AnimationTarget;
struct AnimationInfo;

}   // namespace res

class DrawInfo;
struct BuildResSet;
class Layout;

//---------------------------------------------------------------------------
//! @brief テキストの表示を行うペインです。
//!
//---------------------------------------------------------------------------
class TextBox : public Pane
{
    typedef Pane Base;

public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO(TextBox::Base);

    //! TextBox が使用するタグプロセッサの定義です。
    typedef font::TagProcessorBase<char16> TagProcessor;

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    //! @details
    //! オブジェクトを生成します。
    //! 文字列用バッファの確保、フォントのセット等は行いません。
    //!
    explicit TextBox();

    //! @brief InitStringメソッドにコンストラクタから必要なパラメータを渡すための構造体です。
    //!
    struct InitStringParam {
        u8 textBoxFlag;             //!< res::TextBox::textBoxFlagの値
        Layout* resourceLayout;     //!< リソースを保持しているレイアウト
        const char16* pBlockText;   //!< レイアウトエディタで指定された文字列
        u32 allocStrBufLen;         //!< レイアウトエディタで指定されたバッファサイズ
        u32 resStrLen;              //!< レイアウトエディタで指定された文字列長
    };

    //! @brief コンストラクタです。
    //!
    //! @details
    //! リソースからオブジェクトを生成します。
    //! 文字列関連の初期化は、ここではなくInitStringで行われます。
    //!
    //! @param[in]  pBaseBlock      このペインが所属するレイアウトが保持しているリソースへのポインタです。
    //! @param[in]  pOverrideBlock  上書きを行ったリソースへのポインタです。上書きが行われていないときはNULLになります。
    //! @param[in]  buildArgSet     構築時の引数への参照です。
    //! @param[out] initStringParam InitStringを呼び出す際のパラメータを書き出します。
    //!
    TextBox(
        const res::TextBox* pBaseBlock,
        const res::TextBox* pOverrideBlock,
        const BuildArgSet& buildArgSet,
        InitStringParam* initStringParam);

    //! @brief 文字列関連の初期化を行います。
    //!
    //! @param[in] buildArgSet     構築時の引数への参照です。
    //! @param[in] initStringParam コンストラクタから渡されるパラメータです。
    //!
    virtual void InitString(const BuildArgSet& buildArgSet, const InitStringParam& initStringParam);

    //! @brief コピーコンストラクタです。
    //!
    //! @details
    //! 引数で指定したペインをコピーしたペインを作成します。
    //!
    //! ペインの元々のレイアウトデータの状態ではなく、そのときの状態でコピーします
    //! のでご注意ください。つまり、アニメーションを再生した後はそのままの状態で
    //! コピーします。
    //!
    //! ただし、このメソッドではペインの親子関係はコピーしませんので、ご注意ください。
    //! つまり、ペインは親を持たず、子のリストが空の状態で作られます。
    //! ペインツリーに登録するためには、明示的にペインに AppendChild() してください。
    //!
    //! また、アニメーションについても、何もバインドされていない状態になります。
    //!
    //! @param[in] textBox  コピー元のペインです。
    //!
    explicit TextBox(const TextBox& textBox);

    //! @brief バッファサイズを明示的に指定するコピーコンストラクタです。
    //!
    //! @details
    //! 引数で指定したペインをコピーしたペインを作成します。
    //!
    //! この関数では、バッファサイズはコピーされず、指定したサイズが使われます。
    //! 文字バッファの内容もコピーされませんのでご注意ください。bufferStrLenに0が指定
    //! された場合は、バッファの確保が行われていない状態になります。
    //!
    //! また、コピー元のテキストボックスが文字属性バッファを共有する設定であった
    //! としても、指定したbuferStrlenがコピー元のテキストボックスの文字バッファ
    //! サイズよりも大きい場合は、コピー先は文字属性バッファを共有しない設定に
    //! なりますので、ご注意ください。
    //! レイアウトが保持している共有のバッファが不足する恐れがあるためです。
    //!
    //! ペインの元々のレイアウトデータの状態ではなく、そのときの状態でコピーします
    //! のでご注意ください。つまり、アニメーションを再生した後はそのままの状態で
    //! コピーします。
    //!
    //! ただし、このメソッドではペインの親子関係はコピーしませんので、ご注意ください。
    //! つまり、ペインは親を持たず、子のリストが空の状態で作られます。
    //! ペインツリーに登録するためには、明示的にペインに AppendChild() してください。
    //!
    //! また、アニメーションについても、何もバインドされていない状態になります。
    //!
    //! @param[in] textBox      コピー元のペインです。
    //! @param[in] bufferStrlen 確保する文字列用バッファの長さ(文字数)です。
    //!
    TextBox(const TextBox& textBox, u16 bufferStrlen);

    //! @brief デストラクタです。
    //!
    //! @details
    //! 設定されているフォントオブジェクトは破棄しません。
    //!
    //! @date 2010/01/26 フォントは破棄しない仕様に変更しました。
    virtual ~TextBox();

    //@}

    //----------------------------------------
    //! @name 文字列操作
    //@{

    //! @brief 格納している文字列を取得します。
    //!
    //! @return
    //! 格納している文字列の先頭アドレスを返します。
    //! 文字列用バッファが確保されていなければ NULL を返します。
    //!
    //! @sa SetString
    //! @sa GetStringLength
    //!
    const char16* GetString() const
    {
        return m_TextBuf;
    }

    //! @brief 文字列用バッファを取得します。
    //!
    //! @details
    //! 文字列用バッファを確保するには AllocStringBuffer() を呼び出してください。
    //!
    //! @return
    //! 文字列用バッファの先頭アドレスを返します。
    //! 文字列用バッファが確保されていなければ NULL を返します。
    //!
    //! @sa GetStringBufferLength
    //! @sa AllocStringBuffer
    //!
    const char16* GetStringBuffer() const
    {
        return m_TextBuf;
    }

    //! @brief 格納している文字列の文字数を取得します。
    //!
    //! @return 格納している文字列の長さ(文字数)を返します。
    //!
    //! @sa SetString
    //! @sa GetString
    //!
    u16 GetStringLength() const
    {
        return m_TextLen;
    }

    //! @brief 文字列用バッファの格納可能な文字数を取得します。
    //!
    //! @return 文字列用バッファの長さを返します。
    //!
    //! @sa GetStringBuffer
    //! @sa AllocStringBuffer
    //!
    u16 GetStringBufferLength() const;

    //! @brief 描画用文字列バッファの長さを取得します。
    //!
    //! @return 描画用文字列バッファの長さを返します。
    //!
    //! @sa AllocStringBuffer
    //!
    u16 GetDrawStringBufferLength() const;

    //! @brief 文字列用バッファを確保します。
    //!
    //! @details
    //! 既に確保しているバッファの長さが minLen で指定した
    //! 長さより大きい場合は何もしません。
    //!
    //! 文字列バッファの長さと描画する際に必要なバッファの長さは
    //! 同じ値が指定されます。
    //!
    //! @param[in] minLen   確保する文字列用バッファの長さ(文字数)です。
    //! @param[in] isShareCharAttributeBuffer   DispStringBufferのCharAttributeのバッファをレイアウト内で共有するか否かです。デフォルトではfalseです。
    //!
    //! @sa FreeStringBuffer
    //! @sa GetStringBufferLength
    //! @sa IsShareCharAttributeBuffer
    //!
    //! @date 2010/05/10 引数 isDoubleBuffer を削除しました。
    //! @date 2010/02/19 引数 isDoubleBuffer を追加しました。
    virtual void AllocStringBuffer(u16 minLen, bool isShareCharAttributeBuffer = false);

    //! @brief 文字列用バッファを確保します。
    //!
    //! @details
    //! 引数一つのバージョンと異なり、文字列バッファの長さと、実際に
    //! 描画する際に必要なバッファの長さを分けて指定することができます。
    //!
    //! @param[in] minStrLen     確保する文字列用バッファの長さ(文字数)です。
    //! @param[in] minDrawStrLen 確保する描画用文字列バッファの長さ(文字数)です。
    //! @param[in] isShareCharAttributeBuffer   DispStringBufferのCharAttributeのバッファをレイアウト内で共有するか否かです。デフォルトではfalseです。
    //!
    //! @sa FreeStringBuffer
    //! @sa GetStringBufferLength
    //! @sa GetDrawStringBufferLength
    //! @sa IsShareCharAttributeBuffer
    //!
    virtual void AllocStringBuffer(u16 minStrLen, u16 minDrawStrLen, bool isShareCharAttributeBuffer = false);

    //! @brief 文字列用バッファを解放します。
    //!
    //! @sa AllocStringBuffer
    //!
    virtual void FreeStringBuffer();

    //! @brief 文字列バッファに文字列を格納します。
    //!
    //! @details
    //! 指定した文字列の長さが文字列用バッファの長さを超える場合は、
    //! 文字列用バッファの長さを超える分が切り捨てられます。
    //!
    //! @param[in] str      格納する文字列です。
    //! @param[in] dstIdx   格納位置です。
    //!
    //! @return 格納した文字列の長さを返します。
    //!
    //! @sa GetString
    //! @sa GetStringLength
    //!
    virtual u16 SetString(
        const char16*  str,
        u16             dstIdx = 0);

    //! @brief 文字列バッファに文字列を格納します。
    //!
    //! @details
    //! 文字列の長さは strLen で指定された値です。終端文字を判断しません。
    //!
    //! 指定した文字列の長さが文字列用バッファの長さを超える場合は、
    //! 文字列用バッファの長さを超える分が切り捨てられます。
    //!
    //! @param[in] str      格納する文字列です。
    //! @param[in] dstIdx   格納位置です。
    //! @param[in] strLen   格納する文字列の長さです。
    //!
    //! @return 格納した文字列の長さを返します。
    //!
    //! @sa GetString
    //! @sa GetStringLength
    //!
    //! @date 2010/04/09 strLen の値が 0 だった場合の記述に誤りがありました。値 0 は終端文字までを意味しません。
    virtual u16 SetString(
        const char16*  str,
        u16             dstIdx,
        u16             strLen);

    //@}

    //----------------------------------------
    //! @name 設定／取得
    //@{

    //! @brief テキストの表示色を取得します。
    //!
    //! @param[in] type 表示色の種類です。 lyt::TextColor を指定します。
    //!
    //! @return 表示色を返します。
    //!
    //! @sa SetTextColor
    //! @sa lyt::TextColor
    //!
    const ut::Color8 GetTextColor(u32 type) const
    {
        NW_ASSERTMSG(type < TEXTCOLOR_MAX, "out of bounds: type[%u] < TEXTCOLOR_MAX for TextBox[%s]", type, GetName());
        return m_TextColors[type];
    }

    //! @brief テキストの表示色を設定します。
    //!
    //! @details
    //! 指定の表示色を設定します。
    //!
    //! @param[in] type     表示色の種類です。 lyt::TextColor を指定します。
    //! @param[in] value    表示色です。
    //!
    //! @sa GetTextColor
    //! @sa lyt::TextColor
    //!
    void SetTextColor(u32 type, ut::Color8 value)
    {
        NW_ASSERTMSG(type < TEXTCOLOR_MAX, "out of bounds: type[%u] < TEXTCOLOR_MAX for TextBox[%s]", type, GetName());

        UpdatePTDirty(m_TextColors[type] != value);
        m_TextColors[type] = value;
    }

    //! @brief テキストの表示色を設定します。
    //!
    //! @details
    //! テキストの表示色の上端と下端を同時に設定します。
    //!
    //! @param[in] top      上端の表示色です。
    //! @param[in] bottom   下端の表示色です。
    //!
    //! @sa GetTextColor
    //!
    void SetTextColor(
        ut::Color8 top,
        ut::Color8 bottom
    )
    {
        if (UpdatePTDirty(m_TextColors[TEXTCOLOR_TOP] != top || m_TextColors[TEXTCOLOR_BOTTOM] != bottom))
        {
            m_TextColors[TEXTCOLOR_TOP   ] = top;
            m_TextColors[TEXTCOLOR_BOTTOM] = bottom;
        }
    }

    //! @brief テキストのフォントを取得します。
    //!
    //! @return フォントです。
    //!
    //! @sa SetFont
    //!
    //! @date 2010/01/26 フォントは破棄しない仕様に変更しました。
    const font::Font* GetFont() const;

    //! @brief テキストのフォントを設定します。
    //!
    //! @details
    //! 同時に、フォントサイズを指定されたフォントオブジェクトのサイズに設定します。
    //!
    //! NULL を渡した場合はフォントの設定を解除します。
    //!
    //! @param[in] pFont    フォントオブジェクトへのポインタです。
    //!
    //! @sa GetFont
    //!
    //! @date 2010/01/26 フォントは破棄しない仕様に変更しました。
    void SetFont(const font::Font* pFont);

    //! @brief テキストのフォントサイズを取得します。
    //!
    //! @return テキストのフォントサイズを返します。
    //!
    //! @sa SetFontSize
    //!
    const Size& GetFontSize() const
    {
        return m_FontSize;
    }

    //! @brief テキストのフォントサイズを設定します。
    //!
    //! @param[in] fontSize フォントサイズです。
    //!
    //! @sa GetFontSize
    //!
    void SetFontSize(const Size& fontSize)
    {
        if (UpdatePTDirty(!(m_FontSize == fontSize)))
        {
            m_FontSize = fontSize;
        }
    }

    //! @brief テキストの行間隔を取得します。
    //!
    //! @return テキストの行間隔を返します。
    //!
    //! @sa SetLineSpace
    //!
    f32 GetLineSpace() const
    {
        return m_LineSpace;
    }

    //! @brief テキストの行間隔を設定します。
    //!
    //! @param[in] space    行間です。
    //!
    //! @sa GetLineSpace
    //!
    void SetLineSpace(f32 space)
    {
        if (UpdatePTDirty(m_LineSpace != space))
        {
            m_LineSpace = space;
        }
    }

    //! @brief テキストの文字間隔を取得します。
    //!
    //! @return テキストの文字間隔を返します。
    //!
    //! @sa SetCharSpace
    //!
    f32 GetCharSpace() const
    {
        return m_CharSpace;
    }

    //! @brief テキストの文字間隔を設定します。
    //!
    //! @param[in] space    文字間隔です。
    //!
    //! @sa GetCharSpace
    //!
    void SetCharSpace(f32 space)
    {
        if (UpdatePTDirty(m_CharSpace != space))
        {
            m_CharSpace = space;
        }
    }

    //! @brief テキスト表示基準位置設定の水平位置指定を取得します。
    //!
    //! @return テキスト表示基準位置設定の水平位置指定を返します。
    //!
    //! @sa SetTextPositionH
    //! @sa GetTextPositionV
    //!
    HorizontalPosition GetTextPositionH() const
    {
        return internal::GetHorizontalPosition(m_TextPosition);
    }

    //! @brief テキスト表示基準位置設定の水平位置指定を設定します。
    //!
    //! @param[in] val  テキスト表示基準位置指定です。
    //!
    //! @sa GetTextPositionH
    //! @sa SetTextPositionV
    //!
    void SetTextPositionH(HorizontalPosition val)
    {
        if (UpdatePTDirty(GetTextPositionH() != val))
        {
            internal::SetHorizontalPosition(&m_TextPosition, u8(val));
        }
    }

    //! @brief テキスト表示基準位置設定の垂直位置指定を取得します。
    //!
    //! @return テキスト表示基準位置設定の垂直位置指定を返します。
    //!
    //! @sa SetTextPositionV
    //! @sa GetTextPositionH
    //!
    VerticalPosition GetTextPositionV() const
    {
        return internal::GetVerticalPosition(m_TextPosition);
    }

    //! @brief テキスト表示基準位置設定の垂直位置指定を設定します。
    //!
    //! @param[in] val  テキスト表示基準位置指定です。
    //!
    //! @sa GetTextPositionV
    //! @sa SetTextPositionH
    //!
    void SetTextPositionV(VerticalPosition val)
    {
        if (UpdatePTDirty(GetTextPositionV() != val))
        {
            internal::SetVerticalPosition(&m_TextPosition, u8(val));
        }
    }

    //! @brief 行そろえ指定を取得します。
    //!
    //! @return 行そろえ指定を返します。
    //!
    //! @sa SetTextAlignment
    //!
    TextAlignment GetTextAlignment() const
    {
        return static_cast<TextAlignment>(m_Bits.textAlignment);
    }

    //! @brief 行そろえ指定を設定します。
    //!
    //! @param[in] val  行そろえの指定です。
    //!
    //! @sa GetTextAlignment
    //!
    void SetTextAlignment(TextAlignment val)
    {
        if (UpdatePTDirty(GetTextAlignment() != val))
        {
            m_Bits.textAlignment = val;
        }
    }

    //! @brief タグプロセッサを取得します。
    //!
    //! @return
    //! タグプロセッサを返します。
    //! タグプロセッサが設定されていない場合は NULL を返します。
    //!
    //! @sa SetTagProcessor
    //!
    TagProcessor* GetTagProcessor() const
    {
        return m_pTagProcessor;
    }

    //! @brief タグプロセッサを設定します。
    //!
    //! @details
    //! NULL を渡した場合はタグプロセッサの設定を解除します。
    //!
    //! @param[in] pTagProcessor    タグプロセッサへのポインタです。
    //!
    //! @sa GetTagProcessor
    //!
    void SetTagProcessor(TagProcessor* pTagProcessor)
    {
        if (UpdatePTDirty(m_pTagProcessor != pTagProcessor))
        {
            m_pTagProcessor = pTagProcessor;
        }
    }

    //! @brief テキストの表示色（頂点カラー）を取得します。
    //!
    //! @details
    //! idx が lyt::VERTEXCOLOR_LT のときは上端の色、
    //! idx が lyt::VERTEXCOLOR_LB のときは下端の色を返します。
    //!
    //! idx が lyt::VERTEXCOLOR_RT または lyt::VERTEXCOLOR_RB のときは
    //! 対応する頂点カラーを返しますが表示には使用されていません。
    //!
    //! @param[in] idx  インデックスです。 lyt::VertexColor を指定します。
    //!
    //! @return テキストの表示色を返します。
    //!
    //! @sa SetVtxColor
    //! @sa GetVtxColorElement
    //! @sa lyt::VertexColor
    //!
    virtual const ut::Color8 GetVtxColor(u32 idx) const;

    //! @brief テキストの表示色（頂点カラー）を設定します。
    //!
    //! @details
    //! idx が lyt::VERTEXCOLOR_LT のときは上端の色を、
    //! idx が lyt::VERTEXCOLOR_LB のときは下端の色を変更します。
    //!
    //! idx が lyt::VERTEXCOLOR_RT、 lyt::VERTEXCOLOR_RB のときは
    //! 対応する頂点カラーを変更しますが表示には使用されていません。
    //!
    //! @param[in] idx      インデックスです。 lyt::VertexColor を指定します。
    //! @param[in] value    テキストの表示色です。
    //!
    //! @sa GetVtxColor
    //! @sa SetVtxColorElement
    //! @sa lyt::VertexColor
    //!
    virtual void SetVtxColor(u32 idx, ut::Color8 value);

    //! @brief テキストの表示色（頂点カラー）の成分を取得します。
    //!
    //! @details
    //! idx が lyt::ANIMTARGET_VERTEXCOLOR_LT_* のときは上端の色、
    //! idx が lyt::ANIMTARGET_VERTEXCOLOR_LB_* のときは下端の色の各成分を返します。
    //!
    //! idx が lyt::ANIMTARGET_VERTEXCOLOR_RT_*、 lyt::ANIMTARGET_VERTEXCOLOR_RB_* のときは
    //! 対応する頂点カラーの各成分を返しますが表示には使用されていません。
    //!
    //! このメソッドは、R,G,B など色成分を個別に取得します。色成分をまとめて取得する場合は GetVtxColor を利用してください。
    //!
    //! @param[in] idx  インデックスです。 lyt::AnimTargetPaneColor を指定します。
    //!
    //! @return テキストの表示色の成分を返します。
    //!
    //! @sa SetVtxColorElement
    //! @sa GetVtxColor
    //! @sa lyt::AnimTargetPaneColor
    //!
    virtual u8 GetVtxColorElement(u32 idx) const;

    //! @brief テキストの表示色（頂点カラー）の成分を設定します。
    //!
    //! @details
    //! idx が lyt::ANIMTARGET_VERTEXCOLOR_LT_* のときは上端の色、
    //! idx が lyt::ANIMTARGET_VERTEXCOLOR_LB_* のときは下端の色の各成分を変更します。
    //!
    //! idx が lyt::ANIMTARGET_VERTEXCOLOR_RT_*、 lyt::ANIMTARGET_VERTEXCOLOR_RB_* のときは
    //! 対応する頂点カラーの各成分を変更しますが表示には使用されていません。
    //!
    //! このメソッドは、R,G,B など色成分を個別に設定します。色成分をまとめて設定する場合は SetVtxColor を利用してください。
    //!
    //! @param[in] idx      インデックスです。 lyt::AnimTargetPaneColor を指定します。
    //! @param[in] value    カラーの成分値です。
    //!
    //! @sa GetVtxColorElement
    //! @sa SetVtxColor
    //! @sa lyt::AnimTargetPaneColor
    //!
    virtual void SetVtxColorElement(u32 idx, u8 value);

    //! @brief テキストが描画される矩形を取得します。
    //!
    //! @return 矩形を返します。
    //!
    const ut::Rect GetTextDrawRect() const;

    using Base::GetMaterial;

    //! @brief ペインが持つマテリアルの数を取得します。
    //!
    //! @return マテリアルの数を返します。
    //!
    virtual u8 GetMaterialNum() const;

    //! @brief ペインのマテリアルを取得します。
    //!
    //! @details
    //! idx には 0 のみ指定できます。
    //!
    //! @param[in] idx  インデックスです。
    //!
    //! @return マテリアルへのポインタを返します。
    //!
    //! @sa SetMaterial
    //! @sa UnsetMaterial
    //!
    virtual Material* GetMaterial(u32 idx) const;

    //! @brief マテリアルを設定します。
    //!
    //! @details
    //! 現在設定されているマテリアルが TextBox オブジェクトの
    //! 生成時に同時に生成されたものだった場合には、
    //! そのマテリアルは破棄されます。
    //!
    //! @param[in] pMaterial    マテリアルへのポインタです。
    //!
    //! @sa GetMaterial
    //! @sa UnsetMaterial
    //!
    void SetMaterial(Material* pMaterial);

    //! @brief マテリアルの設定を解除します。
    //!
    //! @details
    //! マテリアルを破棄することなく、マテリアルの設定を解除します。
    //! TextBoxの解放処理の直前で呼ぶことで、
    //! デストラクタ内でマテリアルが解放されることを防ぎます。
    //!
    //! @sa GetMaterial
    //! @sa SetMaterial
    //!
    void UnsetMaterial();

    //! @brief テキストIDを取得します。
    //!
    //! @details
    //! リソースでテキストIDが指定されていた場合は、構築後に
    //! このメソッドでテキストIDを取得することができます。
    //!
    //! テキストIDが指定されていなかった場合はNULLが返されます。
    //!
    //! @return テキストID。null終端されています。
    //!
    const char* GetTextID() const { return m_TextID; }

    //! @brief テキストボックスの幅で折り返す処理が有効か否かを取得します。
    //!
    //! @return テキストボックスの幅で折り返す処理が有効か否か
    //!
    bool IsWidthLimitEnable() const { return m_Bits.widthLimitEnabled; }

    //! @brief テキストボックスの幅で折り返す処理が有効か否かを設定します。
    //!
    //! @details
    //! デフォルトでは有効になっています。
    //!
    //! @param[in] enable   テキストボックスの幅で折り返す処理を有効にするか否か
    //!
    void SetWidthLimitEnable(bool enable) { m_Bits.widthLimitEnabled = enable; }

    //! @brief 斜体の度合いを取得します。
    //!
    //! @return 斜体の度合いを返します。
    //!
    //! @sa SetItalicRatio
    //!
    f32 GetItalicRatio() const
    {
        return m_ItalicRatio;
    }

    //! @brief 斜体の度合いを設定します。
    //!
    //! @param[in] ratio    斜体の度合いです。
    //!
    //! @sa GetItalicRatio
    //!
    void SetItalicRatio(f32 ratio)
    {
        if (UpdatePTDirty(m_ItalicRatio != ratio))
        {
            m_ItalicRatio = ratio;
        }
    }

    //! @brief 影のオフセットを取得します。
    //!
    //! @return 影のオフセットを返します。
    //!
    //! @sa SetShadowOffset
    //!
    const math::VEC2& GetShadowOffset() const
    {
        return m_ShadowOffset;
    }

    //! @brief 影のオフセットを設定します。
    //!
    //! @param[in] offset    影のオフセットです。
    //!
    //! @sa GetShadowOffset
    //!
    void SetShadowOffset(const math::VEC2& offset)
    {
        if (UpdatePTDirty(m_ShadowOffset != offset))
        {
            m_ShadowOffset = offset;
        }
    }

    //! @brief 影のスケールを取得します。
    //!
    //! @return 影のスケールを返します。
    //!
    //! @sa SetShadowScale
    //!
    const math::VEC2& GetShadowScale() const
    {
        return m_ShadowScale;
    }

    //! @brief 影のスケールを設定します。
    //!
    //! @param[in] scale    影のスケールです。
    //!
    //! @sa GetShadowScale
    //!
    void SetShadowScale(const math::VEC2& scale)
    {
        if (UpdatePTDirty(m_ShadowScale != scale))
        {
            m_ShadowScale = scale;
        }
    }

    //! @brief 影の上端の表示色を取得します。
    //!
    //! @return 影の上端の表示色を返します。
    //!
    //! @sa SetShadowTopColor
    //!
    const ut::Color8 GetShadowTopColor() const
    {
        return m_ShadowTopColor;
    }

    //! @brief 影の上端の表示色を設定します。
    //!
    //! @param[in] color    影の上端の表示色です。
    //!
    //! @sa GetShadowTopColor
    //!
    void SetShadowTopColor(ut::Color8 color)
    {
        if (UpdatePTDirty(m_ShadowTopColor != color))
        {
            m_ShadowTopColor = color;
        }
    }

    //! @brief 影の下端の表示色を取得します。
    //!
    //! @return 影の下端の表示色を返します。
    //!
    //! @sa SetShadowBottomColor
    //!
    const ut::Color8 GetShadowBottomColor() const
    {
        return m_ShadowBottomColor;
    }

    //! @brief 影の下端の表示色を設定します。
    //!
    //! @param[in] color    影の下端の表示色です。
    //!
    //! @sa GetShadowBottomColor
    //!
    void SetShadowBottomColor(ut::Color8 color)
    {
        if (UpdatePTDirty(m_ShadowBottomColor != color))
        {
            m_ShadowBottomColor = color;
        }
    }

    //! @brief 影の表示色を設定します。
    //!
    //! @param[in] top       影の上端の表示色です。
    //! @param[in] bottom    影の下端の表示色です。
    //!
    void SetShadowColor(ut::Color8 top, ut::Color8 bottom)
    {
        if (UpdatePTDirty(m_ShadowTopColor != top || m_ShadowBottomColor != bottom))
        {
            m_ShadowTopColor = top;
            m_ShadowBottomColor = bottom;
        }
    }

    //! @brief 影の斜体の度合いを取得します。
    //!
    //! @return 影の斜体の度合いを返します。
    //!
    //! @sa SetShadowItalicRatio
    //!
    f32 GetShadowItalicRatio() const
    {
        return m_ItalicRatio;
    }

    //! @brief 影の斜体の度合いを設定します。
    //!
    //! @param[in] ratio    影の斜体の度合いです。
    //!
    //! @sa GetShadowItalicRatio
    //!
    void SetShadowItalicRatio(f32 ratio)
    {
        if (UpdatePTDirty(m_ShadowItalicRatio != ratio))
        {
            m_ShadowItalicRatio = ratio;
        }
    }

    //! @brief 文字属性バッファを共有する設定か否か
    //!
    //! @details
    //! まだ AllocStringBuffer() が呼び出されていない場合はfalseを返します。
    //!
    //! @return 文字属性バッファを共有する設定か否かを返します。
    //!
    bool IsShareCharAttributeBuffer() const;

    //! @brief ダーティフラグを設定します。
    //!
    //! @details
    //! ペインのサイズ変更など、テキストを再配置する場合などに
    //! isDirty に true を設定してください。
    //!
    //! isDirty 引数に true を設定することで、
    //! 描画前にテキスト用頂点バッファのキャッシュを作成し直します。
    //!
    //! @param[in] isDirty  true を設定すると頂点バッファのキャッシュを作成し直します。
    //!
    void SetDirtyFlag(bool isDirty)
    {
        UpdatePTDirty(isDirty);
    }

    //! @brief ダーティフラグを false にします。
    //!
    //! @details
    //! ダーティフラグを false にすることで、
    //! CalculateMtx() において頂点バッファのキャッシュを作成し直さなくなります。
    void ClearDirtyFlag() { m_Bits.isPTDirty = 0; }

    //! @brief ダーティフラグを取得します。
    //!
    //! @details
    //! CalculateMtx() において頂点バッファのキャッシュを作成し直す場合 true が返ります。
    //!
    //! @return ダーティフラグです。
    //!
    bool GetDirtyFlag() const { return m_Bits.isPTDirty; }

    //! @brief TextWriter の設定を行います。
    //!
    //! @param[out] pWriter  文字列の描画に必要な設定が行われます。
    //!
    virtual void SetupTextWriter(font::WideTextWriter* pWriter);

    //! @brief 文字単位アニメーションのオフセットを設定します。
    //!
    //! @param[in] offset オフセットを指定します。
    //!
    void SetPerCharacterTransformOffset(f32 offset)
    {
        NW_ASSERT_NOT_NULL(m_pPerCharacterTransform);
        m_pPerCharacterTransform->Offset = offset;
        UpdatePTDirty(true);
    }

    //@}

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @return 表示文字列バッファです。
    //!
    font::DispStringBuffer* GetDispStringBuffer() const
    {
        return m_pDispStringBuf;
    }

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[out] pMtx    行列です。
    //!
    void GetTextGlobalMtx(nw::math::MTX34* pMtx) const;

    virtual void CalculateMtx(Pane::CalculateMtxContext& drawInfo, bool isDirtyParentMtx);

protected:
    virtual void DrawSelf(DrawInfo& drawInfo);

    virtual void LoadMtx(DrawInfo& drawInfo);

    void DrawSelfBorder(font::RectDrawer& drawer, const font::DrawContent& drawContent);

    virtual bool InitStringWithTextSearcherInfo(const BuildArgSet& buildArgSet, const TextSearcher::TextInfo& textInfo);

    void InitPerCharacterTransformCurves(const res::AnimationInfo* animInfo);

private:
    void SetFontInfo(font::WideTextWriter* pWriter) const;
    void SetTextPos(font::WideTextWriter* pWriter) const;

    math::VEC2  AdjustTextPos(
                    const Size& size,
                    bool        isCeil
                ) const;

    void Init();
    void InitMaterial();

    u16 SetStringImpl(
        const char16*  str,
        u16             dstIdx,
        u32             strLen);

    //! @brief 文字の位置・テクスチャが変更になったかどうかの状態を更新します。
    bool UpdatePTDirty(bool isChanged)
    {
        m_Bits.isPTDirty |= isChanged ? 1: 0;
        return isChanged;
    }

    // コピー演算子を禁止します。
    const TextBox& operator=( const TextBox& );

    f32 GetLoopFrame(f32 frame, f32 begin, f32 end) const
    {
        if (frame > end)
            return ::std::fmod(frame - begin, end - begin) + begin;
        else if (frame < begin)
            return end - ::std::fmod(begin - frame, end - begin);
        return frame;
    }

    void UpdatePerCharacterTransform();

private:
    char16* m_TextBuf;
    const char* m_TextID;
    ut::Color8 m_TextColors[TEXTCOLOR_MAX];
    const font::Font* m_pFont;
    Size m_FontSize;
    f32 m_LineSpace;
    f32 m_CharSpace;

    TagProcessor* m_pTagProcessor;

    u16 m_TextBufLen;
    u16 m_TextLen;

    //! @brief 内部用機能のため使用禁止です。
    struct Bits
    {
        u8 textAlignment: 2;
        u8 isPTDirty    : 1;    // 位置・テクスチャが変更になったら真
        bool shadowEnabled: 1;
        bool invisibleBorderEnabled: 1;
        bool doubleDrawnBorderEnabled: 1;
        bool widthLimitEnabled: 1;
        bool perCharacterTransformEnabled: 1;
    };

    Bits m_Bits;
    u8 m_TextPosition;

    f32 m_ItalicRatio;

    math::VEC2 m_ShadowOffset;
    math::VEC2 m_ShadowScale;
    ut::Color4u8 m_ShadowTopColor;
    ut::Color4u8 m_ShadowBottomColor;
    f32 m_ShadowItalicRatio;

    Material* m_pMaterial;
    font::DispStringBuffer* m_pDispStringBuf;

    struct CurveInfo {
        const nw::lyt::res::AnimationTarget* Curve;
        AnimTargetPerCharacterTransformCurve CurveType;
    };

    struct PerCharacterTransform
    {
        f32 Offset;
        f32 Width;
        nw::font::PerCharacterTransformInfo* pPerCharacterTransformInfos;
        u8 LoopType;
        u8 OriginV;
        u8 CurveNumber;
        u8 Dummy;
        CurveInfo CurveInfos[1];    // この配列は、CurveNumberの長さがあるものとしてメモリ確保されます。
    };
    PerCharacterTransform* m_pPerCharacterTransform;
};

} // namespace nw::lyt
} // namespace nw

#endif // NW_LYT_TEXTBOX_H_

